(* DELIVER.PAS - General MAIL delivery manager for VMS MAIL.

   Written by Ned Freed, 23-Sep-1985, modified 30-Apr-1986.
   Mail dispatch interface originally written by Kevin Carosso.
   Some modifications by Sheldon Smith, December 1986.
   Rewrite by Ned Freed to use new $GETUAI system service, 15-Dec-1986.
   This change requires the use of VMS 4.4 and Pascal 3.4.

   DELIVER provides a general-purpose MAIL delivery manager similar
   to the MMDF-II MAILDELIVERY system. DELIVER makes it possible
   for users to set up a file containing screening information that
   automatically deals with each incoming message. Actions may be
   selectively taken by DELIVER based on information in the From:,
   To: and Subject: fields of the message header.

   DELIVER operates as a foreign mail interface to VMS MAIL and is
   invoked with a call to LIB$FIND_IMAGE_SYMBOL in MAIL. The shareable
   image containing this code should be placed in SYS$LIBRARY as the file
   DELIVER_MAILSHR.EXE).

   Users may activate DELIVER by setting their forwarding address
   to DELIVER%username, where "username" is the user's own user name.
   In order for DELIVER to perform any useful function a file called
   MAIL.DELIVERY must be present in the user's default login directory.

   DELIVER's operation is only meaningful in outgoing mode; however, rather
   than waste a possible incoming MAIL interface, DELIVER implements a
   rudimentary mail posting mechanism on the incoming side that can be used
   to send messages contained in data files. Full privileges are required
   to run DELIVER in this mode since it is possible to forge From:
   addresses using this mechanism.

  Note:
    The foreign protocol interface within MAIL is undocumented at
    this time.  It may change without notice in a future release
    of VMS.

    The information necessary to write this code comes from the MAIL
    source on the VMS microfiche.  The most useful information is the
    routine NETJOB in module MAIL$MAIL (230-E2), which handles incoming
    foreign mail, and the various routines in module NETSUBS (230-N11),
    most of which deal with outgoing foreign mail.
*)

[inherit ('SYS$LIBRARY:STARLET')] module deliver (output);

const
  (* Debugging control flags *)
  DEBUG_IN  = false;            (* Debug messages produced by receive code. *)
  DEBUG_OUT = false;            (* Debug messages produced by send code.    *)

  LNK_C_OUT_CONNECT  = 0;       (* MAIL protocol link actions.           *)
  LNK_C_OUT_SENDER   = 1;       (* These are defined in MAILSHR.MAR      *)
  LNK_C_OUT_CKUSER   = 2;       (* but because we cannot have external   *)
  LNK_C_OUT_TO       = 3;       (* constants in Pascal, they are         *)
  LNK_C_OUT_SUBJ     = 4;       (* redefined here.                       *)
  LNK_C_OUT_FILE     = 5;
  LNK_C_OUT_CKSEND   = 6;
  LNK_C_OUT_DEACCESS = 7;

  LNK_C_IN_CONNECT = 8;
  LNK_C_IN_SENDER  = 9;
  LNK_C_IN_CKUSER  = 10;
  LNK_C_IN_TO      = 11;
  LNK_C_IN_SUBJ    = 12;
  LNK_C_IN_FILE    = 13;

  LNK_C_IO_READ  = 14;
  LNK_C_IO_WRITE = 15;

  parameter_size     = 256;     (* Size of a single parameter in the
                                   MAIL.DELIVERY file. This is also the
                                   maximum size of lines read from any file. *)
  max_parameters     = 7;       (* Maximum number of parameters that
                                   can appear on a line in a
                                   MAIL.DELIVERY file *)
  min_parameters     = 5;       (* Minimum number of parameters that
                                   can appear on a line in a
                                   MAIL.DELIVERY file *)
  from_parameter     = 1;       (* Position of the From: parameter *)
  to_parameter       = 2;       (* Position of the To: parameter *)
  subject_parameter  = 3;       (* Position of the Subject: parameter *)
  decision_parameter = 4;       (* Position of the decision parameter *)
  action_parameter   = 5;       (* Position of the action parameter *)
  argument_parameter = 6;       (* Position of the argument parameter *)

  stack_size = 10;              (* State mach. stack for messages from MAIL *)

  DCL_line_size = 256;          (* Maximum possible line allowed by DCL. *)

type
  string = varying [parameter_size] of char;

  (* A string descriptor type used to handle the descriptors MAIL hands to
     DELIVER. *)
  longest_possible_string = packed array [1..65536] of char;
  string_descriptor = record
                        length : [word] 0..65535;
                        dclass, dtype : [byte] 0..255;
                        address : ^longest_possible_string;
                      end;

  (* Storage for a single line of MAIL.DELIVERY file information *)
  parameter_block_ptr = ^parameter_block;
  parameter_block = record
                      parameters  : array [1..max_parameters] of string;
                      next        : parameter_block_ptr;
                      any_from    : boolean;
                      any_to      : boolean;
                      any_subject : boolean;
                    end; (* parameter_block record *)

  account_name = packed array [1..8] of char;

  (* Storage for information about a single recipient *)
  user_block_ptr = ^user_block;
  user_block = record
                 username    : varying [12] of char;
                 account     : account_name;
                 directory   : string;
                 uic         : unsigned;
                 rules_list  : parameter_block_ptr;
                 next        : user_block_ptr;
                 copyname    : varying [29] of char;
               end; (* user_block record *)

  (* Possible reasons why MAIL_IO_WRITE will be called. *)
  write_states = (bad_msg, user_check, delivery_check, error_msg);

  (* A stack structure to store information about pending calls to
     MAIL_IO_WRITE. *)
  write_state_stack = record
                        top : integer;
                        store : array [1..stack_size] of write_states;
                      end; (* write_state_check record *)

  (* Record for VMS item lists. *)
  item = record
           len    : [word] 0..65535;
           code   : [word] 0..65535;
           addr   : [long] integer;
           rlen   : [long] integer;
         end; (* item record *)

var
  batch_queue : [readonly] string := 'MAIL$BATCH';
  system_batch_queue : [readonly] string := 'SYS$BATCH';
  batch_log   : [readonly] string := '_NLA0:';

  (* Storage for message header information on the outgoing side *)
  tostring, fromstring, subjectstring : [static] string;

  (* List of active recipients and associated information *)
  user_list, user_list_last : [static] user_block_ptr;
  user_count : [static] integer := 0;

  from_owner : [static] boolean;

  (* Storage for accumulated To: line for incoming messages. *)

  toline : [static] string;

  (* The state machine for MAIL status information. *)

  write_recv_states  : write_state_stack;
  last_error : integer;

  (* Error message codes defined in DELIVER_ERR.MSG *)
  DELIVER__CANACCUAF, DELIVER__NOSUCHUSER, DELIVER__NAMETOOLONG,
  DELIVER__NODEFAULTDIR, DELIVER__TOOMANYPARAMS, DELIVER__TOOFEWPARAMS,
  DELIVER__NOMDFILE, DELIVER__MDIGNORED, DELIVER__NORULES, DELIVER__MESREAERR,
  DELIVER__GETFILERR, DELIVER__MESWRTERR, DELIVER__INTSTKOVR, DELIVER__STKEMPTY,
  DELIVER__BADSTKELE, DELIVER__MESOPNERR, DELIVER__MSGWRTERR,
  DELIVER__MSGREAERR, DELIVER__USERNOEXIST : [external, value] integer;

(* Routine to signal errors *)

procedure LIB$SIGNAL (%IMMED stat : [list, unsafe] integer); extern;

(* Routine to read command line arguments *)

function CLI$GET_VALUE (name : varying [max1] of char;
  var val : varying [max2] of char) : integer; extern;

(* Routine to get symbol values *)

function LIB$GET_SYMBOL (name : varying [max1] of char;
  var result : varying [max2] of char) : integer; extern;

(* Routine to set symbol values *)

function LIB$SET_SYMBOL (name : varying [max1] of char;
  svalue : varying [max2] of char) : integer; extern;

(* create_with_SYSPRV is a Pascal user-action routine for OPEN statements.
   It enables SYSPRV while doing certain OPEN's so we can write files into
   user directories. *)

function create_with_SYSPRV (var fab : FAB$TYPE;
                             var rab : RAB$TYPE;
                             var fil : text) : integer;
var
  stat : integer; ppriv, priv : [quad] array [0..1] of unsigned;

begin (* create_with_SYSPRV *)
  priv[0] := PRV$M_SYSPRV; priv[1] := 0;
  stat := $SETPRV (ENBFLG := 1, PRVADR := priv, PRMFLG := 0, PRVPRV := ppriv);
  if odd (stat) then begin
    FAB.FAB$V_LNM_MODE := PSL$C_EXEC; stat := $CREATE (FAB);
    if odd (stat) then stat := $CONNECT (RAB);
  end;
  priv[0] := uand (priv[0], unot (ppriv[0]));
  priv[1] := uand (priv[1], unot (ppriv[1]));
  $SETPRV (ENBFLG := 0, PRVADR := priv, PRMFLG := 0);
  create_with_SYSPRV := stat;
end; (* create_with_SYSPRV *)

(* open_with_SYSPRV is a Pascal user-action routine for OPEN statements.
   It enables SYSPRV while doing certain OPEN's so we can read system
   files. *)

function open_with_SYSPRV (var fab : FAB$TYPE;
                           var rab : RAB$TYPE;
                           var fil : text) : integer;
var
  stat : integer; ppriv, priv : [quad] array [0..1] of unsigned;

begin (* open_with_SYSPRV *)
  priv[0] := PRV$M_SYSPRV; priv[1] := 0;
  stat := $SETPRV (ENBFLG := 1, PRVADR := priv, PRMFLG := 0, PRVPRV := ppriv);
  if odd (stat) then begin
    FAB.FAB$V_LNM_MODE := PSL$C_EXEC; stat := $OPEN (FAB);
    if odd (stat) then stat := $CONNECT (RAB);
  end;
  priv[0] := uand (priv[0], unot (ppriv[0]));
  priv[1] := uand (priv[1], unot (ppriv[1]));
  $SETPRV (ENBFLG := 0, PRVADR := priv, PRMFLG := 0);
  open_with_SYSPRV := stat;
end; (* open_with_SYSPRV *)

(* find_user_directory locates a user in the system authorization file
   and returns his or her default login directory (which is where a
   MAIL.DELIVERY file must reside). find_user_directory also returns
   the user's UIC and account since this information will be needed
   for creating the delivery batch job. *)

function find_user_directory (username : varying [l1] of char;
  var user_directory : string; var user_uic : unsigned;
  var user_account : account_name) : boolean;

var
  device_with_prefix : packed array [1..16] of char;
  directory_with_prefix : packed array [1..64] of char;
  items : array [1..5] of item; stat : integer;
  ppriv, priv : [quad] array [0..1] of unsigned;

begin (* find_user_directory *)
  if DEBUG_OUT then writeln ('find_user_directory called.');
  find_user_directory := false;
  if length (username) > 12 then
    LIB$SIGNAL (DELIVER__NAMETOOLONG, 2,
                username.length, iaddress (username.body))
  else begin
    with items[1] do begin
      len := 8; code := UAI$_ACCOUNT;
      addr := iaddress (user_account); rlen := 0;
    end; (* with *)
    with items[2] do begin
      len := 16; code := UAI$_DEFDEV;
      addr := iaddress (device_with_prefix); rlen := 0;
    end; (* with *)
    with items[3] do begin
      len := 64; code := UAI$_DEFDIR;
      addr := iaddress (directory_with_prefix); rlen := 0;
    end; (* with *)
    with items[4] do begin
      len := 4; code := UAI$_UIC; addr := iaddress (user_uic); rlen := 0;
    end; (* with *)
    with items[5] do begin
      len := 0; code := 0; addr := 0; rlen := 0;
    end; (* with *)
    (* Enable SYSPRV to check for valid user recipient-name. *)
    priv[0] := PRV$M_SYSPRV; priv[1] := 0;
    $SETPRV (ENBFLG := 1, PRVADR := priv, PRMFLG := 0, PRVPRV := ppriv);
    stat := $GETUAI (usrnam := username, itmlst := items);
    if stat = SS$_NOPRIV then LIB$SIGNAL (DELIVER__CANACCUAF)
    else if not odd (stat) then
      LIB$SIGNAL (DELIVER__NOSUCHUSER, 2,
                  username.length, iaddress (username.body))
    else begin
      user_directory := substr (device_with_prefix, 2,
                                ord (device_with_prefix[1])) +
                        substr (directory_with_prefix, 2,
                                ord (directory_with_prefix[1]));
      if DEBUG_OUT then writeln ('  Default directory: "',
                                 user_directory, '".');
      if DEBUG_OUT then writeln ('  Account: "', user_account, '".');
      if DEBUG_OUT then writeln ('  UIC: ', hex (user_uic, 8, 8), '.');
      if length (user_directory) <= 0 then
        LIB$SIGNAL (DELIVER__NODEFAULTDIR, 2,
                    username.length, iaddress (username.body))
      else find_user_directory := true;
      (* Disable and reestablish former privs. *)
      priv[0] := uand (priv[0], unot (ppriv[0]));
      priv[1] := uand (priv[1], unot (ppriv[1]));
      $SETPRV (ENBFLG := 0, PRVADR := priv, PRMFLG := 0);
    end;
  end;
end; (* find_user_directory *)

(* copy_descr_to_string copies a MAIL string (passed by descriptor) into
   a Pascal VARYING string. *)

procedure copy_descr_to_string (
  var mail_string : string_descriptor;
  var out_string : string; DEBUG_ON : boolean);

var
  index : integer;

begin (* copy_descr_to_string *)
  if DEBUG_ON then writeln ('copy_descr_to_string called.');
  out_string := '';
  if mail_string.length > 256 then index := 256
  else index := mail_string.length;
  for index := 1 to index do
    out_string := out_string + mail_string.address^[index];
  if DEBUG_ON then writeln ('  String copied: "', out_string, '".');
end; (* copy_descr_to_string *)

(* copy_string_to_descr copies a Pascal VARYING string into a MAIL string
   (passed by descriptor). *)

procedure copy_string_to_descr (
  var in_string : string;
  var mail_string : string_descriptor; DEBUG_ON : boolean);

  [asynchronous, unbound, external (LIB$SCOPY_DXDX)]
  function copy_string (var src : varying [max1] of char;
    var dst : string_descriptor) : integer; extern;

begin (* copy_string_to_descr *)
  if DEBUG_ON then writeln ('copy_string_to_descr called.');
  copy_string (in_string, mail_string);
  if DEBUG_ON then writeln ('  String copied: "', in_string, '".');
end; (* copy_string_to_descr *)

(* charupper is a simple function to convert characters to upper case.
   The full DEC Multinational Character Set is accomodated. *)

function charupper (ch : char) : char;

begin (* charupper *)
  if (ch >= 'a') and (ch <= 'z') then
    charupper := chr (ord (ch) + (ord ('A') - ord ('a')))
  else if (ord (ch) >= 224) and (ord (ch) <= 253) then
    charupper := chr (ord (ch) + (192 - 224))
  else charupper := ch;
end; (* charupper *)

(* dispose_rules_list disposes of heap storage associated with
   a list of parameter blocks. *)

procedure dispose_rules_list (var rules_list : parameter_block_ptr);

var
  temp_list : parameter_block_ptr;

begin (* dispose_rules_list *)
  while rules_list <> nil do begin
    temp_list := rules_list; rules_list := rules_list^.next;
    dispose (temp_list);
  end; (* while rules_list <> nil *)
end; (* dispose_rules_list *)

(* dispose_user_list disposes of heap storage associated with
   a list of user name blocks. *)

procedure dispose_user_list (var user_list : user_block_ptr);

var
  temp_list : user_block_ptr;

begin (* dispose_user_list *)
  while user_list <> nil do begin
    temp_list := user_list; user_list := user_list^.next;
    dispose_rules_list (temp_list^.rules_list); dispose (temp_list);
  end; (* while user_list <> nil *)
end; (* dispose_user_list *)

(* read_maildelivery_file reads the contents of a MAIL.DELIVERY file
   and creates a rules_list structure. Any errors are signalled
   by returning FALSE. *)

function read_maildelivery_file (var dfile : text;
  var rules_list : parameter_block_ptr) : boolean;

label
  99;

var
  current, last : parameter_block_ptr; quoted : boolean;
  pindex, lindex, rindex, lcount : integer; line : string;

  procedure addch (ch : char);

  label
    88;

  var
    cindex : integer;

  begin (* addch *)
    if pindex > max_parameters then begin
      if FROM_OWNER then LIB$SIGNAL (DELIVER__TOOMANYPARAMS, 1, lcount);
      goto 99;
    end;
    if current = nil then if (ch = '!') or (ch = ';') then begin
      if DEBUG_OUT then writeln ('  Skipping comment line.');
      lindex := length (line); goto 88;
    end else begin
      new (current);
      with current^ do begin
        for cindex := 1 to max_parameters do parameters[cindex] := '';
        next := nil;
      end; (* with current^ *)
      rindex := rindex + 1;
      if DEBUG_OUT then writeln (' Rule #', rindex:0, '.');
      if last = nil then begin
        last := current; rules_list := current;
      end else begin
        last^.next := current; last := current;
      end;
    end;
    current^.parameters[pindex] := current^.parameters[pindex] + ch;
  88:
  end; (* addch *)

begin (* read_maildelivery_file *)
  if DEBUG_OUT then writeln ('read_maildelivery_file called.');
  read_maildelivery_file := false; last := nil; lcount := 0;
  rindex := 0;
  while not eof (dfile) do begin
    readln (dfile, line); lcount := lcount + 1;
    if DEBUG_OUT then writeln ('  Line from MAIL.DELIVERY: "', line, '".');
    pindex := 1; current := nil; lindex := 1; quoted := false;
    while lindex <= length (line) do begin
      if (not quoted) and (line[lindex] in [' ', chr (9)]) then begin
        if current <> nil then if pindex <= max_parameters then
          if length (current^.parameters[pindex]) > 0 then begin
            if DEBUG_OUT then writeln ('  Parameter #', pindex:0, ' is: "',
                                       current^.parameters[pindex], '".');
            pindex := pindex + 1;
          end;
      end else if line[lindex] = '"' then begin
        if length (line) > lindex then
          if line[lindex+1] = '"' then begin
            addch ('"'); lindex := succ (lindex);
          end else quoted := not quoted
        else quoted := not quoted;
      end else if quoted and (pindex > 5) then addch (line[lindex])
      else addch (charupper (line[lindex]));
      lindex := lindex + 1;
    end; (* while not eoln *)
    if current <> nil then with current^ do begin
      if pindex <= max_parameters then
        if length (parameters[pindex]) > 0 then begin
          if DEBUG_OUT then writeln ('  Parameter #', pindex:0, ' is: "',
                                     parameters[pindex], '".');
          pindex := pindex + 1;
        end;
      pindex := pindex - 1;
      if pindex < min_parameters then begin
        if FROM_OWNER then LIB$SIGNAL (DELIVER__TOOFEWPARAMS, 1, lcount);
        goto 99;
      end;
      any_from      := parameters[from_parameter]      = '*';
      any_to        := parameters[to_parameter]        = '*';
      any_subject   := parameters[subject_parameter]   = '*';
      if parameters[subject_parameter] = '"' then
        parameters[subject_parameter] := '';
    end;
  end; (* while not eof *)
  if FROM_OWNER and (rules_list = nil) then
    LIB$SIGNAL (DELIVER__NORULES)
  else read_maildelivery_file := true;
  99:
  close (dfile);
end; (* read_maildelivery_file *)

(* MAIL_OUT_CONNECT is called by VMS MAIL to initiate a send operation. *)

[global] function MAIL_OUT_CONNECT (var context : unsigned;
  var link_flag : integer;
  var protocol, node : string_descriptor;
  var log_link_error : integer;
  var file_RAT, file_RFM : integer;
  var MAIL$GL_FLAGS : integer;
  var attached_file : string_descriptor) : integer;

begin (* MAIL_OUT_CONNECT *)
  fromstring := ''; tostring := '';
  subjectstring := ''; user_list_last := nil;
  if DEBUG_OUT then writeln ('MAIL_OUT_CONNECT called.');
  MAIL_OUT_CONNECT := SS$_NORMAL;
end; (* MAIL_OUT_CONNECT *)

(* MAIL_OUT_LINE is called by VMS MAIL whenever a single line of stuff
   must be delivered to the DELIVER mail relay. *)

[global] function MAIL_OUT_LINE (var context : unsigned;
  var link_flag : integer;
  var node, line : string_descriptor) : integer;

begin (* MAIL_OUT_LINE *)
  if DEBUG_OUT then writeln ('MAIL_OUT_LINE called.');
  case iaddress (link_flag) of
    (* MAIL is delivering a To: address *)
    LNK_C_OUT_TO     : begin
                         if DEBUG_OUT then writeln ('  OUT_TO option used.');
                         copy_descr_to_string (line, tostring, DEBUG_OUT);
                       end; (* LNK_C_OUT_TO *)
    (* MAIL is delivering a From: address *)
    LNK_C_OUT_SENDER : begin
                         if DEBUG_OUT then
                           writeln ('  OUT_SENDER option used.');
                         copy_descr_to_string (line, fromstring, DEBUG_OUT);
                       end; (* LNK_C_OUT_SENDER *)
    (* MAIL is delivering a Subject: line *)
    LNK_C_OUT_SUBJ   : begin
                         if DEBUG_OUT then writeln ('  OUT_SUBJ option used.');
                         copy_descr_to_string (line, subjectstring,
                                               DEBUG_OUT);
                       end; (* LNK_C_OUT_SUBJ *)
  end; (* case *)
  MAIL_OUT_LINE := SS$_NORMAL;
end; (* MAIL_OUT_LINE *)

(* MAIL_OUT_CHECK is called once with each addressee for the current
   message and once again after the message body has been sent. *)

[global] function MAIL_OUT_CHECK (
  var context : unsigned;
  var link_flag : integer;
  var protocol, addressee : string_descriptor;
  procedure MAIL$READ_ERROR_TEXT) : integer;

var
  usernamebuffer, userdirectory : string;
  maildelivery : text; useruic : unsigned; useraccount : account_name;
  currenttime : [quad] record
                         l0, l1 : unsigned;
                       end;

begin (* MAIL_OUT_CHECK *)
  if DEBUG_OUT then writeln ('MAIL_OUT_CHECK called.');
  case iaddress (link_flag) of
    (* Check out an addressee *)
    LNK_C_OUT_CKUSER : if (addressee.length = 1) and
                          (addressee.address^[1] = chr (0)) then begin
                         (* The null byte indicates that all the addressees
                            have been accomodated. *)
                         if DEBUG_OUT then writeln ('  Terminate user list.');
                         MAIL_OUT_CHECK := SS$_NORMAL;
                       end else begin
                         if DEBUG_OUT then writeln ('  CKUSER option used.');
                         copy_descr_to_string (addressee, usernamebuffer,
                                               DEBUG_OUT);
                         if DEBUG_OUT then
                           writeln ('  Checking out user "',
                                    usernamebuffer, '".');
                         if not find_user_directory (usernamebuffer,
                                                     userdirectory,
                                                     useruic, useraccount) then
                           MAIL_OUT_CHECK := DELIVER__USERNOEXIST
                         else begin
                           if DEBUG_OUT then writeln ('  Trying to open "',
                             userdirectory + 'MAIL.DELIVERY', '".');
                           open (file_variable := maildelivery,
                                 file_name := userdirectory + 'MAIL.DELIVERY',
                                 organization := SEQUENTIAL,
                                 sharing := READONLY,
                                 user_action := open_with_SYSPRV,
                                 error := CONTINUE, history := READONLY);
                           if status (maildelivery) <= 0 then
                             reset (maildelivery, error := CONTINUE);
                           if status (maildelivery) > 0 then begin
                             LIB$SIGNAL (DELIVER__NOMDFILE, 2,
                               usernamebuffer.length,
                               iaddress (usernamebuffer.body));
                             MAIL_OUT_CHECK := DELIVER__NOMDFILE;
                           end else begin
                             if DEBUG_OUT then
                               writeln ('  Adding this user to active list.');
                             user_count := user_count + 1;
                             if user_list_last = nil then begin
                               new (user_list_last);
                               user_list := user_list_last;
                             end else begin
                               new (user_list_last^.next);
                               user_list_last := user_list_last^.next;
                             end;
                             with user_list_last^ do begin
                               FROM_OWNER := fromstring = usernamebuffer;
                               if not read_maildelivery_file (maildelivery,
                                  rules_list) then begin
                                 if FROM_OWNER then LIB$SIGNAL (
                                   DELIVER__MDIGNORED);
                                 dispose_rules_list (rules_list);
                               end;
                               next      := nil;
                               username  := usernamebuffer;
                               directory := userdirectory;
                               uic       := useruic;
                               account   := useraccount;
                               $GETTIM (currenttime);
                               copyname := 'MAIL_' +
                                           hex (currenttime.l0, 8, 8) +
                                           hex (currenttime.l1, 8, 8) +
                                           hex (user_count, 8, 8);
                               if DEBUG_OUT then
                                 writeln ('  Added user "', username,
                                          '"; file code is "',
                                          copyname, '".');
                             end; (* with user_list_last^ *)
                             MAIL_OUT_CHECK := SS$_NORMAL;
                           end;
                         end;
                       end; (* LNK_C_OUT_CKUSER *)
    (* Check out the message send operation *)
    LNK_C_OUT_CKSEND : begin
                         if DEBUG_OUT then writeln ('  CKSEND option used.');
                         MAIL_OUT_CHECK := SS$_NORMAL;
                       end; (* LNK_C_OUT_CKSEND *)
  end; (* case *)
end; (* MAIL_OUT_CHECK *)

(* MAIL_OUT_FILE is called when the body of the message is ready to be
   sent. The message is available as a file and must be read from this
   temporary file using RMS. MAIL_OUT_FILE is where most of the actual
   work DELIVER does takes place. The following steps are taken:

   (1) The mode of the message file is set to record I/O (MAIL sometimes
       leaves the file in block mode).

   (2) The list of users to whom messages are being sent is scanned.
       For each user on the list:

       (a) A copy of the message is placed in the user's default
           directory. The file is created with SYSPRV, so it will
           be owned by that user.

       (b) The user's rules are scanned and checked for matches.

       (c) If any of the rules are satisfied, a command file is also
           created. This files contains some initial symbol definitions
           and then commands to implement each of the user's rules that
           matched. The command file ends with commands that delete the
           copy of the message as well as the commmand file itself.

       (d) A batch job is created to run the command file. Note that this
           means MAIL must be installed with CMKRNL privilege.
*)

[global] function MAIL_OUT_FILE (var context : unsigned;
  var link_flag : integer;
  var protocol : string_descriptor;
  var message_RAB : RAB$TYPE;
  [asynchronous, unbound] procedure UTIL$REPORT_ERROR) : integer;

var
  user_list_scan : user_block_ptr; onehasmatched, match : boolean;
  rules_list_scan : parameter_block_ptr; message_file : text;
  fromupstring, toupstring, subjectupstring, line : string;
  index, lleft, stat : integer;
  ppriv, priv, iosb : [quad] array [0..1] of unsigned;
  items : array [1..8] of item;

  function STR$MATCH_WILD (candidate : varying [l1] of char;
    pattern : varying [l2] of char) : integer; extern;

  function STR$UPCASE (var dststr : varying [l1] of char;
    var srcstr : varying [l2] of char) : integer; extern;

  (* function to read a line from the message file *)

  function get_line (var line : string) : boolean;

  var
    stat : integer;

  begin (* get_line *)
    get_line := false;
    message_RAB.RAB$L_UBF := iaddress (line.body);
    message_RAB.RAB$W_USZ := parameter_size;
    stat := $GET (RAB := message_RAB);
    if odd (stat) then begin
      line.length := message_RAB.RAB$W_RSZ;
      get_line := true;
    end else if stat <> RMS$_EOF then
      LIB$SIGNAL (DELIVER__MESREAERR, 1, stat);
  end; (* get_line *)

  procedure put_string (line : string);

  begin (* put_string *)
    if lleft >= length (line) then begin
      write (message_file, line); lleft := lleft - length (line);
    end;
  end; (* put_string *)

  procedure put_char (ch : char);

  begin (* put_char *)
    if lleft >= 1 then begin
      write (message_file, ch); lleft := pred (lleft);
    end;
  end; (* put_char *)

begin (* MAIL_OUT_FILE *)
  if DEBUG_OUT then writeln ('MAIL_OUT_FILE called.');

  (* Do some fancy footwork with RMS to insure that the file is open
     for sequential access and not block access. MAIL sometimes has
     this file open in block mode. The only way to change modes is
     to disconnect the RAB, diddle the mode bit and then reconnect it. *)
  if DEBUG_OUT then writeln ('  The BIO field of the RAB is set ',
    uand (message_RAB.RAB$L_ROP, RAB$M_BIO) <> 0);
  $DISCONNECT (RAB := message_RAB);
  message_RAB.RAB$L_ROP := uand (message_RAB.RAB$L_ROP, unot (RAB$M_BIO));
  $CONNECT (RAB := message_RAB);

  if DEBUG_OUT then writeln (' Creating upper case copies of header strings.');
  STR$UPCASE (fromupstring,    fromstring);
  STR$UPCASE (toupstring,      tostring);
  STR$UPCASE (subjectupstring, subjectstring);
  if DEBUG_OUT then writeln ('  From: "', fromupstring, '", To: "', toupstring,
                             '", Subject: "', subjectupstring, '".');

  if DEBUG_OUT then writeln (' Pruning the rules list.');
  user_list_scan := user_list;
  while user_list_scan <> nil do begin
    if DEBUG_OUT then writeln ('  User: "', user_list_scan^.username, '".');

    if DEBUG_OUT then writeln ('  Create copy of message in file "',
                               user_list_scan^.directory,
                               user_list_scan^.copyname, '.TEXT".');
    open (file_variable := message_file, history := NEW,
          record_length := parameter_size, record_type := VARIABLE,
          file_name := user_list_scan^.directory +
                       user_list_scan^.copyname + '.TEXT',
          user_action := create_with_SYSPRV, error := CONTINUE,
          organization := SEQUENTIAL);
    if status (message_file) <= 0 then rewrite (message_file);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    $REWIND (RAB := message_RAB);
    while get_line (line) do begin
      writeln (message_file, line, error := CONTINUE);
      if status (message_file) > 0 then
        LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    end; (* while get_line *)
    close (message_file, error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));

    if DEBUG_OUT then writeln ('  Creating command file named "',
                               user_list_scan^.directory,
                               user_list_scan^.copyname, '.COM".');
    open (file_variable := message_file, history := NEW,
          record_length := parameter_size, record_type := VARIABLE,
          file_name := user_list_scan^.directory +
                       user_list_scan^.copyname + '.COM',
          user_action := create_with_SYSPRV, error := CONTINUE,
          organization := SEQUENTIAL);
    if status (message_file) <= 0 then rewrite (message_file);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    writeln (message_file, '$ SET NOON', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    writeln (message_file, '$ DELETE = "DELETE"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    writeln (message_file, '$ MESSAGE_DELETE == "YES"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    write (message_file, '$ FROM == "'); lleft := DCL_line_size - 12;
    for index := 1 to length (fromstring) do
      if fromstring[index] = '"' then put_string ('""')
      else put_char (fromstring[index]);
    writeln (message_file, '"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    write (message_file, '$ QFROM == "'); lleft := DCL_line_size - 13;
    for index := 1 to length (fromstring) do
      if fromstring[index] = '"' then put_string ('""""')
      else put_char (fromstring[index]);
    writeln (message_file, '"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    write (message_file, '$ QQFROM == "'); lleft := DCL_line_size - 14;
    for index := 1 to length (fromstring) do
      if fromstring[index] = '"' then put_string ('""""""""')
      else put_char (fromstring[index]);
    writeln (message_file, '"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    write (message_file, '$ TO == "'); lleft := DCL_line_size - 10;
    for index := 1 to length (tostring) do
      if tostring[index] = '"' then put_string ('""')
      else put_char (tostring[index]);
    writeln (message_file, '"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    write (message_file, '$ QTO == "'); lleft := DCL_line_size - 11;
    for index := 1 to length (tostring) do
      if tostring[index] = '"' then put_string ('""""')
      else put_char (tostring[index]);
    writeln (message_file, '"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    write (message_file, '$ QQTO == "'); lleft := DCL_line_size - 12;
    for index := 1 to length (tostring) do
      if tostring[index] = '"' then put_string ('""""""""')
      else put_char (tostring[index]);
    writeln (message_file, '"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    write (message_file, '$ SUBJECT == "'); lleft := DCL_line_size - 15;
    for index := 1 to length (subjectstring) do
      if subjectstring[index] = '"' then put_string ('""')
      else put_char (subjectstring[index]);
    writeln (message_file, '"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    write (message_file, '$ QSUBJECT == "'); lleft := DCL_line_size - 16;
    for index := 1 to length (subjectstring) do
      if subjectstring[index] = '"' then put_string ('""""')
      else put_char (subjectstring[index]);
    writeln (message_file, '"', error := CONTINUE);
    write (message_file, '$ QQSUBJECT == "'); lleft := DCL_line_size - 17;
    for index := 1 to length (subjectstring) do
      if subjectstring[index] = '"' then put_string ('""""""""')
      else put_char (subjectstring[index]);
    writeln (message_file, '"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    writeln (message_file, '$ MESSAGE_FILE == "',
             user_list_scan^.directory,
             user_list_scan^.copyname, '.TEXT"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    writeln (message_file, '$ COMMAND_FILE == "',
             user_list_scan^.directory,
             user_list_scan^.copyname, '.COM"', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));

    if DEBUG_OUT then writeln ('  Check this user''s delivery list.');
    onehasmatched := false; rules_list_scan := user_list_scan^.rules_list;
    while rules_list_scan <> nil do with rules_list_scan^ do begin
      match := (any_to      or (STR$MATCH_WILD (toupstring,
                                parameters[to_parameter]) = STR$_MATCH)) and
               (any_from    or (STR$MATCH_WILD (fromupstring,
                                parameters[from_parameter]) = STR$_MATCH)) and
               (any_subject or (STR$MATCH_WILD (subjectupstring,
                                parameters[subject_parameter]) = STR$_MATCH));
      case parameters[decision_parameter][1] of
        'A'      : match := true;
        'X'      : match := false;
        'T', 'Y' : match := match;
        'F', 'N' : match := not match;
        '?', 'O' : match := match and (not onehasmatched);
        'B', 'Q' : match := (not match) and (not onehasmatched);
        'E'      : match := match or (not onehasmatched);
        otherwise match := false;
      end; (* decision case *)
      if match then begin
        if DEBUG_OUT then writeln (' Rule matched. From: pattern: "',
          parameters[from_parameter], '", To: pattern: "',
          parameters[to_parameter], '", Subject: pattern: "',
          parameters[subject_parameter], '", Decision character: ',
          parameters[decision_parameter][1], '.');
        onehasmatched := true;
        case parameters[action_parameter][1] of
          (* deliver *)
          'D'  : begin
                   write (message_file, '$ MAIL/NOSELF/SUBJECT="(From: ');
                   write (message_file, '''''QFROM'') ''''QSUBJECT''"');
                   write (message_file, ' ''MESSAGE_FILE'' "_');
                   write (message_file, user_list_scan^.username);
                   writeln (message_file, '"', error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR,
                                 1, status (message_file));
                 end; (* deliver *)
          (* privileged-deliver *)
          'V'  : begin
                   writeln (message_file,'$ PRIV = F$SETPRV("DETACH,SYSPRV")',
                            error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR,
                                 1, status (message_file));
                   write (message_file, '$ MAIL/NOSELF',
                    '/PROTOCOL=DELIVER_MAILSHR/SUBJECT="''''QSUBJECT''"');
                   write (message_file, ' ''MESSAGE_FILE'' "_');
                   write (message_file, user_list_scan^.username);
                   writeln (message_file, '"', error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
                   writeln (message_file,'$ PRIV = F$SETPRV(PRIV)',
                            error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR,
                                 1, status (message_file));
                 end; (* privileged-deliver *)
          (* forward *)
          'F'  : begin
                   write (message_file, '$ MAIL/NOSELF/SUBJECT="(From: ');
                   write (message_file, '''''QFROM'') ''''QSUBJECT''"');
                   writeln (message_file, ' ''MESSAGE_FILE'' ',
                            parameters[argument_parameter], error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
                 end; (* forward *)
          (* privileged-forward *)
          'W'  : begin
                   writeln (message_file,'$ PRIV = F$SETPRV("DETACH,SYSPRV")',
                            error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR,
                                 1, status (message_file));
                   write (message_file, '$ MAIL/NOSELF',
                     '/PROTOCOL=DELIVER_MAILSHR/SUBJECT="''''QSUBJECT''"');
                   writeln (message_file, ' ''MESSAGE_FILE'' ',
                            parameters[argument_parameter], error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
                   writeln (message_file,'$ PRIV = F$SETPRV(PRIV)',
                            error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR,
                                 1, status (message_file));
                 end; (* privileged-forward *)
          (* create, copy *)
          'C'  : begin
                   writeln (message_file, '$ COPY ''MESSAGE_FILE'' ',
                            parameters[argument_parameter],
                            error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
                 end; (* create, copy *)
          (* append *)
          'A'  : begin
                   writeln (message_file, '$ APPEND ''MESSAGE_FILE'' ',
                            parameters[argument_parameter],
                            error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
                 end; (* append *)
          (* execute *)
          'E'  : begin
                   if parameters[argument_parameter][1] <> '$' then
                   write (message_file, '$ ');
                   writeln (message_file, parameters[argument_parameter],
                            error := CONTINUE);
                   if status (message_file) > 0 then
                     LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
                 end; (* execute *)
          (* quit *)
          'Q'  : rules_list_scan := nil;
          otherwise begin end;
        end; (* case *)
      end; (* add commands to implement this matching rule *)
      if rules_list_scan <> nil then rules_list_scan := rules_list_scan^.next;
    end; (* while *)

    if not onehasmatched then begin
      if DEBUG_OUT then writeln ('  No rules matched, just deliver it.');
      write (message_file, '$ MAIL/NOSELF/SUBJECT="(From: ');
      writeln (message_file, '''''QFROM'') ''''QSUBJECT''" ''MESSAGE_FILE'' _',
               user_list_scan^.username, error := CONTINUE);
      if status (message_file) > 0 then
        LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    end;

    if DEBUG_OUT then writeln ('  Finishing up delivery command file.');
    writeln (message_file, '$ IF MESSAGE_DELETE .nes. "NO" then ',
             'DELETE ''MESSAGE_FILE'';', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    writeln (message_file, '$ DELETE ''COMMAND_FILE'';',
             error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    writeln (message_file, '$ LOGOUT', error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));
    close (message_file, error := CONTINUE);
    if status (message_file) > 0 then
      LIB$SIGNAL (DELIVER__MESWRTERR, 1, status (message_file));

    if DEBUG_OUT then writeln ('  Submitting batch job.');
    priv[0] := PRV$M_CMKRNL + PRV$M_SYSPRV; priv[1] := 0;
    $SETPRV (ENBFLG := 1, PRVADR := priv, PRMFLG := 0, PRVPRV := ppriv);
    with items[1] do begin
      len := length (user_list_scan^.username); code := SJC$_USERNAME;
      addr := iaddress (user_list_scan^.username) + 2; rlen := 0;
    end;
    with items[2] do begin
      len := length (batch_log); code := SJC$_LOG_SPECIFICATION;
      addr := iaddress (batch_log) + 2; rlen := 0;
    end;
    with items[3] do begin
      len := 8; code := SJC$_ACCOUNT_NAME;
      addr := iaddress (user_list_scan^.account); rlen := 0;
    end;
    with items[4] do begin
      len := 0; code := SJC$_NO_LOG_SPOOL;
      addr := 0; rlen := 0;
    end;
    with items[5] do begin
      len := 4; code := SJC$_UIC;
      addr := iaddress (user_list_scan^.uic); rlen := 0;
    end;
    with items[6] do begin
      len := length (batch_queue); code := SJC$_QUEUE;
      addr := iaddress (batch_queue) + 2; rlen := 0;
    end;
    line := user_list_scan^.directory + user_list_scan^.copyname + '.COM';
    with items[7] do begin
      len := length (line); code := SJC$_FILE_SPECIFICATION;
      addr := iaddress (line) + 2; rlen := 0;
    end;
    with items[8] do begin
      len := 0; code := 0; addr := 0; rlen := 0;
    end;
    if DEBUG_OUT then writeln ('    Opening job.');
    stat := $SNDJBCW (func := SJC$_ENTER_FILE, itmlst := items, iosb := iosb);
    if (not odd (stat)) or (not odd (iosb[0])) then begin
      with items[6] do begin
        len := length (system_batch_queue);
        addr := iaddress (system_batch_queue) + 2;
      end;
      stat := $SNDJBCW (func := SJC$_ENTER_FILE, itmlst := items, iosb := iosb);
    end;
    if not odd (stat) then LIB$SIGNAL (stat);
    if not odd (iosb[0]) then LIB$SIGNAL (iosb[0]);
    priv[0] := uand (priv[0], unot (ppriv[0]));
    priv[1] := uand (priv[1], unot (ppriv[1]));
    $SETPRV (ENBFLG := 0, PRVADR := priv, PRMFLG := 0);
    user_list_scan := user_list_scan^.next;
  end; (* while *)
  MAIL_OUT_FILE := SS$_NORMAL;
end; (* MAIL_OUT_FILE *)

(* MAIL_OUT_DEACCESS is called to shut down the current send operation. *)

[global] function MAIL_OUT_DEACCESS (var context : unsigned;
  var link_flag : integer) : integer;

begin (* MAIL_OUT_DEACCESS *)
  if DEBUG_OUT then writeln ('MAIL_OUT_DEACCESS called.');
  if user_list <> nil then begin
    if DEBUG_OUT then writeln ('  Deleting user list and associated rules.');
    dispose_user_list (user_list);
  end;
  MAIL_OUT_DEACCESS := SS$_NORMAL;
end; (* MAIL_OUT_DEACCESS *)

(* These routines manipulate a stack in which we maintain state information for
   information being "written" to us when MAIL calls MAIL_IO_WRITE. *)

procedure init_stack (var stack : write_state_stack);

begin (* init_stack *)
  if DEBUG_IN then writeln ('INIT_STACK called.');
  stack.top := 0;
end; (* init_stack *)

procedure push (var stack : write_state_stack; state : write_states);

var
  i : integer;

begin (* push *)
  if DEBUG_IN then writeln ('PUSH called.');
  with stack do begin
    top := succ (top);
    if top > stack_size then LIB$SIGNAL (DELIVER__INTSTKOVR);
    store[top] := state;
  end; (* with *)
  if DEBUG_IN then begin
    writeln ('  after PUSH:');
    for i := stack.top downto 1 do writeln ('    ', stack.store[i]);
  end;
end; (* push *)

procedure pop (var stack : write_state_stack);

var
  i : integer;

begin (* pop *)
  if DEBUG_IN then writeln ('POP called.');
  with stack do begin
    top := pred (top);
    if top < 1 then LIB$SIGNAL (DELIVER__STKEMPTY);
  end; (* with *)
  if DEBUG_IN then begin
    writeln ('  after POP:');
    for i := stack.top downto 1 do writeln ('    ', stack.store[i]);
  end;
end; (* pop *)

function top_of_stack (var stack : write_state_stack) : write_states;

begin (* top_of_stack *)
  if DEBUG_IN then writeln ('TOP_OF_STACK called.');
  top_of_stack := stack.store[stack.top];
  if DEBUG_IN then writeln (' returning ', stack.store[stack.top]);
end; (* top_of_stack *)

(* The incoming mail handling routines are activated by a command line of
   the form:

   $ MAIL/PROTOCOL=DELIVER_MAILSHR/SUBJECT="subject" message.txt address-list

   Everything is specified on the command line except the From: address,
   which is obtained by translating the logical name DELIVER$FROM.

   System privileges are required to use this interface since this routine
   makes it possible to "forge" return addresses. *)

[global] function MAIL_IN_CONNECT (var context : unsigned;
  var link_flag : integer;
  var input_tran : string_descriptor;
  var file_RAT, file_RFM : integer;
  var MAIL$GL_SYSFLAGS : integer;
  var MAIL$Q_PROTOCOL : string_descriptor;
  var pflags : integer) : integer;

begin (* MAIL_IN_CONNECT *)
  if DEBUG_IN then writeln ('MAIL_IN_CONNECT called.');
  toline := '';
  if DEBUG_IN then writeln ('Initializing state stack.');
  init_stack (write_recv_states);
  push (write_recv_states, bad_msg);
  LIB$SET_SYMBOL ('DELIVER$STATUS', '%X00000001');
  MAIL_IN_CONNECT := SS$_NORMAL;
end; (* MAIL_IN_CONNECT *)

(* MAIL calls MAIL_IN_LINE to get single line information from DELIVER. *)

[global] function MAIL_IN_LINE (var context : unsigned;
  var link_flag : integer;
  var line : string_descriptor) : integer;

var
  linebuffer : string; stat : integer;

begin (* MAIL_IN_LINE *)
  if DEBUG_IN then writeln ('MAIL_IN_LINE called.');
  case iaddress (link_flag) of
    (* Return From: information to MAIL *)
    LNK_C_IN_SENDER : begin
                        if DEBUG_IN then writeln ('IN_SENDER option used.');
                        stat := LIB$GET_SYMBOL ('FROM', linebuffer);
                        if not odd (stat) then
                          linebuffer := '<not specified>';
                        copy_string_to_descr (linebuffer, line, DEBUG_IN);
                       end; (* LNK_C_IN_SENDER *)
    (* Return To: information to MAIL *)
    LNK_C_IN_CKUSER : begin
                        if DEBUG_IN then writeln ('IN_CKUSER option used.');
                        stat := CLI$GET_VALUE ('TOLIST', linebuffer);
                        if not odd (stat) then linebuffer := chr (0) else begin
                          if length (toline) > 0 then toline := toline + ',';
                          toline := toline + linebuffer;
                          push (write_recv_states, user_check);
                        end;
                        copy_string_to_descr (linebuffer, line, DEBUG_IN);
                      end; (* LNK_C_IN_CKUSER *)
    (* Return entire To: line to MAIL *)
    LNK_C_IN_TO     : begin
                        if DEBUG_IN then writeln ('IN_TO option used.');
                        copy_string_to_descr (toline, line, DEBUG_IN);
                       end; (* LNK_C_IN_TO *)
    (* Return entire Subject: line to MAIL *)
    LNK_C_IN_SUBJ   : begin
                        if DEBUG_IN then writeln ('IN_SUBJ option used.');
                        stat := CLI$GET_VALUE ('SUBJECT', linebuffer);
                        if not odd (stat) then linebuffer := '';
                        copy_string_to_descr (linebuffer, line, DEBUG_IN);
                       end; (* LNK_C_IN_SUBJ *)
  end; (* case *)
  MAIL_IN_LINE := SS$_NORMAL;
end; (* MAIL_IN_LINE *)

(* MAIL_IN_FILE is called by MAIL to read the body of the message to be
   delivered. This routine gets the file name from the command line, opens
   the file and copies it into MAIL's intermediate file. *)

[global] function MAIL_IN_FILE (var context : unsigned;
  var link_flag : integer;
  var scratch : integer;
  var RAB : RAB$TYPE;
  procedure UTIL$REPORT_IO_ERROR) : integer;

var
  filename, linebuffer : string; message_file : text; stat : integer;

begin (* MAIL_IN_FILE *)
  if DEBUG_IN then writeln ('MAIL_IN_FILE called.');
  (* Get the name of the file containing the message to be delivered. *)
  stat := CLI$GET_VALUE ('FILE', filename);
  if not odd (stat) then begin
    LIB$SIGNAL (DELIVER__GETFILERR, 1, stat);
    MAIL_IN_FILE := DELIVER__GETFILERR;
  end else begin
    open (file_variable := message_file, file_name := filename,
          organization := SEQUENTIAL, sharing := READONLY,
          error := CONTINUE, history := READONLY);
    if status (message_file) <= 0 then
      reset (message_file, error := CONTINUE);
    if status (message_file) > 0 then begin
      LIB$SIGNAL (DELIVER__MESOPNERR);
      MAIL_IN_FILE := DELIVER__MESOPNERR;
    end else begin
      RAB.RAB$L_RBF := iaddress (linebuffer) + 2;
      stat := SS$_NORMAL;
      while (not eof (message_file)) and (odd (stat)) do begin
        readln (message_file, linebuffer, error := CONTINUE);
        if status (message_file) > 0 then begin
          LIB$SIGNAL (DELIVER__MSGREAERR, 1, status (message_file));
          stat := DELIVER__MSGREAERR;
        end else begin
          RAB.RAB$W_RSZ := length (linebuffer);
          stat := $PUT (RAB := RAB);
          if not odd (stat) then
            LIB$SIGNAL (DELIVER__MSGWRTERR, 1, stat);
        end;
      end; (* while *)
      close (message_file, error := CONTINUE);
      MAIL_IN_FILE := stat;
    end;
  end;
  push (write_recv_states, delivery_check);
end; (* MAIL_IN_FILE *)

(* MAIL_IO_WRITE is called by MAIL to tell DELIVER what it thinks of the
   results returned by the various MAIL_IN_ routines. *)

[global] function MAIL_IO_WRITE (var context : unsigned;
  var link_flag : integer;
  line : string_descriptor) : integer;

var
  error_text : string;

  function string_to_integer (var str : string_descriptor) : integer;

  var
    number : packed array [1..4] of char; i : integer;

  begin (* string_to_integer *)
    if str.length <> 4 then string_to_integer := 0 else begin
      for i := 1 to 4 do number[i] := str.address^[i];
      string_to_integer := number :: integer;
    end;
  end; (* string_to_integer *)

begin (* MAIL_IO_WRITE *)
  if DEBUG_IN then writeln ('MAIL_IO_WRITE called.');
  case top_of_stack (write_recv_states) of
    delivery_check : begin
                       if DEBUG_IN then writeln ('Delivery check.');
                       last_error := string_to_integer (line);
                       if DEBUG_IN then writeln (' got a stat : ', last_error);
                       pop (write_recv_states);
                       if not odd (last_error) then
                         LIB$SET_SYMBOL ('DELIVER$STATUS',
                                         '%X' + hex (last_error, 8, 8));
                       if last_error <> SS$_NORMAL then
                         push (write_recv_states, error_msg);
                     end; (* delivery_check *)
    user_check :     begin
                       if DEBUG_IN then writeln ('User check.');
                       last_error := string_to_integer (line);
                       if DEBUG_IN then writeln (' got a stat : ', last_error);
                       pop (write_recv_states);
                       if not odd (last_error) then
                         LIB$SET_SYMBOL ('DELIVER$STATUS',
                                         '%X' + hex (last_error, 8, 8));
                       if last_error <> SS$_NORMAL then
                         push (write_recv_states, error_msg);
                     end; (* user_check *)
    error_msg :      begin
                       if DEBUG_IN then writeln ('Error text.');
                       if (line.length = 1) and (line.address^[1] = chr (0))
                       then begin
                         if DEBUG_IN then
                           writeln (' got a NULL -- popping write_recv_states');
                       end else begin
                         copy_descr_to_string (line, error_text, DEBUG_IN);
                         if DEBUG_IN then
                           writeln ('Error message: "', error_text, '"');
                         if not odd (last_error) then
                           LIB$SET_SYMBOL ('DELIVER$MESSAGE', error_text);
                       end;
                       pop (write_recv_states);
                     end; (* error_text *)
    bad_msg :        begin
                       if DEBUG_IN then writeln ('Unexpected message.');
                       last_error := string_to_integer (line);
                       if DEBUG_IN then
                         writeln (' UNEXPECTED stat : ', last_error);
                       push (write_recv_states, error_msg);
                     end; (* bad_msg *)
    otherwise LIB$SIGNAL (DELIVER__BADSTKELE);
  end; (* case *)
  MAIL_IO_WRITE := SS$_NORMAL;
end; (* MAIL_IO_WRITE *)

[global] function MAIL_IO_READ (var context : unsigned;
  var link_flag : integer;
  var returned_line : string_descriptor) : integer;

begin (* MAIL_IO_READ *)
  if DEBUG_IN then writeln ('MAIL_IO_READ called.');
  MAIL_IO_READ := SS$_NORMAL;
end; (* MAIL_IO_READ *)

(* End of DELIVER.PAS *)
end.
