/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* threads.c - Thread handling */

#include <stdio.h>
#include <string.h>

#include "memory.h"
#include "output.h"
#include "threads.h"
#include "ask_str.h"
#include "language.h"

#ifdef __OS2__
#  define HAVE_THREADS
#endif

TID_REC *first_TID;             /* First thread ID */
TID_REC *last_TID;              /* Last thread ID */
TID_REC *cur_TID;               /* Current thread ID */

unsigned long cur_thread_num;   /* Thread number */
char cur_thread_name[30];       /* Thread name */
char cur_thread_restr[30];      /* Thread restricts usage of .. */

int Threads;                    /* # of threads going on */
int restrict_usage;             /* Don't allow using these functions */

void init_threads(void)
{
    first_TID = NULL;
    last_TID = NULL;
    cur_TID = NULL;
    Threads = 0;
}

void deinit_threads(void)
{
    while (first_TID != NULL)
    {
        cur_TID = (TID_REC *) first_TID->next;
        _free(first_TID);
        first_TID = cur_TID;
    }
}

int begin_thread(void *func, char *name, int restrict, int stack, int *tid)
{
#ifndef HAVE_THREADS
    func = func; name = name; restrict = stack; tid = tid;
    return 0;
#else
#if defined (__OS2__)
    cur_TID = (TID_REC *) _malloc(sizeof(TID_REC));
    if (cur_TID == NULL)
    {
        /* Not enough memory! */
        return 0;
    }

    if (DosCreateThread(&cur_TID->ThreadID,(PFNTHREAD) func,0,0,stack) == 0)
    {
        /* Thread created ok. */
        if (last_TID != NULL) last_TID = cur_TID;
        last_TID = cur_TID;

        strcpy(cur_TID->name,name);
        cur_TID->Restrict = restrict;
        cur_TID->next = NULL;

        restrict_usage |= restrict;
        Threads++;
        *tid = Threads;

        if (first_TID == NULL) first_TID = cur_TID;
        return 1;
    }

    /* Error.. */
    _free(cur_TID);
    return 0;
#endif
#endif
}

void end_thread(int ThreadID)
{
    int num;
    TID_REC *PrevTID, *NextTID;

    num = 1;
    cur_TID = first_TID;
    PrevTID = NULL;
    while (cur_TID != NULL)
    {
        if (num == ThreadID)
        {
            /* Found, remove it. */
            NextTID = (TID_REC *) cur_TID->next;

            if (last_TID == cur_TID)
            {
                /* This was last thread */
                last_TID = (TID_REC *) PrevTID;
            }

            if (PrevTID != NULL)
            {
                PrevTID->next = (TID_REC *) NextTID;
            }
            else
            {
                first_TID = (TID_REC *) NextTID;
            }

            restrict_usage &= ~cur_TID->Restrict;
            _free(cur_TID);
            Threads--;
            break;
        }
        cur_TID = (TID_REC *) cur_TID->next;
        num++;
    }
}

TID get_tid(int ThreadID)
{
    int num;

    num = 1;
    cur_TID = first_TID;
    while (cur_TID != NULL)
    {
        if (num == ThreadID) return cur_TID->ThreadID;
        cur_TID = (TID_REC *) cur_TID->next;
        num++;
    }

    return 0;
}

int remove_thread(int ThreadID)
{
    TID tid;

    tid = get_tid(ThreadID);
    if (tid == 0) return 0;

#ifdef __OS2__
    if (DosKillThread(tid) != 0) return 0; /* It doesn't want to die!! */
    //if (DosWaitThread(&tid,DCWW_NOWAIT) != 0) return 0; /* It doesn't want to die!! */
#endif

    end_thread(ThreadID);
    return 1;
}

/* Check which thread restricts usage of <..> */
int restrictive_thread(int restrict)
{
    int num;

    num = 1;
    cur_TID = first_TID;
    while (cur_TID != NULL)
    {
        if (cur_TID->Restrict & restrict) return num;
        cur_TID = (TID_REC *) cur_TID->next;
        num++;
    }

    restrict_usage &= ~restrict;
    return 0;
}

/* Display threads */
void show_jobs(void)
{
    output(lang[LANG_THREADS_HEADER]);
    cur_thread_num = 1;
    cur_TID = first_TID;
    while (cur_TID != NULL)
    {
        strcpy(cur_thread_name,cur_TID->name);
        if (cur_TID->Restrict & NO_MESSAGES)
            strcpy(cur_thread_restr, lang[LANG_THREADS_RESTR_MSGS]);
        else
            strcpy(cur_thread_restr, lang[LANG_THREADS_RESTR_NONE]);
        output(lang[LANG_THREADS_ENTRY]);
        cur_TID = (TID_REC *) cur_TID->next;
        cur_thread_num++;
    }
}

void kill_job(char *data)
{
    int ThreadID;
    char str[20];

    str[0] = '\0';
    if (!ask_string(lang[LANG_ASK_KILL_THREAD], str, 10, 0, &data)) return;

    if (sscanf(str, "%d", &ThreadID) != 1 || ThreadID == 0) return;

    if (ThreadID > Threads)
    {
        output(lang[LANG_N_THREADS_RUNNING],Threads);
        return;
    }

    if (remove_thread(ThreadID) == 0)
        output(lang[LANG_THREAD_IMMORTAL]);
    else
        output(lang[LANG_THREAD_DIED]);
}
