/*
 * vuohiv.c (last 'v' is for Voyeur)
 *
 * Enables Sysop to snoop VuohiMatik nodes.
 *
 * Based upon ttysnoop v0.12 by Carl Declerck.
 * Some modifications for DayDream by Antti Hyrinen.
 * Further vuohibugs added by ~g/RRR
 *
 * Note: "sockname" is actually a kind of FIFO, not a socket.
 *
 */

#include "struct.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/un.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <grp.h>
#include <utmp.h>
#include <errno.h>
#include <signal.h>

#define TTY_STORE	16
#define BUFF_SIZE	256

#define max(x,y)	((x) > (y) ? (x) : (y))

static struct termios orig_tty_state[TTY_STORE];
static int sttyfds[TTY_STORE];
char buff[BUFF_SIZE];
int outfd;

void stty_orig(void);
void stty_initstore(void);
int stty_raw(int fd);
void closeout(void);
void sighandler(int);
void errorf(char *fmt, ...);

int main (int argc, char *argv[])
{
	fd_set readset;
	int infd, fdmax, quit = 0, n;
	struct winsize ws;
	char sockname[128];
	
	if (argc < 2) {
		errorf ("Usage: %s <node_number>\n", argv[0]);
	}
	sprintf(sockname,"%sVMatik%sr",VMTMP,argv[1]);
	outfd=open(sockname,O_WRONLY);
	
	sprintf(sockname,"%sVMatik%sw",VMTMP,argv[1]);
	infd=open(sockname,O_RDONLY);
	if (infd==-1 || outfd==-1) {
		errorf ("Node not running! (or it is a localnode)\n");
	}
	
	puts("vuohiv. Yum. ctrl-b-h for help!\n");
	
	stty_initstore ();
	atexit (stty_orig);
	if (isatty(STDIN_FILENO))
		stty_raw (STDIN_FILENO);

	/* calc max file descriptor for select() */
	ioctl(STDIN_FILENO,TIOCGWINSZ,&ws);

	buff[0]=2;
	buff[1]=1;
	buff[2]=ws.ws_row;
	
	write(outfd,buff,3);
	
	atexit(closeout);
	signal(SIGINT,sighandler);
	signal(SIGHUP,sighandler);
	signal(SIGTERM,sighandler);
	fdmax = max(STDIN_FILENO, infd);
	
	/* do our thing */
	
	while (!quit)
	{
		FD_ZERO (&readset);
		FD_SET (STDIN_FILENO, &readset);
		FD_SET (infd, &readset);
		
		select (fdmax + 1, &readset, NULL, NULL, NULL);
		
		if (FD_ISSET(STDIN_FILENO, &readset))
		{
			read(STDIN_FILENO, buff, 1);

			if (buff[0]==3) {
				exit(0);
			} else {
				if (write(outfd, buff, 1) < 0)
					quit = 1;
			}
		}
		
		if (FD_ISSET(infd, &readset))
		{
			if ((n = read(infd, buff, BUFF_SIZE)) < 1)
				quit = 1;

			if (n > 0)
				write (STDOUT_FILENO, buff, n);
		}
	}

	return(0);
}

void closeout(void)
{
	buff[0]=2;
	buff[1]=4;
	write(outfd,buff,2);
}

void sighandler(int i)
{
	closeout();
	printf ("\r\nBack at local tty.\r\n");
	exit(0);
}

void errorf (char *fmt, ...)
{
	va_list args;
	
	va_start (args, fmt);
	vfprintf (stderr, fmt, args);
	exit (1);
}


/* init the stty store array */

void stty_initstore (void)
{
	int i;
	
	for (i = 0; i < TTY_STORE; i++)
		sttyfds[i] = -1;
}

/* set tty on fd into raw mode */

int stty_raw (int fd)
{
	struct termios tty_state;
	int i;
	
	if (tcgetattr(fd, &tty_state) < 0)
		return (-1);
	
	for (i = 0; i < TTY_STORE; i++)
		if (sttyfds[i] == -1)
		{
			orig_tty_state[i] = tty_state;
			sttyfds[i] = fd;
			break;
		}
	
	tty_state.c_lflag &= ~(ICANON | IEXTEN | ISIG | ECHO);
	tty_state.c_iflag &= ~(ICRNL | INPCK | ISTRIP | IXON | BRKINT);
	tty_state.c_oflag &= ~OPOST;
	tty_state.c_cflag |= CS8;
	
	tty_state.c_cc[VMIN]  = 1;
	tty_state.c_cc[VTIME] = 0;
	
	if (tcsetattr(fd, TCSAFLUSH, &tty_state) < 0)
		return (-1);
	
	return (0);
}

/* restore all altered ttys to their original state */

void stty_orig (void)
{
	int i;
	
	for (i = 0; i < TTY_STORE; i++)
		if (sttyfds[i] != -1)
		{
			tcsetattr (sttyfds[i], TCSAFLUSH, &orig_tty_state[i]);
			sttyfds[i] = -1;
		}
}
