/*
 * Functions related to uploading (e.g. transferring files TO system)
 * here.
 *
 */
#include "proto.h"
#include <dirent.h>
#include <sys/param.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <sys/mman.h>
#include <signal.h>
#include <utime.h>

#ifndef NO_VFS
#include <sys/vfs.h>
#endif

#include <sys/mount.h>

extern int lmode;

void ul_cleantemp(char *pathname);
void deldir(char *);

int quickmode;

/*
 * Receives files to directory "rec_path". Duplicates will be checked from
 * directories in "pathlist_fname", one dir per line.
 *
 */
int ul_recfiles(char *rec_path, char *pathlist_fname)
{
	char olddir[PATH_MAX];
	char udbuf[PATH_MAX];
	char dszlog[PATH_MAX];
	pid_t bgrun;

	getcwd(olddir,PATH_MAX);
	chdir(rec_path);
		
	bgrun=bg_bgchecker();
	if(!bgrun)
	{
		DDPut("Urgh! Unable to launch bgchecker!\n");
		writelog("Unable to launch bgchecker!\n");
		return(0);
	}

	sprintf(dszlog, "%sdszlog.%d.%d", VMTMP, node, bgrun);	// Unique! 
	
	switch(protocol->Protocol_Type)
	{
		case PROTOCOL_ONEDIR:
			genstdiocmdline(udbuf,protocol->Protocol_ReceiveCommand,
						(pathlist_fname ? pathlist_fname : ""), 
						NULL,NULL, dszlog, NULL);
//			DDPut("Initstring:\n");
//			DDPut(udbuf);
//			DDPut("\n");
			runstdio(udbuf,0,2);
			break;
		case PROTOCOL_BIDIR:
			genstdiocmdline(udbuf, protocol->Protocol_ReceiveCommand,
						(pathlist_fname ? pathlist_fname : ""), 
						rec_path, NULL,dszlog, NULL);
//			DDPut("Initstring:\n");
//			DDPut(udbuf);
//			DDPut("\n");
			runstdio(udbuf,0,4);
			dl_killflood();
			break;
		case PROTOCOL_PSEUDO:
//			eb_checkforftp(2,rec_path,dszlog);
			break;
		default:
			DDPut("Error: Unknown protocol->Protocol_Type!\n");	
			break;
	}
	chdir(olddir);
		
	kill(bgrun, SIGUSR1);			// Notify bgchecker that xfer has ended

	return(1);
}

/*
 * This is the highest level upload function. It sends files to the user.
 *
 */
int ul_upload(int mode)
{
	char	query[QUERY_MAX];
	char	dpath[PATH_MAX];
	char	pathlist[PATH_MAX];
	char udbuf[500];
	int discon=0;
	int ok=1;
	int newarea=0, oldarea=0;

	setprotocol();
	
	changenodestatus("Uploading");

	if(lmode)
	{
		if(access_flags & SECB_COSYSOP)
			return(ul_localupload(&vm_database, Current_FileAreaID));
		else
			return(0);
	}
	
	if (user.user_toggles & TOGGLE_QUICK_UL)
		quickmode=1; 
	else 
		quickmode=0;

/*
	if (conf->CONF_UPLOADAREA==0) {
		DDPut(sd[nouploadsstr]);
		return 0;
	}
*/
/******* Check if upload redirection is active */

	sprintf(query, "area_id=%d AND area_ul_redir!=%d AND area_conf=%d",
			Current_FileAreaID, Current_FileAreaID, conf->CONF_NUMBER);
	newarea=mi_sql_getnum(&vm_database, "area_ul_redir", "vmatik_fileareas",
			query);
	if(newarea)
	{
		oldarea=Current_FileAreaID;
		jc_changefilearea(newarea, JC_ABSOLUTE|JC_SHUTUP);
	}

	sprintf(query, "area_id=%d", Current_FileAreaID);
	mi_sql_getstr(&vm_database, "area_filepath", "vmatik_fileareas",
					query, dpath); 

	if (mode!=UPLOAD_JUSTCHECK) 
	{
		if(!lmode)
			ul_cleandir(currnode->Node_TempDir);
			
		if (mode!=UPLOAD_RZ) 
			TypeFile("upload",TYPE_MAKE|TYPE_CONF);

		if (!ul_freespace(dpath)) 
			ok=0;
		
		if (mode!=UPLOAD_RZ && ok) 
		{
			char go=1;
		
			checkforpartialuploads(1);
			if (protocol->Protocol_Type == PROTOCOL_PSEUDO) 
			{
//				unlink(dszlog);
				
//				if (eb_checkforftp(1,currnode->Node_TempDir,dszlog)) {
//					dl_analyzedszlog(dszlog,udbuf);
					goto ftpgo;
//				}
			}
		
			while (go && ok)
			{
				DDPut(sd[proceedtransferstr]);
				udbuf[0]=0;
				if (!(Prompt(udbuf,1,0))) 
					return(0);
				switch(udbuf[0])
				{
					case 'a':
					case 'A':
						ok=0;
						break;
					case 'd':
					case 'D':
						DDPut(sd[autodisconstr]);
						discon^=1;
						if(discon)
							DDPut(sd[enabledstr]);
						else
							DDPut(sd[disabledstr]);
						DDPut("\n");
						break;
					case 'p':
					case 'P':
					default:
						go=0;
						break;
				}
			}
		}
	
		if(ok)
		{
			int starttimeleft;
		
			starttimeleft=timeleft/60;
			mi_sql_setuser(&vm_database, "user_timeremain",
							user.user_serial_id, '=', starttimeleft);

			sprintf(pathlist, "%spathlist%d", VMTMP, node);
			ul_gen_pathlist(pathlist);
//debug		Prompt(pathlist, 1, HOT_YESNO);

			DDPut(sd[startxferstr]);
			ul_recfiles(currnode->Node_TempDir, pathlist);

			timeleft=mi_sql_getuser(&vm_database, "user_timeremain",
						user.user_serial_id)*60;
			if((starttimeleft*60)<timeleft)			// BGChk gave time reward!
				endtime=time(NULL)+timeleft;
		
			unlink(pathlist);
		}
	}

	if (discon) 
	{
		if (autodisconnect()==1) 
			quickmode=1;
		else 
			discon=0;
	}

ftpgo:

//	DDPut(currnode->Node_TempDir);
	DDPut("\n");

	if(newarea)
		jc_changefilearea(oldarea, JC_ABSOLUTE|JC_SHUTUP);
	
	if(discon) 
		return(2);

	return(0);
}

/*
 * Generates an executable command line "*dest" from input.
 *
 * In "*src", following will be replaced:
 * 
 * %A == Will be replaced with "*arg1"
 * %B == Will be replaced with "*arg2"
 * %C == Will be replaced with "*arg3"
 * %D == Name of the current TTY device
 * %H == User's handle
 * %L == Will be replaced with "*arg4" (usually the dszlog name)
 * %N == Node number if any, else "*no" 
 * %O == User's organization
 * %P == $VMATIK path
 * %R == User's realname
 * %S == User's seclevel
 * %Z == User's zipcity 
 *
 * Note: the arg1,arg2 shit is getting out of hand
 *
 */
int genstdiocmdline(char *dest, char *src, char *arg1, char *arg2, 
					char *arg3, char *arg4, char *no)
{
	char tbuffer[60];
	char *s;

	while(1)
	{
		if (*src=='%') {
			switch(*(src+1))
			{
				case 'a':
				case 'A':
					if(!arg1)
					{
						DDPut("Error: %A specified, but arg1 is NULL\n");
						src+=2;
						break;
					}
					while(*arg1) *dest++=*arg1++;
					src+=2;
					break;
				case 'b':
				case 'B':
					if(!arg2)
					{
						DDPut("Error: %B specified, but arg2 is NULL\n");
						src+=2;
						break;
					}
					while(*arg2) *dest++=*arg2++;
					src+=2;
					break;
				case 'c':
				case 'C':
					if(!arg3)
					{
						DDPut("Error: %C specified, but arg3 is NULL\n");
						src+=2;
						break;
					}
					while(*arg3) *dest++=*arg3++;
					src+=2;
					break;
				case 'd':
				case 'D':
					s=ttyname(serhandle);
					while(*s) *dest++=*s++;
					src+=2;
					break;
				case 'h':
				case 'H':
					s=user.user_handle;
					while(*s) *dest++=*s++;
					src+=2;
					break;
				case 'l':
				case 'L':
					if(!arg4)
					{
						DDPut("Error: %L specified, but arg4 is NULL\n");
						src+=2;
						break;
					}
					while(*arg4) *dest++=*arg4++;
					src+=2;
					break;
				case 'n':
				case 'N':
					s=tbuffer;
					if (!no) {
						sprintf(s,"%d",node);
					} else {
						strcpy(s,no);
					}
					while(*s) *dest++=*s++;
					src+=2;
					break;
				case 'o':
				case 'O':
					s=user.user_organization;
					while(*s) *dest++=*s++;
					src+=2;
					break;
				case 'p':
				case 'P':
					s=origdir;
					while(*s) *dest++=*s++;
					src+=2;
					break;
				case 'r':
				case 'R':
					s=user.user_realname;
					while(*s) *dest++=*s++;
					src+=2;
					break;
				case 's':
				case 'S':
					s=tbuffer;
					sprintf(s,"%d",user.user_securitylevel);
					while(*s) *dest++=*s++;
					src+=2;
					break;
				case 'z':
				case 'Z':
					s=user.user_zipcity;
					while(*s) *dest++=*s++;
					src+=2;
					break;
				default:
					*dest++=*src++;
					break;
			}
		} else {
			if (!*src) 
				break;
			*dest++=*src++;
		}
	}
	*dest=0;
	return(0);
}

/*
 * Renames file "old" to "new". If it fails, it tries
 * newcopy() instead.
 *
 * Returns 0 on success, -1 on failure.
 *
 */
int newrename(char *old, char *new)
{
	if (rename(old,new)==-1) 		// rename won't work, try copy instead
	{
		if(newcopy(old,new)==-1)
			return(-1);
		
		unlink(old);
	}
	
	return(0);
}

/*
 * This copies file "old" to file "new".
 * Original file dates are preserved.
 *
 * Returns 0 on success, -1 on failure
 *
 */
int newcopy(char *old, char *new)
{
	int oldfd;
		
	oldfd=open(old,O_RDONLY);
	if (oldfd!=-1) 
	{
		int newfd;
		struct utimbuf origtimes;	// Original file access times
		struct stat st;
	
		if(fstat(oldfd, &st)!=0)
		{
			close(oldfd);
			return(-1);
		}
		
		origtimes.actime=st.st_atime;
		origtimes.modtime=st.st_mtime;
			
		newfd=open(new,O_WRONLY|O_CREAT,0644);
		if (newfd!=-1) 
		{
			int cnt;
			char *rwbuf;

			rwbuf=(char *)malloc(COPY_BUFSIZE);
				
			while((cnt=read(oldfd,rwbuf,COPY_BUFSIZE)))
			{
				write(newfd,rwbuf,cnt);
			}
			free(rwbuf);
			close(newfd);
			utime(new, &origtimes);			// Restore them
		}
		else
		{
			close(oldfd);
			return(-1);
		}
	}
	else
		return(-1);

	close(oldfd);

	return(0);
}

/*
 * Removes all entries from a directory. Does NOT include subdirs.
 *
 */
void ul_cleandir(char *pathname)
{
	DIR *dirfd;
	struct dirent *ent;
	char tempbuf[PATH_MAX];
	
	if ((dirfd=opendir(pathname)))
	{
		while((ent=readdir(dirfd)))
		{
			sprintf(tempbuf,"%s%s",pathname,ent->d_name);
			unlink(tempbuf);
		}
		closedir(dirfd);
	} else {
		DDPut(sd[tempcleanerrstr]);
	}
}

/*
 * Everything from a directory, including subdirs.
 *
 */
void deldir(char *dir)
{
	DIR *dirfd;
	struct dirent *ent;
	char tempbuf[1024];
	
	if ((dirfd=opendir(dir)))
	{
		while((ent=readdir(dirfd)))
		{
			struct stat st;

			if ( (ent->d_name[0]=='.' && ent->d_name[1]==0) || (ent->d_name[0]=='.' && ent->d_name[1]=='.' && ent->d_name[2]==0)) continue;
			sprintf(tempbuf,"%s/%s",dir,ent->d_name);
			stat(tempbuf,&st);

			if (S_ISDIR(st.st_mode)) {
				deldir(tempbuf);
				rmdir(tempbuf);
			} else {
				unlink(tempbuf);
			}
		}
		closedir(dirfd);
	} 
}

/*
 * Checks if there's enough free space on "path"
 * and in the node's temp directory.
 *
 * Returns: 0 == nope, 1 == yes 
 *
 * Side-effects: Prints the available space.
 *
 */
int ul_freespace(char *path)
{
	bint freesp,freesp2;
	char voukubuf[1024];

	freesp=getfreesp(path);

//	DDPut(path);

	freesp2=getfreesp(currnode->Node_TempDir);
	
	if (freesp < maincfg.CFG_FREEHDDSPACE || 
		freesp2 < maincfg.CFG_FREEHDDSPACE)
	{
		DDPut(sd[notfreespacestr]);
		return(0);	
	} 
		
	sprintf(voukubuf,sd[freespacestr],freesp, freesp2);
	DDPut(voukubuf);

	return(1);	
}

/*
 * Returns the amount of free bytes on a filesystem 
 *
 */
bint getfreesp(char *path)
{
	struct statfs freest;
	
	if (!statfs(path,&freest)) {
		return((bint)freest.f_bsize*(bint)freest.f_bavail);
	}

	return(0);
}

/*
 * Local uploading (so called "adopting"). Files which are not already
 * in the filelist will be added to it.
 *
 */
int ul_localupload(MYSQL *db, int area_id)
{
	int added_files=0, added_bytes=0;
	char	query[QUERY_MAX];
	char 	buffer[PATH_MAX];
	char	filepath[PATH_MAX];
	char 	olddir[PATH_MAX];
	struct dirent *dent;
	int	result, uploader_id;
	DIR *dh;

	DDPut("\n");

	if(user.user_handle[0])
		strcpy(query, user.user_handle);
	else
		strcpy(query, user.user_realname);

	DDPut("Upload as: ");
	Prompt(query, 80, 0);
	if(!query[0])
	{
		DDPut("\n");
		DDPut(sd[euabortedstr]);
		DDPut("\n");
		return(0);
	}

	DDPut("\n");

	uploader_id=mi_findusername(query);
	if(!uploader_id)
	{
		DDPut("No such user...\n\n");
		return(0);
	}

	sprintf(buffer, "area_id=%d", area_id);
	mi_sql_getstr(db, "area_filepath", "vmatik_fileareas", buffer, filepath);

	if((dh=opendir(filepath)))	// Browse the directory
	{
		struct FileType new_file;
		struct stat st;

		getcwd(olddir,PATH_MAX);
		chdir(filepath);

		while((dent=readdir(dh)))
		{
			result=0;
			
			DDPut(dent->d_name);

			if(stat(dent->d_name, &st)==-1)
			{
				DDPut(" ... stat error!!!\n");
				continue;
			}

			if(!(S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))
			   || S_ISDIR(st.st_mode))
			{
				DDPut(" ... skipped, not a file\n");
				continue;
			}

			memset(&new_file, 0, sizeof(struct FileType));
	
			strcpy(new_file.Name, dent->d_name);
			new_file.Size=st.st_size;
			new_file.ByID=uploader_id;
			new_file.UploadDate=st.st_mtime;
	
			if(!(conf->CONF_ATTRIBUTES & CONF_VALIDATE_UPLOADS))
				new_file.Flags|=FILE_VALIDATED;
	
			result=ul_add_file(db, &new_file, area_id);
		
			switch(result)
			{
				case 1:
					added_files++;
					added_bytes+=st.st_size;
					DDPut(" ... added\n");
					break;
				case 0:
					DDPut(" ... exists\n");
					break;
				default:
					DDPut(" ... ERROR!\n");
					break;
			}
		}
		closedir(dh);
		chdir(olddir);
	}
	else
		DDPut("\nError opening dir...\n\n");

	if(added_files)
	{
		mi_sql_setuser(db, "user_filesup", uploader_id, '+', added_files);
		mi_sql_setuser(db, "user_bytesup", uploader_id, '+', added_bytes);
	
		sprintf(query, "UPDATE vmatik_fileareas SET area_newestfile=NOW(),
						area_lasttouch=NOW() WHERE area_id=%d", area_id);
		mysql_query(db,query);
		if(mysql_error(db)[0])
			DDPut(mysql_error(db));
		
		sprintf(buffer, "\nAdded total %d files, %d bytes.\n\n",
						added_files, added_bytes);
		DDPut(buffer);
	}
	else
		DDPut("\nNo files have been added.\n\n");

	return(1);
}

/*
 * Generates a list of paths from the current conference's fileareas.
 *
 */
int ul_gen_pathlist(char *filename)
{
	MYSQL_RES *sql_result;
	MYSQL_ROW sql_row;
	char query[QUERY_MAX];
	FILE *fp;

	fp=fopen(filename, "w");
	if(!fp)
		return(0);

	sprintf(query, "SELECT area_filepath FROM vmatik_fileareas WHERE
			area_conf=%d", conf->CONF_NUMBER);
	mysql_query(&vm_database, query);
	if(mysql_error(&vm_database)[0])
		DDPut(mysql_error(&vm_database));
	sql_result=mysql_use_result(&vm_database);
	if(!sql_result)
	{
		fclose(fp);
		unlink(filename);
		return(0);
	}

	while((sql_row=mysql_fetch_row(sql_result)))
	{
		fprintf(fp, "%s\n", sql_row[0]);
	}

	mysql_free_result(sql_result);

	fclose(fp);

	return(1);
}

/*
 * Adds a file to the table "vmatik_filelist"
 *
 */
int ul_add_file(MYSQL *db, struct FileType *new_file, int area_id)
{
	char		*query=NULL;
	char		*desc_tmp=NULL;

	if(new_file->Description)
	{
		int len;
	
		len=strlen(new_file->Description);
	
		desc_tmp=malloc(2*len);
		mysql_escape_string(desc_tmp, new_file->Description, len);
		
		if(len>QUERY_MAX)
			query=malloc(2*len+QUERY_MAX);
		else
			query=malloc(QUERY_MAX*2);
	}
	else
	{
		query=malloc(QUERY_MAX);
	}

	sprintf(query, "file_name='%s' AND file_area=%d",
					new_file->Name, area_id);
	if(mi_sql_getnum(db, "1", "vmatik_filelist", query)>0)
	{
		if(desc_tmp)
			free(desc_tmp);
		if(query)
			free(query);
		return(0);							// Skipped, already exists
	}

	sprintf(query, "INSERT INTO vmatik_filelist (file_byid, file_toid,
			file_name, file_size, file_area, file_flags, file_integrity,
			file_uldate, file_bytes2uler, file_files2uler, file_description) 
			VALUES (%ld,%ld,'%s',%ld,%d,%ld,%d,FROM_UNIXTIME(%d),%ld,%d,'%s')",
			new_file->ByID, new_file->ToID, new_file->Name, new_file->Size,
			area_id, new_file->Flags, new_file->Integrity, 
			(int)new_file->UploadDate, new_file->Bytes_To_ULer, 
			new_file->Files_To_ULer,
			(desc_tmp ? desc_tmp : ""));
	mysql_query(db, query);
	if(mysql_error(db)[0])
		DDPut(mysql_error(db));
	
	sprintf(query, "UPDATE vmatik_fileareas 
					SET area_lasttouch=NOW()
					WHERE area_id=%d", area_id);
	mysql_query(db, query);
	if(mysql_error(db)[0])
		DDPut(mysql_error(db));

	if(query)
		free(query);
	if(desc_tmp)
		free(desc_tmp);

	return(1);
}
