/*
 * This file has functions related to archive testing,
 * transforming and viewing.
 *
 */
#include "proto.h"
#include <dirent.h>
#include <sys/mount.h>
#include <sys/param.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <sys/mman.h>
#include <signal.h>

#ifndef NO_VFS
#include <sys/vfs.h>
#endif

/*
 * Displays archive/file contents to user.
 *
 * Note: "wholename" must contain the whole path to file.
 *
 */
int ar_viewfile(char *wholename)
{
	struct VMatik_Archiver *arc;
	char parbuf[1024];
	char output_filename[PATH_MAX];

	arc=ar_getarchiver(wholename);
	if (!arc) {
		sprintf(parbuf,sd[viewunkastr],parbuf);
		DDPut(parbuf);
		return(0);
	}

	if (!*arc->ARC_VIEW) {
		sprintf(parbuf,sd[viewnosupstr],arc->ARC_NAME);
		DDPut(parbuf);
		return(0);
	}

	if (arc->ARC_VIEWSEC > user.user_securitylevel) {
		sprintf(parbuf,sd[viewlamerstr],arc->ARC_NAME);
		DDPut(parbuf);
		return(0);
	}
		
	genstdiocmdline(parbuf,arc->ARC_VIEW,wholename,NULL,NULL,NULL,NULL);
	sprintf(output_filename,"%sarcoutp%d",VMTMP,node);
	runstdio(parbuf,output_filename,3);
	DDPut("[0m");
	TypeFile(output_filename,TYPE_NOCODES);
	unlink(output_filename);
	
	return(1);
}

/*
 * Returns an archiver for a certain filetype, or NULL if not found.
 *
 */
struct VMatik_Archiver *ar_getarchiver(char *file)
{
	struct VMatik_Archiver *temp_arch;

	temp_arch=arcs;
	
	while(temp_arch->ARC_FLAGS!=255)
	{
		if (mi_wildcmp(file,temp_arch->ARC_PATTERN)) {
			return(temp_arch);
		}
		temp_arch++;
	}
	
	return(NULL);
}

/*
 * Transforms an archive.
 *
 * Notes: - Result "new_name" MUST have enough space!:
 *          It will contain the new path/filename.
 *        - The "old file" is NOT deleted by this function
 *        - If the new file exist, we suppose the transform
 *          succeeded.
 *
 */
int ar_transform_archive(char *pathname, struct VMatik_Archiver *arc,
						char *new_name)
{
	char tr_cmd[PATH_MAX];
	char outpname[PATH_MAX];
	struct stat st;
	int i;

	if(!arc || !pathname || !new_name)
		return(0);

	if(!(arc->ARC_TRANSFORM_TO[0]) || !(arc->ARC_CMD_TRANSFORM[0]))
		return(0);

	strcpy(new_name, pathname);
	i=strlen(new_name);
	while(i>=0 && new_name[i]!='.')				// Get rid of old extension
		i--;

	if(new_name[i]!='.')						// Odd, extension not found
		return(0);

	strcpy(&new_name[i+1], arc->ARC_TRANSFORM_TO); // Add new extension	
//	DDPut(new_name);
		
	genstdiocmdline(tr_cmd,arc->ARC_CMD_TRANSFORM,pathname,new_name,
					NULL,NULL,NULL);
	sprintf(outpname,"%str_outp%d",VMTMP,node);
	unlink(outpname);
/*
	DDPut(tr_cmd);
	DDPut("\n");
	DDPut(outpname);
	DDPut("\n");
*/
	
	runstdio(tr_cmd,outpname,3);		// No, we don't want to see it 
	unlink(outpname);

	if(stat(new_name, &st)!=0)			// Hmhmh, no new file, we failed
	{
//		DDPut("Jouh!\n");
		return(0);
	}

	return(1);
}

/*
 * Tests an archive.
 *
 * 0 == failure, 1 == passed
 *
 * Bugs: it doesn't check if the archive type is testable or not.
 *
 */
int ar_test_archive(char *pathname, struct VMatik_Archiver *arc)
{
	char arcoutpname[80];
	char arccmd[200];
	char arcbuf[4000];
	FILE *checkd;
	int arc_ok=1;
	
	if(!arc || !pathname)
		return(0);
		
	genstdiocmdline(arccmd,arc->ARC_CMD_TEST,pathname,NULL,NULL,NULL,NULL);
	sprintf(arcoutpname,"%sarcoutp%d",VMTMP,node);
	unlink(arcoutpname);
		
//	DDPut(sd[testingarcstr]);
/*
	if (arc->ARC_FLAGS & ARC_DISPLAY) 
	{
		DDPut(sd[testsepastr]);
		runstdio(arccmd,arcoutpname,0); 
		DDPut(sd[testsepastr]);
	} 
	else
*/
	{
		runstdio(arccmd,arcoutpname,3);			// No, don't show it 
	}

	checkd=fopen(arcoutpname,"r");
	if(!checkd)
		return(0);

	while(fgets(arcbuf,4000,checkd) && arc_ok)
	{
		if (*arc->ARC_CORRUPTED1 && strstr(arcbuf,arc->ARC_CORRUPTED1)) 
			arc_ok=0;
		if (*arc->ARC_CORRUPTED2 && strstr(arcbuf,arc->ARC_CORRUPTED2))
			arc_ok=0;
		if (*arc->ARC_CORRUPTED3 && strstr(arcbuf,arc->ARC_CORRUPTED3))
			arc_ok=0;
	}

	fclose(checkd);

	unlink(arcoutpname);

	if(!arc_ok)
	{
		DDPut(sd[arcbroken2str]);
		return(0);
	}
	
	DDPut(sd[arcokstr]);
	
	return(1);
}		

