/* ix/MBox (tools.c) by Volker Schuermann, 04.12.1993

   This C source code contains the following functions:

   #WH whoami()          who are you ;-)
   #TY tty()             find out your tty number
   #SP stripped()        strip all junk from a string
   #UP upcased()         convert a string to upcased characters
   #ST strcomp()         compare two patterns
   #SC strcopy()         cut a pattern out of a string
   #CY crypted()         encrypt a string
   #DC dateconv()        convert a date string to long
   #DR daterecover()     convert a long to a data string
   #TC timeconv()        convert a time string to long
   #TR timereconv()      convert a long to a time string
   #ML maybe_locked()    check whether a file is locked
   #NS num_stripped()    delete all numeric chars from a string
   #MR mbrename()        internal "rename()" function
   #HL headline()        show a headline
   #TN ttyna()           find out your tty name
   #BC bigcopy()         cut a pattern out of a gib string
   #MO mblock()          lock a file
   #MU mbunlock()        unlock a file
   #MT mytime()          get the time
   #MD mydate()          get the date
   #SN shortname()       convert filename to 12 char strings
   #CL clearline()       delete a line on a terminal
   #IN isin()            check if a char is in a string
   #AN ansi2()           print terminal codes out of "/etc/termcap"
   #TA termansi()        return a terminal code of "/etc/termcap"
   #SG sgrep             find a pattern in a string

   Contact <volkers@unnet.wupper.de> for help! */




#undef _MBOX


#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>
#include <string.h>

#ifdef _SYS7
#include <termcap.h>
#endif

#ifndef UNSIGNED
#if !defined(_MINIX) || defined(_ESTDIO)
#define UNSIGNED unsigned
#else
#define UNSIGNED
#endif
#endif

#ifdef _MBOX
#define EXTERN
#include "mbox.h"
#else
#define VERSION "Tools.C"
#define CR 13
#define TOL01_MSG	"Sonntag"
#define TOL02_MSG	"Montag"
#define TOL03_MSG	"Dienstag"
#define TOL04_MSG	"Mittwoch"
#define TOL05_MSG	"Donnerstag"
#define TOL06_MSG	"Freitag"
#define TOL07_MSG	"Samstag"
#define TOL08_MSG	"Momentchen"
#endif
#ifdef STRING
#undef STRING
#endif

#ifdef LONGSTRING
#undef LONGSTRING
#endif

#define STRING 81
#define LONGSTRING 256

UNSIGNED char headinfo[STRING];



/* #WH - Find the name of your shell account.

   This function returns the name of a shell account. */

UNSIGNED char *whoami()
{
  struct passwd *pw_entry;

  pw_entry = getpwuid(getuid());

  if (pw_entry->pw_uid < 0) return (UNSIGNED char *) "OOPS";

  return (UNSIGNED char *) (pw_entry->pw_name);
}




/* #SC - Cut a part out of a string [st].

   [v] describes the first position in [st], [b] stands for the last
   position. 

   This function returns the part between [v] and [b] as a string. */

UNSIGNED char *strcopy(st, v, b)
UNSIGNED char st[];
int v, b;
{
  static UNSIGNED char s[STRING];

  int i = 0, j;

#ifdef _PROFILE
  profile( "strcopy()" );
#endif
 
  if(b > STRING) b = STRING;

  for (j = v; j <= b; j++) {
	s[i] = st[j];
	i++;
  }
  s[i] = '\0';

  return (UNSIGNED char *) s;
}




/* #SP - Delete all leading and tailing blanks in string [st].

   This function returns the string without blanks. */

UNSIGNED char *stripped(st)
UNSIGNED char st[];
{
  UNSIGNED char *s = st;

  int i = 0;

#ifdef _PROFILE
  profile( "stripped()" );
#endif

  if (st[0] == '\0') 
	return (UNSIGNED char *) "\0";

  if ((st[0] == 10) || (st[0] == 13)) 
	return (UNSIGNED char *) " ";

  while((s[0] < 33) && (s[0] != '\0')) s++;
  i = strlen(s);
  while(s[i] < 33) i--;
  s[(i+1)] = '\0';

  return (UNSIGNED char *) s;
}




/* #UP - Upcaseing all characters in string [st].

   This function returns the upcased string. */

UNSIGNED char *upcased(st)
unsigned char st[];		/* !!! unsigned, NOT UNSIGNED !!! */
{
  static UNSIGNED char s[STRING];

  int i = 0;

#ifdef _PROFILE
  profile( "upcased()" );
#endif

  strcpy(s, (UNSIGNED char *) strcopy(st, 0, STRING));

  while (s[i] != '\0') {
	if (s[i] > 127) {
		if (s[i] == 132) s[i] = 142; /* IBM/DOS, ATARI ST */
		if (s[i] == 148) s[i] = 153;
		if (s[i] == 122) s[i] = 154;
		if (s[i] == 228) s[i] = 196; /* ISO-8859-1, AMIGA */
		if (s[i] == 246) s[i] = 214;
		if (s[i] == 252) s[i] = 220;
	}
	else{
		if (s[i] > 96) s[i] -= 32;
	}
	i++;
  }

  return (UNSIGNED char *) s;
}




/* #BC - Cut a part out of a long string [st].

   [v] describes the first position in [st], [b] stands for the last
   position. 

   This function returns the part between [v] and [b] as a long string. */

UNSIGNED char *bigcopy(st, v, b)
UNSIGNED char st[];
int v, b;
{
  static UNSIGNED char s[LONGSTRING];

  int i = 0, j;

#ifdef _PROFILE
  profile( "bigcopy()" );
#endif

  if (b > LONGSTRING) b = LONGSTRING;

  for (j = v; j <= b; j++) {
	s[i] = st[j];
	i++;
  }
  s[i] = '\0';

  return (UNSIGNED char *) s;
}





/* #ST - Compare two strings [s, t] case insensitive.

  This function returns the number of different characters. */

int strcomp(s, t)
UNSIGNED char s[], t[];
{
  int i = 0, a = 0;
  UNSIGNED char s1, t1;

#ifdef _PROFILE
  profile( "strcomp()" );
#endif

  while (s[i] != '\0') {
	s1 = s[i];
	t1 = t[i];
	if (t1 > 96) 
		if (t1 < 123) t1 -= 32;
	if (s1 > 96)
		if (s1 < 123) s1 -= 32;
	if (t1 != s1) a++;
	i++;
  }

  return (int) a;
}





/* #AN - Get a termcap entry or convert a command to terminal code.

   [code]   ->  'INIT'   = get a complete termcap entry and store it
                command  = find the command and show the code

   [x, y]   ->  if command is 'cm' (Cursor move), than the cursor is
                moved to [y],[x].

   This function returns -1 if terminal is not known, 1 if command 
   couldn't be found, or 0 is everything is ok. */

static UNSIGNED char buf[1024];


int ansi(code)			/* for compatibility reasons */
UNSIGNED char code[];
{

#ifdef _PROFILE
  profile( "ansi()" );
#endif

  ansi2(code, 0, 0);
}

int ansi2(code, x, y)
UNSIGNED char code[];
int x, y;
{
  UNSIGNED char *getenv();
  static UNSIGNED char s[STRING];
  UNSIGNED char *p = s;
  UNSIGNED char *term;

#ifdef _PROFILE
  profile( "ansi2()" );
#endif

#ifdef _MBOX
  term = TERMINAL;
#else
  term = getenv("TERM");
#endif

  if ((x == -1) && (y == -1)) {
	if (tgetent(buf, code) != 1) return -1;
  }
  if (strcomp("INIT", code) == 0) {
	if (tgetent(buf, term) != 1) return -1;
  }
  
  
   
  if (strcomp("cm", code) == 0) {
	if (tgetstr(code, (char **) &p) == 0) return 1;

  if((s[0] == '2') || (s[0] == '5') ||(s[0] == '3')){		/* For the new ''VT100'' termcap enties ;-( */
	strcpy(s, (UNSIGNED char *) strcopy(s, 1, STRING));
  }	
	printf("%s", (UNSIGNED char *) tgoto( s, (x -1), (y -1) ));
	return 0;
  }
  else{
	if (tgetstr(code, (char **) &p) == 0) return 1;
  }

  if((s[0] == '2') || (s[0] == '5') ||(s[0] == '3')){		/* For the new ''VT100'' termcap enties ;-( */
	strcpy(s, (UNSIGNED char *) strcopy(s, 1, STRING));
  }

  printf("%s", s);


  return 0;
}



/* #TA - Get a termcap entry or convert a command to terminal code.

   [code]   ->  'INIT'   = get a complete termcap entry and store it
                command  = find the command and show the code

   This function returns the terminal code or '\0'. */

UNSIGNED char *termansi(code)
UNSIGNED char code[];
{
  UNSIGNED char *getenv();
  UNSIGNED char s[STRING],t[10];
  UNSIGNED char *p = s;
  UNSIGNED char *term;

#ifdef _PROFILE
  profile( "termansi()" );
#endif

#ifdef _MBOX
  term = TERMINAL;
#else
  term = getenv("TERM");
#endif

  t[0]=0;
  if (strcomp("INIT", code) == 0) {
	if (tgetent(buf, term) != 1) return ((UNSIGNED char *) &t);
  }
  
  if (tgetstr(code, &p) == 0) return ((UNSIGNED char *) &t);

 if((s[0] == '2') || (s[0] == '5') ||(s[0] == '3'))
 {           /* For the new vt100 */
   strcpy(s, (UNSIGNED char *) strcopy(s, 1, STRING));
 }
              
  return (UNSIGNED char *) s;
}





/* #MD - Get the date.

   [mode]  ->  0 = tt.mm.yyyy
               1 = tt.nn.yy
               2 = ttt

   This function returns a date string. */


#ifndef TZS
#define TZS	1
#endif

UNSIGNED char *mydate(mode)
int mode;
{
  struct tm *tp;
  time_t tt;
  UNSIGNED char wt[8][11];

  static UNSIGNED char s[STRING];

#ifdef _PROFILE
  profile( "mydate()" );
#endif

#ifndef _MBOX
  strcpy(wt[0], "Sonntag");
  strcpy(wt[1], "Montag");
  strcpy(wt[2], "Dienstag");
  strcpy(wt[3], "Mittwoch");
  strcpy(wt[4], "Donnerstag");
  strcpy(wt[5], "Freitag");
  strcpy(wt[6], "Samstag");
#else
  strcpy(wt[0], TOL01_MSG);
  strcpy(wt[1], TOL02_MSG);
  strcpy(wt[2], TOL03_MSG);
  strcpy(wt[3], TOL04_MSG);
  strcpy(wt[4], TOL05_MSG);
  strcpy(wt[5], TOL06_MSG);
  strcpy(wt[6], TOL07_MSG);
#endif

  time(&tt);
  tzset();
  tp = localtime(&tt);
  if (mode == 0) {
	sprintf(s, "%02.2d.%02.2d.%04.4d", tp->tm_mday, tp->tm_mon + TZS, 1900 + tp->tm_year);
  }
  if (mode == 1) {
	sprintf(s, "%02.2d.%02.2d.%02.2d", tp->tm_mday, tp->tm_mon + TZS, tp->tm_year);
  }
  if (mode == 2) {
	sprintf(s, "%s", wt[tp->tm_wday]);
  }
  return (UNSIGNED char *) s;
}




/* #MT - Get the time.

   [mode]  ->  0 = hh:mm:ss
               1 = hh:mm

   This function returns a time string. */

UNSIGNED char *mytime(mode)
int mode;
{
  struct tm *tp;
  time_t tt;

  static UNSIGNED char s[STRING];


#ifdef _PROFILE
  profile( "mytime()" );
#endif

  time(&tt);
  tp = localtime(&tt);
  sprintf(s, "%02.2d:%02.2d:%02.2d", tp->tm_hour, tp->tm_min, tp->tm_sec);

  if (mode == 1) s[5] = '\0';

  return (UNSIGNED char *) s;
}






/* #CY - Encrypt a string [st].

   This function returns the encrypted string. */

UNSIGNED char *crypted(st)
UNSIGNED char st[];
{
  static UNSIGNED char s[STRING];
  int i = 0, a = 2; 

  strcpy(s, (UNSIGNED char *) strcopy(st, 0, STRING));

  while (s[i] != '\0') {
	s[i] = (s[i] / a) + 32;
	i++;
	a++;
	if (a > 5) a = 2;
  }

  return (UNSIGNED char *) s;
}






/* #DC - Convert a date string [d] into a (long).

   This function returns the date as a (long). */   

long dateconv(d)
UNSIGNED char d[];
{
  UNSIGNED char s[STRING];
  int a, b, i;

  strcpy(s, (UNSIGNED char *) strcopy(d, 0, 1));
  a = atoi(s);
  strcpy(s, (UNSIGNED char *) strcopy(d, 3, 4));
  b = atoi(s);
  strcpy(s, (UNSIGNED char *) strcopy(d, 6, 9));
  i = atoi(s);
  if (i < 1900) i += 1900;

  sprintf(s, "%4.4d%02.2d%02.2d", i, b, a);

  return atol(s);
}




/* #DR - Converting a (long) date into a string.

   This function returns a date string. */

UNSIGNED char *datereconv(l)
long l;
{
  static UNSIGNED char s[STRING];
  static UNSIGNED char t[STRING];


  if (l < 19000000)
	sprintf(s, "%ld", 1900000L + l);
  else
	sprintf(s, "%ld", l);

  strcpy(t, (UNSIGNED char *) strcopy(s, 6, 7));
  strcat(t, ".");
  strcat(t, (UNSIGNED char *) strcopy(s, 4, 5));
  strcat(t, ".");
  strcat(t, (UNSIGNED char *) strcopy(s, 0, 3));

  return (UNSIGNED char *) t;
}




/* #TC - Convert a time string [t] into a (long).

   This function returns the time as a (long). */   

int timeconv(t)
UNSIGNED char t[];
{
  UNSIGNED char s[STRING];
  int i = 0;


  strcpy(s, (UNSIGNED char *) strcopy(t, 0, 1));
  i = 100 * atoi(s);
  strcpy(s, (UNSIGNED char *) strcopy(t, 3, 4));
  i += atoi(s);

  return i;
}





/* #TR - Converting a (long) time into a string.

   This function returns a time string. */

UNSIGNED char *timereconv(i)
int i;
{
  static UNSIGNED char s[STRING];
  static UNSIGNED char t[STRING];


  if (i < 10000)
	sprintf(s, "%04.4d00", i);
  else
	sprintf(s, "%06.6d", i);

  strcpy(t, (UNSIGNED char *) strcopy(s, 0, 1));
  strcat(t, ":");
  strcat(t, (UNSIGNED char *) strcopy(s, 2, 3));
  strcat(t, ":");
  strcat(t, (UNSIGNED char *) strcopy(s, 4, 5));

  return (UNSIGNED char *) t;
}




/* #SN - Converting a long filename [longname] into a 14-char name.

       comp.protocols.tcp-ip.sources.wanted
   ==> comp0ces.wanted

   This function returns the shortened name. */

UNSIGNED char *shortname( longname )
UNSIGNED char longname[];
{
  UNSIGNED char s[STRING];
  UNSIGNED char t[STRING];

  int i = 0, j = 0;

  strcpy(s, longname);

  strcpy(t, longname);

  while(s[i] != '\0'){
	if(s[i] == '/') j = i + 1;
	i++;
  }

  i = strlen(t);
  
  if((i - j) < 15) return (UNSIGNED char *) s;
  
  sprintf(s, "%s%d", (UNSIGNED char *) strcopy(s, j, j + 1), (i - 14));
  strcat(s, (UNSIGNED char *) strcopy(t, (i-12), i));
  s[14] = '\0';

  return (UNSIGNED char *) s;
}




/* #ML - Check whether a file [name] is locked or simply doesn't exist.

   [mode] shouldn't be used!

   This function returns 0 if a file is locked and -1 if file doesn't
   exist. */
 
int maybe_locked(name, mode)
UNSIGNED char name[], mode[];
{
  FILE *fp;
  int i = 0, LCK = 0;
  UNSIGNED char s[STRING];

#ifdef _PROFILE
  profile( "mytime()" );
#endif

  sprintf(s, "%s.LCK", name);

#ifndef _LONGNAMES
  strcpy(s, (UNSIGNED char *) shortname(s));
#endif

  TRY_TWICE:

  fp = NULL;

  do {
	if(fp != NULL) fclose(fp);
	fp = fopen(s, "r");
	i++;
	if (fp != NULL) {
		if(i == 1) printf("%c%s ", CR, TOL08_MSG);
		printf(".");
		sleep(3);
	}
  } while ((i < 8) && (fp != 0));

  if(fp != 0){
	fclose(fp);
#ifdef _MBOX
	mbunlock(name);  /* <--- DAS IST NATUERLICH NICHT KORREKT !!! */
	goto TRY_TWICE;  /* <--- DAS AUCH NICHT !!! ABER ES FUNXT ;-) */
	nerror("tools.c", 566, "maybe_locked", "File locked", name);	
#else
	printf("\n\nDatei >>%s<< gesperrt !!!\n\n", name);
#endif
	exit( -1 );
  }

  i = 0;

  do {
	fp = fopen(name, "r");	/* "r" => "mode", aber VORSICHT ! */
	i++;
	if (fp == NULL) {
		if(i == 1) printf("%c%s ", CR, TOL08_MSG);	
		printf(".");
		sync();
		sleep(3);
	}
  } while ((i < 5) && (fp == NULL));

  if (fp == NULL) return -1;

  fclose(fp);
  return 0;
}




/* #NS - Strip of all numerics of a string [s].

   This function returns the stripped string. */

UNSIGNED char *numstripped(s)
UNSIGNED char s[];
{
  static UNSIGNED char t[STRING];
  int i = 0, a = 0;

  while (s[i] != '\0') {
	if (((s[i] > 64) && (s[i] < 127)) || (s[i] == ' ') || (s[i] == '-')) {
		t[a] = s[i];
		a++;
	}
	i++;
  }
  t[a] = '\0';

  return (UNSIGNED char *) t;
}




/* #MR - Rename file [alt] to file [neu].

   This function returns 0 if everything went ok, or -1 if there are
   any problems. */

int mbrename(alt, neu)
char *alt[], *neu[];
{
  FILE *fp;
  FILE *ff;

  UNSIGNED char s[LONGSTRING];

  if(rename(alt, neu) == NULL) return 0;

  fp = fopen(neu, "w");
  if (fp == NULL) {
	return -1;
  }
  ff = fopen(alt, "r");
  if (ff == NULL) {
	return -2;
  }
  
  while(fgets(s, LONGSTRING, ff) != NULL)
	fputs(s, fp);

  fclose(fp);
  fclose(ff);

  unlink(alt);

  return 0;
}






/* #HL - Display a text [line] as headline. */

void headline(line)
UNSIGNED char line[];
{
  UNSIGNED char ex[255];
  int i;


  strcpy(ex, line);
  i = 0;
  while(line[i] != '\0'){
	if(line[i] == '^'){
		strcpy(ex, (UNSIGNED char *) strcopy(line, 0, (i-1)));
		line[i] = ' ';
	}
	i++;
  }
  strcpy(headinfo, ex);
  sprintf(ex, "%s                                                                          ", line);
  ex[79] = '\0';
  if (ansi2("cl", 0, 0) != -1) {
	if (ansi2("mr", 0, 0) == 1) {
		if (ansi2("so", 0, 0) == 1) { 
			printf("\n\n------  %s  ------\n\n", line);
			return;
		}
	}
	printf("%s\n", ex);
	if (ansi2("me", 0, 0) == 1) {
		ansi2("se", 0, 0);
	}
  }
  else {
	printf("\n\n\n\n\n------  %s  ------\n\n", line);
  }
}




/* #MO - Lock a file [path]. */

void mblock( path )
UNSIGNED char path[];
{
  FILE *fp;
  UNSIGNED char s[STRING];

  sprintf(s, "%s.LCK", path); 

#ifndef _LONGNAMES
  strcpy(s, (UNSIGNED char *) shortname(s));
#endif

  fp = fopen( s, "w" );
  if(fp != NULL){
	fputs(VERSION, fp);
	fclose(fp);
  }
}





/* #MU - Unlock a file [path]. */

void mbunlock( path )
UNSIGNED char path[];
{
  UNSIGNED char s[STRING];
  
  sprintf(s, "%s.LCK", path); 

#ifndef _LONGNAMES
  strcpy(s, (UNSIGNED char *) shortname(s));
#endif

  unlink(s);
}




/* #TY - Find the number of a terminal.

   This function returns the number of a terminal. */

int tty()
{
  UNSIGNED char s[STRING];
  UNSIGNED char t[STRING];
  int i, l, a, b;


  strcpy(s, (UNSIGNED char *) ttyname(0));
  l = strlen(s);
  i = l;

  while ((s[l] != 'y') && (l > 0)) l--;

  l++;
  a = 0;

  for (b = l; b < i; b++) {
	t[a] = s[b];
	a++;
  }
  t[a] = '\0';

  return atoi(t);
}




/* #TN - Find the name of a terminal.

   This function returns the name of a terminal. */

UNSIGNED char *ttyna()
{
  static UNSIGNED char s[STRING];
  static UNSIGNED char t[STRING];
  int i, j, l;


  strcpy(s, (UNSIGNED char *) ttyname(0));

  l = strlen(s) + 1;

  i = 0; 
  while(s[i] != '\0'){
	if(s[i] == '/') j = i + 1;
	i++;
  }
  (void) strcpy(t, (UNSIGNED char *) strcopy(s, j, l));
  
  return (UNSIGNED char *) t;
}




/* #CL - Clear a screenline. */
 
void clearline()
{
  printf("%c", CR);
  if (ansi2("ce", 0, 0) == 1) {
	printf("                                                               ");
  }
  printf("%c", CR);
}




/* #IN - Check if a character [c] is in a string [pattern].

   This function returns 0 if the char isn't included or a number > 0
   otherwise. */

int isin( pattern, c )
UNSIGNED char pattern[];
int c;
{
  int i = 0;
  int ok = 0;

  while((pattern[i] != '\0') && (ok == 0)){
	if(pattern[i] == c) ok++;	
	i++;
  }
  
  return (int) ok;  
}





/* #SG - Check if a pattern [pattern] is in a string [source].

   This function returns 0 if the sign isn't included or a number > 0
   otherwise. */

int sgrep( source, pattern )
UNSIGNED char source[];
UNSIGNED char pattern[];
{
  UNSIGNED char t[STRING];
  int i = 0;
 
  while(source[i] != 0){
	if(source[i] == pattern[0]){
		strcpy(t, (UNSIGNED char *) strcopy(source, i, strlen(source)));
		if(strcomp(pattern, t) == 0) return (int) i;
	}
	i++;
  }

  return (int) 0;
}




/* Position eines Strings in einem anderen -> strpos alias PAS */
int strpos(s, t)
UNSIGNED char *s, *t;
{
  int j=0, i = 0, a = 0;


  while (t[i] != 0) {
	j=0;
	if (t[i] == s[0]){
	    a=i+1;
	    while ((s[j]==t[i])&&(s[j]!=0)) {
	       j++; i++;
	     }
	    if (s[j]==0) return (int) a ; else a='0';
	 }
	i++;
  }

  return ( 0);
}

