/* ripwraps.h: RIPscrip API
 *
 * Eric P. Scott, San Francisco State University, December 1993
 * Copyright 1993 by Eric P. Scott.  All rights reserved.
 *
 * This software is "freeware" but is not in the public domain.
 *
 * The author grants you a nonexclusive royalty-free license to copy,
 * redistribute, and use this software for any purpose provided that
 * source distributions retain this entire copyright and license.
 * You may not sublicense or distribute modified versions to anyone else
 * except as part of another product or program, and modified versions
 * must be so labeled.
 *
 * You are encouraged to give proper credit to the author in binary
 * distributions and associated documentation.
 *
 * This software is provided "as is" with no warranties of any kind,
 * including the warranties of merchantability and fitness for a
 * particular purpose.
 *
 * RIPscrip is a trademark of TeleGrafix Communications, Inc.
 *
 * This software is not supported or endorsed by TeleGrafix.
 *
 */

/*
 * rip_* functions correspond to commands described in _RIPscrip
 * Graphics Protocol Specification_, Revision 1.54, by TeleGrafix
 * Communications, Inc.
 *
 * ripv_* functions are equivalent to their rip_* counterparts,
 * but take an argument vector in place of an inline list.
 *
 * ripio_* functions are specific to this library.
 *
 *
 * 1.0  2/ 8/94  EPS  Initial Release
 * 1.1  4/11/94  EPS  Fixed rip_button()
 *
 */

/* ----- infrastructure ----- */

/* [internal] output a single character */
extern int ripio_putchar(int c);

/* [internal] output a C string; no newline added */
extern int ripio_putstr( char *string);

/* allow user to specify an alternative character output function */
extern void ripio_set_putc(int (*func)(int, void *), void *arg);

/* return current alternative character output function pointer */
extern int (*ripio_get_putc_func(void))(int, void *);

/* return current alternative character output function argument */
extern void *ripio_get_putc_arg(void);

/* set screen width (for command wrapping) */
extern void ripio_set_width(unsigned int width);

/* obtain current screen width setting */
extern unsigned int ripio_get_width(void);

/* set autoflush flag */
extern void ripio_set_autoflush(int autoflush);

/* obtain current autoflush setting */
extern int ripio_get_autoflush(void);

/* [internal] begin a RIPscrip command */
extern void ripio_command(int baggage);

/* end RIPscrip command line */
extern void ripio_flush(void);

/* [internal] output an unsigned int in base 36 */
extern void ripio_meganum(unsigned int value, int digits);

/* [internal] output a text string */
extern void ripio_text( char *string);

/* Query RIPscrip version number */
extern void ripio_query(void);

/* Disables all RIPscrip processing */
extern void ripio_disable(void);

/* Enables RIPscrip processing */
extern void ripio_enable(void);

/* ----- RIPscrip 1.54 functions ----- */

/* Define the size and location of the Text Window */
extern  void rip_text_window(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1, unsigned int wrap, unsigned int size);

/* Define the size & location of the Graphics Window */
extern void rip_viewport(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1);

/* Clear Graphics/Text Windows & reset to full screen */
extern void rip_reset_windows(void);

/* Clears Text Window to current background color */
extern void rip_erase_window(void);

/* Clear Graphics Window to current background color */
extern void rip_erase_view(void);

/* Move text cursor to row & column in Text Window */
extern void rip_gotoxy(unsigned int x, unsigned int y);

/* Move cursor to upper-left corner of Text Window */
extern void rip_home(void);

/* Erase current line from cursor to end of line */
extern void rip_erase_eol(void);

/* Set current Drawing Color for graphics */
extern void rip_color(unsigned int color);
#define RIP_BLACK 0
#define RIP_BLUE 1
#define RIP_GREEN 2
#define RIP_CYAN 3
#define RIP_RED 4
#define RIP_MAGENTA 5
#define RIP_BROWN 6
#define RIP_LIGHTGRAY 7
#define RIP_DARKGRAY 8
#define RIP_LIGHTBLUE 9
#define RIP_LIGHTGREEN 10
#define RIP_LIGHTCYAN 11
#define RIP_LIGHTRED 12
#define RIP_LIGHTMAGENTA 13
#define RIP_YELLOW 14
#define RIP_WHITE 15
#define RIP_BACKGROUND 0

/* Set 16-color Palette from master 64-color palette */
extern void rip_set_palette(unsigned int c1, unsigned int c2, unsigned int c3,
    unsigned int c4, unsigned int c5, unsigned int c6, unsigned int c7,
    unsigned int c8, unsigned int c9, unsigned int c10, unsigned int c11,
    unsigned int c12, unsigned int c13, unsigned int c14,unsigned int c15,
    unsigned int c16);
#define RIP_EGA_BLACK 0
#define RIP_EGA_BLUE 1
#define RIP_EGA_GREEN 2
#define RIP_EGA_CYAN 3
#define RIP_EGA_RED 4
#define RIP_EGA_MAGENTA 5
#define RIP_EGA_BROWN 20
#define RIP_EGA_LIGHTGRAY 7
#define RIP_EGA_DARKGRAY 56
#define RIP_EGA_LIGHTBLUE 57
#define RIP_EGA_LIGHTGREEN 58
#define RIP_EGA_LIGHTCYAN 59
#define RIP_EGA_LIGHTRED 60
#define RIP_EGA_LIGHTMAGENTA 61
#define RIP_EGA_YELLOW 62
#define RIP_EGA_WHITE 63

/* Set color of 16-color Palette from Master Palette */
extern void rip_one_palette(unsigned int color, unsigned int value);

/* Set drawing mode for graphics primitives */
extern void rip_write_mode(unsigned int mode);
#define RIP_COPY_PUT 0
#define RIP_XOR_PUT 1

/* Move the current drawing position to (X,Y) */
extern void rip_move(unsigned int x, unsigned int y);

/* Draw text in current font/color at current spot */
extern void rip_text( char *text_string);

/* Draw text in current font/color at specific spot */
extern void rip_text_xy(unsigned int x, unsigned int y,
     char *text_string);

/* Select current font style, orientation and size */
extern void rip_font_style(unsigned int font, unsigned int direction,
    unsigned int size, unsigned int res);
#define RIP_DEFAULT_FONT 0
#define RIP_TRIPLEX_FONT 1
#define RIP_SMALL_FONT 2
#define RIP_SANS_SERIF_FONT 3
#define RIP_GOTHIC_FONT 4
#define RIP_SCRIPT_FONT 5
#define RIP_SIMPLE_FONT 6
#define RIP_TRIPLEX_SCRIPT_FONT 7
#define RIP_COMPLEX_FONT 8
#define RIP_EUROSTYLE_FONT 9
#define RIP_BOLD_FONT 10
#define RIP_HORIZ_DIR 0
#define RIP_VERT_DIR 1

/* Draws one pixel using current drawing color */
extern void rip_pixel(unsigned int x, unsigned int y);

/* Draw a line in the current color/line style */
extern void rip_line(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1);

/* Draw a rectangle in current color/line style */
extern void rip_rectangle(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1);

/* Draw filled rectangle with fill color/pattern */
extern void rip_bar(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1);

/* Draw circle in current color and line thickness */
extern void rip_circle(unsigned int x_center, unsigned int y_center,
    unsigned int radius);
#define RIP_EGA_ASPECT_RATIO (31.0/40.0)

/* Draw elliptical arc in current color/line style */
extern void rip_oval(unsigned int x, unsigned int y, unsigned int st_ang,
    unsigned int end_ang, unsigned int x_rad, unsigned int y_rad);

/* Draw filled ellipse using current color/pattern */
extern void rip_filled_oval(unsigned int x_center, unsigned int y_center,
    unsigned int x_rad, unsigned int y_rad);

/* Draw circular arc in current color/line thickness */
extern void rip_arc(unsigned int x, unsigned int y, unsigned int st_ang,
    unsigned int end_ang, unsigned int radius);

/* Draw an elliptical arc */
extern void rip_oval_arc(unsigned int x, unsigned int y, unsigned int st_ang,
    unsigned int e_ang, unsigned int radx, unsigned int rady);

/* Draws a circular pie slice */
extern void rip_pie_slice(unsigned int x, unsigned int y,
    unsigned int start_ang, unsigned int end_ang, unsigned int radius);

/* Draws an elliptical pie slice */
extern void rip_oval_pie_slice(unsigned int x, unsigned int y,
    unsigned int st_ang, unsigned int e_ang, unsigned int radx,
    unsigned int rady);

/* Draw a bezier curve */
extern void rip_bezier(unsigned int x1, unsigned int y1, unsigned int x2,
    unsigned int y2, unsigned int x3, unsigned int y3, unsigned int x4,
    unsigned int y4, unsigned int cnt);

/* Draw polygon in current color/line-style */
extern void rip_polygon(unsigned int npoints, ...);

/* Draw filled polygon in current color/fill pattern */
extern void rip_fill_polygon(unsigned int npoints, ...);

/* Draw a Poly-Line (multi-faceted line) */
extern void rip_polyline(unsigned int npoints, ...);

/* Flood fill screen area with current fill settings */
extern void rip_fill(unsigned int x, unsigned int y, unsigned int border);

/* Defines a line style and thickness */
extern void rip_line_style(unsigned int style, unsigned int user_pat,
    unsigned int thick);
#define RIP_SOLID_LINE 0
#define RIP_DOTTED_LINE 1
#define RIP_CENTER_LINE 2
#define RIP_DASHED_LINE 3
#define RIP_USERBIT_LINE 4
#define RIP_NORM_WIDTH 1
#define RIP_THICK_WIDTH 3

/* Set current fill style (predefined) & fill color */
extern void rip_fill_style(unsigned int pattern, unsigned int color);
#define RIP_EMPTY_FILL 0
#define RIP_SOLID_FILL 1
#define RIP_LINE_FILL 2
#define RIP_LTSLASH_FILL 3
#define RIP_SLASH_FILL 4
#define RIP_BKSLASH_FILL 5
#define RIP_LTBKSLASH_FILL 6
#define RIP_HATCH_FILL 7
#define RIP_XHATCH_FILL 8
#define RIP_INTERLEAVE_FILL 9
#define RIP_WIDE_DOT_FILL 10
#define RIP_CLOSE_DOT_FILL 11

/* Set user-definable (custom) fill pattern/color */
extern void rip_fill_pattern(unsigned int c1, unsigned int c2, unsigned int c3,
    unsigned int c4, unsigned int c5, unsigned int c6, unsigned int c7,
    unsigned int c8, unsigned int col);

/* Defines a rectangular hot mouse region */
extern void rip_mouse(unsigned int num, unsigned int x0, unsigned int y0,
    unsigned int x1, unsigned int y1, unsigned int clk, unsigned int clr, 
    unsigned int res,  char *text);

/* Destroys all previously defined hot mouse regions */
extern void rip_kill_mouse_fields(void);

/* Define a rectangular text region */
extern void rip_begin_text(unsigned int x1, unsigned int y1, unsigned int x2,
    unsigned int y2, unsigned int res);

/* Display a line of text in rectangular text region */
extern void rip_region_text(unsigned int justify,  char *text_string);

/* End a rectangular text region */
extern void rip_end_text(void);

/* Copy rectangular image to clipboard (as icon) */
extern void rip_get_image(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1, unsigned int res);

/* Pastes the clipboard contents onto the screen */
extern void rip_put_image(unsigned int x, unsigned int y, unsigned int mode,
    unsigned int res);
#define RIP_OR_PUT 2
#define RIP_AND_PUT 3
#define RIP_NOT_PUT 4

/* Write contents of the clipboard (icon) to disk */
extern void rip_write_icon(unsigned int res,  char *filename);

/* Loads and displays a disk-based icon to screen */
extern void rip_load_icon(unsigned int x, unsigned int y, unsigned int mode,
    unsigned int clipboard, unsigned int res,  char *filename);

/* Button style definition */
extern void rip_button_style(unsigned int wid, unsigned int hgt,
    unsigned int orient, unsigned int flags, unsigned int bevsize,
    unsigned int dfore, unsigned int dback, unsigned int bright,
    unsigned int dark, unsigned int surface, unsigned int grp_no,
    unsigned int flags2, unsigned int uline_col, unsigned int corner_col,
    unsigned int res);

/* Define a Mouse Button */
extern void rip_button(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1, unsigned int hotkey, unsigned int flags,
    unsigned int res,  char *text);

/* Define a text variable */
extern void rip_define(unsigned int flags, unsigned int res,  char *text);

/* Query the contents of a text variable */
extern void rip_query(unsigned int mode, unsigned int res,  char *text);

/* Copy screen region up/down */
extern void rip_copy_region(unsigned int x0, unsigned int y0, unsigned int x1,
    unsigned int y1, unsigned int res, unsigned int dest_line);

/* Playback local .RIP file */
extern void rip_read_scene(unsigned int res,  char *filename);

/* Query existing information on a particular file */
extern void rip_file_query(unsigned int mode, unsigned int res,
     char *filename);

/* Enter block transfer mode with host */
extern void rip_enter_block_mode(unsigned int mode, unsigned int proto,
    unsigned int file_type, unsigned int res,  char *filename);

/* End of RIPscrip Scene */
extern void rip_no_more(void);

/* ----- vector interfaces ----- */

/* Set 16-color Palette from master 64-color palette */
extern void ripv_set_palette(unsigned char colors[16]);

/* Draw polygon in current color/line-style */
extern void ripv_polygon(unsigned int npoints, unsigned short *polypoints);

/* Draw filled polygon in current color/fill pattern */
extern void ripv_fill_polygon(unsigned int npoints,
    unsigned short *polypoints);

/* Draw a Poly-Line (multi-faceted line) */
extern void ripv_polyline(unsigned int npoints, unsigned short *polypoints);

/* Set user-definable (custom) fill pattern/color */
extern void ripv_fill_pattern(unsigned char upattern[8], unsigned int color);
