/*
 *	(c) Copyright 1990, Kim Fabricius Storm.  All rights reserved.
 *
 *	Global declarations and auxiliary functions.
 */

#include <signal.h>
#include <errno.h>
#include <pwd.h>
#include "config.h"

export char *home_directory;
export char *nn_directory;		/* ~/.nn */
export char *news_directory;		/* /usr/spool/news */
export char *news_lib_directory; 	/* /usr/lib/news */
export char *lib_directory;		/* /usr/local/lib/nn */
export char *master_directory;		/* = lib */
export char *help_directory;		/* = lib/help */
export char *bin_directory = BIN_DIRECTORY;

export char *db_directory;	/* /usr/spool/nn or NEWS_DIR/.nn */
export char *db_data_directory;	/* ..../DATA     or undefined    */
export int db_data_subdirs = 0;	/* set if DATA/[0-9]/ exist	 */

export char *news_active;	/* NLIB/active or DB/ACTIVE */

export char *pager;

export char *log_file;			/* = lib/Log */
export char *log_entry_filter = NULL;

export char *temp_file;

#define TMP_DIRECTORY "/usr/tmp"
export char *tmp_directory = TMP_DIRECTORY;

export char version_id[32];


export unsigned short user_eid;
export unsigned short user_id, group_id;
export int process_id;
export int who_am_i;
export int dont_write_console = 0;
export int mail_errors_mode = 2;



init_global()
{
    unsigned short getuid(), getgid();
    int getpid();
    int suspend_nn();
    int i;

#ifdef FAKE_INTERRUPT
    for (i = 0; i < NSIG; i++) fake_keyb_siglist[i] = i == 2 ? 1 : 0;
#endif


    user_id = getuid();

#ifdef HAVE_MULTIGROUP
    ngroups = getgroups(NGROUPS, gidset);	/* Get users's group set */
#endif
    group_id = getegid();
    user_eid = geteuid();

    process_id = getpid();

#ifdef CLIENT_DIRECTORY
    lib_directory = CLIENT_DIRECTORY;
#else
    lib_directory = LIB_DIRECTORY;
#endif

#ifdef NEWS_DIRECTORY
    news_directory = NEWS_DIRECTORY;
#else
    news_directory = "/usr/spool/news";
#endif

#ifdef DB_DIRECTORY
    db_directory = DB_DIRECTORY;
#else
    db_directory = mk_file_name(news_directory, ".nn");
#endif

#ifdef ACCOUNTING
    if (who_am_i == I_AM_ACCT)
	return 0;
#endif

#ifdef DB_DATA_DIRECTORY
    db_data_directory = DB_DATA_DIRECTORY;
#else
#ifdef DB_DIRECTORY
    db_data_directory = mk_file_name(db_directory, "DATA");
#else
    db_data_directory = NULL;
#endif
#endif
#ifndef DB_LONG_NAMES
    if (db_data_directory != NULL)
	db_data_subdirs = file_exist(relative(db_data_directory, "0"), "dx");
#endif

#ifdef NEWS_LIB_DIRECTORY
    news_lib_directory = NEWS_LIB_DIRECTORY;
#else
    news_lib_directory = "/usr/lib/news";
#endif

    /* this may later be changed by nntp_check */
    news_active = mk_file_name(news_lib_directory, "active");

#ifdef MASTER_DIRECTORY
    master_directory = MASTER_DIRECTORY;
#else
    master_directory = LIB_DIRECTORY;
#endif





    return 0;
}

new_temp_file()
{
    static char buf[FILENAME];
    static char *temp_dir = NULL;

    if (temp_dir == NULL)
	if ((temp_dir = getenv("TMPDIR")) == NULL)
	    temp_dir = tmp_directory; /* just to make test above false */
	else
	    tmp_directory = temp_dir;

    sprintf(buf, "%s/nn.XXXXXX", tmp_directory);
    mktemp(buf);
    temp_file = buf;
}


FILE *open_file(name, mode)
char *name;
int mode;
{
    FILE *f;
    int fd;
    extern int errno;

    if ((mode & DONT_CREATE) && !file_exist(name, (char *)NULL))
	return NULL;

    switch (mode & 0x0f) {

     case OPEN_READ:

	f = fopen(name, "r");
	break;

     case OPEN_UPDATE:

/*	f = fopen(name, "r+"); 	-- not reliable on many systems (sigh) */

	if ((fd = open(name, O_WRONLY)) >= 0) {
	    if ((f = fdopen(fd, "w")) != NULL) return f;
	    close(fd);
	}

	/* fall thru */

     case OPEN_CREATE:

	f = fopen(name, "w");
	break;

     case OPEN_APPEND:

	f = fopen(name, "a");
	break;

     case OPEN_CREATE_RW:

	f = fopen(name, "w+");	/* not safe on all systems -- beware */
	break;

     default:

	printf("Illegal mode: open_file(%s, 0x%x)", name, mode);
	sys_error("Illegal mode: open_file(%s, 0x%x)", name, mode);
    }

    if (f) {
	if (mode & OPEN_UNLINK) unlink(name);
	return f;
    }

    if ((mode & MUST_EXIST) == 0) return NULL;

    nerror("global.c", 206, "open_file", "Can't read", name);
    sys_error("Open:Cannot open file %s, mode=0x%x, errno=%d", name, mode, errno);

    return NULL;
}



FILE *open_file_search_path(name, mode)
char *name;
int mode;
{
    FILE *f;

    if (name == NULL) return NULL;

    if (*name == '/') return open_file(name, mode);
    
    f = NULL;
    if (!use_nntp)
	f = open_file(relative(news_lib_directory, name), OPEN_READ);
    if (f == NULL)
	f= open_file(relative(lib_directory, name), OPEN_READ);
    if (f == NULL)
	f = open_file(relative(db_directory, name), OPEN_READ);

    return f;
}
    
fgets_multi(buf, size, f)
char *buf;
int size;
register FILE *f;
{
    register char *s = buf;
    register int c, n = size;
    
    while (--n > 0) {
	c = getc(f);
	if (c == '\\') {
	    if ((c = getc(f)) == NL) continue;
	    *s++ = '\\';
	    if (--n < 0) break;
	}
	if (c == EOF) {
	    *s = NUL;
	    return s != buf;
	}
	if (c == NL) {
	    *s = NUL;
	    return 1;
	}
	*s++ = c;
    }
    buf[30] = NUL;
    printf("\nLine too long \"%s...\" (max %d)", buf, size);
    sys_error("Line too long \"%s...\" (max %d)", buf, size);
}
	
/*
 * 	relative -- concat directory name and file name
 */

char *relative(dir, name)
char *dir, *name;
{
    static char concat_path[FILENAME];

    sprintf(concat_path, "%s/%s", dir, name);
    return concat_path;
}


char *mk_file_name(dir, name)
char *dir, *name;
{
    char *buf;

    buf = newstr(strlen(dir) + strlen(name) + 2);
    sprintf(buf, "%s/%s", dir, name);

    return buf;
}


char *home_relative(dir)
char *dir;
{
    if (dir) {
	if (*dir == '/')
	    return copy_str(dir);
	else {
	    if (*dir == '~' && *++dir == '/') dir++;
	    return mk_file_name(home_directory, dir);
	}
    }
    return NULL;
}


char *substchr(str, c1, c2)
char *str, c1, c2;
{
    char *p;

    if ((p = strchr(str, c1)) != NULL) *p = c2;
    return p;
}

char *copy_str(str)
char *str;
{
    char *new;

    new = newstr(strlen(str) + 1);
    if (new) strcpy(new, str);

    return new;
}

time_t m_time(f)
FILE *f;
{
    struct stat st;

    if (fstat(fileno(f), &st) < 0) return 0;
    return st.st_mtime;
}


time_t file_exist(name, mode)
char *name;
char *mode;
{
    static struct stat statb;
    extern int errno;
    int mask;

    if (name != NULL && stat(name, &statb)) return 0;

    if (mode == NULL) return statb.st_mtime;

	mask = 0700;

    while (*mode) {
	switch (*mode++) {
	case 'd':
	    if ((statb.st_mode & S_IFMT) == S_IFDIR) continue;
	    errno = ENOTDIR;
	    return 0;
	case 'f':
	    if ((statb.st_mode & S_IFMT) == S_IFREG) continue;
	    if ((statb.st_mode & S_IFMT) == 0000000) continue;
	    if ((statb.st_mode & S_IFMT) == S_IFDIR) {
		errno = EISDIR;
		return 0;
	    }
	    break;
	case 'r':
	    if (statb.st_mode & mask & 0444) continue;
	    break;
	case 'w':
	    if (statb.st_mode & mask & 0222) continue;
	    break;
	case 'x':
	    if (statb.st_mode & mask & 0111) continue;
	    break;
	}
	errno = EACCES;
	return 0;
    }

    /* all modes are ok */
    return statb.st_mtime;
}

/*
 * copy_file: copy (or append) src file to dest file.
 *
 * Returns number of characters copied or an error code:
 *  -1: source file not found
 *  -2: cannot create destination
 *  -3: write error
 */

int32 copy_file(src, dest, append)
char *src, *dest;
int append;
{
    register FILE *s, *d;
    register int32 n = 0;
    register int c;

    s = open_file(src, OPEN_READ);
    if (s == NULL) return -1;

    d = open_file(dest, append ? OPEN_APPEND : OPEN_CREATE);
    if (d == NULL) {
	fclose(s);
	return -2;
    }

    n = 0;
    while ((c = getc(s)) != EOF) {
	putc(c, d);
	n++;
    }

    fclose(s);
    if (fclose(d) == EOF) {
	if (!append) unlink(dest);
	return -3;
    }
    return n;
}

/*
 * move_file: move old file to new file, linking if possible.
 *
 * The third arg determines what is acceptable if the old file cannot be
 * removed after copying to the new file:
 *   0: must remove old, else remove new and fail,
 *   1: must remove or truncate old, else remove new and fail,
 *   2: just leave old if it cannot be removed or truncated.
 *	
 * Returns positive value for success, negative for failure:
 *   0: file renamed (link)
 *   1: file copied, old removed
 *   2: file copied, but old file is only truncated.
 *   3: file copied, but old file still exist.
 *  -1: source file not found
 *  -2: cannot create destination
 *  -3: write error
 *  -4: cannot unlink/truncate old
 *  -5: cannot unlink new
 *  -6: cannot link old to new
 *  -9: messy situation: old and new linked on return (cannot happen?)
 */

move_file(old, new, may_keep_old)
char *old, *new;
int may_keep_old;
{
    int32 n;

    if (file_exist(new, (char *)NULL)) {
	if (file_exist((char *)NULL, "d"))
	    return -5;
	if (unlink(new) < 0)	/* careful - new may be directory ? */
	    switch (errno) {
	     case ENOENT:
		break;
	     case EACCES:
		if (file_exist((char *)NULL, "w")) goto do_copy;
	     default:
		return -5;
	    }
    }
    
    if (link(old, new) < 0)
	switch (errno) {
	 case EACCES:	/* can just as well try to copy */
	 case EXDEV:
	    goto do_copy;
	 default:
	    return -6;
	}
    
    if (unlink(old) == 0)
	return 0;

    /* we were able to link but not unlink old	*/
    /* remove new, and attempt a copy instead	*/
    if (unlink(new) < 0) return -9; /* cannot happen? */

 do_copy:
    if ((n = copy_file(old, new, 0)) < 0) return n;
    if (unlink(old) == 0) return 1;
    if (may_keep_old)
	if (n == 0 || truncate(old, (off_t)0) == 0) return 2;
    if (may_keep_old == 2) return 3;
    unlink(new);
    return -4;
}

save_old_file(name, suffix)
char *name, *suffix;
{
    char buf[FILENAME];
    sprintf(buf, "%s%s", name, suffix);
    return move_file(name, buf, 0);
}

#ifdef HAVE_SYSLOG
#include <syslog.h>
#endif /* HAVE_SYSLOG */



char *date_time(t)
time_t t;
{
    char *str;

    if (t == (time_t)0) t = cur_time();
    str = ctime(&t);

    str[16] = 0;
    return str+4;
}

char *plural(n)
long n;
{
    return n != 1 ? "s" : "";
}

/*
 *	memory management
 */

/* #define MEM_DEBUG		/* trace memory usage */

static mem_error(t, bytes)
int t;
int32 bytes;
{
    char buf[200];

    if (t == 1) {
	sprintf(buf, "Alloc failed: unsigned too short to represent %ld bytes",
		(long)bytes);
    } else {
	sprintf(buf, "Out of memory - cannot allocate %ld bytes",
		(long)bytes);
    }
    printf(buf);
    sys_error(buf);
}

char *mem_obj(size, nelt)
unsigned size;
int32 nelt;
{
    unsigned n;
    char *obj, *calloc();

    n = nelt;
    if (n != nelt) mem_error(1, nelt);

    obj = calloc(n, size);
#ifdef MEM_DEBUG
    printf("CALLOC(%u,%u) => %lx\n", n, size, (long)obj);
#endif
    if (obj == NULL) mem_error(2, (int32)(size * nelt));
    return obj;
}

char *mem_str(nelt)
int32 nelt;
{
    unsigned n;
    char *obj, *malloc();

    n = nelt;
    if (n != nelt) mem_error(1, nelt);

    obj = malloc(n);
#ifdef MEM_DEBUG
    printf("MALLOC(%u) => %lx\n", n, (long)obj);
#endif
    if (obj == NULL) mem_error(2, nelt);
    return obj;
}

char *mem_resize(obj, size, nelt)
char *obj;
unsigned size;
int32 nelt;
{
    unsigned n;
    char *realloc(), *obj1;

    if (obj == NULL)
	return mem_obj(size, nelt);

    nelt *= size;

    n = nelt;
    if (n != nelt) mem_error(1, nelt);

    obj1 = realloc(obj, n);
#ifdef MEM_DEBUG
    printf("REALLOC(%lx, %u) => %lx\n", (long)obj, n, (long)obj1);
#endif
    if (obj1 == NULL) mem_error(2, (int32)size);
    return obj1;
}

char *mem_clear(obj, size, nelt)
register char *obj;
unsigned size;
register int32 nelt;
{
    nelt *= size;
    while (--nelt >= 0) *obj++ = NUL;
}

mem_free(obj)
char *obj;
{
#ifdef MEM_DEBUG
    printf("FREE(%lx)\n", (long)obj);
#endif
    if (obj != NULL) free(obj);
}

#ifndef HAVE_MKDIR

mkdir(path, mode)
char *path;
int mode;
{
    char command[FILENAME*2 + 20];

    sprintf(command, "{ mkdir %s && chmod %o %s ; } > /dev/null 2>&1",
	    path, mode, path);
    return system(command) != 0 ? -1 : 0;
}
#endif



time_t cur_time()
{
    time_t t;

    time(&t);
    return t;
}