// Filename:   User.C
// Contents:   the methods for the User object
// Author: Greg Shaw
// Created:    6/1/93

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _USER_C_
#define _USER_C_

#include "bbshdr.h"

// Function:	account_expiration
// Purpose:	return when the account expires next (0L for no expiration)
// Input:	account information
// Output:	(file is written)
// Author:	Greg Shaw
// Created:	8/18/95

time_t User::account_expiration(void)
{
	time_t now;
	time_t exdate;			// date of expiration
	struct tm *nowtm;		// current date (broken down)
	const int DAYSECS = 60*60*24;	// seconds in a day



	time(&now);
	exdate = 0x7fffffff;		// no expiration 'time'
	// have they ever validated?
	if (validated == 0L)
		exdate = validated;
	// days expired?
	else if (card->days_valid != -1)
		exdate = validated+(card->days_valid*DAYSECS); 
	// months expired?
	else if (card->months_valid != -1)
	{
		nowtm = localtime(&now);
		nowtm->tm_min = 0;	// only care about midnight
		nowtm->tm_sec = 0;
		nowtm->tm_hour = 0;
		nowtm->tm_mon += card->months_valid;
		if (nowtm->tm_mon > 11)	// allow for wrap
		{
			nowtm->tm_year += nowtm->tm_mon/12;
			nowtm->tm_mon %= 12;
		}
		exdate = mktime(nowtm);
		if (exdate == -1)
		{
				ap_log("user: month computation for account expiration failed.");
				exdate = now+100;	// I failed, let him in
		}
	}
	// all time used?
	if (card->max_time > 0 && total_timeused/60 >= card->max_time)
	{
		exdate = now-1;	// as of now, account expired
	}
	return(exdate);
}

// Function:   append
// Purpose:    append a new user record to the user file
// Input:  object attributes
// Output: (file is written)
// Author: Greg Shaw
// Created:    7/20/93

int User::append(void)
{
	FILE   *ofile;
	char   finame[255];         // filename
	char   line[255];           // one line in file (255 max)
	char   bbspath[225];        // bbs path

	if (strcpy(bbspath,getenv("BBSDIR")), bbspath == NULL)
	{
		sprintf(finame,"user.save: BBSDIR env var not set for %s",login_name);
		ap_log(finame);
		return(-1);
	}
	strcpy(line,bbspath);
	strcat(line,"/admin/userlog");
	if (ofile = bopen(line,"a+"), ofile == NULL)
	{
		sprintf(line,"user.save: Unable to open userlog file ");
		ap_log(line);
		return(-1);
	}
	//[A login_name firstname lastname state city]
	fprintf(ofile,"[A %s %s %s %s %s %s ]\n", login_name, fname, lname, alias, state, city);
	//[B terminal_type last_logon #_logins downloads uploads card_color total_kused total_timeused validate ]
	fprintf(ofile,"[B %s %ld %d %d %d %s %d %d %ld ]\n", uterm,
	last_login, logins, downloads, uploads, card->colr, total_kused, 
	total_timeused, validated);
	//[C private_msgs public_msgs credited_time color_capable editor lines cols language]
	fprintf(ofile,"[C %d %d %d %d %s %d %d %s]\n",priv_msgs,pub_msgs,
	credited, color_capable, editor, lines, cols, lang);
	//[D flags access_level timelimit timeused_last_call anniversary kused expiration chat_foreground chat_background]
	fprintf(ofile,"[D %lx %d %d %d %ld %d %ld %d %d ]\n",flags, acl, timelimit, timeused, anniversary, kused, expiration,c_fore,c_back);
	bclose(ofile);
	return(0);
};




// Function:   can_access
// Purpose:    return true if the user has access based on the information
//             passed in
// Input:  acl - the access level of the user
//         flags - the flags of the user
//         am - the access level modifier for the user's acccess level
//         fm - the flags modifier
// Output: true if user has access
// Author: Greg Shaw
// Created:    7/27/93

int User::can_access(int acl, int flags, char am, char fm)
{
	int    uacl;                // user's acl
	int uflags;                 // user's flags

	uacl = user.u_acl();
	uflags = user.u_flags();
	if (!((uacl >= acl && am == '>' ) || (uacl < acl && am == '<') ||
		(acl == uacl && am == '=')))
	return(0);                  // no access
	if (flags != 0)
	{
		if ((flags & uflags) && fm == '=')
			return(1);
		if (!(flags & uflags) && fm == '!')
			return(1);
		return(0);
	}
	return(1);                  // access
};

// Function:   change_acl
// Purpose:        change the user's access level (possibly permanently)
// Input:      acl - the timlit to change to
//             perm - permanent or not
// Output:     none
// Author:     Greg Shaw
// Created:        6/8/93

int User::change_acl(unsigned int nacl, char perm)
{
	if (perm)
		acl = nacl;
	tmpacl = nacl;
	return(0);
}

// Function:   change_alias
// Purpose:    change the user's alias
// Input:      alias
// Output:     none
// Author:     Greg Shaw
// Created:    6/22/95

void User::change_alias(char *new_alias)
{
	if (strlen(new_alias) >= MAX_ALIAS_LENGTH)
		new_alias[MAX_ALIAS_LENGTH-1] = 0;	// make sure no corruption

	strcpy(alias, new_alias);
}

// Function:   change_editor
// Purpose:    change the user's editor
// Input:      new_editor
// Output:     none
// Author:     Greg Shaw
// Created:    6/22/95

void User::change_editor(char *new_editor)
{
	new_editor[14] = 0;	// make sure no corruption

	strcpy(editor, new_editor);
}


// Function:   change_flags
// Purpose:        change the user's timelimit (possibly permanently)
// Input:      flags - the flags to change to
//             perm - permanent or not
//         or - or into user flags or and into user flags
// Output:     none
// Author:     Greg Shaw
// Created:        6/9/93

int User::change_flags(unsigned long fl, char perm, char or)
{
	if (perm)
		if (or)
			flags |= fl;
	else
		flags &= fl;
	else
		if (or)
			tmpflags |= fl;
	else
		tmpflags &= fl;
	return(0);
}

// Function:   change_terminal
// Purpose:    change the user's terminal type
// Input:      new terminal type
// Output:     none
// Author:     Greg Shaw
// Created:    6/22/95

void User::change_terminal(char *new_term)
{
	if (strlen(new_term) >= MAX_TERM_LENGTH)
		new_term[MAX_TERM_LENGTH-1] = 0;

	strcpy(uterm,new_term);
}


// Function:   change_tl
// Purpose:        change the user's timelimit (possibly permanently)
// Input:      tl - the timlit to change to
//             perm - permanent or not
// Output:     none
// Author:     Greg Shaw
// Created:        6/8/93

int User::change_tl(unsigned int tl, char perm)
{
	if (perm)
		timelimit = tl;
	else
		tmptl = tl;
	return(0);
}

// Function:	check_access
// Purpose:	check the user's access for time expired, account expired, etc.
// Input:	none
// Output:	0 if user may not continue, 1 if user may continue
///		user record may be modified regarding used/unused time
// Author:	Greg Shaw
// Created:	8/18/95
int User::check_access(void)
{
	time_t now;
	time_t exdate;			// date of expiration
	struct tm *nowtm;		// now (broken down)
	int	newcard;		// new card for updated access
	int	start;			// start of available time
	int	end;			// end of available time
	int	nowtime;		// end of available time
	char   *validprog;		// validation program
	char	tmpstr[255];		// tmp


	time(&now);
	// check for time expired today
	if ((now - expiration)/60 < (waittime()*60))
	{
		if ((timelimit - timeused) < 5)
		{
			clear_scr();
			sstrcrl("NOUNUSEDTIME");
			sprintf(tmpstr,gstrl("TOOSOONERROR"),waittime());
			sstrcr_c(tmpstr);
			cr();
			cr();
			waitcr();
			sstrcrl("THANKSFORCALLING");
			exit(0);
		}
	}
	else
	{
		kused = 0;  // start all counters over
		time(&expiration);
		timeused = 0;
		credited = 0;
	}
	// if user not validated (and there is a validation program)
	// make him go through validation
	validprog = validation_program();
	if (strcmp(validprog,"none"))
	{	
		exdate = account_expiration();
		time(&now);
		if (exdate < now)
		{
			if (exdate == 0L)	// not validated yet
			{
				sstrcrl("YOUARENOTVALIDATED");
				cr();
			}
			else	// account has expired
			{
				sprintf(tmpstr,"Account has expired for %s %s",fname,lname);
				ap_log(tmpstr);
				sstrcrl("YOURACCOUNTEXPIRED");
			}
			sprintf(tmpstr,"Attempting validation of %s %s",fname,lname);
			ap_log(tmpstr);
			sstrcrl("STARTINGVALIDPROGRAM");
			waitcr();
			if (newcard = sysint(validprog,time(NULL)+VALIDATION_TIMEOUT,0),
				newcard >0)
			{	// validation succeeded
				time(&validated);	// save for later		
				newcard -= 1;
				// save new card level
				// and update access
				free(card); // nuke old card
				card = cardinfo(newcard); 
				check_card();
				sprintf(tmpstr,"Updating card to %s for %s %s",card->colr,fname,lname);
				ap_log(tmpstr);
			}
			else
			{
				ap_log("Validation failed");
				sstrcrl("VALIDATIONFAILED");
				waitcr();
				exit(0);
			}
		}
	}
	else
		time(&validated);	// no validation program, set validate
	// check against time-of-day limitations
	nowtm = localtime(&now);
	nowtime = nowtm->tm_hour*100;
	nowtime += nowtm->tm_min;
	start = card->start_time;
	end = card->end_time;
	if (end < start)
	{
		if (nowtime < end)
			nowtime += 2400;
		end += 2400;
	}
	if (!(nowtime > start && nowtime < end))
	{
			sstrcrl("NOTVALIDTIME");
			cr();
			sprintf(tmpstr,gstrl("VALIDTIME"),card->start_time, card->end_time); 
			sstrcr(tmpstr);
			waitcr();
			exit(0);
	}
	// check terminal against list of lines the user is authorized to use
	if (!check_line())
	{
		sstrcrl("NOTVALIDLINE");
		waitcr();
		exit(0);
	}
	return(1);
}

// Function:   check_acl
// Purpose:    Check for a particular file that indicates that the sysop
//		has modified the access of this user.  A re-read of the 
//		userlog is indicated.
// Input:      none
// Output:     (user access may be changed)
// Author:     Greg Shaw
// Created:	5/22/95

int User::check_acl(void)
{
	char tmpstr[255];
	struct stat fistat;         // file status record
	static time_t now;		// only check every 5 seconds
	char *bbsdir;

	if (abs(time(NULL) - now) > 5)
	{
		time(&now);
		bbsdir = getenv("BBSDIR");
		// does file exist?
		sprintf(tmpstr,"%s/admin/%s.acl",bbsdir,login_name);
		if (stat(tmpstr,&fistat) == 0 && S_ISREG(fistat.st_mode))
		{
			// found file  -- reread the userlog 
			get(login_name,1);
		}
	}
	return(0);
}

// Function:   check_card
// Purpose:        check the user's access information vs. the card color he has
// Input:      none
// Output:     (user may be changed)
// Author:     Greg Shaw

int User::check_card(void)
{
	char tmpstr[100];

	if (card = cardinfo(card_color), card == NULL)
	{
		sprintf(tmpstr,"Unable to find card info for color %d ",card_color);
		ap_log(tmpstr);
	}
	if (acl < card->acl)        // only update info if card better
	{
		acl = card->acl;
		tmpacl = card->acl;
	}
	if (timelimit < card->tl)
	{
		timelimit = card->tl;
		tmptl = card->tl;
	}
	flags |= card->flags;
	return(0);
};


// Function:   check_for_ansi
// Purpose:    automatically check for ansi and set the user's terminal 
//		type appropriately.
// Input:      none
// Output:     a report of cursor position
// Author:     Greg Shaw
// Created:    1/1/96

int User::check_for_ansi(void)
{
	time_t	now;
	time_t	then;
	char	c;

	// should return cursor position
	char ansiseq[]  = { 
		ESC,'[','6','n',0 
	};

	sstr(ansiseq);
	time(&then);
	while (time(&now), now - then < 3)
	{
		if (c = gch(1), c == ESC)
		{
			color_capable = 1;
			change_terminal("ansi");
			return(1);
		}
	}
	return(0);
}

// Function:   check_info
// Purpose:    allow the user to change their environment information
// Input:      none
// Output:     the user's information may be changed
// Author:     Greg Shaw
// Created:    1/1/95


int User::check_info(void)
{
	char   loop;                // loop counter
	char   done;                // loop counter
	char   entered;             // entered anything?
	char   tmpstr[255];
	char   anotherstr[50];      // another string
	int    selected;

	loop = 1;
	entered = 0;
	while (loop)
	{
		clear_scr();            // clear screen
		cr();
		cr();
		cr();
		sstrcrl("ENTERFOLLOWING");
		cr();
		sprintf(tmpstr,gstrl("LOGINNAME"),login_name);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("REALNAME"),fname,lname);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("ALIASNAME"),alias);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("CALLINGFROM"),city,state);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("LANGUAGECHOSEN"),lang);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("TERMINALTYPE"),uterm);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("TERMINALCAPABILITIES"),color_capable?gstrl("YES"):gstrl("NO"));
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("EDITOR"),editor);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("LINESCOLUMNS"),lines,cols);
		sstrcr_c(tmpstr);
		cr();
		cr();
		sstrl("CORRECT");
		if (yesno())
		{
			loop = 0;
			continue;
		}
		// show menu
		#ifndef ONE_LANGUAGE_ONLY
		sstrcrl("USEREDITMENU");
		#else
		sstrcrl("USEREDITMENUNOLANG");
		#endif
		cr();
		sstrcrl("RETURNTOEXIT");
		cr();
		sstrl("YOURCHOICE");
		tmpstr[0] = 0;
		gstr(tmpstr,3);
		selected = 0;
		#ifndef ONE_LANGUAGE_ONLY
		if (sscanf(tmpstr,"%d",&selected) != 1 || (selected < 1 || selected > 11))
			continue;
		#else
		if (sscanf(tmpstr,"%d",&selected) != 1 || (selected < 1 || selected > 10))
			continue;
		#endif
		switch(selected)
		{
			case 1:             //  first name
				cr();
				done = 0;
				while (!done)
				{
					cr();
					sstrl("USERFIRSTNAME");
					tmpstr[0] = 0;
					if (gstr(tmpstr,20), strcmp(tmpstr,"") ==
						0)
					continue;
					strcpy(fname,tmpstr);
					sprintf(tmpstr,gstrl("YOUENTERED"),fname);
					sstrcr_c(tmpstr);
					sstrl("CORRECT");
					if (!yesno())
						continue;
					done++;
				}
				break;
			case 2:	// last name
				cr();
				done = 0;
				while (!done)
				{
					cr();
					sstrl("USERLASTNAME");
					tmpstr[0] = 0;
					if (gstr(tmpstr,20), strcmp(tmpstr,"") ==
						0)
					continue;
					strcpy(lname,tmpstr);
					sprintf(tmpstr,gstrl("YOUENTERED"),lname);
					sstrcr_c(tmpstr);
					sstrl("CORRECT");
					if (!yesno())
						continue;
					done++;
				}
				break;
			case 3:	// city
				cr();
				done = 0;
				while (!done)
				{
					sstrl("USERINFOCITY");
					tmpstr[0] = 0;
					if (gstr(tmpstr,20), strcmp(tmpstr,"") == 0)
						continue;
					else
						strcpy(city,tmpstr);
					sprintf(tmpstr,gstrl("YOUENTERED"),city);
					sstrcr_c(tmpstr);
					sstrl("CORRECT");
					if (!yesno())
						continue;
					done++;
				}
				break;
			case 4:	// state
				cr();
				done = 0;
				while (!done)
				{
					sstrl("USERINFOSTATE");
					tmpstr[0] = 0;
					if (gstr(tmpstr,2), strcmp(tmpstr,"") == 0)
						continue;
					else
						strcpy(state,tmpstr);
					sprintf(tmpstr,gstrl("YOUENTERED"),state);
					sstrcr_c(tmpstr);
					sstrl("CORRECT");
					if (!yesno())
						continue;
					done++;
				}
				break;
			case 5:	// alias
				cr();
				sstrcrl("USERINFOALIAS0");
				cr();
				sstrl("USERINFOALIAS1");
				if (yesno())
					strcpy(alias,login_name);
				else
				{
					done = 0;
					while (!done)
					{
						sstrl("USERINFOALIAS2");
						tmpstr[0] = 0;
						if (gstr(tmpstr,20), strcmp(tmpstr,"") == 0)
							continue;
						else
							if (sscanf(tmpstr,"%s%s",alias,anotherstr) != 1)
								continue;
						sprintf(tmpstr,gstrl("YOUENTERED"),alias);
						sstrcr_c(tmpstr);
						sstrl("CORRECT");
						if (!yesno())
							continue;

						done++;
					}
				}
				break;
			case 6:             // terminal type
				done = 0;
				while (!done)
				{
					sstrcrl("USERINFOTERM0");
					cr();
					sstrcrl("USERINFOTERM1");
					cr();
					sstrcrl("USERINFOTERM2");
					cr();
					sstrl("USERINFOTERM3");
					tmpstr[0] = 0;
					if (gstr(tmpstr,20), strcmp(tmpstr,"") == 0)
						continue;
					else
						strcpy(uterm,tmpstr);
					sprintf(tmpstr,gstrl("YOUENTERED"),uterm);
					sstrcr_c(tmpstr);
					sstrl("CORRECT");
					if (!yesno())
						continue;
					// export to environ
					export_hard(1);
					done++;
				}
				break;
			case 7:             // lines on screen
				cr();
				cr();
				done = 0;
				while (!done)
				{
					sstrl("USERINFOLINES");
					tmpstr[0] = 0;
					if (gstr(tmpstr,5), strcmp(tmpstr,"") == 0)
						continue;
					if (tmpstr[0] == 'x')
					{
						lines = 24;
					}
					else
						sscanf(tmpstr,"%d",&lines);
					sprintf(tmpstr,gstrl("YOUENTEREDNUM"),lines);
					sstrcr_c(tmpstr);
					sstrl("CORRECT");
					if (!yesno())
						continue;
					// export to environment
					export_hard(2);
					done++;
				}
				break;
			case 8:             // number of columns on screen
				cr();
				done = 0;
				while (!done)
				{
					sstrl("USERINFOCOLUMNS");
					tmpstr[0] = 0;
					if (gstr(tmpstr,5), strcmp(tmpstr,"") == 0)
						continue;
					if (tmpstr[0] == 'x')
					{
						cols = 80;
					}
					else
						sscanf(tmpstr,"%d",&cols);
					sprintf(tmpstr,gstrl("YOUENTEREDNUM"),cols);
					sstrcr_c(tmpstr);
					sstrl("CORRECT");
					if (!yesno())
						continue;
					// export to environment
					export_hard(3);
					done++;
				}
				break;
			case 9:             // terminal supports color?
				cr();
				cr();
				sstrl("USERINFOCOLOR");
				if (yesno())
					color_capable = 1;
				else
					color_capable = 0;
				cr();
				break;
			case 10:            // editor
				get_editor();
				// export to environment
				export_hard(0);
				done++;
				break;
				#ifndef ONE_LANGUAGE_ONLY
			case 11:            // language
				done = 0;
				while (!done)
				{
					sprintf(tmpstr,gstrl("YOUAREUSINGLANGUAGE"),lang);
					sstrcr_c(tmpstr);
					sstrl("CORRECT");
					if (yesno())
					{
						continue;
					}
					select_lang(lang);
				}
				#endif
		}
	}
	if (save(login_name))
	{
		ap_log("Unable to save user.");
	}
	return(0);
};

// Function:   check_line
// Purpose:    read the $BBSDIR/config/lines file to check user against
//		modem line restrictions
// Input:  	none
// Output: 	0 for failure, 1 for access
// Author: 	Greg Shaw
// Created:    	1/7/95

int User::check_line(void)       // check tty line for security
{
	char tmpstr[255];
	FILE	*infile;	// input file
	char ttystr[25];	// tty name
	char *line;
	char *p;	
	unsigned int lacl;	// access level of line
	int  state;		// state table
	char secmod=0;		// security modifier
	unsigned long lflags;	// flags
	char fmod;		// flags modifier


	// check for lines file
	sprintf(tmpstr,"%s/config/lines",getenv("BBSDIR"));
	if (infile = bopen(tmpstr,"r"), infile == NULL)
		return(1);	// no error for no file found
	// get tty name
	strcpy(ttystr,ttyname(fileno(stdout)));
	if (line = strchr(ttystr,'y'), line != NULL)
	{                   // chop off /dev/tty
		// chop after y
		strcpy(tmpstr,++line);
		// copy back
		strcpy(ttystr,tmpstr);
	}
	// read lines file
	while (!feof(infile))
	{
		line = getline(infile);
		if (strlen(line) == 0)
			continue;
		if (p = strchr(line,'#'), p != NULL)
			*p = 0;	// ignore #'s
		if (p = strchr(line,'|'), p == NULL)
			continue;	// no '|' chars -- nothing valid
		*p = 0;
		if (strcmp(line,ttystr))	// look for our tty
			continue;	// not ours, skip
		// ours -- figure out whether we can continue
		*p = '|';	// revert
		p = line;
		state = 0;
		while (*p != 0)
		{
			if (line = strchr(p,'|'), line == NULL)
				continue;	// no '|' chars -- nothing valid
			*line = 0;
			p = line;
			p++;	// skip null
			switch (state)
			{
			case 0:	// security number
				// get security
				if (sscanf(p,"%d",&lacl) != 1)
					lacl = 0;	// default to no acl
				state++;
				break;
			case 1: // security modifier
				if (*p == '|')
					secmod = '>';
				else
					secmod = *p;
				if (secmod != '>' && secmod != '<' && secmod != '=')
				{
					ap_log("lines file has an error: Unknown security level modifier");
				}
				state++;
				break;
			case 2: // flags
				if (sscanf(p,"%lx",&lflags) != 1)
					lflags = 0;	// default to no flags
				state++;
				break;
			case 3: // flags modifier
				if (*p == '|' || *p == 0)
					fmod = '=';
				else
					fmod = *p;
				if (fmod != '!' && fmod != '=')
				{
					ap_log("lines file has an error: Unknown flags modifier");
				}
				// Ok, we've got everything we need.  do it.
				bclose(infile);
				return(can_access(lacl, lflags, secmod, fmod));
			}
		}			
	}
	bclose(infile);
	return(1);
}


// Function:   delete_user
// Purpose:    delete a user from the BBS
// Input:  name - the login name of the user to delete
// Output: none
// Author: Greg Shaw
// Created:    4/11/94

int User::delete_user(char *name)
{
	FILE *newfile, *oldfile;
	char tmpstr[255];
	char line[255];
	char *ptr;
	char uname[MAX_LOGIN_LENGTH];
	char c;
	int x;

	sprintf(tmpstr,"%s/admin/userlog",getenv("BBSDIR"));
	if (oldfile = bopen(tmpstr,"r"), oldfile == NULL)
	{
		ap_log("Unable to open userlog for read.");
		return(-1);
	}
	strcat(tmpstr,".new");	// tack on .new extension
	if (newfile = bopen(tmpstr,"w"), newfile == NULL)
	{
		ap_log("Unable to open userlog.new for write.");
		return(-1);
	}
	// now find the user in question
	while (strcmp(uname,name) && !feof(oldfile))
	{
		ptr = tmpstr;
		while (c = fgetc(oldfile), !iseol(c) && !feof(oldfile))
			*ptr++ = c;
		*ptr = 0;
		if (strstr(tmpstr,"[A") != NULL)	// first line?
		{
			sscanf(tmpstr,"%*s %s %*s %*s %*s %*s %*s %*s %*s", uname);
			if (strcmp(uname,name))
				fprintf(newfile,"%s\n",tmpstr);
		}
		else
		{
			fprintf(newfile,"%s\n",tmpstr);
		}
	}
	// skip the next 3 lines of the deleted user
	if (!strcmp(uname,name))	// user was found
	{
		for (x=0; x<3; x++)
			while (c = fgetc(oldfile), !iseol(c) && !feof(oldfile));
	}
	else
	{
		sprintf(tmpstr,"Attempted to delete nonexistant user %s",name);
		ap_log(tmpstr);
		bclose(newfile);
		return(-1);
	}
	// dump the rest of the file to new file
	while (c = fgetc(oldfile), !feof(oldfile))
		fputc(c,newfile);
	bclose(oldfile);
	bclose(newfile);
	// rename userlog files
        sprintf(tmpstr,"%s/admin/userlog",getenv("BBSDIR"));
        sprintf(line,"%s/admin/userlog.old",getenv("BBSDIR"));
        rename(tmpstr,line);
        sprintf(tmpstr,"%s/admin/userlog.new",getenv("BBSDIR"));
        sprintf(line,"%s/admin/userlog",getenv("BBSDIR"));
        rename(tmpstr,line);
        return(0);
};




// Function:   display_info
// Purpose:    display the user's information
// Input:  logon - has the user just logged on?  (it gives 'welcome' message
// Output: user information is formatted and printed.
// Author: Greg Shaw
// Created:    7/2793

int User::display_info(int logon)
{
	char   tmpstr[255];         // string
	char   *str2;               // used for time
	double udratio;             // upload/download ratio
	time_t now;

	clear_scr();
	// just logged in?
	if (logon)
		sprintf(tmpstr,gstrl("YOUARE"),fname,lname,city,state);
	else
		sprintf(tmpstr,gstrl("WELCOME"),fname,lname,city,state);
	sstrcr_c(tmpstr);
	str2 = ctime(&login_time);
	str2[strlen(str2)-1] = 0;   // chop \n
	cr();

	// user information header
	sstrcrl("USERINFORMATION");
	str2 = ctime(&last_login);
	str2[strlen(str2)-1] = 0;   // chop \n
	sstr_c("    ");
	sprintf(tmpstr,gstrl("LOGINSLAST"),logins,str2);
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("ALIASEDITOR"),alias,editor);
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("TERMINALINFO"),uterm,color_capable?gstrl("COLOR"):gstrl("MONOCHROME"),lines,cols);
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("TIMELIMITINFO"),timelimit,waittime());
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("CREDITEDTIME"),credited,ansi_color_name(c_fore),
		ansi_color_name(c_back));
	sstrcr_c(tmpstr);
	cr();

	// usage information header
	sstrcrl("ACCOUNTUSAGE");
	sstr_c("    ");
	sprintf(tmpstr,gstrl("TIMEONLINE"), total_timeused, 
		(float)total_timeused/60.0,timeused);
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("DOWNLOADLIST"),downloads,uploads,kused);
	sstrcr_c(tmpstr);
	if (uploads > 0)
		udratio = (double)downloads/uploads;
	else
		udratio = downloads;
	sstr_c("    ");
	sprintf(tmpstr,gstrl("DOWNLOADRATIO"),total_kused,udratio);
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("MESSAGESSENT"),pub_msgs,priv_msgs);
	sstrcr_c(tmpstr);

	cr();
	
	// card information/limits header
	sprintf(tmpstr,gstrl("ACCOUNTINFO"),card->colr);
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("ACCOUNTAVAIL"),
		card->start_time/100,card->start_time%100,
		card->end_time/100,card->end_time%100);
	sstrcr_c(tmpstr);
	sstr_c("    ");
	now = account_expiration();
	str2 = ctime(&now);
	str2[strlen(str2)-1] = 0;   // chop \n
	sprintf(tmpstr,gstrl("TIMELIMITEXP"),card->tl,str2);
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("TOTALTIMELIMIT"),stringornum(card->max_time));
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("DOWNLOADCAP"),stringornum(card->kbytes),waittime());
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("FILESDOWNLOAD"),stringornum(card->max_files));
	sstrcr_c(tmpstr);
	sstr_c("    ");
	sprintf(tmpstr,gstrl("DOWNLOADAMOUNT"),stringornum(card->max_kbytes));
	sstrcr_c(tmpstr);
	cr();
	waitcr();
	return(0);
};

// Method:	edit_chat_colors
// Purpose:	edit the fore/back colors sent when chatting
// Input:	none
// Output:	the foreground and/or background chat colors will be changed
// Author:	Gregory Shaw
// Created:	9/14/95

int User::edit_chat_colors(void)
{
	char tmpstr[255];	// output storage
	int done = 0;	// loop
	int x;		// loop counter
	const char validkeys[] = "fb\n\r";	// valid keys
	char c;			// input char

	while (!done)
	{
		clear_scr();
		// display current color settings
		sstrcrl("CURRENTCOLORSETTINGS");
		sprintf(tmpstr,gstrl("COLORFOREGROUND"),ansi_color_name(c_fore));
		sstrcr(tmpstr);
		sprintf(tmpstr,gstrl("COLORBACKGROUND"),ansi_color_name(c_back));
		sstrcr(tmpstr);
		sprintf(tmpstr,gstrl("RESULTINGCOLOR"),c_fore,c_back);
		sstrcr_c(tmpstr);

		sstrl("ISACCEPTABLE");
		if (yesno())
			done++;
		else
		{
		// prompt to change
		// get new colors
		// verify
			cr();
			sstrl("CHANGEFOREBACK");
			while (c = tolower(gch(1)), c == 0 || strchr(validkeys,c) == NULL);
			cr();
			switch(c)
			{
			case 'f':
			case 'b':
				for(x = BLACK; x<= WHITE; x++)
				{
					sprintf(tmpstr,"%d.  %s",x,ansi_color_name(x));
					sstrcr_c(tmpstr);
				}
				sprintf(tmpstr,gstrl("DEXIT"),x);
				sstrcr_c(tmpstr);
				sstrl("CHANGETO");
				tmpstr[0] = 0;
				gstr(tmpstr,2);
				if (sscanf(tmpstr,"%d",&x) != 1)
					break;
				if (x < BLACK || x > WHITE)
					break;
				if (c == 'f')
					c_fore = x;
				else
					c_back = x;
				break;
			default:
				break;
			}
		}
	}
	return(0);
}

// Method:	exists
// Purpose:	attempt to find a user in the userlog
// Input:	username - the login name of the user to find
// Output:	1 for found, 0 for not found
// Author:	Gregory Shaw
// Created:	5/19/96

int User::exists(char *username)
{
	FILE *userlog;
	char tmpstr[255];
	char *line;
	char *c;

	// abort if environment not configured
	if (getenv("BBSDIR") == NULL)
		return(2);
	sprintf(tmpstr,"%s/admin/userlog",getenv("BBSDIR"));
	if (userlog = bopen(tmpstr,"r"), userlog == NULL)
		return(3);
	while (!feof(userlog))
	{
		line = getline(userlog);
		if (c = strstr(line,"[A"), c != NULL)
		{
			sscanf(line,"%*s%s",tmpstr);
			if (!strcmp(tmpstr,username))
			{
				bclose(userlog);
				return(1);
			}
		}
	}
	bclose(userlog);
	return(0);
}

// Function:   export_hard
// Purpose:    export the user's environment variables
// Input:  num - the item to overwrite the user's environment var
//     0 - editor
//     1 - term
//     2 - lines
//     3 - columns
// Output: one environment var is exported
// Author: Greg Shaw
// Created:    7/27/93
// Notes:  This function will nuke a user's environment variable based
//         on the number passed in.  This is so the user can change
//     his settings and expect them to take effect.

int User::export_hard(int num)
{
	char tmpmsg[100];
	char *outmsg;

	switch (num)
	{
		case 0:                 // editor
			sprintf(tmpmsg,"EDITOR=%s",editor);
			if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
			{
				ap_log("Unable to malloc more environment space.");
				return(0);
			}
			strcpy(outmsg,tmpmsg);
			if (putenv(outmsg) != 0)
				ap_log("Unable to export EDITOR variable.");
			sprintf(tmpmsg,"VISUAL=%s",editor);
			if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
			{
				ap_log("Unable to malloc more environment space.");
				return(0);
			}
			strcpy(outmsg,tmpmsg);
			if (putenv(outmsg) != 0)
				ap_log("Unable to export VISUAL variable.");
			break;
		case 1:
			sprintf(tmpmsg,"TERM=%s",uterm);
			if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
			{
				ap_log("Unable to malloc more environment space.");
				return(0);
			}
			strcpy(outmsg,tmpmsg);
			if (putenv(outmsg) != 0)
				ap_log("Unable to export TERM variable.");
			break;
		case 2:
			sprintf(tmpmsg,"LINES=%d",lines);
			if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
			{
				ap_log("Unable to malloc more environment space.");
				return(0);
			}
			strcpy(outmsg,tmpmsg);
			if (putenv(outmsg) != 0)
				ap_log("Unable to export LINES variable.");
			break;
		case 3:
			sprintf(tmpmsg,"COLUMNS=%d",cols);
			if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
			{
				ap_log("Unable to malloc more environment space.");
				return(0);
			}
			strcpy(outmsg,tmpmsg);
			if (putenv(outmsg) != 0)
				ap_log("Unable to export COLUMNS variable.");
			break;
	}
	return(0);
};


// Function:   export_soft
// Purpose:    export the user's environment variables
// Input:  none
// Output: environment vars are exported if not already available
// Author: Greg Shaw
// Created:    7/2793
// Notes:  This is a good "check" of the user's environment.  However,
//     it should not be used when the user has a wrong setting
//     (too many lines, for example) and wants to set it 'right'.

int User::export_soft(void)
{
	char tmpmsg[100];
	char *outmsg;

	// only export EDITOR if not already exported
	if (outmsg = getenv("EDITOR"), outmsg == NULL)
	{
		sprintf(tmpmsg,"EDITOR=%s",editor);
		if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
		{
			ap_log("Unable to malloc more environment space.");
		}
		strcpy(outmsg,tmpmsg);
		if (putenv(outmsg) != 0)
			ap_log("Unable to export EDITOR variable.");
	}
	// only export VISUAL if not already exported
	if (outmsg = getenv("VISUAL"), outmsg == NULL)
	{
		sprintf(tmpmsg,"VISUAL=%s",editor);
		if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
		{
			ap_log("Unable to malloc more environment space.");
		}
		strcpy(outmsg,tmpmsg);
		if (putenv(outmsg) != 0)
			ap_log("Unable to export VISUAL variable.");
	}
	// only export TERM if not already exported
	if (outmsg = getenv("TERM"), outmsg == NULL)
	{
		sprintf(tmpmsg,"TERM=%s",uterm);
		if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
		{
			ap_log("Unable to malloc more environment space.");
		}
		strcpy(outmsg,tmpmsg);
		if (putenv(outmsg) != 0)
			ap_log("Unable to export TERM variable.");
	}
	// only export LINES if not already exported
	if (outmsg = getenv("LINES"), outmsg == NULL)
	{
		sprintf(tmpmsg,"LINES=%d",lines);
		if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
		{
			ap_log("Unable to malloc more environment space.");
		}
		strcpy(outmsg,tmpmsg);
		if (putenv(outmsg) != 0)
			ap_log("Unable to export LINES variable.");
	}
	else	// already set, so let's use that as 'lines'
	{
		sscanf(outmsg,"%d",&lines);
	}
	// only export COLUMNS if not already exported
	if (outmsg = getenv("COLUMNS"), outmsg == NULL)
	{
		sprintf(tmpmsg,"COLUMNS=%d",cols);
		if (outmsg = (char *) malloc(strlen(tmpmsg)+1), outmsg == NULL)
		{
			ap_log("Unable to malloc more environment space.");
		}
		strcpy(outmsg,tmpmsg);
		if (putenv(outmsg) != 0)
			ap_log("Unable to export COLUMNS variable.");
	}
	return(0);
};

// Function:   forward_email
// Purpose:    create a .forward file for a new user if the forward option set
// Input:	none
// Output: 	a .forward file, depending on mailforward
// Author: 	Greg Shaw
// Created:    	5/25/96

int User::forward_email(void)    // create .forward in user's directory
{
	FILE	*file;
	char	tmpstr[255];

	if (mailforward())
	{
		sprintf(tmpstr,"%s/.forward",getenv("HOME"));
		if (file = bopen(tmpstr,"w"), file == NULL)
		{
			sprintf(tmpstr,"Unable to open user's %s/.forward file for write.",getenv("HOME"));
			ap_log(tmpstr);
			return(-1);
		}
		// write out the forward information
		fprintf(file,"|%s/bin/importmail",getenv("BBSDIR"));
		bclose(file);
	}
	return(0);
}


// Function:   get
// Purpose:        get the user's info from the user list
// Input:      none
// Output:     object is loaded or an error is returned
// Author:     Greg Shaw
// Created:        7/13/93
// Notes:      the user file is a plain text file (to allow editing).
//             its format is such:
//                 [A login_name firstname lastname]
//                 [B terminal_type last_logon #_logins downloads uploads card color]
//                 [C private_msgs public_msgs credited_time color_capable editor lines cols language ]
//                 [D flags access_level timelimit timeused_last_call]
//             the above format includes the brackets and lines (for reference)

int User::get(char *name, int check)
{
	FILE   *ufile;
	char   *u;
	char   finame[255];         // filename
	char   line[255];           // one line in file (255 max)
	char   logname[10];         // user login name
	char   tmpstr[255];         // temporary string
	char   found;               // user found?
	char   linenum;             // used to look for appropriate line
	char   c;                   // input char
	unsigned char  of;          // offset into line
	int    x;
	int    done;                // boolean
	int    numwords;            // number of words found in gecos
	time_t now;                 // current time
	struct passwd *userent;     // user entry in password file

	if (name == NULL)
		// get user's login information
		strcpy(logname,username());
	else
		strcpy(logname,name);
	if (strcpy(finame,getenv("BBSDIR")), finame == NULL)
	{
		sprintf(finame,"user.get: BBSDIR environment variable not set for %s",logname);
		ap_log(finame);
		return(-1);
	}
	// delete key file (we're reading the userlog anyway)
	sprintf(tmpstr,"%s/admin/%s.acl",finame,username());
	unlink(tmpstr);	// nuke file
	strcat(finame,"/admin/userlog");
	linenum = 0;                // look for first line first
	if (ufile = bopen(finame,"r"), ufile != NULL)
	{
		found = 0;
		while (!found && !feof(ufile))
		{
			of = 0;
			while (c = fgetc(ufile), !iseol(c) && !feof(ufile))
				if (c != ']')   // skip trailing left bracket
					line[of++] = c;
			line[of] = 0;       // add null
			switch(linenum)
			{
				case 0:         // first line
					// got line 1?
					if (line[0] == '[' && line[1] == 'A')
					{
						x = sscanf(&line[2],"%s %s %s %s %s %s",login_name,
						fname, lname, alias, state, city);
						if (u = strchr(city,']'), u != NULL)
						{       // convert any left brackets to null
							u++;
							u[0] = 0;
						}
						if (strcmp(login_name,logname) == 0)
						{
							// get next line
							linenum++;
						}
					}
					break;
				case 1:         // second line
					// got line 2?
					if (line[0] == '[' && line[1] == 'B')
					{
						if (sscanf(&line[2],"%s %ld %d %d %d %s %d %d %ld",uterm,
							&last_login, &logins, 
							&downloads, &uploads, 
							card->colr, 
							&total_kused,
							&total_timeused, 
							&validated) != 9)
						{
							ap_log("user.get: bad read from user file.  Corrupted?");
							sprintf(finame,"user.get: bad user record was %s, line B",login_name);
							ap_log(finame);
							return(-1);
						}
						// get next line
						linenum++;
						if (!strcmp(card->colr,"red"))
							card_color = 0;
						else if (!strcmp(card->colr,"blue"))
							card_color = 1;
						else if (!strcmp(card->colr,"green"))
							card_color = 2;
						else if (!strcmp(card->colr,"white"))
							card_color = 3;
						else if (!strcmp(card->colr,"grey"))
							card_color = 4;
						else if (!strcmp(card->colr,"pink"))
							card_color = 5;
						else if (!strcmp(card->colr,"yellow"))
							card_color = 6;
						else if (!strcmp(card->colr,"rose"))
							card_color = 7;
						else if (!strcmp(card->colr,"violet"))
							card_color = 8;
						else if (!strcmp(card->colr,"azure"))
							card_color = 9;
						else if (!strcmp(card->colr,"brown"))
							card_color = 10;
						else if (!strcmp(card->colr,"peach"))
							card_color = 11;
						else if (!strcmp(card->colr,"black"))
							card_color = 12;
					}
					break;
				case 2:         // third line
					// got line 3?
					if (line[0] == '[' && line[1] == 'C')
					{
						if (sscanf(&line[2],"%d %d %d %d %s %d %d %s",&priv_msgs,&pub_msgs,&credited, &color_capable, editor, &lines, &cols, lang) != 8)
						{
							ap_log("user.get: bad read from user file.  Corrupted?");
							sprintf(finame,"user.get: bad user record was %s, line C",login_name);
							ap_log(finame);
							return(-1);
						}
						// get next line
						linenum++;
					}
					break;
				case 3:         // fourth line
					// got line 4?
					if (line[0] == '[' && line[1] == 'D')
					{
						if (sscanf(&line[2],"%lx %d %d %d %ld %d %ld %d %d",&flags,
							&acl, &timelimit,
						&timeused,
						&anniversary,&kused,
						&expiration,&c_fore,
						&c_back) != 9)
						{
							ap_log("user.get: bad read from user file.  Corrupted?");
							sprintf(finame,"user.get: bad user record was %s, line D",login_name);
							ap_log(finame);
							return(-1);
						}
						// get next line
						linenum++;
					}
					break;
			}
			if (linenum == 4)   // got him.
			{
				found++;
				bclose(ufile);
				tmpflags = flags;
				tmpacl = acl;
				tmptl = timelimit;
				// get card info (for check_access)
				check_card();
				if (!check)	// don't check his info
						// (sysop editing only)
					return(0);
				logins++;
				// open the language, but only if different
				if (strcmp(DEFAULT_LANGUAGE,lang) != 0)
					l_open(lang);
				time(&now);
				if (!check_access())
					exit(0);	// should never get here
				start_kused = kused;
				if (name == NULL)
				{
					sprintf(line,"Logon for %s %s from %s,%s",fname,lname,city,state);
					ap_log(line);
					// export his environment variables
					for (x=0; x<2; x++)
						export_hard(x);              // export his environment variables
						// all except LINES and COLUMNS, which are exported SOFT
						export_soft();              // export his environment variables
				}
				return(1);      // return 1 for new user only
			}
		}
	}
	bclose(ufile);              // close just in case
	// didn't find him in user file.  Need to get his information
	if (name != NULL)           // error - unable to find user in userlog
	{
		sprintf(line,"Unable to find %s in userlog.",name);
		ap_log(line);
		return(0);
	}
	strcpy(login_name,logname); // get login name
	sstrcrl("UNABLETOFINDUSER");
	sstrl("WOULDLIKETOREGISTER");
	if (!yesno())
	{
		fflush(stdout);
		sprintf(finame,"user.get: %s decided not to register for bbs.",logname);
		ap_log(finame);
		return(-1);
	}
	cr();
	if (userent = getpwnam((char *)username()), userent == NULL)
	{
		sprintf(finame,"user.get: Unable to get user %s's password entry!",logname);
		ap_log(finame);
		return(-1);
	}
	// get user name
	// check to see if unusual user field in the passwd file
	// don't care about more than 2
	if (numwords = sscanf(userent->pw_gecos,"%s %s %s",fname,lname,tmpstr), numwords < 2)
	{
		// don't know what I got from the passwd file, so,
		// get their first and last names
		clear_scr();
		sstrcrl("USERNAMENOTFOUND");
		cr();
		sstrcrl("PLEASEENTERNAME");
		cr();
		done = 0;
		while (!done)
		{
			cr();
			sstrl("USERFIRSTNAME");
			tmpstr[0] = 0;
			if (gstr(tmpstr,20), strcmp(tmpstr,"") == 0)
				continue;
			strcpy(fname,tmpstr);
			sprintf(tmpstr,gstrl("YOUENTERED"),fname);
			sstrcr_c(tmpstr);
			sstrl("CORRECT");
			if (!yesno())
				continue;
			done++;

		}
		cr();
		done = 0;
		while (!done)
		{
			cr();
			sstrl("USERLASTNAME");
			tmpstr[0] = 0;
			if (gstr(tmpstr,20), strcmp(tmpstr,"") == 0)
				continue;
			strcpy(lname,tmpstr);
			sprintf(tmpstr,gstrl("YOUENTERED"),lname);
			sstrcr_c(tmpstr);
			sstrl("CORRECT");
			if (!yesno())
				continue;
			done++;
		}
	}
	getinfo();                  // get user information
	forward_email();		// create .forward, if applicable
	for (x=0; x<2; x++)
		export_hard(x);              // export his environment variables
		// all except LINES and COLUMNS, which are exported SOFT
		export_soft();              // export his environment variables
	check_card();               // check his access level vs. card
	if (!check_access())
		exit(0);	// should never get here
	append();	// add to userlog
	sprintf(line,"Logon for %s %s from %s,%s",fname,lname, city, state);
	ap_log(line);
	return(0);
}

// Function:   get_editor
// Purpose:    prompt for the editor for the user to use.
// Input:      none
// Output:     the user is prompted for the editor to use.
// Author:     Greg Shaw
// Created:    9/21/96

int User::get_editor(void)
{
	FILE	*infile;
	char	command[MAX_EDITORS][20];	// editor command
	char	text[MAX_EDITORS][255];		// description
	char	key[MAX_EDITORS];		// key to select
	char	*line;
	char	tmpstr[255];
	char	*c,*t;
	char	k;
	int	numcommands;
	int	x;
	int	done;

	clear_scr();
	cr();
	sstrcrl("USERINFOEDITOR");
	sprintf(tmpstr,"%s/config/editors",getenv("BBSDIR"));
	// display file with paging
	if (infile = bopen(tmpstr,"r"), infile == NULL)
		return(-1);
	else	// read the file
	{
		numcommands = 0;
		while (!feof(infile))
		{
			line = getline(infile);
			if (c = strchr(line,'#'), c != NULL)
				*c = 0;	// ignore until end of line on comment
			if (c = strchr(line,'|'), c == NULL)
				continue;
			// get command
			c = line;
			t = command[numcommands];
			while (*c != '|' && *c != 0)
				*t++ = *c++;
			*t = 0;
			c++;
			// get key
			key[numcommands] = *c;
			c++;
			c++;
			t = text[numcommands];
			while (*c != 0)
				*t++ = *c++;
			*t = 0;
			numcommands++;
			if (numcommands>=MAX_EDITORS)
			{
				ap_log("Too many editors in editors file!");
				numcommands--;
			}
		}
		bclose(infile);
		if (numcommands == 0)	// do we have anything to display?
			return(-1);
		for (x=0; x<numcommands; x++)
			sstrcr_c(text[x]);
		done = 0;
		while (!done)
		{
			sstrl("YOURCHOICE");
			tmpstr[0] = 0;
			if (gstr(tmpstr,5), strcmp(tmpstr,"") == 0)
				continue;
			else
			{
				done = 0;
				sscanf(tmpstr,"%c",&k);
				for (x=0; x<numcommands; x++)
					if (key[x] == k)
					{
						strcpy(editor,command[x]);
						done++;
					}
			}
			if (!done)
				continue;
			sprintf(tmpstr,gstrl("YOUVESELECTED"),editor);
			sstrcr_c(tmpstr);
			sstrl("CORRECT");
			if (!yesno())
			{
				done = 0;
				continue;
			}
			// export to environment
			export_hard(0);
			done++;
		}
	}
	return(0);
}

// Function:   getinfo
// Purpose:        prompt the user for information to setup the user object
// Input:      none
// Output:     the user is prompted for their environment
// Author:     Greg Shaw
// Created:        7/25/93

int User::getinfo(void)
{
	char   loop;                // loop counter
	char   done;                // loop counter
	char   entered;             // entered anything?
	char   tmpstr[255];
	char   anotherstr[50];      // another string
	char   finame[255];         // filename
	char   logname[10];         // user login name

	loop = 1;
	entered = 0;
	while (loop)
	{
		clear_scr();            // clear screen
		if (entered)
		{
			cr();
			cr();
			cr();
			sstrcrl("ENTERFOLLOWING");
			cr();
			sprintf(finame,gstrl("LOGINNAME"),login_name);
			sstrcr_c(finame);
			sprintf(finame,gstrl("REALNAME"),fname,lname);
			sstrcr_c(finame);
			sprintf(finame,gstrl("ALIASNAME"),alias);
			sstrcr_c(finame);
			sprintf(finame,gstrl("CALLINGFROM"),city,state);
			sstrcr_c(finame);
			sprintf(finame,gstrl("LANGUAGECHOSEN"),lang);
			sstrcr_c(finame);
			sprintf(finame,gstrl("TERMINALTYPE"),uterm);
			sstrcr_c(finame);
			sprintf(finame,gstrl("TERMINALCAPABILITIES"),color_capable?gstrl("YES"):gstrl("NO"));
			sstrcr_c(finame);
			sprintf(finame,gstrl("EDITOR"),editor);
			sstrcr_c(finame);
			sprintf(finame,gstrl("LINESCOLUMNS"),lines,cols);
			sstrcr_c(finame);
			cr();
			cr();
			sstrl("CORRECT");
			if (yesno())
				loop = 0;
		}
		if (loop)               // still looping?
		{
			#ifndef ONE_LANGUAGE_ONLY
			// get language
			select_lang(lang);
			#else
			strcpy(lang,DEFAULT_LANGUAGE);
			#endif
			// get user's login information
			strcpy(logname,username());
			cr();
			done = 0;
			while (!done)
			{
				sstrl("USERINFOCITY");
				tmpstr[0] = 0;
				if (gstr(tmpstr,20), strcmp(tmpstr,"") == 0)
					continue;
				else
					strcpy(city,tmpstr);
				done++;
			}
			cr();
			done = 0;
			while (!done)
			{
				sstrl("USERINFOSTATE");
				tmpstr[0] = 0;
				if (gstr(tmpstr,2), strcmp(tmpstr,"") == 0)
					continue;
				else
					strcpy(state,tmpstr);
				done++;
			}
			cr();
			done = 0;
			while (!done)
			{
				sstrcrl("USERINFOTERM0");
				cr();
				sstrcrl("USERINFOTERM1");
				cr();
				sstrcrl("USERINFOTERM2");
				cr();
				sstrl("USERINFOTERM3");
				tmpstr[0] = 0;
				if (gstr(tmpstr,20), strcmp(tmpstr,"") == 0 || strchr(tmpstr,' ') != NULL)
					continue;
				else
					strcpy(uterm,tmpstr);
				done++;
			}
			cr();
			cr();
			sstrl("USERINFOCOLOR");
			if (yesno())
				color_capable = 1;
			else
				color_capable = 0;
			cr();
			cr();
			if (strcpy(finame,getenv("BBSDIR")), finame == NULL)
			{
				sprintf(finame,"user.get: BBSDIR env var not set for %s",logname);
				ap_log(finame);
				return(-1);
			}
			get_editor();	// select editor
			cr();
			cr();
			sstrcrl("USERINFOALIAS0");
			cr();
			sstrl("USERINFOALIAS1");
			if (yesno())
				strcpy(alias,logname);
			else
			{
				done = 0;
				while (!done)
				{
					sstrl("USERINFOALIAS2");
					tmpstr[0] = 0;
					if (gstr(tmpstr,20), strcmp(tmpstr,"") == 0)
						continue;
					else
						if (sscanf(tmpstr,"%s%s",alias,anotherstr) != 1)
							continue;
					done++;
				}
			}
			cr();
			cr();
			done = 0;
			while (!done)
			{
				sstrl("USERINFOLINES");
				tmpstr[0] = 0;
				if (gstr(tmpstr,5), strcmp(tmpstr,"") == 0)
					continue;
				if (tmpstr[0] == 'x')
				{
					lines = 24;
				}
				else
					sscanf(tmpstr,"%d",&lines);
				done++;
			}
			cr();
			cr();
			done = 0;
			while (!done)
			{
				sstrl("USERINFOCOLUMNS");
				tmpstr[0] = 0;
				if (gstr(tmpstr,5), strcmp(tmpstr,"") == 0)
					continue;
				if (tmpstr[0] == 'x')
				{
					cols = 80;
				}
				else
					sscanf(tmpstr,"%d",&cols);
				done++;
			}
			entered++;
		}
	}
	export_soft();
	if (save(login_name))
	{
		ap_log("Unable to save user.");
	}
	return(0);
};


// Function:   inactive_list
// Purpose:    list the inactive (people who haven't called lately) users
// Input:  none
// Output: A list of users who haven't called in a specified time
// Author: Greg Shaw
// Created:    4/3/94

int User::inactive_list(void)
{
	char   tmpstr[255];
	char   line[150];
	char   c;
	char   key[MAX_FILENAMELENGTH+1];
	char   uname[10];
	char   fullname[50];
	char   fname[15];
	char   lname[15];
	char   tmp[2][15];
	char   state[50];
	char   city[50];
	char   llogon[10];
	char   from[50];
	char   *sptr;
	int    logins;
	int    uls,dls;
	time_t laston;
	time_t today;
	struct tm *tmdata;
	int    off;
	int    found;
	int    days;
	FILE   *infile;


	clear_scr();
	time(&today);
	// get number of days inactive to search for
	sstrcrl("USERSEARCH0");
	cr();
	sstrl("USERSEARCH1");
	key[0] = 0;
	gstr(key,MAX_FILENAMELENGTH);
	if (key[0] == 0 || sscanf(key,"%d",&days) != 1)
		return(0);
	sprintf(tmpstr,"%s/admin/userlog",getenv("BBSDIR"));
	if (infile = bopen(tmpstr,"r"), infile == NULL)
	{
		ap_log("Unable to open userlog for read.");
		return(0);
	}
	found = 0;
	sstrcrl("USERSEARCH2");
	while (!feof(infile))
	{
		off = 0;
		while (c = fgetc(infile), !iseol(c) && !feof(infile))
			line[off++] = c;
		line[off] = 0;
		if (line[0] == '[')
		{
			if (line[1] == 'A')
			{
				tmp[0][0] = 0;
				tmp[1][0] = 0;
				sscanf(&line[2],"%s %s %s %s %s %s %s %s", uname, fname, lname, alias, state, city,tmp[0], tmp[1]);
				strcat(city,tmp[0]);
				strcat(city,tmp[1]);
				if (sptr = strchr(city,']'), sptr != NULL)
					sptr[0] = 0;
			} else if (line[1] == 'B')
			{
				sscanf(&line[2],"%*s %ld %d %d %d %*s %*d %*d %*d",&laston, &logins,&uls,&dls);
				sprintf(fullname,"%s %s",fname,lname);
				sprintf(from,"%s, %s",city,state);
				tmdata = localtime(&laston);
				strftime(llogon, 12, "%b %d,%Y", tmdata);
				if ((today - laston)/86400 >= days)
				{
					sprintf(tmpstr,"%-15s %-15s %s   %d    %d",fullname,from,llogon,uls,dls);
					sstrcr_c(tmpstr);
					found++;
				}
				if (found == 17)
				{
					found = 0;
					waitcr();
					sstrl("CONTINUELISTING");
					if (!yesno())
					{
						bclose(infile);
						return(0);
					}
					else
						clear_scr();
				}
			}
		}
	}
	cr();
	sstrcrl("NOMOREFOUND");
	bclose(infile);
	waitcr();
	return(0);
};


// Function:   inactive_delete
// Purpose:    delete inactive users on the BBS
// Input:  none
// Output: A list of users who haven't called in a specified time
// Author: Greg Shaw
// Created:    4/21/94

int User::inactive_delete(void)
{
	char   tmpstr[255];
	char   line[150];
	char   c,b;
	char   key[MAX_FILENAMELENGTH+1];
	char   uname[10];
	char   fullname[50];
	char   fname[15];
	char   lname[15];
	char   tmp[2][15];
	char   state[50];
	char   city[50];
	char   llogon[10];
	char   from[50];
	char   *sptr;
	int    x;
	int    logins;
	int    uls,dls;
	int    maxk;
	int    maxt;
	time_t laston;
	time_t today;
	time_t valid;
	struct tm *tmdata;
	int    off;
	int    found;
	int    days;
	FILE   *infile;
	FILE   *outfile;


	clear_scr();
	time(&today);
	// get number of days inactive to search for
	sstrcrl("SEARCHTODELETE0");
	cr();
	sstrl("SEARCHTODELETE1");
	key[0] = 0;
	gstr(key,MAX_FILENAMELENGTH);
	if (key[0] == 0 || sscanf(key,"%d",&days) != 1)
		return(0);
	sprintf(tmpstr,gstrl("WILLDELETE"),days);
	sstrcr_c(tmpstr);
	cr();
	sstrl("SEARCHTODELETE2");
	while (b = tolower(gch(1)), b != 'a' && b != 'i');
	cr();
	if (b == 'i')               // interactively
	{
		sstrcrl("PROMPTENABLED");
		waitcr();
	}
	sprintf(tmpstr,"%s/admin/userlog",getenv("BBSDIR"));
	if (infile = bopen(tmpstr,"r"), infile == NULL)
	{
		ap_log("Unable to open userlog for read.");
		sstrcrl("NOOPENREAD");
		waitcr();
		return(0);
	}
	strcat(tmpstr,".new");      // get 'new' filename
	if (outfile = bopen(tmpstr,"w"), outfile == NULL)
	{
		ap_log("Unable to open userlog.new for write.");
		sstrcrl("NOOPENWRITE");
		waitcr();
		return(0);
	}
	found = 0;
	while (!feof(infile))
	{
		off = 0;
		while (c = fgetc(infile), !iseol(c) && !feof(infile))
			line[off++] = c;
		line[off] = 0;
		if (line[0] == '[')
		{
			if (line[1] == 'A')
			{
				tmp[0][0] = 0;
				tmp[1][0] = 0;
				sscanf(&line[2],"%s %s %s %s %s %s %s %s", uname, fname, lname, alias, state, city,tmp[0], tmp[1]);
				strcat(city,tmp[0]);
				strcat(city,tmp[1]);
				if (sptr = strchr(city,']'), sptr != NULL)
					sptr[0] = 0;
			} else if (line[1] == 'B')
			{
				if (sscanf(&line[2],"%s %ld %d %d %d %s %d %d %ld",uterm,
					&laston,&logins,&uls,
					&dls,card->colr,&maxk,&maxt,&valid) != 9)
				{
					sprintf(tmpstr,"Error found in line B userlog in entry %s %s",fname,lname);
					ap_log(tmpstr);
					sstrcr_c("Error found in userlog.  Clean aborted.");
					waitcr();
					return(0);
				}
				sprintf(fullname,"%s %s",fname,lname);
				sprintf(from,"%s, %s",city,state);
				tmdata = localtime(&laston);
				strftime(llogon, 12, "%b %d,%Y", tmdata);
				if ((today - laston)/86400 >= days)
				{
					sprintf(tmpstr,"%-15s %-15s %s   %d    %d",fullname,from,llogon,uls,dls);
					sstrcr_c(tmpstr);
					// interactive?
					if (b == 'i')
					{
						sstrl("DELETEUSER");
						if (yesno())
							continue;
					}
					else
						continue;
				}
				for (x=0; x<2; x++)
					if (strlen(tmp[x]) < 2)
						tmp[x][0] = 0;
				// save info and next two lines
				fprintf(outfile,"[A %s %s %s %s %s %s %s %s ]\n", uname, fname, lname, alias, state, city, tmp[0], tmp[1]);
				fprintf(outfile,"%s\n",line);
				x = 0;
				while (x < 2)
				{
					if (c = fgetc(infile), iseol(c))
						x++;
					fputc(c,outfile);
				}
			}
		}
	}
	cr();
	bclose(outfile);
	sprintf(tmpstr,"%s/admin/userlog",getenv("BBSDIR"));
	sprintf(line,"%s/admin/userlog.old",getenv("BBSDIR"));
	rename(tmpstr,line);
	sprintf(tmpstr,"%s/admin/userlog.new",getenv("BBSDIR"));
	sprintf(line,"%s/admin/userlog",getenv("BBSDIR"));
	rename(tmpstr,line);
	waitcr();
	return(0);
};


// Function:   list
// Purpose:    return a user record
// Input:  path - the path to the user file
// Output: a list of users on the bbs -
// Author: Greg Shaw
// Created:    6/8/93

int User::list(int search, int sysop)
{
	char   tmpstr[255];
	char   line[150];
	char   c;
	char   key[MAX_FILENAMELENGTH+1];
	char   uname[10];
	char   alias[25];
	char   fname[15];
	char   lname[15];
	char   tmp[2][15];
	char   state[50];
	char   city[50];
	char   *sptr;
	int    logins;
	time_t laston;
	int    off;
	int    found;
	FILE   *infile;


	clear_scr();
	sprintf(tmpstr,"%s/admin/userlog",getenv("BBSDIR"));
	if (infile = bopen(tmpstr,"r"), infile == NULL)
	{
		ap_log("Unable to open userlog for read.");
		return(0);
	}
	if (search)
	{
		sstrcrl("SEARCHFORUSER");
		cr();
		sstrl("SEARCHFORCHARS");
		key[0] = 0;
		gstr(key,MAX_FILENAMELENGTH);
		if (key[0] == 0)
			return(0);
		// search for characters in userlog
		found = 0;
		while (!feof(infile))
		{
			off = 0;
			while (c = fgetc(infile), !iseol(c) && !feof(infile))
				line[off++] = c;
			line[off] = 0;
			if (line[0] == '[')
			{
				if (line[1] == 'A')
				{
					tmp[0][0] = 0;
					tmp[1][0] = 0;
					sscanf(&line[2],"%s %s %s %s %s %s %s %s", uname, fname, lname, alias, state, city,tmp[0], tmp[1]);
					strcat(city,tmp[0]);
					strcat(city,tmp[1]);
					if (sptr = strchr(city,']'), sptr != NULL)
						sptr[0] = 0;
					if (strstr(line,key) != NULL)
					{
						sprintf(tmpstr,gstrl("NAMEFROM"),fname,lname,city,state);
						sstrcr_c(tmpstr);
						sprintf(tmpstr,gstrl("LOGINALIAS"),uname,alias);
						sstrcr_c(tmpstr);
						found++;
					}
				} else if (found && line[1] == 'B')
				{
					sscanf(&line[2],"%*s %ld %d %*d %*d %*s %*d %*d %*d",&laston, &logins);
					sptr = ctime(&laston);
					// chop \n
					sptr[strlen(sptr)-1] = 0;
					sprintf(tmpstr,gstrl("NUMLOGONS"),logins,sptr);
					sstrcr_c(tmpstr);
					waitcr();
					if (sysop)
					{
						sstrl("CONTINUEEDITDELETE");
						while (c = toupper(gch(1)), c != 'D' && c != 'C' && c != 'E');
						cr();
						if (c == 'D')
						{
							bclose(infile);
							delete_user(uname);
							return(0);
						}
						else if (c == 'E')
						{
							bclose(infile);
							if (!get(uname,0))
							{
								check_card();
								sysop_edit();
							}
							else
							{
								sprintf(tmpstr,"Unable to get %s for sysop edit.",uname);
								ap_log(tmpstr);
							}
							return(0);
						}
					}
					else
					{
						sstrl("CONTINUESEARCH");
						if (!yesno())
						{
							bclose(infile);
							return(0);
						}
					}
					found = 0;
				}
			}
		}
		bclose(infile);
	}
	else
	{
		found = 0;
		while (!feof(infile))
		{
			off = 0;
			while (c = fgetc(infile), !iseol(c) && !feof(infile))
				line[off++] = c;
			if (line[0] == '[')
			{
				if (line[1] == 'A')
				{
					tmp[0][0] = 0;
					tmp[1][0] = 0;
					sscanf(&line[2],"%s %s %s %s %s %s %s %s", uname, fname, lname, alias, state, city,tmp[0], tmp[1]);
					strcat(city,tmp[0]);
					strcat(city,tmp[1]);
					if (sptr = strchr(city,']'), sptr != NULL)
						sptr[0] = 0;
					sprintf(tmpstr,"%s %s from %s, %s",fname,lname,city,state);
					sstrcr_c(tmpstr);
					sprintf(tmpstr,gstrl("LOGINALIAS"),uname,alias);
					sstrcr_c(tmpstr);
				} else if (line[1] == 'B')
				{
					sscanf(&line[2],"%*s %ld %d %*d %*d %*s %*d %*d %*d",&laston, &logins);
					sptr = ctime(&laston);
					// chop \n
					sptr[strlen(sptr)-1] = 0;
					sprintf(tmpstr,gstrl("NUMLOGONS"),logins,sptr);
					sstrcr_c(tmpstr);
					cr();
					if (found == 5)
					{
						found = 0;
						waitcr();
						sstrl("CONTINUELISTING");
						if (!yesno())
						{
							bclose(infile);
							return(0);
						}
						clear_scr();
					}
					found++;
				}
			}
		}
		cr();
	}
	cr();
	sstrcrl("NOMOREFOUND");
	waitcr();
	return(0);
}


// Function:   mailavail
// Purpose:    return true if mail for the user is available
// Input:  none
// Output: 1 for mail.  0 for no mail
// Author:     Greg Shaw
// Created:        8/10/93

int User::mailavail(void)
{
	struct stat fistat;         // file status record
	char   tmpstr[255];         // tmpstr
	time_t now = 0L;            // current time
	static time_t  then = 0L;   // previous check time

	if (!checkmail())           // check?
		return(0);              // no, just exit
	time(&now);
	// seconds elapsed?
	if (abs(now - then) > mailchecktime())
	{
		sprintf(tmpstr,"%s/%s",mailspool(),login_name);
		if (stat(tmpstr,&fistat) == 0 && S_ISREG(fistat.st_mode))
		{
			if (mail_check == 0)// just check size the first time
			{
				// save modification time
				mail_check = fistat.st_mtime;
				if (mail_size = fistat.st_size, mail_size > 0)
					return(1);
			}
			else                // check date of last mail update
			{
				if (fistat.st_mtime != mail_check &&
					// new mail (mailbox bigger)
					fistat.st_size > mail_size)
				{
					return(1);
				}
				mail_check = fistat.st_mtime;
				mail_size = fistat.st_size;
			}
		}
	}
	return(0);
};


// Function:   save
// Purpose:    save the current User to the User file
// Input:      path - the path to the User file
// Output:     returns non-zero on failure
// Author:     Greg Shaw
// Created:        6/5/93

int User::save(char *name)
{
	FILE   *ufile;
	FILE   *ofile;
	char   finame[255];         // filename
	char   line[255];           // one line in file (255 max)
	char   *bbspath;            // bbs path
	char   logname[10];         // user login name
	char   found;               // user found?
	char   linenum;             // used to look for appropriate line
	char   c;                   // input character
	unsigned char  of;          // offset into line
	time_t now;
	int	bytes;			// bytes written to file




	last_login = login_time;    // set to new login time
	time(&now);
	// update time used field.
	timeused += (now - login_time)/60;
	total_timeused += timeused;	// total timeused (in hours)
	// don't update unless they've downloaded something (more)
	if (kused != start_kused)
		total_kused += kused-start_kused;
	if (name == NULL)
		// get user's log information
		strcpy(logname,username());
	else
		strcpy(logname,name);
	if (bbspath = getenv("BBSDIR"), bbspath == NULL)
	{
		sprintf(finame,"user.save: BBSDIR env var not set for %s",logname);
		ap_log(finame);
		return(-1);
	}
	sprintf(line,"%s/admin/nuserlog",bbspath);
	if (ofile = bopen(line,"w"), ofile == NULL)
	{
		sprintf(line,"user.save: Unable to open new userlog file: %s",logname);
		ap_log(line);
		ap_log("Please check permissions of admin directory and rocat program.");
		ap_log("See section in the manual pertaining to file permissions.");
		return(-1);
	}
	// change permissions
	chmod(line,S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);
	sprintf(finame,"%s/admin/userlog",bbspath);
	linenum = 0;                // look for first line first
	if (ufile = bopen(finame,"r"), ufile == NULL)
	{
		// create the bugger so that you can continue
		ufile = bopen(finame,"w");
		bclose(ufile);
		if (ufile = bopen(finame,"r"), ufile == NULL)
		{                       // still can't open the file!  AARGH!
			sprintf(line,"user.save: Unable to open userlog file.");
			ap_log(line);
			return(-1);
		}
	}
	else
	{
		found = 0;
		while (!found && !feof(ufile))
		{
			// get first line
			of = 0;
			while (c = fgetc(ufile), !iseol(c) && !feof(ufile))
				if (c != ']')   // skip trailing left bracket
					line[of++] = c;
			line[of] = 0;       // add null
			fflush(ofile);
			// end of file, skip.
			if (!feof(ufile) && of > 1)
			{
				switch(linenum)
				{
					case 0:     // first line
						// got line 1?
						if (line[0] == '[' && line[1] == 'A')
						{
							sscanf(&line[2],"%s %*s %*s %*s %*s %*s",logname);
							if (strcmp(login_name,logname) == 0)
							{
								// get next line
								linenum++;
								bytes = fprintf(ofile,"[A %s %s %s %s %s %s ]\n", login_name, fname, lname, alias, state, city);
							}
							else
								bytes = fprintf(ofile,"%s]\n",line);
							//                 [A login_name firstname lastname state city]
						}
						else
							bytes = fprintf(ofile,"%s]\n",line);
						break;
					case 1:     // second line
						// got line 2?
						if (line[0] == '[' && line[1] == 'B')
						{
							bytes = fprintf(ofile,"[B %s %ld %d %d %d %s %d %d %ld ]\n", uterm,
							last_login, logins,
							downloads, uploads,
							card->colr,
							total_kused,total_timeused,validated);
							//                 [B terminal_type last_logon #_logins downloads uploads]
							// get next line
							linenum++;
						}
						else
							fprintf(ofile,"%s]\n",line);
						break;
					case 2:     // third line
						// got line 3?
						if (line[0] == '[' && line[1] == 'C')
						{
							bytes = fprintf(ofile,"[C %d %d %d %d %s %d %d %s ]\n",priv_msgs,
							pub_msgs, credited,
							color_capable, editor,
							lines, cols, lang);
							// save next line
							linenum++;
							//                 [C private_msgs public_msgs credited_time color_capable editor lines cols language]
						}
						else
							bytes = fprintf(ofile,"%s]\n",line);
						break;
					case 3:     // fourth line
						// got line 4?
						if (line[0] == '[' && line[1] == 'D')
						{
							bytes = fprintf(ofile,"[D %lx %d %d %d %ld %d %ld %d %d ]\n",
							flags, acl,
							timelimit,
							timeused,anniversary,
							kused, expiration, 
							c_fore, c_back);
							// [D flags access_level timelimit timeused_last_call anniversary kused expiration chat_foreground, chat_background ]
							// save next line
							linenum++;
						}
						else
							bytes = fprintf(ofile,"%s]\n",line);
						break;
				}
				// got him.
				if (linenum == 4)
				{
					found++;
				}
			}
		}
		// Ok, got the guy.  copy the rest of the file
		while (c = fgetc(ufile), !feof(ufile))
		{
			fputc(c,ofile);
		}
	}
	// now rename the userfile to old userfile
	bclose(ufile);
	bclose(ofile);
	strcpy(line,bbspath);
	strcat(line,"/admin/userlog.old");
	if (rename(finame, line) != 0)
	{
		sprintf(finame,"user.save: unable to rename userlog to userlog.old");
		ap_log(finame);
		return(-1);
	}
	strcpy(line,bbspath);
	strcat(line,"/admin/nuserlog");
	strcpy(finame,bbspath);
	strcat(finame,"/admin/userlog");
	if (rename(line, finame) != 0)
	{
		sprintf(finame,"user.save: unable to rename new userlog to userlog");
		ap_log(finame);
		return(-1);
	}
	// change permissions
	chmod(finame,S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);
	return(0);
}


// Function:	stringornum
// Purpose:	return 'unlimited' for a negative value or the number
// Input:	num - the number to check
// Output:	unlimited for negative number, or a string with the number
// Author:	Greg Shaw
// Created:	8/24/95
// Note:	since the string to be used is static, it isn't possible
//		to call the function twice in one source line.
//		 ex: sprintf("%s %s",stringornum(a),stringornum(b));

char *User::stringornum(int num)
{
	static char tmpstr[30];	

	if (num < 0)
		return(gstrl("UNLIMITED"));
	else
		sprintf(tmpstr,"%d",num);
	return(tmpstr);
}

// Function:       sysop_edit
// Purpose:        allow the sysop to edit a user
// Input:      firsttime - is this the first logon by user?
// Output:     (user questions)
// Author:     Greg Shaw
// Created:        4/11/94

int User::sysop_edit(void)
{
	char   c;
	char   loop;                // loop counter
	char   changed;             // changed anything?
	char   tmpstr[255];
	char   str1[50], str2[50];
	int    selected;            // which selected?
	int    tmp;
	int    x;
	struct tm *tmdata;          // time stuff

	loop = 1;
	changed = 0;
	while (loop)
	{
		clear_scr();            // clear screen
		cr();
		cr();
		cr();
		sstrcrl("CURRENTUSER");
		cr();
		sprintf(tmpstr,gstrl("LOGINFULLNAME"),login_name,fname,lname,city,state);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("ALIASDOWNLOADS"),alias, downloads, uploads, logins);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("ACLFLAGS"),acl, flags, timelimit);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("TERMTYPECOLOR"),uterm,card->colr,lines,cols);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("TOTALTIMEUSED"),total_timeused,(float)total_timeused/60.0);
		sstrcr_c(tmpstr);
		tmdata = localtime(&anniversary);
		strftime(str1,49,"%c",tmdata);
		tmdata = localtime(&last_login);
		strftime(str2,49,"%c",tmdata);
		sprintf(tmpstr,gstrl("FIRSTLOGON"),str1);
		sstrcr_c(tmpstr);
		sprintf(tmpstr,gstrl("LASTLOGON"),str2,timeused);
		sstrcr_c(tmpstr);
		cr();
		cr();
		sstrl("CORRECT");
		if (yesno())
		{
			if (changed)
			{
				sstrcrl("SAVEUSERINFO");
				if (save(login_name))
				{
					ap_log("Unable to save user.");
				}
				// write key file
				sprintf(tmpstr,"%s/admin/%s.acl",getenv("BBSDIR"),login_name);
				if (creat(tmpstr,0777) == -1)
				{
					ap_log("sysop_edit: Unable to write key file");
				}
			}
			return(0);
		}
		cr();
		clear_scr();
		sstrcrl("EDITUSERMENU");
		sstrcrl("RETURNTOEXIT");
		cr();
		sstrl("YOURCHOICE");
		str1[0] = 0;
		gstr(str1,3);
		selected = 0;
		#ifndef ONE_LANGUAGE_ONLY
		if (sscanf(str1,"%d",&selected) != 1 || (selected < 1 || selected > 17))
			continue;
		#else
		if (sscanf(str1,"%d",&selected) != 1 || (selected < 1 || selected > 16))
			continue;
		#endif
		switch(selected)
		{
			case 1:             //  get login name
				sstrcrl("NONAMECHANGE");
				cr();
				sstrcrl("RETURNTOEXIT");
				waitcr();
				break;          // not done at this time (requires
				// modification of passwd file)
			case 2:             // get first name
				sprintf(tmpstr,gstrl("CURRENTFIRSTNAME"),fname);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGEFIRSTNAME");
				str1[0] = 0;
				gstr(str1,20);
				if (sscanf(str1,"%s",str2) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				strcpy(fname,str2);
				changed++;
				break;
			case 3:             // get last name
				sprintf(tmpstr,gstrl("CURRENTLASTNAME"),lname);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGELASTNAME");
				str1[0] = 0;
				gstr(str1,20);
				if (sscanf(str1,"%s",str2) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				strcpy(lname,str2);
				changed++;
				break;
			case 4:             // get city
				sprintf(tmpstr,gstrl("CURRENTCITY"),city);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGECITY");
				str1[0] = 0;
				gstr(str1,20);
				if (sscanf(str1,"%s",str2) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				strcpy(city,str2);
				changed++;
				break;
			case 5:             // get state
				sprintf(tmpstr,gstrl("CURRENTSTATE"),state);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGESTATE");
				str1[0] = 0;
				gstr(str1,14);
				if (sscanf(str1,"%s",str2) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				strcpy(state,str2);
				changed++;
				break;
			case 6:             // get alias
				sprintf(tmpstr,gstrl("CURRENTALIAS"),alias);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGEALIAS");
				str1[0] = 0;
				gstr(str1,20);
				if (sscanf(str1,"%s",str2) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				strcpy(alias,str2);
				changed++;
				break;
			case 7:             // get downloads
				sprintf(tmpstr,gstrl("CURRENTDOWNLOADS"),downloads);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGEDOWNLOADS");
				str1[0] = 0;
				gstr(str1,4);
				if (sscanf(str1,"%d",&tmp) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				downloads = tmp;
				changed++;
				break;
			case 8:             // get uploads
				sprintf(tmpstr,gstrl("CURRENTUPLOADS"),uploads);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGEUPLOADS");
				str1[0] = 0;
				gstr(str1,4);
				if (sscanf(str1,"%d",&tmp) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				uploads = tmp;
				changed++;
				break;
			case 9:             // get access level
				sprintf(tmpstr,gstrl("CURRENTACL"),acl);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGEACL");
				str1[0] = 0;
				gstr(str1,5);
				if (sscanf(str1,"%d",&tmp) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				acl = tmp;
				tmpacl = tmp;
				changed++;
				break;
			case 10:            // get flags
				sstrcrl("CURRENTFLAGS");
				for (x=0; x<32; x++)
				{
					if (x%8 == 0 && x != 0)
						cr();
					sprintf(tmpstr,"%.2d:%d ",x,flags&1<<x?1:0);
					sstr_c(tmpstr);
				}
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("SETORCLEAR");
				str1[0] = 0;
				gstr(str1,2);
				size_t len;
				for (len=0; len<strlen(str1); len++)
					str1[len] = tolower(str1[len]);
				if (strchr(str1,'s') != NULL)
				{
					sstrl("SETFLAG");
					str1[0] = 0;
					gstr(str1,3);
					if (sscanf(str1,"%d",&tmp) != 1 || (tmp < 0 || tmp > 31))
					{
						sstrcrl("NOTCHANGED");
						waitcr();
						continue;
					}
					flags |= 1<<tmp;
				}
				else if (strchr(str1,'c') != NULL)
				{
					sstrl("CLEARFLAG");
					str1[0] = 0;
					gstr(str1,3);
					if (sscanf(str1,"%d",&tmp) != 1 || (tmp < 0 || tmp > 31))
					{
						sstrcrl("NOTCHANGED");
						waitcr();
						continue;
					}
					flags &= ~(1<<tmp);
				}
				else
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				changed++;
				break;
			case 11:            // get timelimit
				sprintf(tmpstr,gstrl("CURRENTTIMELIMIT"),timelimit);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGETIMELIMIT");
				str1[0] = 0;
				gstr(str1,5);
				if (sscanf(str1,"%d",&tmp) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				timelimit = tmp;
				changed++;
				break;
			case 12:            // get terminal types
				sprintf(tmpstr,gstrl("CURRENTTERMTYPE"),uterm);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGETYPE");
				str1[0] = 0;
				gstr(str1,20);
				if (sscanf(str1,"%s",str2) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				strcpy(uterm,str2);
				changed++;
				break;
			case 13:            // get card color
				switch(card_color)
				{
					case 0:
						strcpy(str2,gstrl("RED"));
						break;
					case 1:
						strcpy(str2,gstrl("BLUE"));
						break;
					case 2:
						strcpy(str2,gstrl("GREEN"));
						break;
					case 3:
						strcpy(str2,gstrl("WHITE"));
						break;
					case 4:
						strcpy(str2,gstrl("GREY"));
						break;
					case 5:
						strcpy(str2,gstrl("PINK"));
						break;
					case 6:
						strcpy(str2,gstrl("YELLOW"));
						break;
					case 7:
						strcpy(str2,gstrl("ROSE"));
						break;
					case 8:
						strcpy(str2,gstrl("VIOLET"));
						break;
					case 9:
						strcpy(str2,gstrl("AZURE"));
						break;
					case 10:
						strcpy(str2,gstrl("BROWN"));
						break;
					case 11:
						strcpy(str2,gstrl("PEACH"));
						break;
					case 12:
						strcpy(str2,gstrl("BLACK"));
				}
				sprintf(tmpstr,gstrl("CURRENTCARDCOLOR"),str2);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrcrl("CHANGECOLOR");
				sstrcrl("COLORMENU");
				cr();
				cr();
				sstrl("YOURCHOICE");
				while (c = toupper(gch(1)), 
					!iseol(c) && 
					c <= 0);
				if (!iseol(c))
				{
					if (isdigit(c) && c != '0')
						card_color = c-'0'-1;
					else if (c >= 'A' && c < 'E')
						card_color = c - 'A' + 9;
					else	// nothing valid
						break;
					free(card);	// nuke old card
					card = cardinfo(card_color);
					check_card();
					changed++;
				}
				cr();
				break;
			case 14:            // get lines
				sprintf(tmpstr,gstrl("CURRENTLINES"),lines);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGELINES");
				str1[0] = 0;
				gstr(str1,4);
				if (sscanf(str1,"%d",&tmp) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				lines = tmp;
				changed++;
				break;
			case 15:            // get columns
				sprintf(tmpstr,gstrl("CURRENTCOLUMNS"),cols);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGECOLUMNS");
				str1[0] = 0;
				gstr(str1,4);
				if (sscanf(str1,"%d",&tmp) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				cols = tmp;
				changed++;
				break;
			case 16:            // get total timelimit
				sprintf(tmpstr,gstrl("CURRENTTOTALTIME"),total_timeused,(float)total_timeused/60.0);
				sstrcr_c(tmpstr);
				cr();
				sstrcrl("RETURNTOEXIT");
				cr();
				sstrl("CHANGETOTALTIME");
				str1[0] = 0;
				gstr(str1,7);
				if (sscanf(str1,"%d",&tmp) != 1)
				{
					sstrcrl("NOTCHANGED");
					waitcr();
					continue;
				}
				total_timeused = tmp;
				changed++;
				break;
#ifndef ONE_LANGUAGE_ONLY
			case 17:            // language
				// get language
				select_lang(lang);
				int done = 0;
				while (!done)
				{
					sprintf(tmpstr,gstrl("YOUAREUSINGLANGUAGE"),lang);
					sstrcr_c(tmpstr);
					sstrl("CORRECT");
					if (yesno())
					{
						done++;
						continue;
					}
					select_lang(lang);
					changed++;
				}
#endif
		}
	}
	return(0);
};

// Function:   timeleft
// Purpose:    return the number of minutes left for the user
// Input:      none
// Output:     minutes left until forced logoff
// Author:     Greg Shaw
// Created:    6/20/95

int User::timeleft(void)
{
	int tl;
	time_t now;

	time(&now);
        tl = (u_timelimit()*60 - (now - user_logon()))/60;
        // subtract time used prior to this logon
        tl -= prevtimeused();  // subtract previous time used
        if (credituploads())
                tl += credit();    // add credited minutes
	if (tl > ((account_expiration() - now)/60))
		tl = (account_expiration() - now)/60;
	return(tl);
}


// Function:   update_callers
// Purpose:    update the callers file
// Input:  	none
// Output: 	the callers file is incremented by one
// Author: 	Greg Shaw
// Created:    	1/3/96

int User::update_callers(void)
{
	FILE	*infile;	// input file
	char	*bbsdir;	// bbs home directory
	char	tmpstr[255];	

	// update the number of callers file
	if (bbsdir = getenv("BBSDIR"), bbsdir == NULL)
	{
		ap_log("user.display_last: BBSDIR environment variable not set.");
		return(-1);
	}
	sprintf(tmpstr,"%s/admin/callers",bbsdir);
	if (infile = bopen(tmpstr,"r"), infile != NULL)
	{
		if (fscanf(infile,"%ld",&callernumber) != 1)
		{
			ap_log("Bad read from callers file.");
			return(-1);
		}
		bclose(infile);
	}
	else
		callernumber = 0;	// must be a new file?
	callernumber++;
	if (infile = bopen(tmpstr,"w"), infile != NULL)
	{
		fprintf(infile,"%ld",callernumber);
		bclose(infile);
	}
	// now fix the permissions
	chmod(tmpstr,S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);
	return(0);
};


// Function:   update_recent_callers
// Purpose:    update the recent_callers file
// Input:  	the old recent_callers file
// Output: 	the last few callers of the bbs, if enabled by sysop
//		the data file is updated automatically
// Author: 	Greg Shaw
// Created:    	1/3/96

int User::update_recent_callers(void)
{
	FILE	*infile;	// input file
	FILE	*outfile;	// output file 
	char	*bbsdir;	// bbs home directory
	char	*line;		// a single line from the users file
	char	tmpstr[255];	
	char	tmpstr2[255];	
	char	ttystr[50];	// tty line
	struct tm *tmdata;
	time_t	now;
	int	lines;		// number of lines read

	// update the number of callers file
	if (bbsdir = getenv("BBSDIR"), bbsdir == NULL)
	{
		ap_log("user.display_last: BBSDIR environment variable not set.");
		return(-1);
	}
	// update the 'recent_callers' file?
	if (showlast())
	{
		sprintf(tmpstr,"%s/admin/recent_callers.new",bbsdir);
		if (outfile = bopen(tmpstr,"w"), outfile == NULL)
		{
			ap_log("Unable to open recent_callers.new file for write.");
			return(-1);
		}
		// add our mark to the new file
		
		// get ttyname
		strcpy(ttystr,ttyname(fileno(stdout)));
		if (line = strchr(ttystr,'y'), line != NULL)
		{                   // chop off /dev/tty
			// chop after y
			strcpy(tmpstr,++line);
			// copy back
			strcpy(ttystr,tmpstr);
		}

		time(&now);
		tmdata = localtime(&now);
		strftime(tmpstr2,50,"%X %x",tmdata);
		fprintf(outfile,gstrl("RECENTCALLERS"),
			fname,lname,city,state,ttystr,tmpstr2,'\n');
		// open existing file (if available)
		sprintf(tmpstr,"%s/admin/recent_callers",bbsdir);
		if (infile = bopen(tmpstr,"r"), infile != NULL)
		{
			// read the first 4 lines -- ignore the fifth
			lines = 0;
			while (lines < 4 && !feof(infile))
			{
				line = getline(infile);
				if (strlen(line) > 0) // get anything?
				{
					lines++;
					fprintf(outfile,"%s\n",line);
				}
			}
			bclose(infile);

		}
		bclose(outfile);
		// now rename the files
		// I really should check the status of this
		// nuke the original
		sprintf(tmpstr,"%s/admin/recent_callers",bbsdir);
		unlink(tmpstr);
		// rename the new file
		sprintf(tmpstr,"%s/admin/recent_callers.new",bbsdir);
		sprintf(tmpstr2,"%s/admin/recent_callers",bbsdir);
		rename(tmpstr,tmpstr2);
		chmod(tmpstr2,S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);

	}
	return(0);
};

// Function:   constructor
// Purpose:        initialize all object variables
// Input:      none
// Output:     (object is initialized)
// Author:     Greg Shaw
// Created:        6/1/93

User::User()
{
	card = NULL;
	if (card_color = def_card(), card_color == -1)
	{
		ap_log("Unable to get default card color.");
		sstrcr("The BBS has a configuration problem.\n");
		sstrcr("The BBS is unable to find the default card color in bbsinfo.\n");
		exit(0);
	}
	if (card = cardinfo(card_color), card == NULL)
	{
		ap_log("Unable to get card information.");
		sstrcr("The BBS has a configuration problem.\n");
		sstrcr("The BBS is unable to find the card information.\n");
		exit(0);
	}
	fname[0] = 0;
	lname[0] = 0;
	alias[0] = 0;
	login_name[0] = 0;
	editor[0] = 0;
	city[0] = 0;
	state[0] = 0;
	uterm[0] = 0;
	time(&last_login);
	time(&login_time);
	time(&anniversary);
	time(&expiration);
	validated = 0L;
	mail_check = 0;
	logins = 1;
	lines = 24;
	cols = 80;
	downloads = 0;
	uploads = 0;
	priv_msgs = 0;
	pub_msgs = 0;
	credited = 0;
	flags = 0;
	tmpflags = 0;
	acl = card->acl;
	tmpacl = acl;
	timelimit = card->tl;
	tmptl = timelimit;
	credited = 0;
	timeused = 0;
	total_kused = 0;
	total_timeused = 0;
	kused = 0;
	color_capable = 0;
	c_fore = def_foreground();
	c_back = def_background();
}


// Function:   destructor
// Purpose:        clean up the object
// Input:      none
// Output:     none ( null destructor at this point )
// Author:     Greg Shaw
// Created:        6/1/93

User::~User()
{
}


#endif                          // _USER_C_




