/* vt100.h - VT100 emulation header file
 *
 *   vt100_create         - make new terminal structure
 *   vt100_destroy        - free a vt_info structure
 *   vt100_resize         - change the size of a terminal
 *   vt100_reset          - reset terminal to initial state
 *   vt100_interpret      - interpret a null-terminated string
 *   vt100_interpret_buf  - interpret buffer of given length
 *   vt100_interpret_char - interpret single character
 *   vt100_outcall_set    - set the terminal response function
 *   vt100_callback_set   - set the callback function for a terminal
 *   vt100_callback_ptr   - specify private pointer for callback
 *   vt100_callback_mask  - set mask determining which callbacks allowed
 *   vt100_get_size       - read terminal size
 *   vt100_get_pos        - read cursor position
 *   vt100_get_newpos     - read cursor position we're about to move to
 *   vt100_get_text       - extract text from (X,Y) to (X+L,Y)
 *   vt100_get_attr       - extract attributes, likewise
 *   vt100_get_xtitle     - read xterm title string
 *   vt100_get_clrtype    - find which kind of clear is being done
 *   vt100_ansify         - turn text and attrs into one ANSI text string
 *
 * The contents of a "vt_info" structure should not be directly accessed.
 *
 * Coordinates have their origin as (1,1), at the top left of the screen.
 * Text is stored as unsigned characters; attributes are stored as unsigned
 * short integers, and contain colour information as well as things like bold
 * and underline flags - to extract this information use the bitmasks and
 * shifts described below.
 *
 * For instance, background colour is equal to
 * (attr & VT_BACKGROUND_MASK) >> VT_BACKGROUND_SHIFT.
 *
 * The callback function for a terminal is called by vt100_interpret() or
 * vt100_interpret_buf() if any of the events specified by the mask given
 * by vt100_callback_mask() - a bitwise OR of any of VT_ACTION_* - occur.
 * The function's first parameter is the vt_info, the second is the reason
 * it's being called (any ONE of the VT_ACTION_* set in the mask), and the
 * third is the pointer passed to vt100_callback_ptr(), or zero if none was
 * set.
 *
 * If the callback function returns zero, interpretation continues. If it
 * returns 1, interpretation is halted and the interpreter returns with a
 * value indicating that there is data still to process (see below). If it
 * returns anything other than 0 or 1, the interpreter bails out with a
 * return code indicating that an error occurred.
 *
 * The interpreter functions will return a negative value on error, zero if
 * all data passed was processed, and a positive integer - the number of
 * bytes left to process - if not all of the buffer could be dealt with
 * (usually because a callback function returned 1).
 *
 * When creating a vt_info structure you also need to specify a "vt_outcall"
 * function. This will be called by the interpreter if a string needs to be
 * sent back to the originator of the data stream - i.e. responses to
 * terminal status queries and the like. If you specify a null pointer here,
 * no function will be called. It takes one parameter, a pointer to the
 * null-terminated string.
 *
 * $Id: vt100.h,v 1.1.1.1 1999/12/02 20:03:07 ivarch Exp $
 */

#ifndef _IV_VT100_H
#define _IV_VT100_H

#define VT_ACTION_BELL		0x00001	/* bell */
#define VT_ACTION_LF		0x00002	/* linefeed */
#define VT_ACTION_CR		0x00004	/* carriage return */
#define VT_ACTION_RI		0x00008	/* reverse index (move up) */
#define VT_ACTION_ID		0x00010	/* terminal ID requested */
#define VT_ACTION_SAVE_CUR	0x00020	/* save cursor */
#define VT_ACTION_RESTORE_CUR	0x00040	/* restore cursor */
#define VT_ACTION_RESET		0x00080	/* terminal to be reset */
#define VT_ACTION_STATUS	0x00100	/* terminal status requested */
#define VT_ACTION_CURSOR	0x00200	/* cursor status requested */
#define VT_ACTION_GOTO		0x00400	/* cursor to be explicitly moved */
#define VT_ACTION_DECALIGN	0x00800	/* screen to be filled with E's */
#define VT_ACTION_CLEAR		0x01000	/* line or screen to be cleared */
#define VT_ACTION_SCROLL_UP	0x02000	/* screen to scroll up */
#define VT_ACTION_SCROLL_DOWN	0x04000	/* screen to scroll down */
#define VT_ACTION_INS_LINE	0x08000	/* line(s) to be inserted */
#define VT_ACTION_DEL_LINE	0x10000	/* line(s) to be deleted */
#define VT_ACTION_INS_CHAR	0x20000	/* char(s) to be inserted */
#define VT_ACTION_DEL_CHAR	0x40000	/* char(s) to be deleted */
#define VT_ACTION_XTITLE	0x80000	/* xterm title changed */
#define VT_ACTION_RENDITION	0x100000	/* set graphics rendition */
#define VT_ACTION_SET_MODE	0x200000	/* set mode */
#define VT_ACTION_RESET_MODE	0x400000	/* reset mode */
#define VT_ACTION_MAX		0x800000

#define VT_FOREGROUND_MASK	0x0700	/* mask for foreground from attr */
#define VT_FOREGROUND_SHIFT	8	/* bits to shift right */
#define VT_BACKGROUND_MASK	0x7000	/* mask for background from attr */
#define VT_BACKGROUND_SHIFT	12	/* bits to shift right */

#define VT_COLOUR_BLACK		0
#define VT_COLOUR_RED		1
#define VT_COLOUR_GREEN		2
#define VT_COLOUR_YELLOW	3
#define VT_COLOUR_BLUE		4
#define VT_COLOUR_MAGENTA	5
#define VT_COLOUR_CYAN		6
#define VT_COLOUR_WHITE		7

#define VT_ATTR_BOLD		0x0800
#define VT_ATTR_FAINT		0x8000
#define VT_ATTR_STANDOUT	0x0001
#define VT_ATTR_UNDERLINE	0x0002
#define VT_ATTR_BLINK		0x0004
#define VT_ATTR_REVERSE		0x0008

typedef struct VT_Info * vt_info;	/* opaque VT100 information type */
typedef int (* vt_callback) (vt_info, long, void *);	/* callback function */
typedef void (* vt_outcall) (char *);		/* string response function */

vt_info vt100_create (int, int, vt_outcall);	/* create new terminal */
void vt100_destroy (vt_info);		/* remove terminal from memory */

int vt100_reset (vt_info);		/* reset terminal to initial state */
int vt100_resize (vt_info, int, int);	/* resize terminal */

int vt100_interpret (vt_info, char *);			/* interpret string */
int vt100_interpret_buf (vt_info, char *, int);		/* interpret buffer */
int vt100_interpret_char (vt_info, unsigned char);	/* interpret char */

int vt100_outcall_set (vt_info, vt_outcall);	/* change output function */
int vt100_callback_set (vt_info, vt_callback);	/* set up callback function */
int vt100_callback_ptr (vt_info, void *);	/* give function a parameter */
int vt100_callback_mask (vt_info, long);	/* set callback mask */

int vt100_get_size (vt_info, int *, int *);
int vt100_get_pos (vt_info, int *, int *);
int vt100_get_newpos (vt_info, int *, int *);
int vt100_get_text (vt_info, unsigned char *, int, int, int);
int vt100_get_attr (vt_info, unsigned short *, int, int, int);
char * vt100_get_xtitle (vt_info);
int vt100_get_clrtype (vt_info);
 
int vt100_ansify (char *, unsigned char *, unsigned short *, int, int);

#endif	/* _IV_VT100_H */

/* EOF */
