/****************************************************************************
 * *
 * *  MODULE : List.hh
 * *
 * *  Copyright (c) 1997 Jukka Vuokko
 * *  See file COPYING for more information about copyrights.
 * * 
 * * $Id: List.hh,v 1.3 1998/04/13 14:42:19 jvuokko Exp $
 * *
 ****************************************************************************
 * *
 * * Template class for handling two directional linked list.
 * *
 * *
 * *
 * *
 * *
 ***************************************************************************/

#ifndef __LIST_H__
#define __LIST_H__

//#include "string_class.hh"
#include <stdlib.h>  // malloc

#include "my_types.h"

/// function used to compare data of nodes in list during sorting.
typedef int compare_function_t (const void*, const void*);


/**
 *  List node.
 */
class ListNode {
public:
           /// id-code of node.
        int id; 
           /// pointer to data of node               
        void *sp;              
           /// pointer to previos node
        ListNode *prev;  
           /// pointer to next node
        ListNode *next;  
};

/**
 * Class for double linked list.
 */
template <class Type> class List {
public:
           /**
            * Constructor. Initializes all pointers to 0.
            */
        List ();
           /**
            * Destructor. Destroys contents of the whole list including
            * datanodes.
            */
        ~List ();
           /**
            * Destroy contens of the list and free memory of each datanode.
            */
        void destroy();         
           /**
            * Leave list, but do not free memory of the datanodes.
            */
        void leave();           
           /**
            * Leave current node, but don't free it.
            */
        void leavenode();    
           /**
            * Check the state of the list.
            *
            * @return true if list has some members, false if list is empty.
            */
        bool check () const; 

           /**
            * Check if current position is at the first item of the list.
            *
            * @return true if first, false if not
            */
        bool is_first() const { return current==first_of_list ? true : false; }

           /**
            * Check if current position is at the last item of the list.
            *
            * @return true if last, false if not
            */
        bool is_last() const { return current==last_of_list ? true : false; } 

           /**
            * Add new node to the list.
            * New node will go next after current. 
            * After operation, current position will be added node.
            *
            * @param sp Pointer to new data item.
            * @return false if something goes wrong.
            */
        bool add (const Type *sp);    

           /**
            * Insert new node to list, before current node. After operation,
            * position in list will be at insterted node.
            *
            * @param sp Pointer to new data item.
            * @return false if something goes wrong.
            */
        bool insert (const Type *sp); // insert node before current

           /**
            * Set current position in list to next node.
            *
            * @return false if there are no next node.
            */
        bool next ();           // go to next node

           /**
            * Set current position in list to previous node.
            *
            * @return false if there are no previous node.
            */
        bool prev ();           // previous node

           /**
            * Set current position to start of list.
            *
            * @return false if list is empty.
            */
        bool first ();          // go to start of list

           /**
            * Set current position in list to last node of the list.
            *
            * @return false if list is empty.
            */
        bool last ();           // go to last node of list

           /**
            * Count size of the list.
            *
            * @return number of items stored to the list.
            */
        int count_items () const;     // return number of items in list

           /**
            * Save current postion in the list.
            *
            * NOTE: Use this and function restore\_position() very carefully, 
            *       because they keeps recorded only latest event.
            *
            *       e.g. next won't work:
            *
              \begin{verbatim}

                   my_list.save_position();
                   my_list.next();
                   search_from_list (my_list); <-- uses also save_position()
                   my_list.restore_position();                
            
              \end{verbatim}
            *
            *       After this code, position in the list will be that what 
            *       was saved in the procedure search\_from\_list(), not the 
            *       position that was stored before call.
            *
            *       So {\bf avoid nested calls } to these methods.
            */
        void save_position(); 

           /**
            * Restore latest saved position.
            */
        void restore_position();
        
           /** 
            * Get pointer to data of the current node. 
            *
            * @return Pointer to data or 0 if list is empty.
            */
        Type* get () const;    
        
           /**
            * Sort list using given function to compare data.
            *
            * Pointers to data are passed using two void** pointers, so compare
            * function need to cast them to right type.
            *
            * e.g. if type of datanode is int, compare function must do :
             \begin{verbatim}
                      int **ptr_a = (int**) a;
                      int **ptr_b = (int**) b;
                      return *ptr_a - *ptr_b;
             \end{verbatim}
            *
            * @param Pointer to comparison function.
            */
        void sort (compare_function_t compare);
        
        // Copy contents of list to other
        const List<Type>& List<Type>::operator = (const List<Type>& src);

           /**
            * Removes current node from list. After removing, current 
            * node will be node that was next of removed, or if removed
            * removed node was last of list, then current will be previous.
            *
            * @param delete_all If true, then free memory of the node.
            *
            * @return true if success, false if not.
            */
        bool remove (bool delete_all = true);

           /** @name Direct data manipulators. */
           //@{
        
           /**
            * Remove all data of the current node.
            *
            * @return true if success, false if failed.
            */
        bool remove_data_of_node(); 

           /**
            * Put new data to current node. Initializes data pointer of the
            * current node using given pointer.
            *
            * @param data Pointer to new data element.
            * @return true if success, false if not.
            */
        bool put_data_to_node (Type *data); 
           //@}

private:
        ListNode *current;        // current node
        ListNode *first_of_list;  // first node
        ListNode *last_of_list;   // last node
        ListNode *save;           // saved position
        int items;              // number of items in list
        int id_code;            // id code of this list
};


//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: add
//
// description: Adds new node to list. New node will go next after current.
//              After operation, current will point node that was added.
//
//
// returns: false if something goes wrong
//
//**************************************************************************/
template <class Type> bool
List <Type>::add (const Type *sp)
{
        ListNode *newnode = new ListNode;
        if (newnode == 0)
                return false;
        newnode->sp = (void*) sp;
        newnode->id = id_code;
        
        if (current == 0) { /* lista on tyhj */
                current = newnode;
                current->next = 0;
                current->prev = 0;
                first_of_list = newnode;
                last_of_list = newnode;
        } else if (current == last_of_list){ /* nykyinen on listan viimeinen */
                newnode->next = 0;
                newnode->prev = current;
                current->next = newnode;
                current = newnode;
                last_of_list = newnode;
        } else {                       /* listn johonkin listan keskelle */
                current->next->prev = newnode;
                newnode->next = current->next;
                current->next = newnode;
                newnode->prev = current;
                current = newnode;
        }
        items++;
        id_code++;
        return true;
}


//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: add
//
// description: Adds new node to list, before current node. After operation,
//              current points to inserted node.
//
//
// returns: false if something goes wrong
//
//**************************************************************************/
template <class Type> bool
List <Type>::insert (const Type *sp)
{
        ListNode *newnode = new ListNode;
        if (newnode == 0)
                return false;
        newnode->sp = (void*) sp;
        newnode->id = id_code;
        
        if (current == 0) { /* lista on tyhj */
                current = newnode;
                current->next = 0;
                current->prev = 0;
                first_of_list = newnode;
                last_of_list = newnode;
        } else if (current == first_of_list){ /* nykyinen on listan eka */
                newnode->next = current;
                newnode->prev = 0;
                current->prev = newnode;
                current = newnode;
                first_of_list = newnode;
        } else {                       /* listn johonkin listan keskelle */
                current->prev->next = newnode;
                newnode->prev = current->prev;
                current->prev = newnode;
                newnode->next = current;
                current = newnode;
        }
        items++;
        id_code++;
        return true;
}


template <class Type> bool
List<Type>::remove_data_of_node()
{
        if (current != 0) {
                Type *tmp = (Type*) current->sp;
                // poistetaan alkion data
                if (tmp != 0) {
                        delete tmp;
                }
                tmp = 0;
                return true;
        }
        return false;
}        

template <class Type> bool
List<Type>::put_data_to_node (Type *data)
{
        if (current != 0) {
                current->sp = (void*) data;
                return true;
        }
        return false;
}        

//**************************************************************************/
//  CLASS: List
//  MEMBER FUNCTION: leave
//**************************************************************************/ 
//
//  Removes list, but does not free memory from it's data.
//
//
//**************************************************************************/
template <class Type>
void List<Type>::leave()
{
        first();
        while (remove(false) != false);
        first_of_list = 0;
        current = 0;
        last_of_list = 0;
        items = 0;
        id_code = 0;
}

//**************************************************************************/
//  CLASS: List
//  MEMBER FUNCTION: leavenode
//**************************************************************************/ 
//
//  Removes current node from list, but leaves data of it to memory.
//  This is useful, if list contains pointers to data, which is used or/and
//  modified directly by other process.
//
//**************************************************************************/
template <class Type>
void List<Type>::leavenode()
{
        remove(false);
}

//**************************************************************************/
//  CLASS: List
//  MEMBER FUNCTION: operator=
//**************************************************************************/ 
//
//  Copy contents of given list to current list.
//  NOTE: if datatype contains pointers, then use of this might be dangerous.
//
//
//  RETURN: reference to current list.
//**************************************************************************/
// template <class Type>
// const List<Type>& List<Type>::operator = (const List<Type>& src)
// {
//         if (this == &src) {
//                 return *this;
//         }
//         destroy();
//         ListNode *ptr = src.first_of_list;
//         Type* item;
//         Type* tmp;
//         if (ptr != 0) {
//                 do {
//                         item = new Type;
//                         // kopioidaan toisen listan alkion sislt
//                         // HUOM! pit varoa ettei tyypin sisiset
//                         // pointterit kopioidu sellaisenaan!!!
//                         tmp = (Type*) ptr->sp;
//                         *item = *tmp;
//                         add (item);
//                 } while ((ptr = ptr->next) != 0);
//         }
//         return *this;
// }

//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: List
//
// description: constructor for class. Initializes all pointers to 0.
//
//
//
//**************************************************************************/
template <class Type> 
List <Type>::List ()
{
        current = 0;
        first_of_list = 0;
        last_of_list = 0;
        save = 0;
        items = 0;
        id_code = 0;
}

//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: check
//
// description: Checks if list has nodes.
//
// returns: false if list is empty, true if not.
//
//**************************************************************************/
template <class Type>
bool List <Type>::check () const
{
        if (current == 0)
                return false;
        return true;
}

//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: remove
//
// description: Removes current node from list. After removing, current 
//              node will be node that was next of removed, or if removed
//              removed node was last of list, then current will be previous.
//
// returns: false if list is empty, or if list is empty after removing node.
//
//**************************************************************************/
template <class Type> bool
List <Type>::remove (bool delete_all)
{
        ListNode *tmp;
        Type *rm;
        bool rc;
        if (check() == false)
                return false;

        rc = true;
        /* seuraavan alkion linkit */
        tmp = current->next;
        if (tmp != 0)
                tmp->prev = current->prev;
        /* edeltvn linkit */
        tmp = current->prev;
        if (tmp != 0)
                tmp->next = current->next;
        /* poistettava alkio */
        tmp = current;
        /* current-osoitin osoittaa edellist, jos sellainen on. */
        if (tmp->prev != 0 && tmp->next != 0) {
                /* poistettava on listan keskell */
                current = current->prev;
        } else if (tmp->next == 0 && tmp->prev != 0) {
                /* poistettava on viimeinen */
                last_of_list = tmp->prev;
                current = tmp->prev;
        } else if (tmp->prev == 0 && tmp->next != 0) {
                /* poistettava on ensimminen */
                first_of_list = tmp->next;
                current = tmp->next;
        } else {
                /* poistettava on listan ainoa alkio */
                first_of_list = 0;
                last_of_list = 0;
                current = 0;
                rc = false;
        }

        // poistetaan alkion data
        if (delete_all == true) {
                rm = (Type*) tmp->sp;
                if (rm != 0) {
                        delete rm;
                }
        }
        // poistetaan itse alkio
        delete tmp;
        items--;
        return rc;
}

//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: next
//
// description: Sets current position in list to next node.
//
// returns: false if there are no next node.
//
//**************************************************************************/
template <class Type> bool
List <Type>::next ()
{
        if (check() == false) {
                return false;
        }
        if (current->next == 0)
                return false;
        current = current->next;
        return true;
}
//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: prev
//
// description: Sets current postition in list to previous node.
//
// returns: 0 if there are no previous node.
//
//**************************************************************************/
template <class Type> bool
List <Type>::prev ()
{
        if (check() == false) {
                return false;
        }
        if (current->prev == 0)
                return false;
        current = current->prev;
        return true;
}
//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: first
//
// description: Sets current position to start of list.
//
//
// returns: false if list is empty.
//
//**************************************************************************/
template <class Type> bool
List <Type>::first ()
{
        if (first_of_list == 0)
                return false;
        current = first_of_list;
        return true;
}
//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: last
//
// description: Sets current position in list to last node of the list.
//
//
//
// returns: false if list is empty.
//
//**************************************************************************/
template <class Type> bool
List <Type>::last ()
{
        if (last_of_list == 0)
                return false;
        current = last_of_list;
        return true;
}
//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: get
//
// Returns pointer to data of current node. Returned value is 0 if
// list is empty.
//
//
//**************************************************************************/
template <class Type> Type*
List <Type>::get () const
{
        if (check() == false) {
                return 0;
        }
        return (Type*) current->sp;
}

//**************************************************************************/
//  CLASS: List
//  MEMBER FUNCTION: save_position
//**************************************************************************/ 
//
// Saves current position in list.
//
// NOTE: Use this and function restore_position() very carefully, because
//       they keeps recorded only latest event.
//
//       e.g. next won't work:
//
//       my_list.save_position();
//       my_list.next();
//       search_from_list (my_list); // this function uses also save_position()
//       my_list.restore_position();                
//
//       After this code, position in list will be that what is saved in
//       procedure search_from_list(), not position that was stored before
//       call.
//       So avoid nested calls to these methods.
//
//**************************************************************************/
template <class Type> void
List <Type>::save_position()
{
        save = current;
}

//**************************************************************************/
//  CLASS: List
//  MEMBER FUNCTION: restore_position
//**************************************************************************/ 
//
//  Restorese latest saved position. 
//
//**************************************************************************/
template <class Type> void
List <Type>::restore_position()
{
        if (save != 0) {
                current = save;
        }
}

//**************************************************************************/
//
// CLASS: List
// MEMBER FUNCTION: ~List
//
// description: destructor for class. Destroys contents whole list
//              (including datanodes).
//
//
//
//**************************************************************************/
template <class Type> 
List <Type>::~List ()
{
        destroy();
}

//**************************************************************************/
//  CLASS: List
//  MEMBER FUNCTION: destroy
//**************************************************************************/ 
//
// Destroys contents of whole list and removes data.  
// 
//
//**************************************************************************/
template <class Type>
void List <Type>::destroy()
{
        if (first () == true) {
                while (remove() != false);
        }
        first_of_list = 0;
        current = 0;
        last_of_list = 0;
        save = 0;
        items = 0;
        id_code = 0;
}

//**************************************************************************/
//  CLASS: List
//  MEMBER FUNCTION: count_items
//**************************************************************************/ 
//
// 
//  RETURN: number of items stored in list.
//**************************************************************************/
template <class Type> int
List <Type>::count_items() const
{
        return items;
}



//**************************************************************************/
//  CLASS: List
//  MEMBER FUNCTION: sort
//**************************************************************************/ 
//
// This function sorts contents of list using given compare function. 
// Pointers to data are passed using two void** pointers, so compare
// function need to cast them to right type.
//
// e.g. if type of datanode is int, compare function must do :
//          int **ptr_a = (int**) a;
//          int **ptr_b = (int**) b;
//          return *ptr_a - *ptr_b;
//
//**************************************************************************/
template <class Type> void
List<Type>::sort(compare_function_t compare)
{
        void **ptrtable = (void**) malloc (items * sizeof (void*));

        int i = 0;

        if (items > 1) {
                first();
                do {
                        *(ptrtable+i) = current->sp;
                        i++;
                } while (next() == true);
                
                qsort (ptrtable, items ,sizeof (void*), compare);

                i=0;
                first();
                do {
                        current->sp = ptrtable[i];
                        i++;
                } while (next() == true);
        }
        free (ptrtable);
}


#endif
