// $Id: window.cc,v 1.6 1998/05/25 20:06:08 jvuokko Exp $

/*****************************************************************************
 * *
 * *      MODULE:     window.cc
 * *                  ---------
 * ***************************************************************************
 * *
 * *
 * *      COPYRIGHT (C) 1997 JUKKA VUOKKO. ALL RIGHTS RESERVED
 * ***************************************************************************
 * *
 * *      Functions for handling window and statusline.
 * *
 *****************************************************************************/

#include "terminal_io.hh"

// no debugging stuff are currently needed from this module
#undef DEBUG
#define DEBUG(ex)


//**************************************************************************/
// CLASS: Statusline
// MEMBER FUNCTION: init
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Statusline::init( int maxlen )
{
        width = maxlen;
        infotext = "";
        extra_info = "";
        percent = NO_PERCENT;

        if (sline) {
                delete[] sline;
        }
        sline = new char[width + 1];

        memset( sline, '-', width );
        sline[width] = 0;
}

//**************************************************************************/
// CLASS: Statusline
// MEMBER FUNCTION: 
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
char*
Statusline::get_sline()
{
        int len;
        memset( sline, '-', width );
        sline[width] = 0;

        len = (2 * width) / 3;

        memcpy( sline, infotext.c_str(), len < infotext.length()
                ? len : infotext.length());

        len = (width / 3);

        memcpy( sline + width - len, extra_info.c_str(),
                len < extra_info.length() ? len : extra_info.length());


        
        if ( percent >= 0 ) {
                char tmp[10];
                if ( ALL_PERCENT == percent ) {
                        sprintf(tmp, "--All" );
                } else if ( TOP_PERCENT == percent ) {
                        sprintf(tmp, "--Top" );
                } else if ( BOT_PERCENT == percent ) {
                        sprintf (tmp, "--Bot" );
                } else {
                        sprintf(tmp, "--%d%%", percent );
                }
                int cnt = strlen( tmp ) >= 5 ? 5 : strlen( tmp );

                if (width > 6 ) {
                        memcpy( sline + width - 6, tmp, cnt );
                }
        }
        sline[width] = 0;

        return sline;
}


//**************************************************************************/
// CLASS: Window
// MEMBER FUNCTION: Window
//**************************************************************************/ 
//
// Creates a new window.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS: handle  Handle to the term. screen.
//             x, y    Top left corner of the window
//             w, h    Width and height of the window
//             style   (optional) Style of the window. WIN_DEFAULT, WIN_BORDER,
//                     WIN_TITLEBAR. These can be combined.
//             title   Text of the titlebar. optional.
//
// RETURN: 
//**************************************************************************/
Window::Window( Terminal_screen *handle, int x, int y, int w, int h,
                int style, const char* title)
{
        _mainwin = handle;
        _width = w;
        _height = h;
        _xbeg = x;
        _ybeg = y;
        _xmax = _width - 1;
        _ymax = _height- 1;
        _xmin = 0;
        _ymin = 0;
        _style = style;
        _border_attr = DEFAULT_ATTR;
        _sline = 0;
        if ( (_style & WIN_TITLEBAR) || (_style & WIN_BORDERS) ) {
                ++_ymin;
        }

        if ( (_style & WIN_SLINE) || (_style & WIN_BORDERS) ) {
                --_ymax;
                _sline = new Statusline( _width );
        }
                
        if (_style & WIN_BORDERS ) {
                --_xmax;
                ++_xmin;
        }

        _x = _xmin;
        _y = _ymin;

        if (title) {
                _title = "[ ";
                _title += title;
                _title += " ]";
        }

        _cwidth = _xmax - _xmin + 1;
        _size = _width * _height;
        _win = new attr_t[ _size ];
        _attr = DEFAULT_ATTR;
        attr_memset( _win, DEFAULT_ATTR | ' ', _size );

        if (_style != WIN_DEFAULT) {
                draw_borders();
        }

        is_visible = true;
        _border_changed = false;
        DEBUG("Window : _xmin = " << _xmin << " _xmax = " << _xmax );
        DEBUG("Window : _ymin = " << _ymin << " _ymax = " << _ymax );
        DEBUG("         _width = " << _width );
        DEBUG("         _xbeg = " << _xbeg );
}


//**************************************************************************/
// CLASS: Window
// MEMBER FUNCTION: draw_borders
//**************************************************************************/ 
//
// Draws box around the window and/or titlebar of the window.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Window::draw_borders()
{
        if ( _style & WIN_BORDERS || _style & WIN_TITLEBAR) {
                attr_memset( _win, _border_attr | '-', _width );
        }

        if ( _style & WIN_BORDERS || _style & WIN_SLINE ) {
                attr_memset( _win + (_height-1) * _width, _border_attr | '-',
                             _width );
        }
        if ( _style & WIN_BORDERS ) {
                for (int i = _ymin; i <= _ymax; ++i) {
                        _win[ i * _width ] = (_border_attr | '|' );
                        _win[ i * _width + ( _width - 1 ) ] =
                                (_border_attr | '|' );
                }
        }

        if ( (_style & WIN_TITLEBAR) && _title.is_empty() == false ) {
                int y = _y;
                int x = _x;
                int len = _title.length();
                _y = 0;
                _x = ( (_width - len )/ 2) - 1;
                if (x < 0 ) {
                        len += x; // note, x is negative --> ( len - x )
                }
                attr_t tmp_attr = _attr;
                _attr = _border_attr;
                addbytes( _title.c_str(), len );
                _attr = tmp_attr;
                _y = y;
                _x = x;
        }

        if ( (_style & WIN_SLINE) && _sline != 0) {
                char *sline = _sline->get_sline();
                int y = _y;
                int x = _x;
                _x = 0;
                _y = _height - 1;
                int len = strlen(sline);
                len = len < _width ? len : _width;
                attr_t tmp_attr = _attr;
                _attr = _border_attr;
                addbytes( sline, len );
                _attr = tmp_attr;
                _y = y;
                _x = x;
        }
        _border_changed = false;
}

void
Window::set_border_colors( int fg, int bg )
{
        _border_attr = DEFAULT_ATTR;
        if (fg & 0xf) {
                _border_attr &= ~ FCOLOR_BITS;
        }

        _border_attr |= FOREGROUND( fg ) | ATTRIBUTES( fg );
        
        if (bg & 0xf) {
                _border_attr &= ~ BCOLOR_BITS;
        }
        _border_attr |= BACKGROUND( bg ) | ATTRIBUTES( bg );

        draw_borders();
}


//**************************************************************************/
// CLASS: Window
// MEMBER FUNCTION: scroll_down
//**************************************************************************/ 
//
// Scrolls contents of the current windot one line down.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Window::scroll_down()
{
        int y;
        int yy;
        int w = _xmax * sizeof( attr_t );
        for ( yy = _ymin, y = _ymin + 1; y <= _ymax; ++y, ++yy ) {
                memcpy( _win + _xmin + yy * _width,
                        _win + _xmin + y * _width, w );
        }
        attr_memset( _win + _xmin + (_ymax * _width), DEFAULT_ATTR | 0x20,
                     _xmax );
        _x = _xmin;
        _y = _ymax;
}


//**************************************************************************/
//  CLASS: Window
//  MEMBER FUNCTION: clear
// 
// 
//  DESCRIPTION: Clears window and moves cursor to top left.
// 
// 
//**************************************************************************/
void
Window::clear()
{
        _x = _xmin; _y = _ymin;
        _attr = DEFAULT_ATTR;
        attr_memset( _win, _attr | 0x20, _size );
        if (_style != WIN_DEFAULT) {
                _border_changed = true;
                //draw_borders();
        }
}

//**************************************************************************/
//  CLASS: Window
//  MEMBER FUNCTION: gotoxy
// 
// 
//  DESCRIPTION: Moves cursor to given position. Note: 0,0 is left and top.
// 
// 
//**************************************************************************/
bool
Window::gotoxy (int x, int y)
{
        x += _xmin;
        y += _ymin;
        if (x > _xmax || y > _ymax || x < _xmin || y < _ymin ) {
                return false;
        }
        _x = x;
        _y = y;
        return true;
}

//**************************************************************************/
//  CLASS: Window
//  MEMBER FUNCTION: forward
// 
// 
//  DESCRIPTION: Moves cursor n position forward.
// 
// 
//**************************************************************************/
void
Window::forward (const int n)
{
        if ( _x + n <= _xmax ) {
                _x += n;
        } else {
                _x = _xmin;
                down();
        }
}

//**************************************************************************/
//  CLASS: Window
//  MEMBER FUNCTION: back
// 
// 
//  DESCRIPTION: Moves cursor n position bacward.
// 
// 
//**************************************************************************/
void
Window::back (const int n)
{
        if (_x - n >= _xmin) {
                _x -= n;
        } else {
                _x = _xmax;
                up();
        }
}
//**************************************************************************/
//  CLASS: Window
//  MEMBER FUNCTION: clr_eol
// 
// 
//  DESCRIPTION: Clears end of line.
// 
// 
//**************************************************************************/
void
Window::clr_eol()
{
        int x = _x;
        int y = _y;
        addch( ' ', _xmax - _x + 1);
        _x = x;
        _y = y;
}

//**************************************************************************/
//  CLASS: Window
//  MEMBER FUNCTION: empty_line
// 
// 
//  DESCRIPTION: Draws empty line
// 
// 
//**************************************************************************/
void
Window::clr_line()
{
        int x = _x;
        int y = _y;
        _x = _xmin;
        addch( ' ', _cwidth );
        _y = y;
        _x = x;
}

//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: textcolor
// 
// 
//  DESCRIPTION: Sets color for text.
// 
// 
//**************************************************************************/
void
Window::textcolor (int color)
{
        if (color & 0xf) {
                _attr &= ~ FCOLOR_BITS;
        }

        _attr |= FOREGROUND( color ) | ATTRIBUTES( color );
}

//**************************************************************************/
// CLASS: 
// MEMBER FUNCTION: 
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Window::textcolor (int fg, int bg)
{
        textcolor (fg);
        
        back_color (bg);
}

//**************************************************************************/
// CLASS: 
// MEMBER FUNCTION: 
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Window::back_color (int color)
{
        if (color & 0xf ) {
                _attr &= ~ BCOLOR_BITS;
        }
        _attr |= BACKGROUND( color ) | ATTRIBUTES( color );
}

//**************************************************************************/
//  CLASS: Window
//  MEMBER FUNCTION: getline
// 
// 
//  DESCRIPTION: Reads line using line editor to given buffer, or
//               if buffer is NULL, then allocates space for line.
//               When using existing buffer, then size of buffer must be
//               max+1 and buffer must contains null terminated string!
//               Line editor knows commands ^K, ^E, ^A, ^L and ^H
// 
//  RETURNS: pointer to line
// 
//**************************************************************************/
char*
Window::getline (char* buffer, int max, int minlen)
{
        int len = 0;
        int c;
        int x = 0;
        if (buffer == NULL) {
                buffer = new char [max+1];
                memset (buffer,0,max+1);
        }
        len = strlen (buffer);
        x = len;
        print (buffer);

        for ( ;; ) {
          loop:
                _mainwin->refresh();
                c = _mainwin->get_ch();
                switch (c) {
                case CODE_CTRL_K:
                        clr_eol();
                        len = x;
                        buffer[len+1] = 0;
                        goto loop;
                case CODE_CTRL_A:
                        back (x);
                        x = 0;
                        goto loop;
                case CODE_CTRL_E:
                        forward (len - x);
                        x = len;
                        goto loop;
                case CODE_CTRL_L:
                        back(x);
                        clr_eol();
                        print( buffer );
                        back(len-x);
                        goto loop;
                }
                if ((len < max &&  c >= ' ' && c < 127) || c == 0x8f
                    || c == 0x8e || c == 0x99 || c == 0xc5 || c == 0xc4
                    || c == 0xd6 || c == 0xf6 || c == 0xe4 || c == 0xe5
                    || c == 0x94 || c == 0x84 || c == 0x86) {
                            // listn tilaa uudelle merkille
                        if (x < len) {
                                memmove (buffer+x+1, buffer+x, len-x+1);
                                clr_eol();
                                buffer[len+1] = 0;
                                buffer[x] = (char) c;
                                print( buffer+x );
                                x++;
                                len++;
                                back (len-x);
                        } else {
                                buffer[x] = (char) c;
                                print ("%c", (char) c );
                                len++;
                                x++;
                                buffer[len] = 0;
                        }
                } else if ((c == CODE_RIGHT || c == CODE_CTRL_F) && x < len) {
                        x++;
                        forward();
                } else if ((c == CODE_LEFT  || c == CODE_CTRL_B) && x > 0) {
                        x--;
                        back();
                } else if (c == LINE_FEED && len >= minlen) {
                        break;
                } else if (c == CODE_BS && x > 0) {
                        if (x == len) {
                                print ("\b \b");
                                buffer[len-1]='\0';
                                x--;
                                len--;
                        } else {
                                memmove (buffer+x-1, buffer+x, len-x+1);
                                buffer[len-1]='\0';
                                len--;
                                x--;
                                back();
                                clr_eol();
                                print( buffer+x );
                                back (len - x);
                        }
                }
        }
        buffer[len] = '\0';
        return buffer;
}


//**************************************************************************/
//  CLASS: Window
//  MEMBER FUNCTION: down
// 
// 
//  DESCRIPTION: Moves cursor n lines down.
// 
//**************************************************************************/
bool
Window::down (int n)
{
        if (_y + n <= _ymax) {
                _y += n;
                return true;
        }
        return false;
}

//**************************************************************************/
//  CLASS: Window
//  MEMBER FUNCTION: up
// 
// 
//  DESCRIPTION: Moves cursor n lines up
// 
// 
//**************************************************************************/
bool
Window::up (int n)
{
        if (_y - n >= _ymin) {
                _y -= n;
                return true;
        }
        return false;
}


//**************************************************************************/
// CLASS: Window
// MEMBER FUNCTION: print
//**************************************************************************/ 
//
// Prints formatted string to the current window.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: Number of printed characters.
//**************************************************************************/
int
Window::print( const char* fmt, ...)
{
        va_list ap;
        int rc;
        
        va_start( ap, fmt );

        rc = vprint( fmt, ap );

        va_end( ap );

        return rc;
}
//**************************************************************************/
// CLASS: Window
// MEMBER FUNCTION: vprint
//**************************************************************************/ 
//
// Prints formatted string to the window.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: Number of printed characters.
//**************************************************************************/
int
Window::vprint( const char* fmt, va_list ap )
{
        char buffer[BUFSIZ];
        int num = vsprintf( buffer, fmt, ap );
        assert( num < BUFSIZ );
        return addbytes( buffer, num );
}

//**************************************************************************/
// CLASS: Window
// MEMBER FUNCTION: addbytes
//**************************************************************************/ 
//
// Adds given bytes to the current window.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS: 
//
// RETURN: true if success.
//**************************************************************************/
bool
Window::addbytes( const char* bytes, int count )
{
        static char tabspace[] = "        ";
        register unsigned int ch;
        register int x,y;
        unsigned char *ptr = (uchar_t*) bytes;
        x = _x;
        y = _y;

        while (count) {
                --count;
                ch = (unsigned int ) *ptr;

                ++ptr;

                switch (ch) {
                        
                case '\n':
                        if ( y < _ymax ) {
                                ++y;
                                x = _xmin;
                        } else if ( y == _ymax ) {
                                _y = y; _x = x;
                                scroll_down();
                                y = _y; x = _x;
                        } else {
                                _x = x; _y = y;
                                return false;
                        }
                        break;
                case '\r':
                        x = _xmin;
                        break;
                case '\b':
                        if ( x > _xmin ) {
                                --x;
                        }
                        break;
                case '\t':
                        _x = x; _y = y;
                        if (false == addbytes( tabspace, 8 ) ) {
                                return false;
                        }
                        x = _x; y = _y;
                        break;
                case 27:
                        _x = x; _y = y;
                        if (false == addbytes( "^[", 2 ) ) {
                                return false;
                        }
                        x = _x; y = _y;
                        break;
                default:
                        if ( ch < 32 ) {
                                _x = x; _y = y;
                                if (false == addbytes("^?", 2 ) ) {
                                        return false;
                                }
                                x = _x; y = _y;
                                break;
                        }
			if (y < 0 || x < 0 || _y >= _height) {
				return false;
			}
			
			if (x > _xmax ) {
				if ( y < _ymax ) {
					x = _xmin;
					++y;
				} else {
					_y = y; _x = x;
					scroll_down();
					y = _y; x = _x;
				}
			}
                        _win[ y * _width + x] = _attr | ch;
			++x;
                        break;

                }
        }
        _x = x;
        _y = y;
        return true;
                                
}

//**************************************************************************/
// CLASS: Window
// MEMBER FUNCTION: addch
//**************************************************************************/ 
//
// Adds given character to the window.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS: ch  - character
//             n   - number of chacacters to put, default is 1.
//
// RETURN: true if success.
//**************************************************************************/
bool
Window::addch( char ch, int n )
{
        bool rc = true;
        while (n && rc == true) {
                --n;
                rc = addbytes( &ch, 1 );
        }
        return rc;
}



