// $Id: messagedata.hh,v 1.14 1998/07/18 19:51:31 jvuokko Exp $
/*****************************************************************************
 * *
 * *      MODULE:     messagedata.hh
 * *                  --------------
 * ***************************************************************************
 * *
 * *      COPYRIGHT (C) 1997 JUKKA VUOKKO
 * *      More information about copyrights in file COPYING.
 * *
 * ***************************************************************************
 * *
 * *      Class for handling contents of a single article.
 * *
 *****************************************************************************/


#ifndef __MESSAGEDATA_HH__
#define __MESSAGEDATA_HH__

#include <fstream.h>
#include "../utilib/my_types.h"
#include "../utilib/List.hh"
#include "../utilib/String.hh"
#include "time.hh"

//--------------------------------------------------------------------------
// offsets in file
#define SIZE_OFFSET      0
#define SUBJECT_OFFSET   4
#define WRITER_OFFSET    85
#define RECEIVER_OFFSET  166
#define DATE_OFFSET      247
#define TIME_OFFSET      258
#define STATUS_OFFSET    264
#define NUMBER_OFFSET    265
#define REFERENCE_OFFSET 269
#define GROUP_OFFSET     273
#define ARTICLE_OFFSET   319


//**************************************************************************/
//
// CLASS: Message
//
// DERIVED FROM:
//
// for handling single article.
//
//**************************************************************************/
class Message {
public:
        Message();
        Message (fstream* file);  
        ~Message() {};

        Message (const Message&); // not implemented
        const Message& operator = (Message&); 
        
        // adding and removing lines
        bool add_line (const String&);   // add new line to end of message
        bool insert_line (const String&);// insert new line to start of message
        void add_tagline (const String&); // add tagline to message
#ifndef TAGB
        void strip_tagline();         // remove exist tagline from article
#else
  void strip_tagline(String&);      // remove existing tagline from article and return it
#endif
        void remove_tail();           //remove empty lines from end of message

        // send contents of message to given output stream
        bool save_to_file (ofstream&, const String&, const String&, const int);

        // functions for moving around within text
        void save_position();       // save current position
        void restore_position();    // restore previous saved position
        bool first_line();          // go at first line of article
        bool last_line();           // last line
        bool next_line();           // go to next line
        bool prev_line();           // previous line
        bool check_lines();         // check if is empty or not

        // get data from message
        const String& getline() const; // get current line
        int get_group_number() const;  // get number of group
        bool is_killed() const;        // check if message is killed
        byte_t get_status() const;     // get status of message
        int get_number() const;        // get number of message
        int get_reference_num() const; // get reference message number
        int get_number_in_thread() const;  // get message's number in thread
        const String& get_subject() const; // get subject
        const String& get_writer() const;  // get name of author
        const String& get_receiver() const;// get name of receiver
        const String& get_date() const;    // get date of article
        const String& get_jmrdate() const;
        const String& get_time() const;    // get time
        const String& get_stripped_subject() const; // get stripped subject
        int get_num_of_lines() const;      // get size of message in lines
        const char* get_cmpdate() const;   // get date formatted for comparison


        // functions for manipulate private data members
        void set_status_bits (const int);  // set given status bits on 
        void reset_status_bits (const int);// reset given bits
        void inv_status_bits (const int);  // invert given bits
        void init_status (const int=0);    // init status with given value
        void invert_kill_flag();           // invert kill flag
        void set_subject (const String&);  // set subject
        void set_writer (const String&);   // set author
        void set_receiver (const String&); // set receiver
        void set_date (const String&);     // set date 
        void set_time (const String&);     // time
        void set_number (const int);       // message's number
        //void set_index (const ulong_t);    // index of message
        void set_reference_num (const int);// reference message's number
        void set_group_number (const int); // number of group
        void set_number_in_thread (const int); // message's number in thread
        void set_charset (const int);      // set current character set
        void translate (const int);        // translate to given charset.
        bool remove_line();                // remove current line
        bool remove_lines();               // remove all lines
        bool unload_lines();               // unload contents of article,
                                           // if fstrean handle is given.
        bool search(char*, int);
protected:
        List<String> msg_lines;   // contents of message in list
private:
        void set_cmpdate();    // set up date formatted for comparison
        void set_cmptime();    // set up time formatted for comparison
        
        byte_t message_status; // status of message
        int characterset;      // characterset of message (see string_class)
        int message_number;    // numeber of message
        int reference_msg_num; // reference message number
        int number_in_thread;  // number of message in thread
        int group_number;      // number of group
        Date date;             // date 
        String time;           // time in hh:mm form
        String receiver;       // receiver
        String writer;         // author
        String subject;        // subject
        String stripped_subject;// subject without possible prefix 'Re:'
        char cmpdate[13];      // date and time for comparing articles
        bool killed;           // is message killed or not?
        
        fstream *msg_file;        // fstream handle to file where article is
        streampos offset_in_file; // offset of article in file.
};


// HUOM! tt ei saa kytt, jos artikkelin sislt on tyhj, tai crash!
inline const String&
Message::getline() const
{
        assert (msg_lines.check() == true);
        return *(msg_lines.get());
}

inline bool
Message::is_killed() const
{
        return killed;
}

inline byte_t
Message::get_status() const
{
        return message_status;
}

inline int
Message::get_number() const
{
        return message_number;
}

inline int
Message::get_reference_num() const
{
        return reference_msg_num;
}

inline int
Message::get_group_number() const
{
        return group_number;
}

inline int
Message::get_number_in_thread() const
{
        return number_in_thread;
}

inline const String&
Message::get_subject() const
{
        return subject;
}

inline const String&
Message::get_writer() const
{
        return writer;
}

inline const String&
Message::get_receiver() const
{
        return receiver;
}

inline const String&
Message::get_date() const
{
        return date.get_date();
}

inline const String&
Message::get_jmrdate() const
{
        return date.get_jmrdate();
}

inline const String&
Message::get_time() const
{
        return time;
}


inline const String&
Message::get_stripped_subject() const
{
        return stripped_subject;
}


inline int
Message::get_num_of_lines() const
{
        return msg_lines.count_items();
}


inline const char*
Message::get_cmpdate() const
{
        return cmpdate;
}



inline void
Message::set_status_bits (const int st)
{
        message_status |= (byte_t) st;
}

inline void
Message::reset_status_bits (const int st)
{
        message_status &= ~(byte_t)st;
}

inline void
Message::inv_status_bits (const int st)
{
        message_status ^= (byte_t)st;
}

inline void
Message::init_status (const int i)
{
        message_status = (byte_t) i;
}

inline void
Message::invert_kill_flag()
{
        killed = (killed == false ? true : false);
}

inline void
Message::set_writer (const String& str)
{
        writer = str;
}

inline void
Message::set_receiver (const String& str)
{
        receiver = str;
}

inline void
Message::set_date (const String& str)
{
        date = str;
        set_cmpdate();
}

inline void
Message::set_time (const String& str)
{
        time = str;
        set_cmptime();
}


inline void
Message::set_number (const int n)
{
        message_number = n;
}

inline void
Message::set_reference_num (const int n)
{
        reference_msg_num = n;
}

inline void
Message::set_group_number (const int n)
{
        group_number = n;
}

inline void
Message::set_number_in_thread (const int n)
{
        number_in_thread = n;
}


inline void
Message::set_charset (const int n)
{
        characterset = n;
}




inline bool
Message::last_line()
{
        return msg_lines.last();
}

inline bool
Message::next_line()
{
        return msg_lines.next();
}

inline bool
Message::prev_line()
{
        return msg_lines.prev();
}

inline bool
Message::remove_line()
{
        return msg_lines.remove();
}

inline void
Message::save_position()
{
        msg_lines.save_position();
}

inline void
Message::restore_position()
{
        msg_lines.restore_position();
}

inline bool
Message::check_lines()
{
        return msg_lines.check();
}

inline bool
Message::remove_lines()
{
        msg_lines.destroy();
        return true;
}
#endif



