// $Id: messagedata.cc,v 1.20 1998/07/18 19:51:31 jvuokko Exp $
/*****************************************************************************
 * *
 * *      MODULE:     messagedata.cc
 * *                  --------------
 * ***************************************************************************
 * *
 * *      COPYRIGHT (C) 1997 JUKKA VUOKKO
 * *      More information about copyrights in file COPYING.
 * *
 * ***************************************************************************
 * *
 * *      Functions for handling contents of single message.
 * *
 *****************************************************************************/
#ifdef __WATCOMC__
#include <unistd.h>
#include <io.h>  // access stuff
#include <stdio.h> // _popen, _pclose
#endif

#ifdef __EMX__
#include <unistd.h>
#include <io.h> // access
#endif
#include "jmr.hh"
#include "messagedata.hh"
#include "linereader.hh"

// Settings of program
extern settings_t Settings;

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: Message
//**************************************************************************/ 
//
// Constructor for class. 
// 
// 
//**************************************************************************/
Message::Message()
{
        characterset = ISO_LATIN_CHARSET;  // this is just a default...
        message_status = 0;
        message_number = 0;
        reference_msg_num = 0;
        group_number = 0;
        killed = false;
        msg_file = NULL;
        offset_in_file = 0;
        memset (cmpdate, 0, 13);
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: Message
//**************************************************************************/ 
//
// Constructor for class. Initializes class with given fstream handle.
// Position in file stream must be at start of article (i.e. before
// field 'size').
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
Message::Message (fstream *file)
{
        characterset = ISO_LATIN_CHARSET;  // this is just a default...
        message_status = 0;
        message_number = 0;
        reference_msg_num = 0;
        group_number = 0;
        killed = false;
        msg_file = file;
        offset_in_file = file->tellg();
        memset (cmpdate, 0, 13);
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: first_line
//**************************************************************************/ 
//
// Sets position in article to first line. If lines of article are not
// in memory, then reads lines from given file stream. If lines are readed
// from file, then character set of lines are translated from LOG_CHARSET
// to current character set of article.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: false if message is empty or contents of message cannot be read
//         from file.
//**************************************************************************/
bool
Message::first_line()
{
        // contents of article is in memory, then use it
        if (msg_lines.check() == true || msg_file == NULL) {
                return msg_lines.first();
        }
        
        // otherwise read lines from file to list
        int c;
        String *tmp;
        char *buffer;
        unsigned int size;
        byte_t little_endian_value[4];

        // luetaan tietueen koko
        msg_file->clear(); // clear status bits of stream
        msg_file->seekg (offset_in_file, ios::beg);
        msg_file->read ((char*) little_endian_value, 4);
	if (msg_file->fail()) {
                DEBUG ("Fatal error at message database");
		return false;
	}
        size = get_le_int32 (little_endian_value);
        
        
	// varataan tila puskurille
        buffer = new char [size+1];
        
        size -= SUBJECT_OFFSET;
        msg_file->read (buffer + SUBJECT_OFFSET, size);
        if (msg_file->fail()) {
                DEBUG ("Fatal error at message database, checkpoint 2");
                return false;
        }
        
        // read article lines
        c = ARTICLE_OFFSET;
        while ((unsigned int) c < size) {
                tmp = new String;
                c += tmp->put (buffer + c) +1;
                tmp->convert_charset (LOG_CHARSET, characterset);
                msg_lines.add (tmp);
        }
        delete[] buffer;
        return msg_lines.first();
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: unload_lines
//**************************************************************************/ 
//
// If lines are stored to file, then this function removes lines from
// memory. 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: true if lines was removed from memory. false if lines cannot
//         removed.
//**************************************************************************/
bool
Message::unload_lines()
{
        if (msg_file == NULL) {
                return false;
        }
        msg_lines.destroy();
        return true;
}


//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: set_subject
//**************************************************************************/ 
//
// Inits subject of message with given string, and creates stripped
// subject from it. Max. length of stripped subject is 21 characters.
//
//
// EXTERNAL VARIABLE REFERENCES
//   IN :  Settings.max_bbs_subject_len
//   OUT: 
// 
//**************************************************************************/
void
Message::set_subject (const String &str)
{
        int i = 0;
        subject = str;
        stripped_subject = str;

        // strip out possible prefix 'Re: ' from subject
        strip_re_prefix (stripped_subject);

        // then leave only 21 first letters to string stripped_subject.
        // Stripped subject is useful for sorting messages to thread
        // order. Limit 21 comes from QWK's short subjects and every BBS
        // has allways subjects that are shorten by QWK-readers...
        i = stripped_subject.length();
        i += 4;
        if (i > Settings.max_bbs_subject_len) {
                i -= Settings.max_bbs_subject_len;
                stripped_subject.ncut_end (i);
                stripped_subject.cut_tail();
        }
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: add_tagline
//**************************************************************************/ 
//
// Adds taglines to end of message.
//
//  Example:
//  ___
//  * jmr 0.5.13.3 [Unix] *  This is permanent tagline here.
//  ... The official tagline is placed here.
//
//  Usually it looks like:
//  ___
//  * jmr 0.5.13.3 [Unix] *
//  ... This message has been UNIXized for your protection.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN : Settings.ptagline 
//   OUT: 
// 
//**************************************************************************/
void
Message::add_tagline (const String &tag)
{
        String *s;

        remove_tail(); // remove trailing empty lines from message
        
        msg_lines.last();

        // Add empty line
        s = new String;
        msg_lines.add (s);
        
        // Add tear-line
        s = new String;
        *s = TAGLINE_INDICATOR;
        msg_lines.add (s);

        if ( tag.is_empty() && Settings.signaturefile != "" ) {
                if ( ACCESS( Settings.signaturefile.c_str(), R_OK ) == 0 ) {
                        Linereader l( Settings.signaturefile );
                        List<String>* sig = l.read_all_lines();
                        if ( sig && sig->first() == true ) {
                                do {
                                        s = new String;
                                        *s = *sig->get();
                                        msg_lines.add( s );
                                } while ( sig->next() );
                                delete sig;
                        }
                } else {
                        String tmp = "Cannot open signaturefile `";
                        tmp += Settings.signaturefile;
                        tmp += "'";
                        system_error( tmp.c_str() );
                }

        } else if ( tag.is_empty() && Settings.tagliner != "" ) {
                FILE *fp = popen( Settings.tagliner.c_str(), "r" );
                if ( fp ) {
                        char buffer[1024];
                        while ( fgets( buffer, 1023, fp ) ) {
                                msg_lines.add( new String( buffer ) );
                        }
                        pclose( fp );
                } else {
                        DEBUG( "Tagliner failed!" );
                        String tmp = "Cannot execute tagliner `";
                        tmp += Settings.tagliner;
                        tmp += "'";
                        system_error( tmp.c_str() );
                }
        } else {
                   // add tag of jmr and persistent tagline. Format is:
                   // * jmr x.y.z.f [platform] * persistent tagline here
                s = new String;
                   // *s = TAG + Settings.ptagline;
                *s = Settings.tag + Settings.ptagline;
                msg_lines.add (s);
                
                DEBUG("Adding tagline");
                if (tag.is_empty() == false) {
                        DEBUG("Tagline is not empty");
                        s = new String;
#ifdef TAGB
                        *s = TAGLEADER + tag;
#else
                        *s = "... " + tag;
#endif
                        msg_lines.add (s);
                }
        }

}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: add_line
//**************************************************************************/ 
//
// Adds new line to end of article.
// 
// RETURN: false if something goes wrong.
//**************************************************************************/
bool
Message::add_line (const String &line)
{
        String *tmp = new String;
        *tmp = line;
        msg_lines.last();
        return msg_lines.add (tmp);
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: insert_line
//**************************************************************************/ 
//
// Insert new line to start of article.
// 
// RETURN: false if something goes wrong
//**************************************************************************/
bool
Message::insert_line (const String &line)
{
        String *tmp = new String;
        *tmp = line;
        msg_lines.first();
        return msg_lines.insert (tmp);
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: remove_tail
//**************************************************************************/ 
//
// Removes empty lines from end of message.
// 
//**************************************************************************/
void
Message::remove_tail()
{
        String *s;
        while (msg_lines.last() == true) {
                s = msg_lines.get();
                s->cut_tail();
                if (s->is_empty() == false) {
                        break;
                } else {
                        msg_lines.remove();
                }
        };
        msg_lines.first();
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: strip_tagline
//**************************************************************************/ 
//
// Remove old tagline from message. This should be used only with messages
// writed by jmr.
//
//**************************************************************************/
#ifndef TAGB
void
Message::strip_tagline()
#else
void
Message::strip_tagline(String& stag)
#endif
{
        bool found = false;
        String *s;
        if (msg_lines.last() == true) {
                do {
                        s = msg_lines.get();
                        if (*s == TAGLINE_INDICATOR) {
                                //if (s->find (TAG) != NOTHING){
                                found = true;
                                break;
                        }
                } while (msg_lines.prev() == true);
                if (found == true) {
                        do {
#ifdef TAGB
			  char *tagtmp = msg_lines.get()->c_str();
			  if( 0 == strncmp(tagtmp, TAGLEADER, TAGLEADERLEN)) {
			    stag.copy(tagtmp,TAGLEADERLEN);
			  }
#endif
                                msg_lines.remove();
			  
                        } while (msg_lines.next() == true);
                }
        }
        remove_tail();
        msg_lines.first();
#ifdef TAGB
	// return *retstr;
#endif
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: save_to_file
//**************************************************************************/ 
//
// Writes contents of message to given output file stream
// 
// RETURN: true
//**************************************************************************/
bool
Message::save_to_file (ofstream& fout, const String& bbs,
                           const String& grp_name, const int grp_num)
{
        fout << "===============================================================================\n";
        fout <<"\n      BBS: " << bbs;
        fout <<"\n    Group: " << grp_name << " (" << grp_num <<")";
        fout <<"\n     Date: " << date.get_date() <<" " << time;
        fout <<"\n   Number: " << message_number;
        fout <<"\n   Writer: " << writer;
        fout <<"\n Receiver: " << receiver;
        fout <<"\n  Subject: " << subject;
        fout <<"\n";
        fout << "- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -\n\n";

        // write contents of article to file
        msg_lines.save_position();
        if (first_line() == false ) {
                return false;
        }
        do {
                assert (msg_lines.get());
                fout << *(msg_lines.get()) << "\n";
        } while (msg_lines.next() == true);
        fout <<"\n";
        msg_lines.restore_position();
        return true;
}


//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: translate
//**************************************************************************/ 
//
// Translates contents of article to given character set.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Message::translate (const int dest)
{
        if (dest == characterset) {
                return;
        }
        subject.convert_charset (characterset, dest);
        stripped_subject.convert_charset (characterset, dest);
        writer.convert_charset (characterset, dest);
        receiver.convert_charset (characterset, dest);
        if (msg_lines.first() == true) {
                do {
                        msg_lines.get()->convert_charset (characterset, dest);
                } while (msg_lines.next() == true);
                msg_lines.first();
        }
        characterset = dest;
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: set_cmpdate
//**************************************************************************/ 
//
// Sets up date formatted for fast comparison. Format will be yyyymmddHHMM,
// where yyyy is year, mm is month, dd is day, HH is hour and MM is minute.
//
// NOTE: time must set up using function set_cmptime.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Message::set_cmpdate()
{
        char *ptr = date.get_jmrdate().c_str();
        // ptr points now date formatted :  mm-dd-yyyy
        
        // set year
        cmpdate[0] = ptr[6];
        cmpdate[1] = ptr[7];
        cmpdate[2] = ptr[8];
        cmpdate[3] = ptr[9];

        // set month
        cmpdate[4] = ptr[0];
        cmpdate[5] = ptr[1];

        // set day
        cmpdate[6] = ptr[3];
        cmpdate[7] = ptr[4];

        // and add nul for safety...
        cmpdate[12] = 0x00;
}

//**************************************************************************/
// CLASS: Message
// MEMBER FUNCTION: set_cmptime
//**************************************************************************/ 
//
// Sets up time for fast comparison.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Message::set_cmptime()
{
        char *ptr = time.get();

        // set hour
        cmpdate[8] = ptr[0];
        cmpdate[9] = ptr[1];

        // set minute
        cmpdate[10] = ptr[3];
        cmpdate[11] = ptr[4];
        cmpdate[12] = 0x00;
}

bool
Message::search (char *pattern, int mode)
{
        bool rc = false;
        if (mode & SUBJECT_FL) {
                rc = subject.match(pattern);
        }
        if (rc == false && (mode & RECEIVER_FL)) {
                rc = receiver.match(pattern);
        }
        if (rc == false && (mode & AUTHOR_FL)) {
                rc = writer.match(pattern);
        }
        if (rc == false && (mode & ARTICLE_FL)
            && first_line() == true) {
                do {
                        rc = getline().match(pattern);
                } while (rc==false && next_line()==true);
                unload_lines();
        }
        
        return rc;
}



const Message&
Message::operator = (Message& src)
{
        if (this == &src) {
                return *this;
        }
        message_status = src.message_status;
        characterset = src.characterset;
        message_number = src.message_number;
        reference_msg_num = src.reference_msg_num;
        number_in_thread = src.number_in_thread;
        group_number = src.group_number;
        date = src.date;
        time = src.time;
        receiver = src.receiver;
        writer = src.writer;
        subject = src.subject;
        stripped_subject = src.stripped_subject;
        memcpy( cmpdate, src.cmpdate, 13 );
        killed = src.killed;
        
        DEBUG("-------------------- Message::operator = ----------------");
        if (msg_lines.first() == true) {
                DEBUG("  **** Need to remove contents of exist message!");
                msg_lines.destroy();
        }

        DEBUG("subject: "<<subject);
        DEBUG("writer : "<<writer);
        DEBUG("receiver:"<<receiver);
        DEBUG("-- Contents of message: ");
        msg_file = src.msg_file;
        offset_in_file = src.offset_in_file;

        if (src.msg_lines.check() == true) {
                src.msg_lines.save_position();
                src.msg_lines.first();
                String *tmp_str = 0;
                do {
                        assert( src.msg_lines.get() );
                        tmp_str = new String( src.msg_lines.get()->c_str() );
                        msg_lines.add( tmp_str );
                        DEBUG(tmp_str->c_str());
                } while (src.msg_lines.next() == true);
                msg_lines.first();
                src.msg_lines.restore_position();
        }

        DEBUG("--------------------------------------------------------");
        return *this;
}


