// $Id: mail.cc,v 1.23 1998/05/25 20:13:02 jvuokko Exp $
/*****************************************************************************
 * *
 * *      MODULE:     mail.cc
 * *                  ---------
 * ***************************************************************************
 * *
 * *
 * *      COPYRIGHT (C) 1997 JUKKA VUOKKO. ALL RIGHTS RESERVED
 * ***************************************************************************
 * *
 * *      This module contains functions of classes Group and Mail.
 * *
 *****************************************************************************/

#include <fstream.h>
#include <stdio.h>  // remove()
#ifdef __WATCOMC__
#   include <io.h>  // access()
#else
#   include <unistd.h> // F_OK, rename()
#endif
#include <time.h>
#include "mail.hh"
#include "qwk.hh"
#include "jmr.hh"


/************************ GLOBAL VARIABLES *********************************/
extern settings_t Settings;
extern Terminal_screen Screen;
extern Window* Wscript;


// This array defines rules for keywords in a resource file.
// "*" means that the keyword must have a value.
// ""             a value of the keyword can be empty.
// "$"            a value must be a number.
//
// Format of an array:
//   Keyword, rule1, rule2, ruleN,
//   "@NEXT",
//   .
//   .
//   .
//   "@NEXT", "@END"
static const char *bbs_setting_array[] = {
        RC_QUOTEPREFIX,"*",
        "@NEXT",
        RC_BBSCHARSET, "latin1", "ibmpc", "7bit",
        "@NEXT",
        RC_PTAG, "",
        "@NEXT",
        RC_EXTRACTDIR, "",
        "@NEXT",
        RC_REPLYLOGSIZE, "$",
        "@NEXT",
        RC_DATABASESIZE, "$",
        "@NEXT",
        RC_FOLLOWUPSTR, "",
        "@NEXT",
        RC_REPLYSTR, "",
        "@NEXT",
        RC_TAG, "",
        "@NEXT",
        "@END"
};



/***************************************************************************
 * FUNCTION : cmp_date
 ***************************************************************************
 *
 * description : Compares date and time fields of a two Message class
 *
 * return : <0, 0 or >0
 ***************************************************************************/
int cmp_date (const void *a, const void *b)
{
        Message **aa, **bb;

        aa = (Message**) a;
        bb = (Message**) b;

        return strcmp ((*aa)->get_cmpdate(), (*bb)->get_cmpdate());
}


/***************************************************************************
 * FUNCTION : cmp_subject
 ***************************************************************************
 *
 * description : Function for compare two subject field of a Message_t
 *               structure.
 *
 * return : <0, if first is less, 0 if equal, >0 if last is less than first.
 ***************************************************************************/
int cmp_subject (const void *a, const void *b)
{
        int rc;
        Message **aa, **bb;
        Message *cc, *dd;
        const String *s1, *s2;
        aa = (Message**) a;
        bb = (Message**) b;
        cc = aa[0];
        dd = bb[0];
        s1 = &cc->get_subject();
        s2 = &dd->get_subject();
        if ((rc = s1->compare (*s2)) == 0) {
                rc = cmp_date (a, b);
        }
        return rc;
}
/***************************************************************************
 * FUNCTION : cmp_thread
 ***************************************************************************
 *
 * description : Function for compare two stripped subject field of a
 *               Message_t structure.
 *
 *
 * return : <0, 0 or >0
 ***************************************************************************/
int cmp_thread (const void *a, const void *b)
{
        int rc;
        Message **aa, **bb;
        Message *cc, *dd;
        aa = (Message**) a;
        bb = (Message**) b;
        cc = aa[0];
        dd = bb[0];
        
        rc = cc->get_stripped_subject().compare( dd->get_stripped_subject() );
        if ( !rc ) {
                rc = cmp_date (a, b);
        }
        return rc;
}


/***************************************************************************
 * FUNCTION : cmp_writer
 ***************************************************************************
 *
 * description : compares writer-fields of a two Message_t structure
 *
 * return : guess what :-)
 ***************************************************************************/
int cmp_writer (const void *a, const void *b)
{
        int rc;
        Message **aa, **bb;
        Message *cc, *dd;
        const String *s1, *s2;
        aa = (Message**) a;
        bb = (Message**) b;
        cc = aa[0];
        dd = bb[0];
        s1 = &cc->get_writer();
        s2 = &dd->get_writer();
        if ((rc = s1->compare (*s2)) == 0) {
                rc = cmp_date (a, b);
        }
        return rc;
}

/***************************************************************************
 * FUNCTION : cmp_receiver
 ***************************************************************************
 *
 * description : compares receiver-fields
 *
 * return : <0, 0 or >0
 ***************************************************************************/
int cmp_receiver (const void *a, const void *b)
{
        int rc;
        Message **aa, **bb;
        Message *cc, *dd;
        const String *s1, *s2;
        aa = (Message**) a;
        bb = (Message**) b;
        cc = aa[0];
        dd = bb[0];
        s1 = &cc->get_receiver();
        s2 = &dd->get_receiver();
        if ((rc = s1->compare (*s2)) == 0) {
                rc = cmp_date (a, b);
        }
        return rc;
}


//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: Group
//**************************************************************************/ 
//
// Constructor for class.
// 
//**************************************************************************/
Group::Group (
              const int& g_num,     // number of group
              const String& g_name) // name of group
{
        is_mirror = false;
        number = g_num;
        unread_msgs = 0;
        total_msgs = 0;
        status_ = DEFAULT_FL;
        name = g_name;
        numberstr.put (number, 3);
}

//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: ~Group
//**************************************************************************/ 
//
// Destructor for class. Removes contents of all articles of group, but
// if group is marked as mirrored, then removes only list of messages,
// not contents of that list...
// 
//**************************************************************************/
Group::~Group()
{
        DEBUG("Deleting group: "<<number<<"   "<<name);
        if (is_mirror == true) {
                DEBUG("Mirrored!");
                messages.leave();
        } else {
                DEBUG("Normal");
                messages.destroy();
        }
        DEBUG("OK");
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: add
//**************************************************************************/ 
//
// Adds new article to end of exist articles.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: true if success.
//**************************************************************************/
bool
Group::add (const Message *msg)
{
        ++total_msgs;
        if (! (msg->get_status() & READ_ST)) {
                ++unread_msgs;
        }
        messages.last();
        return messages.add (msg);
}
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: remove_article
//**************************************************************************/ 
//
// Removes current article. If group is mirrored, then does not free
// memory of listnodes.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Group::remove_article()
{
        Message *msg;

        msg = messages.get();

        if (! (msg->get_status() & READ_ST) ) {
                --unread_msgs;
        }
        --total_msgs;
        
        if (is_mirror == false) {
                messages.remove();
        } else {
                messages.leavenode();
        }
}

//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: sort
//**************************************************************************/ 
//
// Sorts contents of group to using given order
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Group::sort (const int order)
{
        switch (order) {
        case SUBJECT_SORTED:
                messages.sort(cmp_subject);
                break;
        case DATE_SORTED:
                messages.sort(cmp_date);
                break;
        case WRITER_SORTED:
                messages.sort(cmp_writer);
                break;
        case RECEIVER_SORTED:
                messages.sort(cmp_receiver);
                break;
        case THREAD_SORTED:
                messages.sort(cmp_thread);
                break;
        }

        sort_order = order;
}

//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: update
//**************************************************************************/ 
//
// Updates statistics. Counts amount of unread articles.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Group::update()
{
        Message *msg;
        messages.save_position();
        unread_msgs = 0;
        total_msgs = 0;
        if (messages.first() == true) {
                do {
                        msg = messages.get();
                        if (! (msg->get_status() & READ_ST)) {
                                ++unread_msgs;
                        }
                        ++total_msgs;
                } while (messages.next() == true);
        }
        messages.restore_position();
}

//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: mark_all_read
//**************************************************************************/ 
//
// Marks all articles as read.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Group::mark_all_read()
{
        Message *msg;
        messages.save_position();
        if (messages.first() == true) {
                do {
                        msg = messages.get();
                        msg->set_status_bits (READ_ST);
                } while (messages.next() == true);
        }
        unread_msgs = 0;
        messages.restore_position();
}
                
//**************************************************************************/
// CLASS: Group
// MEMBER FUNCTION: unload
//**************************************************************************/ 
//
// Removes lines (from memory) of those articles, that have valid handle 
// to a file, where lines are stored.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Group::unload()
{
        if (messages.first() == true) {
                do {
                        messages.get()->unload_lines();
                } while (messages.next() == true);
                messages.first();
        }
}




//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: Mail
//**************************************************************************/ 
//
// Constructor for class. This should be called in first constructor of 
// derived classes.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
//
// PARAMETERS:
//      char *packet_name  Name of mail packet (QWK etc). Name must in
//                         format <bbsid>.XXX.
//                         
// RETURN: 
//**************************************************************************/
Mail::Mail (const char *packet_name)
{
	DEBUG("Entering Mail::Mail");
        tmp_db_file = NULL;
        tmp_log_file = NULL;

        num_tagged_groups = 0;
        
        // get id of bbs from name of packet.
        bbsid = packet_name;
        int i = bbsid.findch ('.');
        if (i != NOTHING) {
                bbsid.ncut_end (bbsid.length() - i);
        }
        bbsid.s_downcase();

        have_new_replies = false;
        new_articles = 0;
        articles = 0;
        dead_flag = false;
        replyhandler = new Replyhandler;

        // read bbs related settings from resource file, if it exist.
        local_settings.set( bbsid );
        
        // read contents of message database and replylog
        Wscript->enable();
        Wscript->print( "\nReading message database..." );
        Screen.refresh();
        read_message_database();

        Screen.refresh();
        // if previous session was killed, then read replies from
        // emergency file
        if (ACCESS (DEADJMR_FILENAME, F_OK) == 0) {
                read_deadjmr();
        }
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: create
//**************************************************************************/ 
//
// Creates new Mail-object, using child class, that matches with type
// of given mail packet.
//
// This function extracts contents of zipped mail packet, and solves
// type of packet by checking extracted files.
//
// If open_new_packet is false, then opens only exist message database, and
// does not open new mail packet.
//
// EXTERNAL VARIABLE REFERENCES
//   IN :  Settings.unzipcmd
//   OUT: 
// 
// RETURN: Pointer to created object, or 0 if failed.
//**************************************************************************/
Mail*
Mail::create (
              const char* packet_name,
              bool open_new_packet)
{
	DEBUG("Enterting Mail::create");
        String tmp;
        Mail *mail = 0;
        

        DEBUG("Clearing working directory");
        clear_workdir();
        Wscript->enable();
        Wscript->print( "\n----------------------------------------------\n");
        Wscript->print( "Opening mail packet/database: %s", packet_name );
        // extract mail packet
        if (open_new_packet == true) {
                tmp = Settings.unzipcmd + " " + Settings.qwkpath + packet_name;
                jmrsystem (tmp.get());
        }

        // check if type of packet is QWK.
        if (open_new_packet == false || ACCESS ("control.dat", F_OK) == 0) {
                // QWK-packet
		DEBUG("Creating Qwk object");
                mail = new Qwk (packet_name, open_new_packet);
        } else {
                handle_error ("Not a QWK packet", HALT_FL);
        }
#ifdef USE_DEBUG
        DEBUG( "GROUPS:" );
        mail->grouplist.first();
        do {
                DEBUG( mail->grouplist.get()->get_numstr() << " -- '"
                       << mail->grouplist.get()->get_name() << "'" );
        } while ( mail->grouplist.next() );
        mail->grouplist.first();
#endif
        return mail;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: open_dead
//**************************************************************************/ 
//
// Creates new Mail object using killed session.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 0 if failed, otherwise returns pointer to created object
//**************************************************************************/
Mail*
Mail::open_dead()
{
        Mail *mail = 0;
        char line[81];
        if (ACCESS (DEADJMR_FILENAME, F_OK) != 0) {
                return 0;
        }

        // read bbsid of killed session
        ifstream fin (DEADJMR_ID_FILENAME);
        if (fin.fail()) {
                return 0;
        }
        fin.getline (line, 80);
        fin.close();

        // check type of killed session
        if (ACCESS ("control.dat", F_OK) == 0) {
                // Type was QWK
                mail = new Qwk (line, true);
        } else {
                // if type of killed session is unknown, then read only
                // contents of message database and keep new replies
                // of killed session
                mail = new Qwk (line, false);
        }
        return mail;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: init
//**************************************************************************/ 
//
// Moves messages from list 'packet_articles' to message database, and
// sets up all necessary values. Function also removes deadjmr-files.
//
// NOTE: this function must be called before returned from constructor
//       of derived class!
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT:  Settings.is_pcboard, Settings.max_subject_len
// 
// RETURN: 
//**************************************************************************/
void
Mail::init()
{
        Wscript->enable();
        DEBUG("Moving messages to database");
        move_messages_to_base();
        DEBUG("OK");
        DEBUG("Setting up personal mail");
        set_personal_mail();
        DEBUG("OK");
        Wscript->print( "\nCreating article threads..." );
        Screen.refresh();

        clock_t start_time = clock();

        thread_sort_articles();

        clock_t end_time = clock();

        Wscript->print("OK (sorted in %.2f seconds)",
                       float((end_time - start_time) /(float) CLOCKS_PER_SEC));
        
        go_group_number (NEW_REPLIES_NUMBER);
        if (count_articles() > 0) {
                have_new_replies = true;
        }

        // check if bbs is useing PCBOARD-software, that allows
        // long subjects with QWK-packets.
        if (door.ifind ("pcboard") != NOTHING) {
                Settings.is_pcboard = true;
                Settings.max_subject_len = JMR_MAX_SUBJECT_LEN;
        } else {
                Settings.is_pcboard = false;
                Settings.max_subject_len = MAX_SUBJECT_LEN;
        }
        remove (DEADJMR_FILENAME);
        remove (DEADJMR_ID_FILENAME);
        write_recovery_file();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: emergency_exit
//**************************************************************************/ 
//
// Writes new replies to file 'deadjmr.msg' and sets flag
// dead_flag to true. Also writes ID of bbs to file 'deadjmr.id'.
//
// This function should be called if program is terminated abnormally.
//
// EXTERNAL VARIABLE REFERENCES
//   IN :  -
//   OUT:  -
// 
// RETURN: -
//**************************************************************************/
void
Mail::emergency_exit()
{
        dead_flag = true;
        write_recovery_file();
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: write_recovery_file
//**************************************************************************/ 
//
// Writes new replies to recovery file 'deadjmr.msg' and writes ID of the
// bbs to file 'deadjmr.id'.
//
// NOTES: This saves and restores postion in list, so do not call this
//        when you have saved group position.
//
// EXTERNAL VARIABLE REFERENCES
//   IN :  -
//   OUT:  -
// 
// PARAMETERS: -
//
// RETURN: -
//**************************************************************************/
void
Mail::write_recovery_file( void )
{
        DEBUG("Creating recovery files");
        String fname = DEADJMR_ID_FILENAME;

        // write bbsid to file 'deadjmr.id'
        ofstream fout (fname.get());
        if (fout.fail()) {
                system_error ("Cannot write recovery file.. Sad but true.",
                              HALT_FL);
        }
        fout << bbsid;
        fout.close();

        // then write new replies to file 'deadjmr.msg'.
        fname = DEADJMR_FILENAME;
        
        if (false == dead_flag) {
                grouplist.save_position();
        }
        go_group_number( NEW_REPLIES_NUMBER );
        write_log( fname, NEW_REPLIES_NUMBER, count_articles() +1, true );
        if (false == dead_flag) {
                grouplist.restore_position();
        }
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: ~Mail
//**************************************************************************/ 
//
// Destructor for class. Writes message database, replylog (if session
// was not killed).
// Removes temporary files.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
Mail::~Mail()
{
        Wscript->enable();
        if (dead_flag == false) {

                Wscript->print( "\nWriting message database..." );
                Screen.refresh();
                write_message_database();
                Wscript->print("OK");
                Screen.refresh();
        }
        delete replyhandler;
        DEBUG("Removing database");
        grouplist.destroy();
        DEBUG("Database removed!");
        DEBUG("Closing files");
        if (tmp_db_file != 0) {
                tmp_db_file->close();
                remove (tmp_db_name.get());
                delete tmp_db_file;
        }
        if (tmp_log_file != 0) {
                tmp_log_file->close();
                remove (tmp_log_name.get());
                delete tmp_log_file;
        }

        // restore original settings
        local_settings.restore();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: move_messages_to_base
//**************************************************************************/ 
//
// Moves contents of packet_articles list to message database. If some of
// articles exist already in database, then ask from user, if it is ok to
// replace all exiest articles with newer.
//
// After function, list 'packet_articles' will be empty.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Mail::move_messages_to_base()
{
        Group *grp, *tmp_grp;
        Message *msg, *tmp_msg;
        int number;
        bool replace = false;
        bool load = true;
        new_articles = 0;
        
        if (packet_articles->first() == false) {
                DEBUG( "** No new articles from packet! ** ");
                return;
        }
        
        // go through every group readed from packet
        do {
                grp = packet_articles->get();
                assert (grp != 0);

                if (false == go_group_number (grp->get_number())) {
                        tmp_grp = new Group (grp->get_number(),
                                             grp->get_name());
                        grouplist.add (tmp_grp);
                } else {
                        tmp_grp = grouplist.get();
                        if ( tmp_grp->get_name() != grp->get_name() ) {
                                tmp_grp->set_name( grp->get_name() );
                        }
                }
                
                assert (tmp_grp != 0);
                if (grp->first_article() == false) {
                        continue;
                }
                // go through every article within a single group from packet
                do {
                        msg = grp->get_article();

                        assert (msg != 0);
                        
                        number = msg->get_number();

                        if (REPLYLOG_NUMBER == tmp_grp->get_number()
                            || NEW_REPLIES_NUMBER==tmp_grp->get_number()) {
                                tmp_grp->add (msg);
                                continue;
                        } else if (false == tmp_grp->first_article()) {
                                tmp_grp->add (msg);
                                ++new_articles;
                                continue;
                        } else if (false == load) {
                                continue;
                        }

                        do {
                                tmp_msg = tmp_grp->get_article();
                                if (tmp_msg->get_number() == number) {
                                        if (false == replace) {
                                                load = ask_replace();
                                                if (false == load) {
                                                        break;
                                                }
                                                replace = true;
                                        }
                                        // remove old one, if exist.
                                        tmp_grp->remove_article();
                                        break;
                                }
                        } while (true == tmp_grp->next_article());
                        // add new article to group
                        if (true == load) {
                                tmp_grp->add (msg);
                                ++new_articles;
                        }
                } while (true == grp->next_article());
                
        } while (true == packet_articles->next());
        

        packet_articles->leave();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: ask_replace
//**************************************************************************/ 
// 
// Asks if user wants to load contents of mail packet, even if articles
// are already in message database.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: true if it's ok to load packet.
//**************************************************************************/
bool
Mail::ask_replace() const
{
        Wscript->enable();
        Wscript->print( "\n\nArticle is already in database. " );
        Wscript->print( "\nReload contents of mail packet to database (y/n) y" );
        int c = get_yesno ('y');
        if (c == 'y') {
                Wscript->print( "\bYes" );
                Screen.refresh();
                return true;
        }
        Wscript->print( "\bNo!" );
        Screen.refresh();
        return false;
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: read_message_database
//**************************************************************************/ 
//
// Reads contents of message database and replylog. Does not read lines
// of articles to memory, reads only headers at this point. Lines are
// readed only when it is necessary (using Message::first_line() method).
//
// Before reading, original database and replylog is copied to temporaly
// files.
//
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Mail::read_message_database()
{
        Group *grp;
        String file = Settings.datadir + bbsid + ".gdb";
        String mfile = Settings.datadir + bbsid + ".mdb";
        String logfile = Settings.datadir + bbsid + ".log";
        
        // remove exist articles.
        // List should already be empty before this but...
        grouplist.destroy();

        // add basegroups to begin of list.
        grp = new Group (PERSONAL_NUMBER, PERSONAL_TEXT);
        grp->set_mirror();
        grouplist.add (grp);
        grp = new Group (REPLYLOG_NUMBER, REPLYLOG_TEXT);
        grouplist.add (grp);
        grp = new Group (NEW_REPLIES_NUMBER, NEW_REPLIES_TEXT);
        grouplist.add (grp);
        grp = new Group (TAGGED_ARTICLES_NUMBER, TAGGED_ARTICLES_TEXT);
        grp->set_mirror();
        grouplist.add (grp);
        grp = new Group (SEARCH_RESULTS_NUMBER, SEARCH_RESULTS_TEXT);
        grp->set_mirror();
        grouplist.add (grp);
        
        DEBUG("Reading message database and replylog");
        if (ACCESS (file.get(), F_OK) == 0 &&
            ACCESS (mfile.get(), F_OK) == 0) {
                DEBUG("Database exist!");
                tmp_db_name = get_tmpname();
                if (copy_file (mfile.get(), tmp_db_name.get()) == false) {
                        char buffer[256];
                        sprintf (buffer, "Cannot create temp file : %s\n",
                                 tmp_db_name.get());
                        system_error(buffer, HALT_FL );
                }
                read_groups_from_db();
                read_articles_from_db();
                Wscript->print("OK");
        } 

        if (ACCESS (logfile.get(), F_OK) == 0) {
                tmp_log_name = get_tmpname();
                if (copy_file (logfile.get(), tmp_log_name.get()) == false) {
                        char buffer[256];
                        sprintf (buffer, "Cannot create temp file : %s\n",
                                 tmp_db_name.get());
                        system_error (buffer, HALT_FL);
                }
                read_replylog();
        }
}
//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: read_groups_from_db
//**************************************************************************/ 
//
// Reads data of bbs, names and numbers of groups from group database
// (<bbsid>.gdb).
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Mail::read_groups_from_db()
{
        int pos = 0;
        int num;
        String name;
        String file;
        Group *grp;
        char *buffer;
        int size = 0;

        assert (bbsid.is_empty() == false);
        file = Settings.datadir + bbsid + ".gdb";
        
        DEBUG("Reading groups : "<< file.get());
        fstream fin (file.get(), ios::in
#ifndef __unix__
                     | ios::binary
#endif
                     );
        if (fin.fail()) {
                system_error ("Cannot open message database", HALT_FL);
        }
        fin.seekg (0, ios::end);
        size = fin.tellg();
        fin.seekg (0);
        
        // read whole file to buffer
        buffer = new char [size + 1];
        fin.read (buffer, size);
        fin.close();

        // get information about bbs from buffer
        pos += bbsname.put (buffer) + 1;
        DEBUG( "Readed from gdb: '" << bbsname << "'" );
        pos += city.put (buffer + pos) +1;
        DEBUG( "Readed from gdb: '" << city << "'" );
        pos += phone.put (buffer + pos) +1;
        DEBUG( "Readed from gdb: '" << phone << "'" );
        pos += sysop.put (buffer + pos) +1;
        DEBUG( "Readed from gdb: '" << sysop << "'" );
        pos += door.put (buffer + pos) +1;
        pos += time.put (buffer + pos) +1;
        pos += username.put (buffer + pos) +1;
        username.convert_charset (LOG_CHARSET, Settings.syscharset);
        username.s_capitalize();
        Settings.username = username;

        if ( door.ifind( "pcboard" ) != NOTHING ) {
                Settings.is_pcboard = true;
                Settings.max_subject_len = JMR_MAX_SUBJECT_LEN;
        }

        
        // get groups from buffer
        while (pos < size) {
                num = atoi (buffer + pos);
                pos += strlen (buffer + pos) + 1;
                pos += name.put (buffer + pos) +1;
                name.convert_charset (LOG_CHARSET, Settings.syscharset);
                DEBUG( "Read group from db: '" << name << "'" );
                grp = new Group (num, name);
                DEBUG( "Adding group from db: '" << name << "'" );
                grouplist.add (grp);
        }
        
        delete[] buffer;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: read_articles_from_db
//**************************************************************************/ 
//
// Read articles from temporaly copy of message database. Does not
// read lines of articles to memory, they should be read only when
// it is necessary. This program is not memory monster like Microsoft...
//
// File opened here must be keep opened until the destructor closes it.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  Settings.syscharset
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Mail::read_articles_from_db()
{
        Message *msg;
        Group *grp;
        Group *tag;
        char idcode[4];
        
        go_group_number (TAGGED_ARTICLES_NUMBER);
        tag = grouplist.get();

        assert (bbsid.is_empty() == false);
        assert (tmp_db_file == NULL);


        DEBUG("Reading articles : "<< tmp_db_name.get());
        tmp_db_file = new fstream (tmp_db_name.get(), ios::in
#ifndef __unix__
                                   | ios::binary
#endif
                                   );
        if (tmp_db_file->fail()) {
                system_error ("Cannot open message database", HALT_FL);
        }
        tmp_db_file->read ((char*) idcode, 4); // skip over id code
        
        while (NULL != (msg=read_msg_from_log(*tmp_db_file, HEADER_ONLY_FL))) {
                if (go_group_number (msg->get_group_number()) == true) {
                        grp = grouplist.get();
                } else {
                        grp = new Group (msg->get_group_number(), "Unknown");
                        grouplist.add (grp);
                }
                msg->set_charset (LOG_CHARSET);
                msg->translate (Settings.syscharset);
                grp->add (msg);
                if (msg->get_status() & TAGGED_ST) {
                        tag->add (msg);
                }
#ifdef HASH_HACK
                char a[30];
                char b[15];
//                unsigned int hkey = ((unsigned) msg->get_group_number()) << 17;
//                hkey |= msg->get_number();
                msg->get_subject().get_at( a, 10 );
                msg->get_writer().get_at( b, 10 );
                strcat( a, b ); 
                unsigned int hkey = msg->get_group_number() * msg->get_number();
/*                unsigned int foo = 1;
                for ( int x = 0; a[x]; ++x ) {
                        foo = a[x] + 31 * foo;
                }
                hkey *= foo; */
                cerr << "Hashkey: " << hkey << endl;
#endif
        }
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: read_deadjmr
//**************************************************************************/ 
//
// Read replies of killed session.
// NOTE: contents of message database must be read before this!
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Mail::read_deadjmr()
{

        fstream fin;
        Message *msgdata;
        Group *grp;
        char idcode[4];

        fin.open (DEADJMR_FILENAME, ios::in
#ifndef __unix__
                  | ios::binary
#endif
                  );
        if (fin.fail()) {
                return;
        }
        
        // skip over idcode 
        fin.read ((char*) idcode, 4);

        go_group_number (NEW_REPLIES_NUMBER);
        grp = grouplist.get();
        
        msgdata = read_msg_from_log (fin);
        while (msgdata != NULL) {
                msgdata->set_charset (LOG_CHARSET);
                msgdata->translate (Settings.syscharset);
                grp->add (msgdata);
                msgdata = read_msg_from_log (fin);
        }
        fin.close();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: write_message_database
//**************************************************************************/ 
//
// Writes new message database and replylog
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Mail::write_message_database()
{
        write_groups_to_db();
        write_articles_to_db();
        write_replylog();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: write_groups_to_db
//**************************************************************************/ 
//
// Writes information of bbs and numbers and names of each group to
// database file <bbsid>.gdb
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: true
//**************************************************************************/
bool
Mail::write_groups_to_db()
{
        Group *grp;
        String str;
        String file;
        
        file = Settings.datadir + bbsid + ".gdb";

        fstream fout (file.get(), ios::out | ios::trunc
#ifndef __unix__
                       | ios::binary
#endif
                       );
        if (fout.fail()) {
                system_error ("Cannot write message database", HALT_FL);
        }

        // Write information about bbs to file
        DEBUG( "writing to gdb: '" << bbsname.get() << "'" );
        fout.write (bbsname.get(), bbsname.length()+1);
        DEBUG( "writing to gdb: '" << city.get() << "'" );
        fout.write (city.get(), city.length()+1);
        DEBUG( "writing to gdb: '" << phone.get() << "'" );
        fout.write (phone.get(), phone.length()+1);
        DEBUG( "writing to gdb: '" << sysop.get() << "'" );
        fout.write (sysop.get(), sysop.length()+1);
        fout.write (door.get(), door.length()+1);
        fout.write (time.get(), time.length()+1);
        username.convert_charset (Settings.syscharset, LOG_CHARSET);
        fout.write (username.get(), username.length()+1);

        // write number and name of each group
        grouplist.first();
        do {
                grp = grouplist.get();
                if (grp->get_number() < BASEGROUP_NUMBER) {
                        fout.write (grp->get_numstr().get(),
                                    grp->get_numstr().length()+1);
                        str = grp->get_name();
                        str.convert_charset (Settings.syscharset, LOG_CHARSET);
                        fout.write (str.get(), str.length()+1);
                        DEBUG( "Writing group to gdb: '"<<str<<"'");
                }
        } while (grouplist.next() == true);

        
        fout.close();

        return true;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: write_articles_to_db
//**************************************************************************/ 
//
// Writes articles to message database.
//
// Before writing, all articles are copied to one list and sorted in order
// of date. Only Settings.databasesize newest articles are writed to file.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  Settings.databasesize, Settings.datadir
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
bool
Mail::write_articles_to_db()
{
        Group *grp;
        Message *msg;
        List<Message> msglist;
        String file;
        char idcode[4];

        DEBUG("Writing message database");
        
        file = Settings.datadir + bbsid + ".mdb";
        fstream fout (file.get(), ios::out | ios::trunc
#ifndef __unix__
                      | ios::binary
#endif
                      );
        if (fout.fail()) {
                system_error ("Cannot write message database", HALT_FL);
        }

        put_le_int32 ((byte_t*) idcode, LOG_IDCODE);
        fout.write (idcode, 4);

        // add all articles to single list
        grouplist.first();
        do {
                grp = grouplist.get();
                if (grp->get_number() < BASEGROUP_NUMBER
                    && grp->first_article() == true) {
                        do {
                                msg = grp->get_article();
                                msglist.add (msg);
                        } while (grp->next_article() == true);
                }
        } while (grouplist.next() == true);
        
        // sort articles by date
	DEBUG("Sorting articles by date...");
#ifdef USE_DEBUG
        clock_t start_time = clock();
#endif
	msglist.sort( cmp_date );
#ifdef USE_DEBUG
        clock_t end_time = clock();
        cerr <<"Sorted in "<< float((end_time - start_time) / (float) CLK_TCK)
	  << " seconds";
#endif
	DEBUG("OK");
        
        // Check if size of database is larger than limit.
        msglist.first();
        int i = msglist.count_items() - Settings.databasesize;
        DEBUG("Trashing "<<i<<" oldest articles");

        // If size is too big, then mark oldest i articles
        // with killed flag.
        if (i > 0 ) {
                do {
                        msg = msglist.get();
                        // If article is not tagged and is not marked as 
                        // killed, then mark it killed now.
                        if (! (msg->get_status() & TAGGED_ST) &&
                            !msg->is_killed()) {
                                msg->invert_kill_flag();
                                --i;
                        }
                } while (i > 0 && msglist.next());
        }

        // Write database, skip killed articles.
        msglist.first();
        do {
                msg = msglist.get();
		if (msg->is_killed() == true &&
		    (msg->get_status() & TAGGED_ST) ) {
			msg->reset_status_bits( TAGGED_ST);
		} else if (msg->is_killed() ) {
                        continue;
                }
                msg->first_line();
                msg->translate (LOG_CHARSET);
                write_msg_to_log (fout, *msg);
                // remove lines of article from memory
                msg->unload_lines();
        } while (msglist.next() == true);
        
        msglist.leave();
        fout.close();

        DEBUG("OK");
        return true;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: read_replylog
//**************************************************************************/ 
//
// Reads old replies of current bbs from temporaly replylog file.
// Does not read lines of articles to memory.
//
// NOTE: Variable tmp_log_name must be set before use this!
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  Settings.datadir, Settings.syscharset
//   OUT: 
// 
//**************************************************************************/
void
Mail::read_replylog()
{
        Message *msgdata;
        Group *grp;
        byte_t idcode[4];
        Message* (*read_func)(fstream&, int);

        assert (tmp_log_file == NULL);
        assert (tmp_log_name.is_empty() == false);

        DEBUG("Reading replylog : "<<tmp_log_name.get());
        tmp_log_file = new fstream (tmp_log_name.get(), ios::in
#ifndef __unix__
                                    | ios::binary
#endif
                                    );
        if (tmp_log_file->fail()) {
                return;
        }
        Wscript->enable();
        Wscript->print( "\nReading replylog..." );
        Screen.refresh();
        // select right function for reading data from log
        tmp_log_file->read ((char*) idcode, 4);
        if (get_le_int32 (idcode) != LOG_IDCODE) {
                Wscript->print( "\nObsolete log format detected. Converting...");
                Screen.refresh();
                read_func = read_msg_from_obsolete_log;
        } else {
                read_func = read_msg_from_log;
        }

        go_group_number (REPLYLOG_NUMBER);
        grp = grouplist.get();
        
        msgdata = read_func (*tmp_log_file, HEADER_ONLY_FL);
        while (msgdata != NULL) {
                msgdata->set_charset (LOG_CHARSET);
                msgdata->translate (Settings.syscharset);
                grp->add (msgdata);
                msgdata = read_func (*tmp_log_file, HEADER_ONLY_FL);
        }
        Wscript->print( "OK" );
        Screen.refresh();
}

//**************************************************************************/
//  CLASS: Mail
//  MEMBER FUNCTION: write_replylog
// 
// 
//  Writes replylog of current bbs to file <bbsid>.log.
// 
// 
//  EXTERNAL REFERENCES:
//  IN :  Settings.datadir, Settings.replylogsize
//  OUT: 
//**************************************************************************/
void
Mail::write_replylog()
{
        String fname = Settings.datadir + bbsid + ".log";
        write_log (fname, REPLYLOG_NUMBER, Settings.replylogsize);
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: write_log
//**************************************************************************/ 
// Writes contents of given group to given file. If group has more articles
// than given maximum count is, then writes only newest <maxsize> articles.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
//**************************************************************************/
void
Mail::write_log (const String &fname,   // name of file
                 const int group_number,// number of group
                 const int maxsize,     // max. number of articles
                 bool  keep_charset)   // should charset restored after writing
{
        int size,i;
        Message *msg;
        char idcode[4];

        go_group_number (group_number);
        // check size of log. Return immediately if there's no replies
        size = count_articles();
        if (size == 0) {
                return;
        }

        DEBUG("Writing replies to log");
        first_article();
        DEBUG("Size of logfile is : "<<size);
        DEBUG("Max. size is : "<<maxsize);
        // skip over oldest articles, if count of articles > maxsize
        if (size > maxsize) {
                sort (DATE_SORTED);
                i = size - maxsize;
                DEBUG("Trashing "<<i<<" articles...");
                while (i) {
                        next_article();
                        i--;
                }
        }

        
        fstream file (fname.get(), ios::out | ios::trunc
#ifndef __unix__
                      | ios::binary
#endif
                      );
        if (file.fail()) {
                system_error ("Cannot write log");
                return;
        }

        // write contents of log
        put_le_int32 ((byte_t*)idcode, LOG_IDCODE);
        file.write (idcode, 4); // tunniste
        do {
                msg = get_article();
                if (msg->is_killed() == false) {
                        // mark article as unread
                        // msg->reset_status_bits (READ_ST);
                        msg->first_line();
                        msg->translate (LOG_CHARSET);
                        write_msg_to_log (file, *msg);
                        if ( keep_charset == true ) {
                                msg->translate( Settings.syscharset );
                        }
                        msg->unload_lines();
                }
        } while (next_article() == true);
        file.close();
}



//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: set_personal_mail
//**************************************************************************/ 
//
// Copies pointer of each article, which's receiver matches with
// Settings.username to group 'Personal Mail'. (Personal articles are mirrored
// to group 'Personal Mail').
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  Settings.username
//   OUT: 
// 
// RETURN: 
//**************************************************************************/
void
Mail::set_personal_mail()
{
        Group *personal, *grp;
        Message *msg;
        go_group_number (PERSONAL_NUMBER);
        personal = grouplist.get();

        assert (personal != 0);
        
        grouplist.first();

        do {
                grp = grouplist.get();
                if (grp->get_number() >= BASEGROUP_NUMBER) {
                        continue;
                }
                if (grp->first_article() == false) {
                        continue;
                }
                
                do {
                        msg = grp->get_article();

                        assert (msg != NULL);
                        
                        if (Settings.username == msg->get_receiver()) {
                                personal->add (msg);
                        }
                } while (grp->next_article() == true);

        } while (grouplist.next() == true);

}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: thread_sort_articles
//**************************************************************************/ 
//
// Sorts contents of basegroups by subject, and other groups by threads.
// 
//**************************************************************************/
void
Mail::thread_sort_articles ()
{
        // contents first 4 groups is sorted by subject (basegroups)
        grouplist.first();
        assert (grouplist.first() == true);

        for (int i = 0; i < FIRST_GROUP_FROM_PACKET; ++i) {
                sort (SUBJECT_SORTED);
                grouplist.next();
        }

        // and contents of rest groups is sorted by threads
        do {
                sort (THREAD_SORTED);
                articles += count_articles();
        } while (true == grouplist.next());
        grouplist.first();
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: sort
//**************************************************************************/ 
//
// Sorts messages of current group to given order. After this function,
// current position within group will be first article.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
//**************************************************************************/
void
Mail::sort (const int order)
{
        Group *tmp;
        int num;

        tmp = grouplist.get();
        assert (tmp != 0);
        DEBUG("Sorting group: '" << grouplist.get()->get_name() <<"'");
        tmp->sort (order);

        num = tmp->get_number();
        if (num == PERSONAL_NUMBER) {
                return;
        }
        tmp->first_article();
}


//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: operator []
//**************************************************************************/ 
//
// Moves position in list of groups to given index and returns reference
// to that group. If index is out of range, then returned group will be
// last group of list.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: reference to group
//**************************************************************************/
const Group&
Mail::operator [] (int index)
{
        grouplist.first();
        
        for (int i = 0; i < index; i++) {
                grouplist.next();
        }
        
        assert (grouplist.get() != 0);
        
        return *(grouplist.get());
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: go_group_index
//**************************************************************************/ 
//
// Goes to group of given index.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: false if given index is out of range
//**************************************************************************/
bool
Mail::go_group_index (const int n)
{
        if (n >= grouplist.count_items()) {
                return false;
        }
        (*this)[n];
        return true;
}
        

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: go_group_number
//**************************************************************************/ 
//
// Moves position in list of groups to group, that have number matches
// with given value. If group not found, then position will be last group
// of list.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// RETURN: true if group exist. false if not.
//**************************************************************************/
bool
Mail::go_group_number(const int num)
{
        Group *tmp;
        grouplist.first();
        do {
                tmp = grouplist.get();
                if (tmp->get_number() == num)
                        return true;
        } while (grouplist.next() == true);
        
        return false;
}



//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: add_reply
//**************************************************************************/ 
//
// Adds new reply to end of list of new replies.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
//**************************************************************************/
void
Mail::add_reply (Message *reply)
{
        grouplist.save_position();
        go_group_number (NEW_REPLIES_NUMBER);
        grouplist.get()->add (reply);
        grouplist.restore_position();
        have_new_replies = true;
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: kill_reply
//**************************************************************************/ 
//
// Removes given reply from list of new replies.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
//**************************************************************************/
void
Mail::kill_reply (const Message *msg)
{
        Message *tmp;
        Group *grp;

        grouplist.save_position();
        go_group_number (NEW_REPLIES_NUMBER);
        grp = grouplist.get();
        grouplist.restore_position();
        
        grp->first_article();
        do {
                tmp = grp->get_article();
                if (msg == tmp) {
                        grp->remove_article();
                        break;
                }
        } while (grp->next_article() == true);
}

//**************************************************************************/
// CLASS: Mail
// MEMBER FUNCTION: tag_article
//**************************************************************************/ 
//
// Marks current message as tagged, or untags message if it's
// already tagged.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
//**************************************************************************/
void
Mail::tag_article (Message *msg)
{
        Message *tmp;
        Group *grp;
        int number;

        if (!msg) {
                return;
        }
        
        msg->inv_status_bits (TAGGED_ST);
        
        grouplist.save_position();
        go_group_number (TAGGED_ARTICLES_NUMBER);
        grp = grouplist.get();
        // add message to tagged list list
        if (msg->get_status() & TAGGED_ST) {
                grp->add (msg);
        } else {
                // remove message from tagged article list
                grp->first_article();
                number = msg->get_number();
                do {
                        tmp = grp->get_article();
                        if (tmp->get_number() == number) {
                                grp->remove_article();
                                break;
                        }
                } while (grp->next_article() == true);
        }
        grouplist.restore_position();
}


int
Mail::search (char* pattern, int mode)
{
        int count = 0;
        
        assert( mode & ALL_FL || mode & TAGGED_FL || mode & CURRENT_FL );
        
        // search
        if (mode & ALL_FL) {
                count = search_all( pattern, mode );
        } else if (mode & TAGGED_FL) {
                count = search_tagged( pattern, mode );
        } else {
                count = search_current( pattern, mode );
        }
                
        return count;
}


int
Mail::search_all (char* pattern, int mode)
{
        Group *grp;
        Group *search_grp;
        Message *msg;
        int count = 0;
        int searched_articles = 0;
        float percent = 0;

        save_group_pos();
        
        // remove results of previous search
        go_group_number (SEARCH_RESULTS_NUMBER);
        search_grp = grouplist.get();
        if (search_grp->first_article() == true) {
                do {
                        search_grp->remove_article();
                } while (search_grp->first_article() == true);
        }
        

        go_group_index(FIRST_GROUP_FROM_PACKET);
        do {
                Screen.bottom();
                Screen.print( "%d%% of articles searched", (int) percent );
                Screen.refresh();
                grp = grouplist.get();
                if (grp->first_article() == false) {
                        continue;
                }
                do {
                        msg = grp->get_article();
                        if (msg->search( pattern, mode ) == true) {
                                ++count;
                                search_grp->add (msg);
                        }
                        ++searched_articles;
                } while (grp->next_article() == true);
                percent = (float) searched_articles / (float) articles;
                percent *= 100;
        } while (grouplist.next() == true);

        restore_group_pos();
        return count;

}

int
Mail::search_current (char* pattern, int mode)
{
        Group *grp;
        Group *search_grp;
        Message *msg;
        int count = 0;
        
        save_group_pos();
        
        // remove results of previous search
        go_group_number (SEARCH_RESULTS_NUMBER);
        search_grp = grouplist.get();
        if (search_grp->first_article() == true) {
                do {
                        search_grp->remove_article();
                } while (search_grp->first_article() == true);
        }
        
        restore_group_pos();

        grp = grouplist.get();
        if (grp->first_article() == false) {
                return 0;
        }

        do {
                msg = grp->get_article();
                if (msg->search( pattern, mode ) == true) {
                        ++count;
                        search_grp->add( msg );
                }
        } while (grp->next_article() == true);

        return count;
}


int
Mail::search_tagged (char* pattern, int mode)
{
        int count = 0;
        Group *grp;
        Group *search_grp;
        Message *msg;
        int searched_articles = 0;
        int all_articles = 0;
        float percent = 0;

        save_group_pos();
        
        // remove results of previous search
        go_group_number (SEARCH_RESULTS_NUMBER);
        search_grp = grouplist.get();
        if (search_grp->first_article() == true) {
                do {
                        search_grp->remove_article();
                } while (search_grp->first_article() == true);
        }
        
        go_group_index(FIRST_GROUP_FROM_PACKET);
        // count amount of articles to search
        do {
                grp = grouplist.get();
                if (grp->is_tagged() == false) {
                        continue;
                }
                if (grp->first_article() == false) {
                        continue;
                }
                all_articles += grp->count_articles();
        } while (grouplist.next() == true);

        // search
        go_group_index( FIRST_GROUP_FROM_PACKET);
        do {
                Screen.bottom();
                Screen.print("%d%% of articles searched", (int) percent );
                Screen.refresh();
                grp = grouplist.get();
                if (grp->is_tagged() == false) {
                        continue;
                }
                if (grp->first_article() == false) {
                        continue;
                }
                do {
                        msg = grp->get_article();
                        if (msg->search( pattern, mode ) == true) {
                                ++count;
                                search_grp->add (msg);
                        }
                        ++searched_articles;
                } while (grp->next_article() == true);
                percent = (float) searched_articles / (float) all_articles;
                percent *= 100;
        } while (grouplist.next() == true);

        restore_group_pos();
        return count;
}


void
Mail::tag_group()
{
        if (grouplist.get()->get_number() >= BASEGROUP_NUMBER) {
                return;
        }
        grouplist.get()->tag();
        if (grouplist.get()->is_tagged() == true) {
                ++num_tagged_groups;
        } else {
                --num_tagged_groups;
        }
}

int
Mail::count_tagged_groups() const
{
        return num_tagged_groups;
}




//**************************************************************************/
// CLASS: bbs_settings
// MEMBER FUNCTION: set
//**************************************************************************/ 
//
// This reads settings defined in file <bbsid>.rc, and replaces
// original settings using them.
// Contents of original Settings-structure are stored and them
// can be restored when these settings are no longer needed.
//
// EXTERNAL VARIABLE REFERENCES
//   IN :  Settings.jmrdir
//   OUT:  Settings
// 
// PARAMETERS: id of the current bbs.
//
// RETURN: true if new settings was defined, false if not
//**************************************************************************/
bool
bbs_settings::set (String& bbsid)
{
        rcfname = Settings.jmrdir + bbsid + ".rc";
        if ( ACCESS( rcfname.get(), F_OK) ) {
                save_flag = false;
                return false;
        }
        if (read_resources() == false) {
                save_flag = false;
                return false;
        }

        String value;

        saved = Settings;
        save_flag = true;
        
        DEBUG("Setting up bbs related resources");
        value = trie.get_value( RC_BBSCHARSET );
        if ( value.is_empty() == false ) {
                if (value == "ibmpc") {
                        Settings.bbscharset = IBM_PC_CHARSET;
                } else if (value == "latin1") {
                        Settings.bbscharset = ISO_LATIN_CHARSET;
                } else {
                        Settings.bbscharset = ASCII_7BIT_CHARSET;
                }
        }

        value = trie.get_value( RC_PTAG );
        if (value.is_empty() == false) {
                Settings.ptagline = value;
        }

        value = trie.get_value( RC_TAG );
        if (value.is_empty() == false) {
                Settings.tagline = value;
        }

        value = trie.get_value( RC_QUOTEPREFIX );
        if (value.is_empty() == false) {
                value.replace_ch( '_', ' ' );
                Settings.quoteprefix = value;
        }

        value = trie.get_value( RC_REPLYLOGSIZE );
        if (value.is_empty() == false) {
                Settings.replylogsize = atoi( value.get() );
        }

        value = trie.get_value( RC_DATABASESIZE );
        if (value.is_empty() == false) {
                Settings.databasesize = atoi( value.get() );
        }

        value = trie.get_value( RC_EXTRACTDIR );
        if (value.is_empty() == false) {
                correct_path( value );
                Settings.extractdir = value;
        }

        value = trie.get_value( RC_FOLLOWUPSTR );
        if (value.is_empty() == false) {
                Settings.followup = value;
        }
        
        value = trie.get_value( RC_REPLYSTR );
        if (value.is_empty() == false) {
                Settings.reply = value;
        }
        return true;
}
//**************************************************************************/
// CLASS: bbs_settings
// MEMBER FUNCTION: read_resources
//**************************************************************************/ 
//
//  This function reads contents of a resource file to a tree structure.
//  If syntax of the resource file is not correct, then program is halted.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: true if syntax of the file was right.
//**************************************************************************/
bool
bbs_settings::read_resources()
{
        String tmp;
        char line[MAX_LINE_LEN+1];
        int linenum = 1;
        bool rc = true;
        
        ifstream rcfile( rcfname.get(), ios::in );
        if (rcfile.fail()) {
                String err_str = "Cannot read resource file : " + rcfname;
                system_error (err_str.get());
                return false;
        }
        Wscript->enable();
        Wscript->print("\nParsing resource file: \'%s\'....", rcfname.c_str());
        Screen.refresh();
        do {
                rcfile.getline (line, MAX_LINE_LEN);
                tmp = line;

                // remove preceeding spaces
                tmp.cut_first();

                // if line is not empty and it is not commented,
                // then check syntax of line and get value
                // of keyword.
                if (tmp.is_empty() == false && tmp[0] != '#') {
                        // remove trailing spaces
                        tmp.cut_tail();

                        // parse the line
                        if (parse_rcline( tmp, linenum ) == false) {
                                rc = false;
                        }
                }
                ++linenum;
        } while (!rcfile.eof());
        rcfile.close();

        if (rc == false) {
                handle_error("Fatal errors in bbs related resource file. Game over - Insert a coin.",
                             HALT_FL);
        }
        Wscript->print( "OK" );
        Screen.refresh();
        return true;
}
//**************************************************************************/
// CLASS: bbs_settings
// MEMBER FUNCTION: parse_rcline
//**************************************************************************/ 
//
//  Checks that syntax of a line is right, and initializes keyword that
//  is defined in a line with a value given to it.
//  If syntax error occures, then false is returned.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS: line  Contents of the current line
//             num   Number of the line
//
// RETURN: true if syntax of the line was right, false if not.
//**************************************************************************/
bool
bbs_settings::parse_rcline (String& line, int num)
{
        int n=0;
        String tmp;

        // check that assigment operator is in it's place
        n = line.findch ('=');
        if (n == NOTHING) {
                String error_str ;
                error_str = "Assigment operator is missing at line ";
                error_str += num;
                handle_error( error_str.get(), REPORT_FL);
                return false;
        }

        // Then get word before an assigment operator and check
        // if it matches with some of keywords in a trie
        tmp.copy( line, 0, n);
        tmp.cut_tail();
        if (false == trie.goto_keyword( tmp )) {
                String error_str = "At line ";
                error_str += num;
                error_str += ": Illegal keyword : \'" + tmp + "\'";
                handle_error( error_str.get(), REPORT_FL);
                return false;
        }

        // Check that given value is legal, and store it
        // in a trie.
        tmp.copy( line, n + 1 );
        tmp.cut_first();
        tmp.cut_tail();
        if (false == trie.set_value( tmp )) {
                String error_str = "Invalid value at line ";
                error_str += num;
                error_str += ": \'" + tmp + "\'";
                handle_error( error_str.get(), REPORT_FL );
                return false;
        }

        return true;
}

//**************************************************************************/
// CLASS: bbs_settings
// MEMBER FUNCTION: restore
//**************************************************************************/ 
//
// This function restores original settings.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: Settings
// 
// PARAMETERS:
//
// RETURN: always true
//**************************************************************************/
bool
bbs_settings::restore ()
{
        if (save_flag == true) {
                DEBUG("Restoring original settings.");
                Settings = saved;
                save_flag = false;
        }
        return true;
}

//**************************************************************************/
// CLASS: bbs_settings
// MEMBER FUNCTION: bbs_settings
//**************************************************************************/ 
//
// Constructor for class. Initializes syntax trie using values defined
// in static array bbs_setting_array
//
// EXTERNAL VARIABLE REFERENCES
//   IN :  bbs_setting_array
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
bbs_settings::bbs_settings()
{
        save_flag = false;
        trie.init( bbs_setting_array );
}








