/*
 * Copyright (C) 1995  Peter L Jones
 * See file COPYING before you even think about using this program.
 */
static char version[] = "$Id: useradd.c,v 1.6 1995/12/19 21:55:27 thanatos Exp $";
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <errno.h>
#include <unistd.h>
#include <getopt.h>
#include <dirent.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/*
 * I have not written supplementary groups code yet
 * - please update /etc/group by hand :-(
 * (options are accepted and ignored)
 */
#define NO_SUPPGROUPS

#define DEFAULT_SKELDIR  "/etc/skel"
#define DEFAULT_SHELL "/bin/sh"
#define DEFAULT_GID (100)

#define PASSWD_FILE "/etc/passwd"
#define GROUP_FILE "/etc/group"
const char HOME_PARENT[] = "/home";
const long MIN_USER_ID = 100;
const short HOME_DIR_MODE = 0755;
const char SHADOW_SYSV[] = "/etc/shadow";
const char SHADOW_BSD[] = "/etc/master.passwd";	/* I got this name from "adduser" */

/*
 * Private local functions
 */
static void Unlock(void);
static int Shadow(void);
static int Parse(int argc, char *argv[]);
static int Validate(void);
static int Create(void);

static void Usage(void);
static void Long_usage(void);
static int MakeHome(void);
static int LockPW(void);
static int BackupPW(void);
static int AddUser(void);
#ifndef NO_SUPPGROUPS
static int LockGrp(void);
static int BackupGrp(void);
static int AddGroups(void);
#endif

static struct mainOptions {
	int	mkhome;
	int	noskel;
	int	multi;
	uid_t	num_uid;
	gid_t	num_gid;
	const char*	shell;
	const char*	skeldir;
	const char*	group;
	const char*	comment;
	const char*	homedir;
	const char*	suppgroups;
	const char*	uid;
	const char*	username;
	const char*	pname;
} Opt = { 0, 0, 0, 0, DEFAULT_GID, DEFAULT_SHELL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, };


int main(int argc, char *argv[]) {
	Opt.pname = argv[0];
	if (atexit(Unlock)) {
		perror(argv[0]);
		return 1;
	}
	if (Shadow()) return 2;
	if (Parse(argc, argv)) return 3;
	if (Validate()) return 4;
	if (Create()) return 5;
	return 0;
}


static int Create(void) {
	if (LockPW()) return 1;
	if (BackupPW()) return 1;
	if (AddUser()) return 1;
	if (Opt.mkhome && MakeHome()) return 1;
#ifndef NO_SUPPGROUPS
	if (Opt.suppgroups) {
		if (LockGrp()) return 1;
		if (BackupGrp()) return 1;
		if (AddGroups()) return 1;
	}
#endif
	return 0;
}

static int AddUser(void) {
	FILE *PASSWD = fopen(PASSWD_FILE, "a");
	static struct passwd pw;
	
	if (!PASSWD) {
		fprintf(stderr,"%s: could not append to %s: %s\n", Opt.pname, PASSWD_FILE, strerror(errno));
		return 1;
	}
	pw.pw_name = strdup(Opt.username);
	pw.pw_passwd = strdup("!");
	pw.pw_uid = Opt.num_uid;
	pw.pw_gid = Opt.num_gid;
	pw.pw_gecos = strdup(Opt.comment);
	pw.pw_dir = strdup(Opt.homedir);
	pw.pw_shell = strdup(Opt.shell);
	putpwent(&pw, PASSWD);			/* if this doesn't update, why isn't it const? */
	fclose(PASSWD);
	free(pw.pw_name);
	free(pw.pw_passwd);
	free(pw.pw_gecos);
	free(pw.pw_dir);
	free(pw.pw_shell);
	return 0;
}

static int MakeHome(void) {
	char buffer[1024];
	if (mkdir(Opt.homedir,HOME_DIR_MODE) && (errno != EEXIST)) {
		fprintf(stderr,"%s: could not create home directory '%s': %s\n", Opt.pname, Opt.homedir, strerror(errno));
		return 1;
	}
	if (!Opt.noskel) {
		sprintf(buffer, "/bin/cp -rp %s/. %s", Opt.skeldir, Opt.homedir);
		if (system(buffer)) return 1;
	}
	sprintf(buffer, "/bin/chown -R %d.%d %s", Opt.num_uid, Opt.num_gid, Opt.homedir);
	if (system(buffer)) return 1;
	return 0;
}

#ifndef NO_SUPPGROUPS
static int AddGroups(void) {
	return 0;
}

static int BackupGrp(void) {
	if (Opt.suppgroups && system("/bin/cp " GROUP_FILE " " GROUP_FILE ".OLD")) return 1;
	chmod(GROUP_FILE ".OLD", 0);	/* don't care if it fails */
	return 0;
}

static int LockGrp(void) {
	static int fd;
	if ((fd = open(GROUP_FILE "~",O_CREAT | O_EXCL, 0)) < 0) {
		fprintf(stderr,"%s: cannot create group file lock: %s\n", Opt.pname, strerror(errno));
		remove(PASSWD_FILE "~");
		return 1;
	} else close(fd);
	return 0;
}
#endif

static int Validate() {
	static int i;
	static struct passwd *pw;
	static char *p;
	static long id;

	/* ensure username is valid */
	if ( (i = strlen(Opt.username)) == 0) {
		fprintf(stderr,"%s: null username not allowed.\n",Opt.pname);
		return 1;
	}
	if (i > 8) {
		fprintf(stderr,"%s: username '%s' is too long (more than 8 characters).\n",Opt.pname,Opt.username);
		return 1;
	}
	if (Opt.username[0] == '-') {	/* can't start with - (for login) */
		fprintf(stderr,"%s: username ('%s') must not start with a dash.\n",Opt.pname,Opt.username);
		return 1;
	}
	if (strpbrk(Opt.username," /'\"\\")) {	/* can't contain space, dir. sep., quoting chars */
		fprintf(stderr,"%s: invalid characters in username '%s'.\n",Opt.pname,Opt.username);
		return 1;
	}

	if (!Opt.homedir) {
		/* default it now we know username is okayish... */
		static char buffer[9 + sizeof("/home/")];
		sprintf(buffer,"/home/%s",Opt.username);
		Opt.homedir = buffer;
	}

	if (Opt.mkhome) {
		static char buffer[FILENAME_MAX];
		static DIR *d;
		if (Opt.skeldir) {
			/* ensure skels dir exists, as it's been given */
			sprintf(buffer,"%s/.",Opt.skeldir);
			if ((d = opendir(buffer)) == NULL) {
				fprintf(stderr,"%s: skeleton directory %s: %s\n", Opt.pname, Opt.homedir, strerror(errno));
				return 1;
			} else closedir(d);
		} else if (!Opt.noskel) {
			/* don't moan if there's no skeleton */
			Opt.skeldir = DEFAULT_SKELDIR;
			sprintf(buffer,"%s/.",Opt.skeldir);
			if ((d = opendir(buffer)) == NULL) Opt.noskel++;
			else closedir(d);
		}
	}

	if (!Opt.comment) Opt.comment = "";

	if (Opt.uid) {
		id = strtol(Opt.uid,&p,0);

		if (p && (*p == '\0')) {
			/* numeric */
			Opt.num_uid = id;
		} else {
			/* non-numeric */
			if (!getpwnam(Opt.uid)) {
				fprintf(stderr,"%s: '%s' is not an existing user name.\n", Opt.pname, Opt.uid);
				return 1;
			}
			Opt.num_uid = pw->pw_uid;
		}
	}
	
	if (Opt.group) {
		id = strtol(Opt.group,&p,0);

		if (p && (*p == '\0')) {
			/* numeric */
			if (!getgrgid(id)) {
				fprintf(stderr,"%s: '%s' is not an existing group number.\n", Opt.pname, Opt.group);
				return 1;
			}
			Opt.num_gid = id;
		} else {
			/* non-numeric */
			static struct group *g;
			if (!(g = getgrnam(Opt.group))) {
				fprintf(stderr,"%s: '%s' is not an existing group name.\n", Opt.pname, Opt.group);
				return 1;
			}
			Opt.num_gid = g->gr_gid;
		}
	}

#ifndef NO_SUPPGROUPS
	if (Opt.suppgroups) {
		/* this should be a comma-separated list of existing groups */
		/* It's probably a good idea to split these out into a more useable format at this stage */
	}
#endif

	if (Opt.uid) {
		if (!Opt.multi && getpwuid(Opt.num_uid)) {
			fprintf(stderr,"%s: '%d' (%s) is not a unique user number.\n", Opt.pname, Opt.num_uid, Opt.uid);
			return 1;
		}
	} else {
		for (Opt.num_uid = MIN_USER_ID; getpwuid(Opt.num_uid); Opt.num_uid++);
		if (getpwuid(Opt.num_uid)) {
			fprintf(stderr,"%s: cannot get a unique user number.\n", Opt.pname);
			return 1;
		}
	}

	if (getpwnam(Opt.username)) {
		fprintf(stderr,"%s: '%s' is not a unique user name.\n", Opt.pname, Opt.username);
		return 1;
	}
	
	return 0;
}

static int Parse(int argc, char *argv[]) {
	int c;
	char short_options[] = "hVmKok:d:s:c:g:G:u:";
	struct option long_options[] = {
		{"help",		no_argument,       0, 'h'},
		{"version",		no_argument,       0, 'V'},
		{"make-homedir",	no_argument,       0, 'm'},
		{"no-skeleton",		no_argument,       0, 'K'},
		{"nonunique",		no_argument,       0, 'o'},
		{"skeldir",		required_argument, 0, 'k'},
		{"directory",		required_argument, 0, 'd'},
		{"shell",		required_argument, 0, 's'},
		{"comment",		required_argument, 0, 'c'},
		{"group",		required_argument, 0, 'g'},
		{"suppgroups",		required_argument, 0, 'G'},
		{"uid",			required_argument, 0, 'u'},
		{0, 0, 0, 0}
	};

	while ((c = getopt_long(argc, argv, short_options,
			long_options, NULL)) != -1) switch (c) {
		case 'm':	Opt.mkhome++; break;
		case 'K':	Opt.noskel++; break;
		case 'o':	Opt.multi++; break;
		case 'k':	Opt.skeldir=optarg; break;
		case 'd':	Opt.homedir=optarg; break;
		case 's':	Opt.shell=optarg; break;
		case 'c':	Opt.comment=optarg; break;
		case 'g':	Opt.group=optarg; break;
		case 'G':	Opt.suppgroups=optarg; break;
		case 'u':	Opt.uid=optarg; break;

		case 'V':
			printf("%s\n",version);
			exit(0);

		case 'h':
			Long_usage();
			exit(0);

		default:
			Usage();
			return 1;
	}

	if (optind != argc-1) {
		fprintf(stderr,"%s: user name not specified\n", Opt.pname);
		Usage();
		return 1;
	}

	Opt.username=argv[optind];

	return 0;
}

static void Usage(void) {
	fprintf(stderr,"Try `%s --help' for more information.\n",Opt.pname);
}

static void Long_usage(void) {
	fprintf(stderr,"Usage: %s [option ...] username\n", Opt.pname);
	fprintf(stderr,
		"where 'option' is:\n"
		"  -h,          --help\n"
		"  -V,          --version\n"
		"  -m,          --make-homedir\n"
		"  -K,          --no-skeleton\n"
		"  -o,          --nonunique\n"
		"  -k skeldir,  --skeldir=skeldir    (default: /etc/skel)\n"
		"  -s progname, --shell=progname     (default: /bin/sh)\n"
		"  -g gid,      --group=gid          (default: 100)\n"
		"  -d homedir,  --directory=homedir  (default: /home/username)\n"
		"  -c realname, --comment=realname\n"
#ifndef NO_SUPPGROUPS
		"  -G list,     --suppgroups=list\n"
#endif
		"  -u id,       --uid=id\n"
	);
}

static int Shadow(void) {
	/* if either /etc/shadow or /etc/master.passwd exists, this is shadow system - we don't work! */
	if (!access(SHADOW_SYSV,F_OK) || !access(SHADOW_BSD,F_OK)) {
		fprintf(stderr,"%s: Detected a SHADOW PASSWORD file on this system.  "
			"If you are not using shadow passwords, please remove the shadow file.\n", Opt.pname);
		return 1;
	}
	return 0;
}

static int BackupPW(void) {
	if (system("/bin/cp " PASSWD_FILE " " PASSWD_FILE ".OLD")) return 1;
	chmod(PASSWD_FILE ".OLD", 0);
	return 0;
}

static int LockPW(void) {
	static int fd;
	if ((fd = open(PASSWD_FILE "~",O_CREAT | O_EXCL, 0)) < 0) {
		fprintf(stderr,"%s: cannot create password lock: %s\n", Opt.pname, strerror(errno));
		return 1;
	} else close(fd);
	return 0;
}

static void Unlock(void) {
	remove(PASSWD_FILE "~");
	remove(GROUP_FILE "~");
}
