#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>

#include <daydream.h>
#include <ddcommon.h>

#ifdef HAVE_STROPTS_H
#include <stropts.h>
#endif

static int childon;
static int readpty;

static void childhan(int);
static void readhan(int);
static int getpty(char *);
static pid_t kid;

void stdioout(const char *s)
{
	char fname[15] = "";
	int fd;

	strlcpy(fname, "/tmp/dd.XXXXXX", sizeof(fname));
	if ((fd = mkstemp(fname)) == -1) {
		fprintf(stderr, "%s: %s\n", fname, strerror(errno));
		return;
	}
	if (runstdio(s, fd, 3)) 
		TypeFile(fname, TYPE_NOCODES);
	close(fd);
	unlink(fname);
}

/* Splits the command line at whitespaces for execv(). */
static void chop_args(char *cmd, char **args)
{
	int quoteon, i = 1;
	char *s;
	
	s = cmd;
	args[0] = s;
	quoteon = 0;

	while (*s) {
		if (quoteon) {
			if (*s == '\"') {
				quoteon = 0;
				*s = 0;

			}
			s++;
		} else if (*s == '\"') {
			args[i - 1] = s + 1;
			s++;
			quoteon = 1;
		} else if (*s == ' ') {
			*s = 0;
			s++;
			args[i] = s;
			i++;
		} else
			s++;
	}
	args[i] = 0;
}

#define max(_a, _b) ((_a) > (_b) ? (_a) : (_b))

#define PTY_MODE_STDIO	1
#define PTY_MODE_PIPE	2
#define PTY_MODE_SMODEM 4

static void master_loop(int pipefd[2], int ptyfd, int mode, int outfd)
{
	time_t huptime = 0;
	int bytes_read;
	
	if (mode == PTY_MODE_PIPE)
		close(pipefd[1]);

	signal(SIGCHLD, childhan);
	while (!readpty)
		usleep(4000);

	for (;;) {
		char stbuf[1024];
		unsigned char pala;
		fd_set rset;
		struct timeval tv;
		int highfd = 0, i;

		/* If 'huptime' is not zero, the child has been signaled
		 * with SIGHUP and we're just counting seconds.
		 */
		if (huptime != 0) {
			if (mode != PTY_MODE_SMODEM)
				break;

			if (waitpid(kid, NULL, WNOHANG) > 0)
				break;
			
			if (huptime < time(NULL) - 20) {
				kill(kid, SIGKILL);
				/* there's wait() after the loop */
				break;
			}
		}
			
		FD_ZERO(&rset);
		/* Serial line should not be read if it is handled by
		 * programs like ZModem (PTY_MODE_PIPE) or SModem.
		 */
		if (!lmode && mode == 1) {
			FD_SET(serhandle, &rset);
			highfd = max(highfd, serhandle);
		}

		if (console_active()) 
			highfd = console_select_input(highfd, &rset);

		if (mode == PTY_MODE_PIPE) {
			FD_SET(pipefd[0], &rset);
			highfd = max(highfd, pipefd[0]);
		} else {
			FD_SET(ptyfd, &rset);
			highfd = max(highfd, ptyfd);
		}
			
		tv.tv_sec = 0;
		tv.tv_usec = 50e3;

		if (select(highfd + 1, &rset, NULL, NULL, &tv) == -1) {
			if (errno == EINTR) {
				if (!childon)
					break;
				continue;
			}

			huptime = time(NULL);
			kill(kid, SIGHUP);
			continue;
		}

		if (mode == PTY_MODE_PIPE) {
			if (!checkcarrier())
				kill(kid, SIGKILL);
		} else if (mode == PTY_MODE_SMODEM) {
			if (!huptime) {
				kill(kid, SIGHUP);
				huptime = time(NULL);
			} else {
				if (huptime < (time(NULL) - 20)) {
					kill(kid, SIGKILL);
					break;
				}
			}
		} else if (mode == PTY_MODE_STDIO && !checkcarrier())
			break;
		
		
		if (mode != PTY_MODE_PIPE && mode != PTY_MODE_SMODEM &&
			       	FD_ISSET(serhandle, &rset) && !lmode) {
			read(serhandle, &pala, 1);
			if (mode != 3)
				safe_write(ptyfd, &pala, 1);
			keysrc = 1;
		}
		
		if (mode == PTY_MODE_PIPE && FD_ISSET(pipefd[0], &rset)) {
			bytes_read = read(pipefd[0], stbuf, sizeof(stbuf));
			if (console_active()) {
				for (i = 0; i < bytes_read; i++) {
					if (stbuf[i] == 10)
						console_putc('\r');
					console_putc(stbuf[i]);
				}
			}
		}

		if (console_active() && console_pending_input(&rset)) {
			pala = console_getc();
			if (mode != 2 && mode != 3)
				safe_write(ptyfd, &pala, 1);
			keysrc = 2;
			if (pala == 2) {
				handlectrl(0);
			}
		}

		if (mode != 2 && FD_ISSET(ptyfd, &rset)) {
			int r = read(ptyfd, &stbuf, 1023);
			if (r < 0)
				break;
			stbuf[r] = 0;
			if (*stbuf) {
				if (mode != 2) {
					if (mode == 4) {
						int t;
						int ans;
						ans = ansi;
						ansi = 1;
						t = userinput;
						userinput = 0;
						DDPut(stbuf);
						userinput = t;
						ansi = ans;
					} else if (mode != 3)
							DDPut(stbuf);
				} else {
					if (*stbuf == 10) {
						safe_write(ptyfd, "\r", 1);
					}
					safe_write(ptyfd, &stbuf, strlen(stbuf));
				}
				if (outfd != -1 && mode != 3) {
					safe_write(outfd, &stbuf, strlen(stbuf));
				}
			}
		}
		
		if (!childon) 
			break;
	}
	
	waitpid(-1, 0, WNOHANG);
}

int runstdio(const char *command, int outfd, int inp)
{
	int ptyfd = 0;
	int pipefd[2];

	char dcmd[400];
	char *args[100];
	char ptyname[20];

	genstdiocmdline(dcmd, command, NULL, NULL);

	chop_args(dcmd, args);
	
	if (inp != 2)
		if ((ptyfd = getpty(ptyname)) < 0) {
			DDPut("*AAAARGH* fatal error: Tell Sysop to make more accessable pty's!\n");
			return 0;
		}
	childon = 1;
	readpty = 0;

	if (inp == 2) {
		pipe(pipefd);
	} else {

		struct winsize ws;

		ws.ws_col = 80;
		if (inp == 4) {
			ws.ws_row = maincfg.CFG_LOCALSCREEN;
		} else {
			ws.ws_row = user.user_screenlength;
		}
		ws.ws_xpixel = ws.ws_ypixel = 0;
		ioctl(ptyfd, TIOCSWINSZ, &ws);

	}
	signal(SIGUSR2, readhan);
	kid = fork();

	switch (kid) {
	case 0:
		if (inp != 2) 
			close(ptyfd);

		if (inp != 2 && setsid() == -1) {
			log_error(__FILE__, __LINE__, 
				"setsid() failed: %s (%d)",
				strerror(errno), errno);
		}
		

		if (inp == 2) {
			kill(getppid(), SIGUSR2);
			fcntl(serhandle, F_SETFD, 0);
			dup2(pipefd[1], 2);
			setbuf(stderr, 0);
			close(pipefd[0]);
		} else if (inp == 3 && outfd != -1) {
			if ((ptyfd = open(ptyname, O_RDWR | O_NONBLOCK)) < 0)
				_exit(1);
			set_blocking_mode(ptyfd, 0);
#ifdef HAVE_STROPTS_H
			if (isastream(ptyfd)) 
				if (ioctl(ptyfd, I_PUSH, "ptem") < 0 ||
				    ioctl(ptyfd, I_PUSH, "ldterm") < 0) {
					close(ptyfd);
					_exit(1);
				}
#endif
			kill(getppid(), SIGUSR2);
			dup2(ptyfd, STDIN_FILENO);
			dup2(outfd, STDOUT_FILENO);
			dup2(outfd, STDERR_FILENO);
		} else {
			if ((ptyfd = open(ptyname, O_RDWR | O_NONBLOCK)) < 0)
				_exit(1);
			set_blocking_mode(ptyfd, 0);
#ifdef HAVE_STROPTS_H			
			if (isastream(ptyfd)) 
				if (ioctl(ptyfd, I_PUSH, "ptem") < 0 ||
				    ioctl(ptyfd, I_PUSH, "ldterm") < 0) {
					close(ptyfd);
					_exit(1);
				}
#endif
			kill(getppid(), SIGUSR2);
			dup2(ptyfd, STDIN_FILENO);
			dup2(ptyfd, STDOUT_FILENO);
			dup2(ptyfd, STDERR_FILENO);
		}

#ifdef HAVE_TIOCSCTTY
		if (inp != 2 && ioctl(ptyfd, TIOCSCTTY, 0) == -1) {
			log_error(__FILE__, __LINE__, 
				"ioctl(..., TIOCSCTTY, ...) failed: %s (%d)",
				strerror(errno), errno);
		}
#endif

		execv(args[0], &args[0]);		
		syslog(LOG_WARNING, "execv(\"%s\"...) failed: %m", args[0]);
		kill(getpid(), SIGKILL);
		_exit(0);
		break;
	case -1:
		syslog(LOG_ERR, "fork() failed: %m");
		DDPut("fork() failed\n");
		break;
	default:
		master_loop(pipefd, ptyfd, inp, outfd);
		break;
	}
	if (inp != 2) {
		close(ptyfd);
	} else {
		close(pipefd[0]);
	}
	return 1;

}

static void childhan(int signum)
{
	while ((waitpid(-1, NULL, WNOHANG)) > 0);
	if (!ispid(kid))
		childon = 0;
}

static void readhan(int signum)
{
	readpty = 1;
}

#ifdef HAVE_PTY_UTILS
static int getpty_unix98(char *name)
{
	int pty;
	char *ttyn;
	
	pty = getpt();
	if (pty < 0)
		return -1;
	
	if (grantpt(pty) < 0 || unlockpt(pty) < 0)
		goto close_pty;

	ttyn = ptsname(pty);
	if (!ttyn)
		goto close_pty;
	
	strcpy(name, ttyn);
	return pty;
	
close_pty:
	close(pty);
	return -1;
}
#endif 
static int getpty(char *name)
{
	register int pty, tty;
	char *pty_dev = "/dev/ptc", *tt;

#ifdef HAVE_PTY_UTILS
	/* first try to use Unix98 allocation */
	 if ((pty = getpty_unix98(name)) != -1) {
		set_blocking_mode(pty, 0);
		return pty;
	}
#endif 
	/* look for a SYSV-type pseudo device */
	if ((pty = open(pty_dev, O_RDWR | O_NONBLOCK)) >= 0) {
		if ((tt = ttyname(pty)) != NULL) {
			strcpy(name, tt);
			set_blocking_mode(pty, 0);
			return pty;
		}
		close(pty);
	}

	/* scan Berkeley-style */
	strcpy(name, "/dev/ptyp0");
	while (access(name, 0) == 0) {
		if ((pty = open(name, O_RDWR | O_NONBLOCK)) >= 0) {
			name[5] = 't';
			if ((tty = open(name, O_RDWR | O_NONBLOCK)) >= 0) {
				close(tty);
				set_blocking_mode(pty, 0);
				return pty;
			}
			name[5] = 'p';
			close(pty);
		}

		/* get next pty name */
		if (name[9] == 'f') {
			name[8]++;
			name[9] = '0';
		} else if (name[9] == '9')
			name[9] = 'a';
		else
			name[9]++;
	}
	return -1;
}

