#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <time.h>
#include <unistd.h>
#include "bb.h"
#include "editor.h"
#include "mail.h"
#include "output.h"
#include "varlist.h"

#define NNTP_PORT 119

extern varlist list;
extern struct line_t *first, *current;
extern int ansi;
extern char yes_key, no_key, quit_key, cont_key;

void nntp_stop();
int read_news();

int nntp_socket, posting_ok, nntp_connected;
static char *nntp_buf;

struct article_t {
  char *id, *from, *subject, *date;
  struct article_t *next;
} *first_article = NULL, *last_article = NULL;

void add_article()
{
  if(first_article == NULL) {
    first_article = (struct article_t *)malloc(sizeof(struct article_t));
    last_article = first_article;
  }
  else {
    last_article -> next = (struct article_t *)malloc(sizeof(struct article_t));
    last_article = last_article -> next;
  }
  
  last_article -> next = NULL;
  last_article -> id = NULL;
  last_article -> from = NULL;
  last_article -> subject = NULL;
  last_article -> date = NULL;
}

void kill_articles()
{
  struct article_t *tmp = first_article, *next;
  
  while(tmp) {
    next = tmp -> next;
    free(tmp -> id);
    if(tmp -> from)
      free(tmp -> from);
    if(tmp -> subject)
      free(tmp -> subject);
    if(tmp -> date)
      free(tmp -> date);
    free(tmp);
    tmp = next;
  }

  first_article = last_article = NULL;
}

int nntp_getline()
{
  int t;
  char c;
  
  for(t = 0;t < 1024;t++) {
    if(!read(nntp_socket, &c, 1))
      break;
    nntp_buf[t] = c;
    if(c == '\n')
      break;
  }
  
  nntp_buf[t] = 0;
  
  if(nntp_buf[t - 1] == '\r')
    nntp_buf[t - 1] = 0;
  
  return t;
}

int nntp_reply()
{
  char reply[4];
    
  if(!nntp_getline())
    return 0;
  
  dprintf(">NNTP reply: %s\n", nntp_buf);
  
  memcpy(reply, nntp_buf, 3);
  reply[3] = 0;
  
  return atoi(reply);
}

int nntp_command(char *s)
{
  dprintf(">NNTP command: %s\n", s, strlen(s));
  write(nntp_socket, s, strlen(s));
  return nntp_reply(); 
}

int nntp_connect()
{
  char *hostname;
  struct hostent *host;
  struct sockaddr_in sname;
  int tmp;

  hostname = string("nntpserver"); 
  host = gethostbyname(hostname);

  if(host == NULL) {
    dprintf(">nntp_connect: couldn't resolve \"%s\"\n", hostname);
    return 1;
  }
   
  nntp_socket = socket(PF_INET, SOCK_STREAM, 0);
  sname.sin_family = AF_INET;
  sname.sin_port = htons(NNTP_PORT);
  memcpy(&sname.sin_addr, host -> h_addr_list[0], sizeof(struct in_addr));
  
  if(connect(nntp_socket, (struct sockaddr *)&sname, 
     sizeof(struct sockaddr_in)) == -1) {
    dprintf(">nntp_connect: connect error\n");
    close(nntp_socket);
    return 1;
  }

  tmp = nntp_reply();

  // My INN needs this, probably because the config is screwed up ;-)
  // We should not get kicked off for sending an illegal command, so it
  // probably won't hurt.
  nntp_command("MODE READER\r\n");  

  if(tmp == 200) 
    posting_ok = 1;
  else if(tmp == 201)
    posting_ok = 0;
  else {
    close(nntp_socket);
    return 1;
  }

  return 0; 
}

int nntp_start()
{
  if(nntp_connected)
    return 0;

  nntp_buf = (char *)malloc(1024);

  if(nntp_connect()) {
    free(nntp_buf);
    return 1;
  }

  nntp_connected = 1;
  return 0;
}

void nntp_stop()
{
  nntp_command("QUIT\r\n");
  free(nntp_buf);
  nntp_connected = 0;
  close(nntp_socket);
}

int change_group(char *group)
{
  char buf[256];
  sprintf(buf, "GROUP %s\r\n", group);
  return nntp_command(buf);
}

int read_new_news()
{
  char date_string[20], cmd[256], *group;
  struct tm *date;
  time_t last;
  int tmp;
  
  if(exists("newsdate"))
    last = number("newsdate");
  else
    last = (time_t)0; /* Jan 1, 1970 00:00:00 */
  
  date = localtime(&last);
  
  if(exists("newsgroup"))
    group = string("newsgroup");
  else
    group = "test";
  
  if(change_group(group) == 411)
    return 2;
  
  strftime(date_string, 20, "%y%m%d %H%M%S\r\n", date);
  sprintf(cmd, "NEWNEWS %s %s", group, date_string);
  tmp = nntp_command(cmd);
  
  if(tmp != 230)
    return 1;

  kill_articles();

  for(;;) {
    nntp_getline();
    if(*nntp_buf == '.')
      break;
    add_article();
    last_article -> id = (char *)malloc(strlen(nntp_buf) + 1);
    strcpy(last_article -> id, nntp_buf);
  }
  
  return read_news();
}

int get_headers()
{
  struct article_t *tmp = first_article;
  char cmd[256];
  
  while(tmp) {
    sprintf(cmd, "HEAD %s\r\n", tmp -> id);
    if(nntp_command(cmd) != 221)
      return 1;
      
    for(;;) {
      if(!nntp_getline())
        break;
        
      if(!strcmp(nntp_buf, "."))
        break;
      else if(!strncmp(nntp_buf, "From: ", 6)) {
        tmp -> from = (char *)malloc(strlen(nntp_buf) + 1);
        strcpy(tmp -> from, nntp_buf);
      }
      else if(!strncmp(nntp_buf, "Subject: ", 9)) {
        tmp -> subject = (char *)malloc(strlen(nntp_buf) + 1);
        strcpy(tmp -> subject, nntp_buf);
      }
      else if(!strncmp(nntp_buf, "Date: ", 6)) {
        tmp -> date = (char *)malloc(strlen(nntp_buf) + 1);
        strcpy(tmp -> date, nntp_buf);
      }
    }    

    tmp = tmp -> next;
  }
  
  return 0;
}


void nntp_writeline(struct line_t *line)
{
  if(line -> length)
    write(nntp_socket, line -> text, line -> length);
  write(nntp_socket, "\r\n", 2);
}

int post_article()
{
  char tmp[80], subject[80], group[80], buf[160];
  int width = number("rows"), reply;
  struct line_t *line;
  
  if(!posting_ok)
    return 2;
  
  width >>= 1;
  width -= 10;
  
  strncpy(subject, string("subject"), width);
  strncpy(group, string("newsgroup"), width);
  
  subject[width] = 0;
  group[width] = 0;
  
  sprintf(tmp, "$white$Newsgroup: $yellow$%s $white$($yellow$%s$white$)",
    group, subject); 

  list.add_sys("editstatus", tmp);
  load_signature(1);
  edit();
  line = first;
  
  reply = nntp_command("POST\r\n");
  if(reply != 340) {
    edit_cleanup();
    return 1;
  }

  if(strchr(string("email"), '@'))
    sprintf(buf, "From: %s <%s>\r\n", string("user"), string("email"));
  else
    sprintf(buf, "From: %s <%s@%s>\r\n", string("user"), string("email"),
            string("bbsdomain"));
  write(nntp_socket, buf, strlen(buf));

  sprintf(buf, "Newsgroups: %s\r\n", string("newsgroup"));
  write(nntp_socket, buf, strlen(buf));

  sprintf(buf, "Subject: %s\r\n", string("subject"));
  write(nntp_socket, buf, strlen(buf));

  if(exists("organization")) {
    sprintf(buf, "Organization: %s\r\n", string("organization"));
    write(nntp_socket, buf, strlen(buf));
  }

  sprintf(buf, "Message-Id: <BBBBS.%d.%s@%s>\r\n", number("today"), 
          initials("user"), string("bbsdomain"));
  write(nntp_socket, buf, strlen(buf));

  write(nntp_socket, "\r\n", 2);
  
  while(line) {
    if(*(line -> text) == '.')
      write(nntp_socket, ".", 1);
    nntp_writeline(line);
    line = line -> next;
  }   
  
  edit_cleanup();
  
  reply = nntp_command("\r\n.\r\n");
  if(reply == 240)
    return 0;
  if(reply == 441)
    return 3; 

  return 1;
}

int print_header(struct article_t *a)
{
  output_cooked("\\c");
  output_cooked(string("newshead"));
  
  //output_raw(a -> id);
  //output_cooked("\n");

  output_raw(a -> date);
  output_cooked("\n");

  output_raw(a -> from);
  output_cooked("\n");

  output_raw(a -> subject);
  output_cooked("\n");
  
  output_cooked("\n");
  
  return 4;
}

void nntp_flush()
{
  while(nntp_getline() && strcmp(nntp_buf, "."));
}

int reply(struct article_t *a, int inc)
{
  char buf[256];
    
  sprintf(buf, "BODY %s\r\n", a -> id);
  if(nntp_command(buf) != 222)
    return 1;

  current = first = NULL;
    
  if(inc)
  for(;;) {
    if(!nntp_getline())
      break;
    
    if(!strcmp(nntp_buf, "."))
      break;
      
    insert_line();
    if(current -> next)
      current = current -> next;
      
    sprintf(current -> text, ">%s", nntp_buf);
    current -> length = strlen(nntp_buf) + 1;
  }
  
  if(strncmp(a -> subject + 9, "Re: ", 4))
    sprintf(buf, "Re: %s", a -> subject + 9);
  else
    strcpy(buf, a -> subject + 9);
  
  list.add_sys("subject", buf);
  return post_article(); 
}

int read_news()
{
  struct article_t *article;
  int lines, width, y, end, key, forever, quit = 0;
  char cmd[256];
  
  if(get_headers()) {
    nntp_stop();
    return 1;
  }

  article = first_article;
  if(article == NULL)
    return 4;
 
  width = number("rows");
  lines = number("lines");

  while(!quit) {
   
  sprintf(cmd, "BODY %s\r\n", article -> id);
  if(nntp_command(cmd) != 222) {
    nntp_stop();
    return 1;
  }

  end = 0;
  forever = 0;
  y = 1;
  
  while(!end) {
    y += print_header(article);
    output_cooked(string("newsbody"));
    
    while(y < lines) {
      if(!nntp_getline()) {
        nntp_stop();
        return 1;
      }
      
      if(!strcmp(nntp_buf, ".")) {
        end = 1;
        break;
      }
      
      nntp_buf[width] = 0;
      output_raw(nntp_buf);
      output_cooked("\n");
      if(!forever)
        y++;
    }
    
    if(!end)  
      output_cooked(string(posting_ok ? "newsprompt1r" : "newsprompt1"));
    else
      output_cooked(string(posting_ok ? "newsprompt2r" : "newsprompt2"));
    
    key = read_char();
    
    if(!end && key == 'c')
      forever = 1;
    else if(key == 'n')
      end += 2;
    else if(key == 'q')
      quit = 1;
    else if(posting_ok && key == 'r') {
       if(!end)
         nntp_flush();

      if(!end)
        kill_line(strlen(string(posting_ok ? "newsprompt1r" : "newsprompt1")));
      else
        kill_line(strlen(string(posting_ok ? "newsprompt2r" : "newsprompt2")));
      output_cooked(string("includemsg"));
      if(yesno())
        reply(article, 1);
      else
        reply(article, 0);
      end = 1;  
    }

    y = 1;
  }
  
  if(end == 2) { // >-]
    while(nntp_getline())
    if(!strcmp(nntp_buf, "."))
      break;
  }
  
  article = article -> next;
  if(!article)
    quit = 1;

  }
  
  kill_articles();
  
  return 0;
}
