#include <fcntl.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <termios.h>
#include <unistd.h>
#include "areas.h"
#include "bb.h"
#include "cmdline.h"
#include "doors.h"
#include "files.h"
#include "inc.h"
#include "lock.h"
#include "mail.h"
#include "menus.h"
#include "modem.h"
#include "news.h"
#include "output.h"
#include "status.h"
#include "varlist.h"
#include "users.h"

#define VERSION_STRING "0.73 Beta"

#define EXIT_NORMAL    0
#define EXIT_INIT      1
#define EXIT_FATAL     2
#define EXIT_TERM      3
#define EXIT_TIMEOUT   4
#define EXIT_EXPIRED   5
#define EXIT_HUNGUP    6
#define EXIT_FAULT     7
#define EXIT_BADLOGIN  8
#define EXIT_NODES     9
#define EXIT_CONNECT  10

int num_options = OPTIONS;

struct option_t options[OPTIONS] = {
  { "-node",     OPTION_INT },
  { "-modem",    OPTION_STRING },
  { "-remote",   OPTION },
  { "-debug",    OPTION },
  { "-spy",      OPTION_STRING },
  { "-user",     OPTION_STRING },
  { "-nodetect", OPTION },
  { "-nogetty",  OPTION }
};

struct value_t values[OPTIONS];

extern struct status_t *status_p;
extern int node, timeout, newuser, nntp_connected, login_time;
extern void chat(int);
extern int login();
extern void log_args(char *, va_list);
extern void init_protocols();
extern void init_archivers();
extern void init_getty();

int online = 0, remote = 0, dev_locked = 0, in_dir = 0, debug;
varlist list;
char dev_lock[100], *home_dir = NULL;

vmodem *modem = NULL, *spymodem = NULL;

void dprintf(char *fmt, ...) 
{
  if(!debug)
    return;
   
  va_list args;
  va_start(args, fmt);
  if(debug)
    vfprintf(stderr, fmt, args);
  va_end(args);
}

void unlock_device()
{
  if(!dev_locked)
    return;
  unlink(dev_lock);
  dev_locked = 0;
}

void lock_device()
{
  int lck;
  char *dev = string("device"), *dir;
   
  if(strncmp(dev, "/dev/", 5))
    return;
 
  if(!exists("lockdir"))
    return;
   
  dir = string("lockdir");
   
  strcpy(dev_lock, dir);
  if(dir[strlen(dir) - 1] != '/')
    strcat(dev_lock, "/");
  strcat(dev_lock, "LCK..");
  strcat(dev_lock, dev + 5);
  
  dprintf(">lockfile: %s\n", dev_lock);

  lck = open(dev_lock, O_RDWR | O_CREAT | O_EXCL, 0666);
  if(lck == -1)
    if(errno == EEXIST)
      init_error("%s is locked", dev);
    else
      init_error("could not create lockfile \"%s\"", dev_lock);

  close(lck);
  dev_locked = 1;
  atexit(unlock_device);
}

void logoff()
{
  if(online && status_p -> busy != BUSY_LOGIN) {
    list.add("lastcall", (int)login_time);
    save_user();
  }
}

void cleanup()
{
  if(status_p)
    busy(BUSY_SHUTDOWN);
  if(nntp_connected)
    nntp_stop();
  kill_door();  
  kill_locks();
  unlock_device();
  if(modem)
    delete modem;
  close_log();
  if(status_p)
    killstatus();
}

void fatal_error(char *fmt, ...)
{
  va_list args;
   
  if(online)
    output_cooked("\n\n$errormsg$\n");
 
  if(in_dir) {
    chdir(home_dir);
    in_dir = 0;
  }
  
  list.add("exit", "error");
  logoff();
  
  va_start(args, fmt);
  if(!remote) {
    fprintf(stderr, "bb node %d: ", node);
    vfprintf(stderr, fmt, args);
    fprintf(stderr, "\n");
  }
   
  log("fatal error");
  log_args(fmt, args);
  va_end(args);
  
  cleanup();
  exit(EXIT_FATAL);
}

void init_error(char *fmt ...)
{
  va_list args;
   
  va_start(args, fmt);
  
  if(!remote) {
    fprintf(stderr, "bb: ");
    vfprintf(stderr, fmt, args);
    fprintf(stderr, "\n");
  }
      
  va_end(args);
  cleanup();
  exit(EXIT_INIT);
}

void terminate(int sig)
{
  if(modem && online)
    output_cooked("\n\n$termmsg$\n");

  list.add_sys("exit", "kill");
 
  if(in_dir) {
    chdir(home_dir);
    in_dir = 0;
  }

  logoff();
  if(!remote)
    fprintf(stderr, "bb: terminating node %d (%d)\n", node, sig);
  
  log("terminated by signal %d", sig);
  cleanup();
  exit(EXIT_TERM);
}

void timed_out()
{
  output_cooked("\n\n$timeoutmsg$\n");
  list.add("exit", "timeout");

  if(in_dir) {
    chdir(home_dir);
    in_dir = 0;
  }

  logoff();

  log("user timed out on input");
  cleanup();
  exit(EXIT_TIMEOUT);
}

void expired()
{
  output_cooked("\n\n$expiredmsg$\n");   
  list.add("exit", "expired");

  if(in_dir) {
    chdir(home_dir);
    in_dir = 0;
  }

  logoff();
 
  log("time is up");
  cleanup();
  exit(EXIT_EXPIRED);
}

void hung_up(int sig)
{
  if(!online)
    fatal_error("no carrier");
   
  list.add("exit", "hangup");

  if(in_dir) {
    chdir(home_dir);
    in_dir = 0;
  }

  logoff();
  log("user hung up");
  cleanup();
  exit(EXIT_HUNGUP);
}

void tick(int sig)
{
  static variable *online = NULL;
  int tmp; 

  if(online == NULL)
    online = list.add_sys("online", 0);
   
  if(sig != 42) {
    if(number("clock") && online && status_p -> busy != BUSY_LOGIN) {
      if(status_p -> busy != BUSY_WAITING) {
        online -> i++;
	tmp = list["timeleft"] -> get_i();
        --tmp;
        *list["timeleft"] = tmp;
        if(!tmp)
	  expired();
      }
    }
  }
   
  if(/*remote &&*/!locked && online && status_p -> busy != BUSY_LOGIN)
    save_user(); // Don't ask. (I don't know either.)
   
  alarm(60);
  signal(SIGALRM, tick);
}

void cont(int sig)
{
  modem -> init();
}

void suspend(int sig)
{
  modem -> restore();
  raise(SIGTSTP);
}

void default_vars()
{
  list.add_sys("timeoutmsg", "*** You have been inactive for too long ***");
  list.add_sys("errormsg", "*** A fatal error has occurred ***");
  list.add_sys("termmsg", "*** This node is being terminated ***");
  list.add_sys("expiredmsg", "*** Your time is up ***");
  list.add_sys("chatmsg", "The sysop is now online.");
  list.add_sys("enteryourname", "Please enter your name: ");
  list.add_sys("areyounew", "Are you a new user? (y/N) ");
  list.add_sys("enterpasswd", "Please enter your password: ");
  list.add_sys("numberoflines", "Number of lines (Enter for default 24): ");
  list.add_sys("numberofrows", "Number of rows (Enter for default 80): ");
  list.add_sys("doyouhaveansi", "Does your terminal support ANSI (Y/n)? ");
  list.add_sys("notriesleft", "Sorry, your three tries are up.");
  list.add_sys("badusername", "You are not allowed to use that name.");
  list.add_sys("user", "nobody");
  list.add("passwd", "none");
  list.add("ansi", 0);
  list.add_sys("bbversion", VERSION_STRING);
  list.add("timelimit", 30);
  list.add_sys("exitcode", 0);
  list.add_sys("esc", "\033");
  list.add_sys("quote", "\"");
  list.add_sys("maxnodes", 666);
  list.add_sys("datefmt", "M d y");
  list.add_sys("date", "Nail it to the counter, Lord Ferguson, "
	               "and damn the cheesemongers!");
  list.add_sys("today", 42);
  list.add_sys("filename", "");
  list.add_sys("fileuser", "");
  list.add_sys("filesize", 0);
  list.add_sys("filenum", 0);
  list.add_sys("red", "");
  list.add_sys("bred", "");
  list.add_sys("blue", "");
  list.add_sys("bblue", "");
  list.add_sys("cyan", "");
  list.add_sys("bcyan", "");
  list.add_sys("green", "");
  list.add_sys("bgreen", "");
  list.add_sys("purple", "");
  list.add_sys("bpurple", "");
  list.add_sys("black", "");
  list.add_sys("brown", "");
  list.add_sys("yellow", "");
  list.add_sys("darkgrey", ""); //Bloody Canadians...
  list.add_sys("darkgray", "");
  list.add_sys("grey", "");
  list.add_sys("gray", "");
  list.add_sys("white", "");

  list.add_sys("filefmt", "$white$$3r filenum$ $grey$"
	       "$11r filedate$ $30l filename$ $10r filesize$");

  list.add_sys("markedfmt", "$yellow$$3r filenum$ "
	       "$11r filedate$ $30l filename$ $10r filesize$ $white$(Marked)"
	       "$grey$");

  list.add_sys("fileprompt1", "$white$More? $yellow$[Y]$white$es, $yellow$"
               "[N]$white$o, $yellow$[M]$white$ark "
	       "$yellow$[U]$white$nmark$grey$ ");

  list.add_sys("fileprompt2", "$white$End of list. $yellow$[M]$white$ark "
	       "$yellow$[U]$white$nmark$grey$ ");

  list.add_sys("moreprompt", "$yellow$More?$grey$");
  list.add_sys("yesprompt", "$white$[Y]$yellow$es$grey$");
  list.add_sys("noprompt", "$white$[N]$yellow$o$grey$");
  list.add_sys("quitprompt", "$white$[Q]$yellow$uit$grey$");
  list.add_sys("contprompt", "$white$[C]$yellow$ontinuous$grey$");
  list.add_sys("nextprompt", "$white$[N]$yellow$ext$grey$");
  list.add_sys("deleteprompt", "$white$[D]$yellow$elete$grey$");
  list.add_sys("replyprompt", "$white$[R]$yellow$eply$grey$");

  list.add_sys("markprompt", "$white$Mark:$grey$ ");
  list.add_sys("unmarkprompt", "$white$Unmark:$grey$ ");

//  list.add_sys("nodefmt", "$3r nodenum$ $30l username$ $30l userbusy$");  
  list.add_sys("pressanykey", "$yellow$Press any key to continue.$grey$");
  list.add_sys("smtphost", "localhost");
  list.add_sys("homedir", "");
  list.add_sys("guest", 0);
  list.add_sys("chatreason", "No reason.");
  list.add_sys("spyterm", "none");
  list.add_sys("chatterm", "none");
  list.add("protocol", "none");
  list.add_sys("subject", "None");
  list.add_sys("baudrate", 0);
  list.add_sys("caller", "unknown");
  list.add_sys("mprotocol", "unknown");
  list.add_sys("filesort", "byname");
  list.add_sys("clock", 1);
  list.add_sys("filedate", 0);
  list.add_sys("newshead", "$yellow$");
  list.add_sys("newsbody", "$grey$");
  list.add_sys("includemsg", "$white$Include original message (Y/n)? ");

  list.add_sys("newsprompt1", "$yellow$[M]$white$ore $yellow$[N]$white$ext"
               " $yellow$[C]$white$ontinuous $yellow$[Q]$white$uit ");

  list.add_sys("newsprompt2", "$white$End of article. $yellow$[N]$white$ext "
               "$yellow$[Q]$white$uit "); 
 
  list.add_sys("newsprompt1r", "$yellow$[M]$white$ore $yellow$[N]$white$ext "
               "$yellow$[R]$white$eply $yellow$[C]$white$ontinuous "
               "$yellow$[Q]$white$uit ");
               
  list.add_sys("newsprompt2r", "$white$End of article. $yellow$[N]$white$ext "
               "$yellow$[R]$white$eply $yellow$[Q]$white$uit ");
  list.add_sys("nntpserver", getenv("NNTPSERVER") ? getenv("NNTPSERVER") : 
               "localhost");
               
  list.add_sys("yesterday", 0);
  list.add_sys("dayago", 0);
  list.add("lastcall", 0);
  list.add_sys("lastmenu", "menus/login");
}

void segfault(int sig)
{
  list.add("exit", "error");
   
  if(online)
    output_cooked("\nDON'T PANIC!\n");
  
  killstatus();
  log("AAAAAAAAAAAARGH!");
  cleanup();
  exit(EXIT_FAULT);
}

int main(int argc, char *argv[])
{ 
  int ret_val = EXIT_NORMAL;
  struct status_t tmp;
   
  if(cmdline(argc, argv)) {
    fprintf(stderr, "\nBlack Box BBS %s Copyright 1996-1997 Walter Heukels\n"
	    "Usage: bb [-remote] [-node n] [-modem m] [-spy t] [-nodetect] "
	    "[-user u]\n\n"
	    "-remote:   disable signals\n"
	    "-node:     force node number to n\n"
	    "-modem:    use modem device m\n"
	    "-spy:      show session on terminal device t\n"
	    "-nodetect: do not attempt to detect ANSI terminal\n"
	    "-user:     don't ask for username, use u instead\n\n",
	    VERSION_STRING);
    exit(EXIT_INIT);
  }
   
  if(values[DEBUG].used)
    debug = 1;

  if(values[REMOTE].used && values[MODEM].used)
    init_error("You can not use both -modem and -remote");
  
  // Clean this up.
  home_dir = (char *)malloc(1024);
  if(getcwd(home_dir, 1024) == NULL) {
    perror("getcwd");
    init_error("could not get current directory");
  }
  list.add_sys("homedir", home_dir);
  free(home_dir);
  
  signal(SIGTERM, terminate);
  signal(SIGINT,  terminate);
  signal(SIGSEGV, segfault);
  signal(SIGUSR1, chat);
  signal(SIGHUP,  hung_up);

  if(init_pool(1))
    init_error("error initializing shared memory pool");
   
  if(values[NODE].used) {
    node = values[NODE].i;
    if(isanode(node))
      init_error("node %d already exists", node);
  }
  else {
    node = probe_node();
    if(node == -1)
      init_error("all nodes are used");
  }
   
  list.add_sys("node", node);
  
  init_inc();
  init_log();
   
  if(initstatus())
    init_error("Maximum of %d nodes reached (Wow!)", MAX_NODES);
     
  busy(BUSY_INIT);

  list.add_sys("pid", status_p -> pid);
  default_vars();
   
  read_config();
  init_areas();
  init_protocols();
  init_archivers();
  if(!values[NOGETTY].used)
    init_getty();
    
  if(values[REMOTE].used) {
    list.add_sys("devtype", "remote");
    list.add_sys("device", "/dev/tty");
    modem = new localmodem("/dev/tty");
    remote = 1;
  } 
  else if(values[MODEM].used) {
    list.add_sys("devtype", "modem");
    list.add_sys("device", values[MODEM].s);
    lock_device();
    get_initstring();
    modem = new realmodem(values[MODEM].s);
  } 
  else {
    list.add_sys("devtype", "local");
    list.add_sys("device", "/dev/tty");
    modem = new localmodem("/dev/tty");
  }
   
  dprintf(">initializing modem\n");
  if(modem -> init())
    init_error("could not initialize modem");

  if(modem -> type == DEV_MODEM)
    modem -> reset();
 
  if(!remote && modem -> type == DEV_LOCAL) {
    modem -> sig(1);
    signal(SIGCONT, cont);
    signal(SIGTSTP, suspend);
  }
   
  if(!values[NODE].used && node >= list["maxnodes"] -> get_i()) {
    dump_file("data/busy");
    delete modem;
    exit(EXIT_NODES);
  }
   
  dprintf(">loading newuser\n");
  if(!load_user("newuser", &list))
    init_error("no \"newuser\" entry in user file");

  if(!list["timeout"])
    list.add("timeout", 240);

  busy(BUSY_WAITING);
  
  if(modem -> type == DEV_MODEM)
    log("ok, waiting for call");
  
  if(remote)
    log("ok, spawned for remote call");
   
  if(modem -> wait4call()) {
    log("bad connect");
    cleanup();
    return EXIT_CONNECT;
  }
   
  if(values[SPY].used) {
    spymodem = new localmodem(values[SPY].s);
    if(spymodem -> init())
      init_error("error initializing spy modem"); 
    list.add_sys("spyterm", spymodem -> device);
  }
  else 
    spymodem = NULL;
   
  tick(42);

  timeout = list["timeout"] -> get_i();
  dprintf(">timeout = %d, logging in\n", timeout);

  srandom(time(0));
 
  if(!values[NODETECT].used)
    output_raw("\033Z");

  if(login()) {
    busy(BUSY_ONLINE);
    do_menu("menus/login");
    list.add("exit", "normal");
    logoff();
    log("logoff");
  }
  else 
    ret_val = EXIT_BADLOGIN;
   
  cleanup();
  
  if(remote)
    usleep(1000000);
  
  return ret_val;
}

