#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>
#include "bb.h"
#include "editor.h"
#include "lock.h"
#include "modem.h"
#include "output.h"
#include "smtp.h"
#include "tokens.h"
#include "varlist.h"

extern char *out_buf;

#define MF_READ    1
#define MF_DELETED 2

#define MSG_NEXT 1
#define MSG_QUIT 2
#define MSG_READ 3
#define MSG_KILL 4

#define MSG_MAX 250

struct {
  char *filename;
  unsigned char flags;
} message[MSG_MAX];

extern varlist list;
extern char token[1024], quit_key, cont_key;
extern struct line_t *first, *current;

void save_mail(char *);

extern vmodem *modem;

int load_signature(int def)
{
  char tmp[80];
  int t;
  FILE *sig;
  
  sprintf(tmp, "signatures/%s", string("user"));
  for(t = 0;t < 80 && tmp[t];t++)
    if(tmp[t] == ' ')
      tmp[t] = '_';
  
  sig = fopen(tmp, "r");
  
  if(sig == NULL)
    if(def)
      sig = fopen("signatures/default", "r");
    else
      return 1;
  
  if(sig == NULL)
    return 1;
    
  while(current && current -> next)
    current = current -> next;

  while(fgets(tmp, 159, sig)) {
    insert_line();
    if(current -> next)
      current = current -> next;
    if(tmp[strlen(tmp) - 1] == '\n')
      tmp[strlen(tmp) - 1] = 0;
    sprintf(current -> text, tmp);
    current -> length = strlen(current -> text);
  }
  
  fclose(sig);
  return 0;
}

void edit_signature()
{
  int t;
  struct line_t *first_line, *last_line, *tmp;
  FILE *sig;
  char sig_file[80];
  
  if(load_signature(0))
    insert_line();

  list.add_sys("editstatus", "$white$Editing signature");    
  edit();

  first_line = first;
  while(first_line) {
    strip(first_line);
    if(first_line -> length)
      break;
      
    first_line = first_line -> next;
  }

  if(first_line == NULL) {
    edit_cleanup();
    return;
  }
      
  tmp = first_line;
  while(tmp) {
    strip(tmp);
    if(tmp -> length)
      last_line = tmp;
      
    tmp = tmp -> next;
  }

  tmp = first_line;

  sprintf(sig_file, "signatures/%s", string("user"));
  for(t = 0;t < 80 && sig_file[t];t++)
    if(sig_file[t] == ' ')
      sig_file[t] = '_';

  sig = fopen(sig_file, "w");
  if(sig == NULL)
    fatal_error("could not open %s for writing", sig_file);

  for(;;) {
    fwrite(tmp -> text, 1, tmp -> length, sig);
    fputc('\n', sig);
    if(tmp == last_line)
      break;
    tmp = tmp -> next;
  }
  
  fclose(sig);
  edit_cleanup();
}

void edit_mail(char *to)
{
  char tmp[160];
    
  sprintf(tmp, "$white$Private E-mail to: $yellow$%s $white$($yellow$%s$white$)", 
          to, string("subject"));

  list.add_sys("editstatus", tmp); 

  current = first;
  insert_line();

  load_signature(1);
  
  edit();
}

int mail_smtp()
{
  int sd, reply;
  char *buf = NULL, *text, *from;
  struct line_t *line;
   
  if(!buf)
    buf = (char *)malloc(1024);
   
  edit_mail(string("netuser"));
  line = first;
   
  sd = smtp_connect(string("smtphost"));
  if(sd == -1) {
    edit_cleanup();
    free(buf);
    return 1;
  }

  sprintf(buf, "HELO %s", getenv("HOSTNAME"));
  reply = smtp_command(buf);
  if(reply != 250) {
    edit_cleanup();
    close(sd);
    free(buf);
    return 1;
  }
    
  from = string("email");
   
  sprintf(buf, "MAIL FROM:<%s>", from);
  reply = smtp_command(buf);
  if(reply != 250) {
    edit_cleanup();
    close(sd);
    free(buf);
    return 1;
  }

  sprintf(buf, "RCPT TO:<%s>", string("netuser"));
  reply = smtp_command(buf);
  if(reply != 250) {
    edit_cleanup();
    close(sd);
    free(buf);
    return 1;
  }

  reply = smtp_command("DATA");
  if(reply != 354) {
    edit_cleanup();
    close(sd);
    free(buf);
    return 1;
  }

  sprintf(buf, "To: %s", string("netuser"));
  smtp_write(buf);
   
  sprintf(buf, "From: %s <%s>", string("user"), from);
  smtp_write(buf);
 
  sprintf(buf, "Subject: %s", string("subject"));
  smtp_write(buf);
  
  sprintf(buf, "X-Mailer: BlackMail %s", string("bbversion"));
  smtp_write(buf);

  while(line) {
    text = line -> text;
    if(*text == '.')
      *text = ' ';       

    if(line -> length)
      write(sd, text, line -> length);

    write(sd, "\n", 1);
    line = line -> next;
  }

  reply = smtp_command(".");
  if(reply != 250) {
    edit_cleanup();
    close(sd);
    free(buf);
    return 1;
  }
   
  smtp_command("QUIT");
  close(sd);
  edit_cleanup();
  free(buf);  
  
  return 0;
}

int mail_user()
{
  FILE *users;
  char *touser, filename[200];
  int type;
  struct tm *time_p;
  time_t caltime;
   
  if(!exists("touser"))
    return 1;
   
  touser = string("touser");
   
  bb_lock("data/users");
  users = fopen("data/users", "r");

  for(;;) {
    type = read_token(users);
    if(type == TOKEN_EOF) {   
      bb_unlock();
      fclose(users);
      return 2;
    }
      
    if(!strcasecmp(touser, token))
      break;
  }

  bb_unlock();

  edit_mail(string("touser"));
  
  caltime = time(NULL);
  time_p = localtime(&caltime);
   
  sprintf(filename, "mail/");
  strcat(filename, initials("user"));
  strcat(filename, "2");
  strcat(filename, initials("touser"));
  strcat(filename, "-");
  sprintf(filename + strlen(filename) - 1, "%02d-%02d-%4d@%02d:%02d:%02d",
	  time_p -> tm_mon + 1, time_p -> tm_mday, time_p -> tm_year + 1900,
	  time_p -> tm_hour, time_p -> tm_min, time_p -> tm_sec);
   
  save_mail(filename);
  edit_cleanup();
  return 0;
}

int check_mail()
{
  FILE *index;
  int type;
  char *user;
   
  user = string("user");
   
  index = fopen("mail/Index", "r");
  if(!index) {
    perror("fopen");
    fatal_error("cannot open mail index");
  }
   
  bb_lock("mail/Index");
  
  for(;;) { 
    type = read_token(index);
     
    if(type == TOKEN_EOF) {
      fclose(index);
      bb_unlock();
      return 0;
    }
    
    if(!strcmp(user, token)) {
      type = read_token(index); // filename
      type = read_token(index); // read/unread
      if(!strcmp(token, "read")) {
	skip_line(index);
	continue;
      }

      fclose(index);
      bb_unlock();
      return 1;
    }
     
    skip_line(index);
  }
}

void save_mail(char *file)
{
  FILE *fp;
  char buf[200];
  int use_gzip = 0;
  struct line_t *line = first;
   
  if(!exists("gzip"))
    fp = fopen(file, "w");
  else {
    strcat(file, ".gz");
    sprintf(buf, "%s > %s", string("gzip"), file);
   
    fp = popen(buf, "w");
    if(!fp)
      fatal_error("error executing gzip");
   
    use_gzip = 1;
  }

  fprintf(fp, "Date: %s\n", string("date"));
  fprintf(fp, "To: %s\n", string("touser"));
  fprintf(fp, "From: %s\n", string("user"));
  fprintf(fp, "Subject: %s\n\n", string("subject"));
     
  while(line) {
    fwrite(line -> text, 1, line -> length, fp);
    fprintf(fp, "\n");
    line = line -> next;
  }
   
  if(use_gzip)
    pclose(fp);
  else
    fclose(fp);
  
  bb_lock("mail/Index");
   
  fp = fopen("mail/Index", "a+");
  if(!fp) {
    perror("fopen");
    fatal_error("cannot open mail index");
  }
   
  fprintf(fp, "\"%s\" \"%s\" unread\n", string("touser"), file);
  fclose(fp);
  bb_unlock();
}

int read_msg(char *file)
{
  FILE *msg;
  static char linebuf[160];
  int quit = 0, lines, rows, l = 0, forever = 0, use_gzip = 0;
  char c, buf[200], prompt[1024];

  output_cooked("\\c");
   
  lines = number("lines");
  rows = number("rows");
  
  if(file[strlen(file) - 1] == 'z') {
    sprintf(buf, "%s -c -d %s", string("gzip"), file);
    msg = popen(buf, "r");
    if(!msg)
      fatal_error("error executing gzip");
  }
  else
    msg = fopen(file, "r");
 
  for(;;) {
    if(!fgets(linebuf, 160, msg)) {
      quit = MSG_READ;
      break;
    }
     
    linebuf[rows - 1] = 0;
    if(linebuf[strlen(linebuf) - 1] == '\n')
      linebuf[strlen(linebuf) - 1] = 0;
     
    if(linebuf[strlen(linebuf) - 1] == '\r')
      linebuf[strlen(linebuf) - 1] = 0;
     
    if(!forever)
      l++;
     
    output_raw(linebuf);
    output_cooked("\n");
     
    if(l == (lines - 1)) {
      cook("$moreprompt$ $yesprompt$ $noprompt$ $deleteprompt$ "
	   "$quitprompt$");
      strcpy(prompt, out_buf);
      output_cooked(prompt);

      flush();
      c = tolower(read_char());

      if(c == 'n')
	quit = MSG_NEXT;
      else if(c == quit_key)
	quit = MSG_QUIT;
      else if(c == 'd')
        quit = MSG_KILL;
      else if(c == cont_key)
	forever = 1;
       
      output_cooked("\n");
      l = 0;
    }
     
    if(quit)
      break;
  }

  if(quit == MSG_READ) {
    cook("$nextprompt$ $deleteprompt$ $quitprompt$");
    strcpy(prompt, out_buf);
    output_cooked(prompt);

    flush();
    c = tolower(read_char());
    //if(c == 'r')
    //  return MSG_REPLY;
    if(c == 'd')
      return MSG_KILL;
    if(c == quit_key)
      return MSG_QUIT;
    return MSG_NEXT;
  }
 
  if(use_gzip)
    pclose(msg);
  else
    fclose(msg);
  
  return quit;
}

void read_mail(int all)
{
  int m, type, msgs = 0, t;
  char *user = string("user"), dest[40];
  FILE *index, *new_index;
   
  bb_lock("mail/Index");
  index = fopen("mail/Index", "r");

  for(;;) {
    type = read_token(index); //user
    if(type == TOKEN_EOF)
      break;
     
    if(strcmp(token, user)) {
      skip_line(index);
      continue;
    }
     
    read_token(index); // filename     
    message[msgs].filename = (char *)malloc(strlen(token) + 1);
    strcpy(message[msgs].filename, token);
    
    read_token(index); // read/unread
    if(!strcmp(token, "read"))
      message[msgs].flags |= MF_READ;
    else
      message[msgs].flags = 0;
     
    msgs++;
    skip_line(index);
  }
  
  fclose(index);
  bb_unlock();

  if(!msgs)
    return;
   
  for(t = 0;t < msgs;t++) {
    if(all || !message[t].flags & MF_READ)
      m = read_msg(message[t].filename);
    else
      continue;
     
    message[t].flags |= MF_READ;

    //if(m == MSG_REPLY)
    //  reply_msg(t);
     
    if(m == MSG_QUIT)
      break;

    if(m == MSG_KILL)
      message[t].flags |= MF_DELETED;
  }
  
  bb_lock("mail/Index");
  index = fopen("mail/Index", "r");
  new_index = fopen("mail/Index.tmp", "w");
   
  for(;;) {
    type = read_token(index); // user
    if(type == TOKEN_EOF)
      break;
     
    strcpy(dest, token);
      
    read_token(index); // filename

    if(!strcmp(dest, user)) {
       for(t = 0;t < msgs;t++)
	if(!strcmp(message[t].filename, token))
	  break;
     
      if(t == msgs)
        fprintf(new_index, "\"%s\" \"%s\" unread\n", user, token);
      else if(message[t].flags & MF_DELETED)
	remove(message[t].filename);
      else
        fprintf(new_index, "\"%s\" \"%s\" %s\n", user, token, 
	        (message[t].flags & MF_READ) ? "read" : "unread");      
    }
    else {
      fprintf(new_index, "\"%s\" \"%s\" ", dest, token);
      read_token(index);
      fprintf(new_index, "%s\n", token);
    }
     
    skip_line(index);
  }
   
  fclose(index);
  fclose(new_index);
   
  rename("mail/Index.tmp", "mail/Index");
  bb_unlock();

  for(t = 0;t < msgs;t++)
    free(message[t].filename);
}
