#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "editor.h"
#include "output.h"
#include "varlist.h"

void insert_line();

extern varlist list;
extern char *out_buf;
extern int ansi;

/* lines = LINES-3 (!) */
static int width, maxx = 255, old_xpage, old_ypage;
static int lines, xpage = 0, ypage = 0, yjump;
static int cx, cy, old_cx, old_cy, tx;
static char edit_status[160];

struct line_t *current = NULL, *first = NULL;

void strip(struct line_t *);
void go_down(int);
void go_up(int);
void delete_line(struct line_t *);

void move_to(int x, int y)
{
  char buf[20];
  sprintf(buf, "\033[%d;%dH", y + 1, x + 1);
  output_raw(buf);
}

void header()
{
  int x, t;
  static char *txt = "BlackEdit v1.01 Control-Z to save";
  x = (width - strlen(txt)) / 2;

  output_raw("\033[1;37;44m");

  for(t = 0;t < x;t++)
    output_raw(' ');

  output_raw(txt);

  for(t = 0;t < x;t++)
    output_raw(' ');

  output_raw("\033[2;1H\033[0m");
  output_raw(edit_status);
  output_raw("\033[0m");
}

static void redraw()
{
  int t, n, x;
  struct line_t *line = first;

  output_raw("\033[0m\033[2J\033[1;1H");
  header();

  for(t = 0;t < ypage;t++)
    if(line -> next)
      line = line -> next;

  for(t = 0;t < lines;t++) {
    move_to(0, t + 2);

    if(line) {
      x = 0;
      for(n = xpage;n < line -> length;n++) {
        if(x++ >= width)
          break;
        output_raw(line -> text[n]);
      }
    }
     
    if(line)
      line = line -> next;
  }

  move_to(cx, cy);
}

void redraw_partial()
{
  int y = cy, x, n;
  struct line_t *line = current;
  
  move_to(0, cy);
   
  for(;y <= lines;y++) {
    output_raw("\033[K");
    if(line) {
      x = 0;
      for(n = xpage;n < line -> length;n++) {
        if(x++ >= width)
          break;
        output_raw(line -> text[n]);
      }
      line = line -> next;
    }
    if(y != lines)
      output_raw("\n");
  }

  move_to(cx, cy);
}

void insert_line()
{
  struct line_t *tmp;

  tmp = (struct line_t *)malloc(sizeof(struct line_t));
  tmp -> text = (char *)malloc(maxx + 2);
  tmp -> length = 0;
  tmp -> text[0] = 0;
   
  if(current) {
    if(current -> next)
      current -> next -> last = tmp;

    tmp -> next = current -> next;
    tmp -> last = current;
    current -> next = tmp;
  }
  else {
    tmp -> next = NULL;
    tmp -> last = NULL;
    first = current = tmp;
  }
}

int wrap()
{
  int n = 0, t, w;

  if(current -> length < width || tx < (current -> length - 1))
    return 0;
   
  for(t = width;t > 0;t--)
    if(current -> text[t] == ' ')
      break;

  w = t;
  if(!w)
    return 0;

  strip(current);

  insert_line();
  for(++t;t < current -> length;t++)
    current -> next -> text[n++] = current -> text[t];

  current -> length = w;
  current -> next -> length = n;
  cx = n;
  redraw_partial();
  go_down(1);
  return 1;
}

void insert_char(char c)
{
  int t, x = cx;

  if(!current)
    insert_line();

  if(tx > current -> length) {
    for(t = current -> length;t < tx;t++)
      current -> text[t] = 32;

    current -> length = tx + 1;
    current -> text[tx] = c;

    if(tx == width && wrap())
      return;
    else {
      output_raw(c);
      cx++;
      old_cx++;
    }
  }
  else {
    if(current -> length == maxx)
      current -> length--;

    for(t = current -> length;t > tx;t--)
      current -> text[t] = current -> text[t - 1];

    current -> text[tx] = c;
    current -> length++;

    if(tx == width && wrap())
      return;

    for(t = tx;t < current -> length && x < width;t++, x++)
      output_raw(current -> text[t]);

    cx++;
  }
}

int input()
{
  int c, n;

  c = read_char();
   
  if(c == 2)          // ^b
    c = CURSOR_LEFT;
  else if(c == 6)     // ^f
    c = CURSOR_RIGHT;
  else if(c == 16)    // ^p
    c = CURSOR_UP;
  else if(c == 14)    // ^n 
    c = CURSOR_DOWN;
  else if(c == 5)     // ^e
    c = CURSOR_END;
  else if(c == 1)     // ^a
    c = CURSOR_HOME;
  else if(c == 27) { /* ANSI cursor keys */
    c = read_char();
    if(c != '[')
      return UNKNOWN;
    
    c = read_char();
     
    if(c == 'A')
      c = CURSOR_UP;
    else if(c == 'B')
      c = CURSOR_DOWN;
    else if(c == 'D')
      c = CURSOR_LEFT;
    else if(c == 'C')
      c = CURSOR_RIGHT;
    else if(c == '1' || c == '4') {
      n = read_char();
      if(n == '~')
	return c == '1' ? CURSOR_HOME : CURSOR_END;
      else
	return UNKNOWN;
    }
    else
      c = UNKNOWN;
  }

  return c;
}

void enter()
{
  int t;
  struct line_t *newline;

  if(!current)
    return;

  strip(current);
   
  insert_line();
  newline = current -> next;

  xpage = cx = 0;

  if(tx < current -> length) {
    newline -> length = 0;
    for(t = tx;t < current -> length;t++) {
      newline -> text[t - tx] = current -> text[t];
      newline -> length++;
    }
    current -> length = tx;
    strip(current);
  }

  redraw_partial();
  go_down(1);
}

void backspace()
{
  int t, x = cx, n;
  struct line_t *last;

  if(tx) {
    tx--;
    cx--;
    move_to(cx, cy);

    for(t = tx;t < (current -> length - 1);t++, x++) {
      current -> text[t] = current -> text[t + 1];
      if(x <= width)
        output_raw(current -> text[t]);
    }

    current -> length--;
    if(x <= width)
      output_raw(' ');
  }
  else {
    if(current == first)
      return;

    last = current -> last;

    if((current -> length + last -> length) > maxx)
      return;

    n = last -> length;
    cx = n;
    for(t = tx;t < current -> length;t++)
      last -> text[n++] = current -> text[t];

    last -> length = n;

    go_up(1);
    delete_line(current -> next);
    redraw_partial();
  }
}

void strip(struct line_t *line)
{
  while(line -> length && line -> text[line -> length - 1] == ' ')
    line -> length--;
}

void delete_line(struct line_t *line)
{
  if(!line)
    return;

  if(line -> last)
    line -> last -> next = line -> next;

  if(line -> next)
    line -> next -> last = line -> last;

  free(line -> text);
  free(line);
}

void go_down(int n)
{
  if(!current)
    return;
   
  while(n--) {
    if(current -> next == NULL)
      break;

    current = current -> next;
    cy++;
    if(cy >= lines) {
      cy -= yjump;
      ypage += yjump;
    }
  }
}

void go_up(int n)
{
  if(!current)
    return;
   
  while(n--) {
    if(!current -> last)
      break;

    current = current -> last;
    cy--;
    if(cy < 2) {
      ypage -= yjump;
      cy += yjump;
    }
  }
}

void qnd_ascii_edit()
{
  struct line_t *tmp;
  
  output_cooked("\n\nBlackEdit v0.00(Q&D) Enter a single . on a line to exit\n");
  output_cooked(edit_status);
  output_cooked("\n");
  
  for(;;) {
    input_alpha(current -> text, width, 0);
    current -> length = strlen(current -> text);
    if(!strcmp(current -> text, "."))
      break;
    insert_line();
    go_down(1);
    output_cooked("\n");
  }
  
  tmp = current -> last;
  free(current -> text);
  free(current);
  tmp -> next = NULL;
} 

void edit()
{
  int c;
  struct line_t *tmp;
   
  cy = 2;
  lines = number("lines") - 3;
  width = number("rows");
  yjump = lines - 3;
  xpage = ypage = 0;
  old_cy = ~cy;
  //current = tmp = first;

  cook(string("editstatus"));
  //strncpy(edit_status, out_buf, width - 1);
  strncpy(edit_status, out_buf, 160);
  
#if 0
  while(tmp) {
    tmp -> length = strlen(tmp -> text);
    tmp = tmp -> next;
  } 

  while(current -> next)
    go_down(1);
 
  cx = current -> length;
#endif

  current = first;
  if(!current)
    insert_line();
    
  cx = 0;

  if(!ansi) {
    qnd_ascii_edit();
    return;
  }
    
  output_raw("\033[0m");
     
  redraw();
 
  for(;;) {
    if(cx > maxx)
      cx = maxx;

    if(cx < 0) {
      if(xpage) {
        xpage -= 8;
        cx += 8;
      }
      else {
	if(current -> last) {
	  go_up(1);
	  cx = current -> length + 1;
	}
	else
          cx = 0;	 
      }
    }

    if(cx > width) {
      while(cx > width) {
        xpage += 8;
        cx -= 8;
      }
    }
     
    if(cx != old_cx || cy != old_cy) {
      move_to(cx, cy);
      old_cx = cx;
      old_cy = cy;
    }

    if(xpage != old_xpage || ypage != old_ypage) {
      old_xpage = xpage;
      old_ypage = ypage;
      redraw();
    }

    tx = cx + xpage;

    c = input();

    if(c == 23 || c == 26) /* ^W/^Z */
      break;
    else if(c == 12) /* ^L */
      redraw();
    else if(c == CURSOR_LEFT)
      cx--;
    else if(c == CURSOR_RIGHT)
      cx++;
    else if(c == CURSOR_UP)
      go_up(1);
    else if(c == CURSOR_DOWN)
      go_down(1);
    else if(c == CURSOR_END && current)
      cx = current -> length;
    else if(c == CURSOR_HOME)
      cx = xpage = 0;
    else if(c == 13)
      enter();
    else if(c == 8 || c == 127)
      backspace();
    else if(c == 25) { /* ^Y */
      if(current -> last) {
        go_up(1);
        delete_line(current -> next);
        go_down(1);
      }
      else if(current == first && first -> next) {
        tmp = first -> next;
        delete_line(first);
        tmp -> last = NULL;
        first = current = tmp;
      }
      redraw_partial();
    }
    else if(isprint(c) && cx <= maxx)
      insert_char(c);
  }
}

void edit_cleanup()
{
  struct line_t *tmp;
  
  current = first;
   
  for(;;) {
    tmp = current -> next;
    free(current -> text);
    free(current);
    if(!tmp)
      break;
    current = tmp;
  }
  
  current = first = NULL;
}
