#include <ctype.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <termios.h>
#include <unistd.h>
#include "bb.h"
#include "modem.h"
#include "output.h"
#include "status.h"
#include "tokens.h"
#include "varlist.h"

#undef USER
#define USER 0
#define SYSOP 1

// Always use powers of 2 for arbitrary limits ;-)
#define MACROS 128

static int width[2], x[2], w[2], color;
static char *word[2], macros_loaded = 0, bold[2];

extern int node, ansi;
extern struct status_t *status_p;
extern char token[1024], *out_buf;
extern varlist list;

void chat(int);
void init_macros();

int macros = 0, in_macro = 0;
char *macro[MACROS], *text[MACROS];
vmodem *sysop;
extern vmodem *modem, *spymodem;

int isendmacro(char c)
{
  return c == '!' || c == ',' || c == '?' || c == ' ' || c == '\'' 
     || c == '\"' || c == '.' || c == '\r';
}

void single(char *s, int who)
{
  if(who == SYSOP)
    output_raw(s);
  else
    sysop -> raw(s);
}

void both(char *s) 
{
  output_raw(s);
  sysop -> raw(s);
}

int do_macro(int who)
{
  int t, m;
  char tmp[161], *txt, buf[20];
  void wrap(char, int);
  variable *v;
   
  tmp[0] = 0;
   
  for(t = 0;t < w[who];t++)
    if(!isendmacro(word[who][t])) {
      tmp[t] = word[who][t];
      tmp[t + 1] = 0;
    }
   
  if(!t)
    return 0;
   
  if(*tmp == '$' && (v = list[tmp + 1]))
    if(v -> get_type() == STRING)
      txt = v -> get_s();
    else {
      sprintf(buf, "%d", v -> get_i());
      txt = buf;
    }
  else {
    for(m = 0;m < macros;m++) {
      if(!strcasecmp(macro[m], tmp))  
        break;
     }
     if(m == macros)
       return 0;
     txt = text[m];
  }
   
  for(t = 0;t < w[who];t++) {
    single("\010 \010", who);
    x[who]--;
  }
   
  cook(txt);
   
  w[who] = 0;
  *word[who] = 0;  
  for(t = 0;t < strlen(out_buf);t++)
    wrap(out_buf[t], who);

  return 1;
}

void wrap(char c, int who)
{
  int t;
   
  if(c == 8 || c == 127) {
    if(!x[who])
      return;
     
    w[who]--;
    if(w[who] < 0)
      w[who] = 0;
     
    word[who][w[who]] = 0;
        
    single("\010 \010", who);
    
    x[who]--;
  }
  else if(isprint(c)) {
    word[who][w[who]] = c;
    w[who]++;
    word[who][w[who]] = 0;
    x[who]++;
     
    if(x[who] == width[who]) {       // wrapped 
      if(w[who] == width[who]) {
	in_macro++; wrap('\r', who); in_macro--; // >-) 
	if(c != 32)
	  wrap(c, who);
	return;
      }
       
      for(t = 0;t < w[who] - 1;t++) 
        single("\010 \010", who);
       
      single("\n", who);	 
      single(word[who], who);
      
      x[who] = strlen(word[who]);     
    } else                           // not wrapped 
    if(who == SYSOP)
      output_raw(c);
    else
      sysop -> raw(c);
  }

  if(isendmacro(c)) {
    if(!in_macro) {
      in_macro = 1;
      if(do_macro(who)) {
        w[who] = 0;
	*word[who] = 0;
        if(c != 13)
	  wrap(c, who);
      }
      in_macro = 0;
    }
     
    if(c == '\r') {
      single("\n", who);     
      x[who] = 0;
    }    
     
    w[who] = 0;
    *word[who] = 0;
  }
   
}

void print(char c, int who)
{
  if(who == USER && color != USER) {
    sysop -> raw(bold[who] ? "\033[1;" : "\033[0;");
    sysop -> raw("36m");
    if(ansi) {
      output_raw(bold[who] ? "\033[1;" : "\033[0;");
      output_raw("36m");
    }
    color = USER;
  }
  else if(who == SYSOP && color != SYSOP) {
    sysop -> raw(bold[who] ? "\033[1;" : "\033[0;");
    sysop -> raw("32m");
    if(ansi) {
      output_raw(bold[who] ? "\033[1;" : "\033[0;");   
      output_raw("32m");
    }
    color = SYSOP;
  }
   
  wrap(c, USER);
  wrap(c, SYSOP);
}

void chat_core()
{
  int fd_max, ret, modemfd, sys_fd;
  fd_set rdfd;
  char c;
  
  init_macros();
   
  color = 42;
  modemfd = modem -> fd;
  sys_fd = sysop -> fd;
   
  x[USER] = x[SYSOP] = 0;
  w[USER] = w[SYSOP] = 0;
  bold[USER] = bold[SYSOP] = 0;
   
  width[SYSOP] = list["rows"] -> get_i() - 2;
  width[USER] = getenv("COLUMNS") ? atoi(getenv("COLUMNS")) - 2 : 78;
   
  word[USER]  = (char *)malloc(width[USER]);
  word[SYSOP] = (char *)malloc(width[SYSOP]);
  
  output_cooked("\\c");
  output_cooked(string("chatmsg"));
  output_cooked("\\n\\n");
    
  cook("\033[0m\033[2J\033[HYou can talk to $user$ now.\n\n");
  single(out_buf, USER);
   
  if(sys_fd > modemfd)
    fd_max = sys_fd + 1;
  else
    fd_max = modemfd + 1;
   
  for(;;) {  
    FD_ZERO(&rdfd);
    FD_SET(modemfd, &rdfd);
    FD_SET(sys_fd, &rdfd);
    ret = select(fd_max, &rdfd, NULL, NULL, NULL);
     
    if(ret == -1) //interrupted
      continue;
    
    if(FD_ISSET(modemfd, &rdfd)) {
      read(modemfd, &c, 1);
      if(c == 2) {
        color = 42;	 
	bold[USER] ^= 1;
      }
      else
       print(c, USER);
    }
   
    if(FD_ISSET(sys_fd, &rdfd)) {
      read(sys_fd, &c, 1);
      if(c == 4) {
	free(word[USER]);
	free(word[SYSOP]);
	return;
      }
      else if(c == 2) {
	color = 42;
	bold[SYSOP] ^= 1;
      }
      else
        print(c, SYSOP);
    }     
  }
}

void chat(int sig)
{
  int old_busy = status_p -> busy;
   
  if(status_p -> busy == BUSY_CHATTING) {
    signal(SIGUSR1, chat);
    return;
  }
 
  if(!list["chatterm"]) {
    signal(SIGUSR1, chat);
    return;
  }

  sysop = new localmodem(list["chatterm"] -> get_s());
  sysop -> init();
   
  busy(BUSY_CHATTING); 
   
  chat_core();

  signal(SIGUSR1, chat);
  busy(old_busy);  
  delete sysop;
}

void init_macros()
{
  FILE *macrofile;
  int type;
   
  if(macros_loaded)
    return;
   
  macros_loaded = 1;
   
  dprintf(">Loading chat macros: ");
   
  macrofile = fopen("data/macros", "r");
  if(!macrofile) {
    dprintf("no macro file\n");
    return;
  }
   
  for(;;) {
    type = read_token(macrofile);
    if(type == TOKEN_COMMENT || type == TOKEN_EOL)
      continue;
    
    if(type == TOKEN_EOF)
      break;
     
    if(type != TOKEN_TEXT)     
      fatal_error("error in macro file");
     
    macro[macros] = new char [strlen(token) + 1];
    strcpy(macro[macros], token);
    
    type = read_token(macrofile);
    text[macros] = new char [strlen(token) + 1];
    strcpy(text[macros], token);
    macros++;
  }

  dprintf("%d macros loaded.\n", macros);
  fclose(macrofile);
}

int page()
{
  char *pager, *term;
  int t, delay, tries, result, old_busy = status_p -> busy, sys_fd;
  struct timeval tv;
  fd_set rdfd;
  pid_t pid;
   
  log("page by %s (%s)", string("user"), string("chatreason"));
   
  if(!list["pager"] || !list["pagedelay"] || !list["pagetries"])
    return 1;

  term = string("chatterm");
  if(!strcmp(term, "none") || (spymodem && !strcmp(term, string("spyterm"))))
    term = "auto";
   
  sysop = new localmodem(term);
  *list["chatterm"] = sysop -> device;
     
  sysop -> init();
  sys_fd = sysop -> fd;
   
  cook("\n[Chat request from $user$ (node $node$) on $chatterm$: $chatreason$]\n");
  single(out_buf, USER);

  /*
  for(t = 0;t < SHM_MAX;t++) {
    if(t == node) // this node
      continue;
     
    if(!getstatus(t, &tmp)) // non-existent/dead node
      continue;
     
    if(tmp.busy == BUSY_PAGING || tmp.busy == BUSY_CHATTING)
      return 3;
  }*/

  for(t = 0;t < MAX_NODES;t++) {
    if(t == node)
      continue;

    if(status_pool -> status[t].pid == 1)
      continue;

    if(status_pool -> status[t].busy == BUSY_PAGING
       || status_pool -> status[t].busy == BUSY_CHATTING)
      return 3;
  }
   
  pager = list["pager"] -> get_s();
  delay = list["pagedelay"] -> get_i();
  tries = list["pagetries"] -> get_i();

  busy(BUSY_PAGING);
   
  for(t = 0;t < tries;t++) {
    output_raw(".");
 
    pid = fork();
    if(pid == 0) { // child
      system(pager);
      raise(SIGKILL);
    }
    else if(pid == -1) { // error
      perror("fork");
      return 1;
    }
    else { // parent
      FD_ZERO(&rdfd);
      FD_SET(sys_fd, &rdfd);
      tv.tv_sec = delay;
      tv.tv_usec = 0;
    
      result = select(sys_fd + 1, &rdfd, NULL, NULL, &tv);
      if(result && result != -1)
        break;
    
      waitpid(pid, NULL, WNOHANG);
    }
  }

  if(t == tries)
    return 2;
  else {
    sysop -> read_char();
    busy(BUSY_CHATTING);
    chat_core();
    busy(old_busy);
    return 0;
  }
}

