/* 
File: bbutmp.c 
Desc: Handlers for special bbs-utmp
  All routines return 0 for success or -1 for fail.
  Routines include:

	init_btmp	Read utmp and btmp together, clearing dead entries
			and initializing ours to the given username arg.  If
			this arg is null, leave the username alone but still
			return the offset.  Stores our btmp offset
			in "thisbtmp".
	enter_btmp	Set our btmp program-name to string argument.
	clear_btmp	Zero out our btmp entry.
	read_btmp	Dump all the active btmp entries.  This constitutes the
			bare metal of "bbswho".

   If this is compiled with -DINIT, main() creates a new empty btmp file.
   If compiled with -DWHO, main() calls read_btmp; this is bbswho.
Source: Yet another Hobbitization (Hobbit, pirmann)

NAME                      LOGIN@ READING/DOING           FROM HOST
Ensign Ro                 11:13  Music                   p0:middlebury.edu..
1234567890123456789012345 12345  12345678901234567890123 12 1234567890123456
26 for name   remember to null terminate
6 for time
24 for what
5 for tty
17 for host
*/

#include <utmp.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <fcntl.h>
#include <time.h>
#include "citadel.h"
#include "bbutmp.h"

int
init_btmp (uname, hname, thistty)
char *uname;
char hname[17];
char thistty[9];
{
  int count;
  long atime;
  short wflag;
  int fp, bfp;
  struct tm *tmstruct;
  char ltime[6];

  if ((fp = open ("/etc/utmp", O_RDONLY)) == -1) 
    return (-1);

/* Since we in theory are only munging our own entry, non-exclusive access
   should be okay; the rest of citadel seems to survive this way...-Hobbit
   Yeah right--pirmann */

  if ((bfp = open ("btmp",(O_RDWR))) == -1) 
    {
      close (fp);
      return (-1);
    }
  
  while (flock (bfp, LOCK_EX|LOCK_NB) < 0)
    sleep (1);  /* get a lock on the btmp file */

  count = 0;
  wflag = 0;

  while ((read (fp, &entry, REC)) != 0) 
    {
      time (&atime);

      /* Copy down relevant utmp fields to btmp fields, ensuring that 
	 they're all properly terminated...  */
      strncpy (bent.bt_line, entry.ut_line, 5);
      bent.bt_line[5] = '\0';

      /* Is this our entry?  */
      if ((strcmp (bent.bt_line, thistty)) == 0) 
	{
	  if (uname) 	/* if given arg, stuff it in */
	    {		
	      strncpy (bent.bt_name, uname, 26);
	      bent.bt_name[26] = '\0';
	    } 
	  else 
	    {
	      lseek (bfp, (count * BREC), L_SET); /* if null arg, suck in */
	      read (bfp, &bent, BREC);		/* old name so we have it */
	    }
	  thisbtmp = count * BREC;		/* our offset */
	  wflag = 1;
	}


/* If the utmp username for this line is null, then we might have a bogus
   btmp entry.  Zero it out just to be sure.  This fixes the clowns who just
   hang up instead of logging out, which otherwise leave turds in btmp. */
      if (entry.ut_name[0] == 0) 
	{
	  bent.bt_name[0] = '\0';
	  wflag = 1;
	}
/* If this is our entry, or if fixing an empty one, write it out.  */
      if (wflag) 
	{
/* Going to write this one, snarf up the rest of the fields first */
	  bent.bt_pid = getpid();
	  tmstruct = localtime(&atime);
	  sprintf (ltime,"%2.2d:%2.2d",tmstruct->tm_hour, tmstruct->tm_min);
	  ltime[5] = '\0';
	  strncpy (bent.bt_time,ltime, 5);
	  strncpy (bent.bt_host,entry.ut_host, 16);
	  bent.bt_host[16] = '\0';
	  if (strlen(bent.bt_host) > 1) 
	      strncpy (hname, bent.bt_host, 16);
	  strncpy (bent.bt_prog, login, 24);
	  bent.bt_prog[24] = '\0';
	  lseek (bfp, (count * BREC), L_SET);
	  if ((write (bfp, &bent, BREC)) == -1) 
	    {
	      barf ("can't write btmp!");
	      close (fp);
	      flock (bfp, LOCK_UN);
	      close (bfp);
	      return (-1);
	    }
	  wflag = 0;
	}  /* if wflag */
      count++;
    } /* while utmp records */

  close (fp);
  flock (bfp, LOCK_UN);
  close (bfp);
  hname [17] = '\0';        /* always null terminate the entry */
  return (0);
}

int
enter_btmp (prog)
char *prog;
{
  int bfp;

/* All of this should error-check, but we'll finesse that for now */
  if ((bfp = open ("btmp", O_RDWR)) == -1)
    return (-1);

  while (flock (bfp, LOCK_EX|LOCK_NB) < 0)
    sleep (1);  /* get a lock on the btmp file */

  lseek (bfp, thisbtmp, L_SET);
  read (bfp, &bent, BREC);
  if (!strcmp("Enter msg",prog))
    bent.bt_prog[0] = '+';
  else
    {
      bent.bt_prog[0] = ' ';
      strncpy (&bent.bt_prog[1], prog, 23); /* must be one less! */
    }
  lseek (bfp, thisbtmp, L_SET);
  write (bfp, &bent, BREC);
  flock (bfp, LOCK_UN);
  close (bfp);
  return (0);
}

clear_btmp()
{
  int bfp;

  if ((bfp = open ("btmp", O_RDWR)) == -1)
    return (-1);

  while (flock (bfp, LOCK_EX|LOCK_NB) < 0)
    sleep (1);  /* get a lock on the btmp file */


  lseek (bfp, thisbtmp, L_SET);
  read (bfp, &bent, BREC);
  bent.bt_name[0] = '\0';
  lseek (bfp, thisbtmp, L_SET);
  write (bfp, &bent, BREC);
  flock (bfp, LOCK_UN);
  close (bfp);
  return (0);
}

read_btmp()
{
  int bfp;

  if ((bfp = open ("btmp", O_RDONLY)) == -1)
    {
      barf ("can't read bbs-utmp file!");
      return (-1);
    }     
#ifdef WHO
  printf ("The following users are logged into QuartzBBS.\n\r\n\r");
#endif

  printf ("NAME                      LOGIN@   READING/DOING           TTY/HOST");
#ifdef WHO
  printf ("\n\r");
#else
  printf ("\n");
#endif
  while (read (bfp, &bent, BREC) != 0) 
    if (bent.bt_name[0] != '\0') 
      {
	printf ("%-25.25s %-5s   %-23.23s  %-2.2s:%-16s", bent.bt_name, 
		bent.bt_time, bent.bt_prog, &bent.bt_line[3],
		bent.bt_host);
#ifdef WHO
	printf ("\n\r");
#else
	printf ("\n");
#endif
      }
  close (bfp);
  return (0);
}

alt_read_btmp()
{
  int bfp;

  if ((bfp = open ("btmp", O_RDONLY)) == -1)
    {
      barf ("can't read bbs-utmp file!");
      return (-1);
    }     
  printf ("NAME                      LOGIN@ TTY     PID   FROM HOST\n");

  while (read (bfp, &bent, BREC) != 0) 
    if (bent.bt_name[0] != '\0') 
      {
	printf ("%-25.25s %-5s  %-5.5s  %5d  %-16s\n",
		bent.bt_name, bent.bt_time, bent.bt_line,
		bent.bt_pid, bent.bt_host);
      }
  close (bfp);
  return (0);
}


static
barf(str)
char *str;
{
  printf ("?btmp: %s\n", str);
/*  exit(0); */
}

#ifdef INIT
main() {
  system ("touch btmp");	/* ick, but ... */
  exit (init_btmp());
}
#endif

#ifdef WHO
main() {
  chdir (BBSDIR);
  exit (read_btmp());
}
#endif

