/***************************************************************************
*
*  PRIVATE #includes
*
***************************************************************************/
#define INCL_DOSFILEMGR
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "stdcode.h"
#include "comm.h"
#include "system.h"
#include "mystring.h"
#include "files.h"
#include "os.h"

#define MSG_SUBSYSTEM
#include "message.h"
#include "m_squish.h"

/***************************************************************************
*
*  PRIVATE #defines
*
***************************************************************************/

#define MSG_MAPPATH "msgmap\\"

/***************************************************************************
*
*  GLOBAL variables
*
***************************************************************************/

t_Area*     msg_FirstArea_PS    = NULL;
t_Area*     msg_ThisArea_PS     = NULL;
char*       msg_Username_PC     = NULL;

/***************************************************************************
*
*  PRIVATE functions
*
***************************************************************************/

PRIVATE t_RetCode msg_ReadMapFile  ( void );
PRIVATE t_RetCode msg_ReadMapLine  ( char* );
PRIVATE t_RetCode msg_WriteMapLine ( FILE* );
PRIVATE t_RetCode msg_MakeNewArea  ( char* );
PRIVATE t_RetCode msg_RemakeMap    ( void );

/*==========================================================================
=
=  GLOBAL functions
=
===========================================================================*/

/***************************************************************************
*
*  Function:    MSG_Init
*  Description: Initialize message DLL:s
*
***************************************************************************/
t_RetCode MSG_Init( void )
{
    CALL( SQUISH_Init() );

    return OK;
}

/***************************************************************************
*
*  Function:    MSG_Exit
*  Description: Deinitialize MSGAPI.DLL
*
***************************************************************************/
t_RetCode MSG_Exit( void )
{
    DBG( "MSG_Exit", 0 );

    if ( msg_Username_PC )
	free( msg_Username_PC );

    if ( msg_FirstArea_PS )
	CALL( MSG_WriteMapFile() );

    CALL( SQUISH_Exit() );

    return OK;
}

/***************************************************************************
*
*  Function:    MSG_SetAreaSize
*  Description: Set the size (in messages) of the current area
*
***************************************************************************/
t_RetCode MSG_SetAreaSize( int MsgCount_I )
{
    CALL( SQUISH_SetAreaSize( MsgCount_I ) );

    return OK;
}

/***************************************************************************
*
*  Function:    MSG_SetUser
*  Description: Copy the user id/name to use for the map file
*
***************************************************************************/
t_RetCode MSG_SetUser( char* Username_PC )
{
    int     i;
    char*   Name_PC;

    if (!Username_PC)
	return OK;

    i = strlen( Username_PC );
    if (!i)
	return OK;

    Name_PC = (char*) malloc( i + 1 );
    if (!Name_PC)
	RET_ERR( "Out of memory!", i + 1 );

    strcpy( Name_PC, Username_PC );
    msg_Username_PC = Name_PC;

    return OK;
}

/***************************************************************************
*
*  Function:    MSG_GetUser
*  Description: 
*
***************************************************************************/
char* MSG_GetUser( void )
{
    if (msg_Username_PC)
	return msg_Username_PC;
    return "";
}

/***************************************************************************
*
*  Function:    MSG_MsgId
*  Description: Generate a unique MSGID
*
***************************************************************************/
t_RetCode MSG_MsgId( int* MsgId_PI )
{
    int       Ms_I;
    time_t    Time_I;
    struct tm*    Time_PS;
    struct tm     Time_S;
    int       MsgId_I, Year_I, SubSecs_I;

    Time_I = time( NULL );
    Time_PS = localtime( &Time_I );

    /* Get closest 5-year flat */
    Year_I = Time_PS->tm_year - ( Time_PS->tm_year % 5 );
    memset( &Time_S, 0, sizeof( Time_S ) );
    Time_S.tm_year = Year_I;
    SubSecs_I = mktime( &Time_S );

    /* Fetch ms count */
    CALL( OS_Milliseconds( &Ms_I ) );

    Ms_I /= 100;

    /* Create id */
    MsgId_I = ( Time_I - SubSecs_I ) * 10 + Ms_I;

    *MsgId_PI = MsgId_I;

    return OK;
}


/***************************************************************************
*
*  Function:    MSG_GotoArea
*  Description: Close current area and open new area
*
***************************************************************************/
t_RetCode MSG_GotoArea( char* Filename_PC, char* Tagname_PC, int AreaType_I )
{
    int     Status_I, Len_I;
    t_Area* Ptr_PS;

    if ( msg_FirstArea_PS ) {
	switch( msg_ThisArea_PS->Type_I ) {
	  case MSG_TYPE_SQUISH:
	    CALL( SQUISH_CloseArea() );
	    break;

	  default:
	    RET_ERR("Unsupported area type!", msg_ThisArea_PS->Type_I );
	}
    }

    if (!msg_FirstArea_PS)
	CALL( msg_ReadMapFile() );

    /* find area struct */
    for (msg_ThisArea_PS = msg_FirstArea_PS;
	 msg_ThisArea_PS;
	 msg_ThisArea_PS = msg_ThisArea_PS->Next_PS ) {

	if (!strcmp( Tagname_PC, msg_ThisArea_PS->Tag_PC ))
	    break;
    }

    switch ( AreaType_I ) {
      case MSG_TYPE_SQUISH:
	CALL( SQUISH_OpenArea( Filename_PC ) );
	break;

      default:
	RET_ERR("Unsupported area type!", AreaType_I );
    }

    /* if this is a new area, create a new area struct */
    if ( !msg_ThisArea_PS ) {
	CALL( msg_MakeNewArea( Tagname_PC ) );
    }
    else {
	/* has the area changed? */
	if ( SQUISH_NewMail() )
	    CALL( SQUISH_RemakeMap() );
    }
    msg_ThisArea_PS->Type_I = AreaType_I;

    return OK;
}

/***************************************************************************
*
*  Function:    MSG_DelMsg
*  Description: Deletes a message
*
****************************************************************************/
t_RetCode MSG_DelMsg( int MsgNo_I )
{
    CALL( SQUISH_DelMsg( MsgNo_I ) );

    return OK;
}

/***************************************************************************
*
*  Function:    MSG_Member
*  Description: Get/set info about whether user is a member of this area
*
****************************************************************************/
t_RetCode MSG_Member( char* Tagname_PC, int Command_I, int* Result_PI )
{
    t_Area* Area_PS;

    *Result_PI = 0;

    for ( Area_PS = msg_FirstArea_PS; Area_PS; Area_PS = Area_PS->Next_PS )
	if (!strcmp( Tagname_PC, Area_PS->Tag_PC ))
	    break;

    if (!Area_PS)
	return OK;

    switch ( Command_I ) {
      case -1:
	break;

      case 1:
	Area_PS->Map_PS->Member_B = TRUE;
	break;

      case 0:
	Area_PS->Map_PS->Member_B = FALSE;
	break;

      default:
	RET_ERR( "Unsupported command code", Command_I );
    }

    *Result_PI = Area_PS->Map_PS->Member_B ? 1 : 0;

    return OK;
}

/***************************************************************************
*
*  Function:    MSG_AreaInfo
*  Description: Get info about current message area
*
*               AreaInfo_I is one of the following
*
*  #define    Value Description           Example
*  ------------------------------------------------------------------------
*    	         0 - Total number of msgs  400
*                1 - Number of unread msgs 113
*                2 - Highest message #     6500
* 
****************************************************************************/
t_RetCode MSG_AreaInfo( int AreaInfo_I, int* Result_PI )
{
    CALL( SQUISH_AreaInfo( AreaInfo_I, Result_PI ) );

    return OK;
}


/***************************************************************************
*
*  Function:    MSG_MarkMessage
*  Description: Mark a message as (un)read
*
***************************************************************************/
t_RetCode MSG_MarkMessage( int MsgNo_I, int Command_I, t_Bool* Read_PB )
{
    int First_I, Count_I;

    *Read_PB = TRUE;

    if ( !MsgNo_I )
	return OK;

    MsgNo_I--;

    /* check for newly arrived mail */
    if (SQUISH_NewMail())
	msg_RemakeMap();

    if ( MsgNo_I > THISMAP->MsgCount_I )
	RET_ERR( "Too high message!", MsgNo_I );
    
    if ( MSG_NOP != Command_I ) {
	THISMAP->Read_PB[ MsgNo_I ] =
	    MSG_MARK == Command_I ? TRUE : FALSE;
	THISMAP->Touched_B  = TRUE;
    }
    
    *Read_PB = THISMAP->Read_PB[ MsgNo_I ];

    return OK;
}


/***************************************************************************
*
*  Function:    MSG_NextMessage
*  Description: UMSGID shell for FindNextMessage
*
***************************************************************************/
t_RetCode MSG_NextMessage( int MsgNo_I, int* NextMsgNo_PI )
{
    CALL( SQUISH_NextMessage( MsgNo_I, NextMsgNo_PI ) );
    
    return OK;
}

/***************************************************************************
*
*  Function:    MSG_CreateMessage
*  Description: Clear message buffer
*
***************************************************************************/
t_RetCode MSG_CreateMessage( void )
{
    CALL( SQUISH_CreateMessage() );
    return OK;
}

/***************************************************************************
*
*  Function:    MSG_SendMessage
*  Description: Post message in message base
*
***************************************************************************/
t_RetCode MSG_SendMessage( void )
{
    CALL( SQUISH_SendMessage() );
    return OK;
}

/**************************************************************************
*
*  Function:    MSG_PutMessageFlag
*
***************************************************************************/
t_RetCode MSG_PutMessageFlag( int Flag_I, int Action_I )
{
    CALL( SQUISH_PutMessageFlag( Flag_I, Action_I ) );
    return OK;
}

/***************************************************************************
*
*  Function:    MSG_PutMessage
*  Description: Put part of a message into message buffer
*
*               MsgPart_I is one of the following:
*
*  #define    Value Description           Example
*  ------------------------------------------------------------------------
*  M_BODY       0 - Message body filename "c:\\tmp\\node01.msg"
*  M_FROM       1 - From Name             "Joe User"
*  M_TO         2 - To Name               "Sysop"
*  M_SUBJECT    3 - Subject               "How to create a message"
*  M_ORIG       4 - From Address          "39:162/102.0"
*  M_DEST       5 - To Address            "2:201/328.0"
*  M_REPLYTO    7 - Reply to              "14171"
*  M_CONTROL    9 - Control info          "PID: Squish v1.01"
*
*  Flag_I is used for resetting reply counter
*
***************************************************************************/
t_RetCode MSG_PutMessage(int MsgPart_I, char* Data_PC )
{
    CALL( SQUISH_PutMessage( MsgPart_I, Data_PC ) );
    return OK;
}

/**************************************************************************
*
*  Function:    MSG_GetMessageFlag
*
***************************************************************************/
t_RetCode MSG_GetMessageFlag ( int MsgNo_I, int Flag_I, int* Result_PI )
{
    CALL( SQUISH_GetMessageFlag(  MsgNo_I, Flag_I, Result_PI ) );
    return OK;
}

/***************************************************************************
*
*  Function:    MSG_GetMessage
*  Description: Get part of a message
*
*               MsgPart_I is one of the following:
*
*  #define    Value Description           Example
*  ------------------------------------------------------------------------
*  M_BODY       0 - Message body          "Hi!\n  I just got your..."
*  M_FROM       1 - From Name             "Joe User"
*  M_TO         2 - To Name               "Sysop"
*  M_SUBJECT    3 - Subject               "How to create a message"
*  M_ORIG       4 - From Address          "39:162/102.0"
*  M_DEST       5 - To Address            "2:201/328.0"
*  M_DATE       6 - Date/time of writing  "94-03-10 12:22"
*  M_REPLYTO    7 - Reply to              "171"
*  M_REPLIES    8 - Replies               "172"
*  M_CONTROL    9 - Control info          "PID: Squish v1.01"
*  M_REPYCNT   10 - Number of replies     "4"
*          11 - Unique message id     "37682"
*
*  Flag_I is used for resetting reply counter
*
***************************************************************************/
t_RetCode MSG_GetMessage(int MsgNo_I,
             int MsgPart_I,
             int Flag_I,
             char** Text_PPC )
{
    CALL( SQUISH_GetMessage( MsgNo_I, MsgPart_I, Flag_I, Text_PPC ) );
    return OK;
}


/***************************************************************************
*
*  Function:    MSG_WriteMapFile
*  Description: Write map of read messages in all areas
*
***************************************************************************/
t_RetCode MSG_WriteMapFile( void )
{
    FILE*   OutFile_PS;
    char    Filename1_AC[128];
    char    Filename2_AC[128];
    t_Area* LastArea_PS = msg_ThisArea_PS;

    if ( !THISMAP->Read_PB )
	return OK;

    /* create output map filename */
    sprintf( Filename2_AC, MSG_MAPPATH "%s.new", msg_Username_PC );
    OutFile_PS = fopen( Filename2_AC, "w" );
    if ( NOT OutFile_PS )
	RET_ERR( "Can't create new user message map file!", errno );

    for ( msg_ThisArea_PS = msg_FirstArea_PS;
	 msg_ThisArea_PS;
	 msg_ThisArea_PS = msg_ThisArea_PS->Next_PS ) {

	msg_WriteMapLine( OutFile_PS );
    }

    fclose( OutFile_PS );

    /* make a backup */
    sprintf( Filename1_AC, MSG_MAPPATH "%s", msg_Username_PC );
    sprintf( Filename2_AC, MSG_MAPPATH "%s.bak", msg_Username_PC );
    rename( Filename1_AC, Filename2_AC );

    /* rename new master file */
    sprintf( Filename1_AC, MSG_MAPPATH "%s.new", msg_Username_PC );
    sprintf( Filename2_AC, MSG_MAPPATH "%s", msg_Username_PC );

    if ( ! rename( Filename1_AC, Filename2_AC ) ) {
	/* remove backup */
	sprintf( Filename1_AC, MSG_MAPPATH "%s.bak", msg_Username_PC );
	remove( Filename1_AC );
    }

    msg_ThisArea_PS = LastArea_PS;

    return OK;
}

/*==========================================================================
=
=  LOCAL functions
=
===========================================================================*/

/***************************************************************************
*
*  Function:    msg_ReadMapFile
*  Description: Read map file of unread messages
*
***************************************************************************/
PRIVATE t_RetCode msg_ReadMapFile( void )
{
    FILE*   File_PS;
    char*   Line_PC;
    int     Size_I = 4096;
    t_Bool  More_B;
    char    Filename_AC[250];

    /* open map file */
    sprintf( Filename_AC, MSG_MAPPATH "/%s", msg_Username_PC );
    File_PS = fopen( Filename_AC, "r" );

    /* if no map file exists */
    if ( NOT File_PS )
	return OK;

    /* make space for line */
    Line_PC = (char*) malloc( Size_I );
    if ( NOT Line_PC )
	RET_ERR( "Out of memory!", Size_I );

    /* step through map file */
    while (1) {
	CALL( STR_ReadLine( File_PS, &Line_PC, &Size_I, &More_B ) );
	if (!More_B)
	    break;
	CALL( msg_ReadMapLine( Line_PC ) );
    }

    fclose( File_PS );

    free( Line_PC );

    return OK;
}

/***************************************************************************
*
*  Function:    msg_ReadMapLine
*  Description: Read a line from map of unread messages
*
***************************************************************************/
PRIVATE t_RetCode msg_ReadMapLine( char* Line_PC )
{
    char  Tagname_AC[80];
    char* Ptr_PC = Line_PC;
    char* Tmp_PC;
    int   i, First_I = 1;
    int    FirstMsg_I, MsgCount_I, Dummy_I;
    t_Bool Member_B;

#if 0
    FirstMsg_I = THISMAP->FirstMsg_I;
    MsgCount_I = THISMAP->MsgCount_I;

    if (!MsgCount_I)
	MsgCount_I = 500;
#else
    MsgCount_I = 500;
#endif

    /* member? */
    if ( '*' == *Ptr_PC ) {
	Ptr_PC++;
	Member_B = TRUE;
    }
    else
	Member_B = FALSE;

    /* copy tagname */
    Tmp_PC = Ptr_PC;
    i = 0;
    while( ':' != *Tmp_PC ) {
	Tagname_AC[i] = *Tmp_PC;
	i++;
	Tmp_PC++;
    }
    Tagname_AC[i] = 0;

    /* find area struct */
    for (msg_ThisArea_PS = msg_FirstArea_PS;
	 msg_ThisArea_PS;
	 msg_ThisArea_PS = msg_ThisArea_PS->Next_PS ) {

	if (!strcmp( Tagname_AC, msg_ThisArea_PS->Tag_PC ))
	    break;
    }

    /* if this is a new area, create a new area struct */
    if ( !msg_ThisArea_PS )
	CALL( msg_MakeNewArea( Tagname_AC ) );

    THISMAP->Member_B  = Member_B;
    THISMAP->Touched_B = TRUE;

    /* move across the "FREXXNET: " tag name part of the line */
    Ptr_PC = strchr( Ptr_PC, ':' ) + 1;

    while ( 1 ) {
	int FirstNum_I, NextNum_I, MsgNo_I;
	t_Bool  Tmp_B;

	/* find out the first message number */
	FirstNum_I = strtol( Ptr_PC, &Ptr_PC, 10 );
	if ( 0 == FirstNum_I )
	    break;

	if ( First_I ) {
	    FirstMsg_I = FirstNum_I;
	    THISMAP->FirstMsg_I = FirstNum_I;
	    First_I = 0;
	}

	/* mark message as read */
	MsgNo_I = FirstNum_I - FirstMsg_I;

	if ( MsgNo_I >= MsgCount_I ) {
	    t_Bool* Tmp_PB;
	    int OldSize_I = MsgCount_I;

	    MsgCount_I += MsgNo_I * 2;
	    Tmp_PB = (t_Bool*) malloc( MsgCount_I );
	    if ( !Tmp_PB )
		RET_ERR( "Out of memory!", MsgCount_I );
	    memset( Tmp_PB, 0, MsgCount_I );
	    memcpy( Tmp_PB, THISMAP->Read_PB, OldSize_I );
	    free( THISMAP->Read_PB );
	    THISMAP->Read_PB = Tmp_PB;
	    THISMAP->MsgCount_I = MsgCount_I;
	}

	THISMAP->Read_PB[ MsgNo_I ] = TRUE;

	/* is it a range? */
	if ( '-' == Ptr_PC[0] ) {
	    int Msg_I;

	    Ptr_PC++;

	    NextNum_I = strtol( Ptr_PC, &Ptr_PC, 10 );
	    if ( 0 == NextNum_I )
		break;

	    MsgNo_I = NextNum_I - FirstMsg_I;
	    if ( MsgNo_I >= MsgCount_I ) {
		t_Bool* Tmp_PB;
		int OldSize_I = MsgCount_I;

		MsgCount_I += MsgNo_I * 2;
		Tmp_PB = (t_Bool*) malloc( MsgCount_I );
		if ( !Tmp_PB )
		    RET_ERR( "Out of memory!", MsgCount_I );
		memset( Tmp_PB, 0, MsgCount_I );
		memcpy( Tmp_PB, THISMAP->Read_PB, OldSize_I );
		free( THISMAP->Read_PB );
		THISMAP->Read_PB = Tmp_PB;
		THISMAP->MsgCount_I = MsgCount_I;
	    }

	    /* mark all messages in range */
	    for ( Msg_I = FirstNum_I + 1; Msg_I <= NextNum_I; Msg_I++ ) {
		MsgNo_I = Msg_I - FirstMsg_I;
		THISMAP->Read_PB[ MsgNo_I ] = TRUE;
	    }
	}

	/* skip the trailing comma */
	Ptr_PC++;
    }

    return OK;
}

/***************************************************************************
*
*  Function:    msg_WriteMapLine
*  Description: Create a one-line map of read messages
*
***************************************************************************/
PRIVATE t_RetCode msg_WriteMapLine( FILE* OutFile_PS )
{
    CALL( SQUISH_WriteMapLine( OutFile_PS ));
    return OK;
}

/***************************************************************************
*
*  Function:    msg_MakeNewArea
*  Description: Create a new node in area list
*
***************************************************************************/
PRIVATE t_RetCode msg_MakeNewArea( char* Tagname_PC )
{
    int     Len_I;
    char*   TmpStr_PC;

    /* create new area struct */

    msg_ThisArea_PS = (t_Area*) malloc( sizeof( t_Area ) );
    if (!msg_ThisArea_PS)
	RET_ERR( "Out of memory!", sizeof( t_Area ) );

    Len_I = strlen( Tagname_PC ) + 1;
    TmpStr_PC = (char*) malloc( Len_I );
    if (!TmpStr_PC)
	RET_ERR( "Out of memory!", Len_I );
    strcpy( TmpStr_PC, Tagname_PC );
    msg_ThisArea_PS->Tag_PC = TmpStr_PC;

    msg_ThisArea_PS->Next_PS = msg_FirstArea_PS;
    msg_FirstArea_PS = msg_ThisArea_PS;

    /* Create map */

    THISMAP = (t_AreaMap*) malloc( sizeof( t_AreaMap ) );
    if (!THISMAP)
    RET_ERR( "Out of memory!", sizeof( t_AreaMap ) );

    Len_I = 500;
    THISMAP->Read_PB    = (t_Bool*) malloc( Len_I );
    if ( !THISMAP->Read_PB )
	RET_ERR( "Out of memory!", Len_I );
    memset( THISMAP->Read_PB, 0, Len_I );

    THISMAP->FirstMsg_I = 0;
    THISMAP->MsgCount_I = Len_I;
    THISMAP->Member_B   = FALSE;
    THISMAP->Touched_B  = FALSE;

    return OK;
}

/***************************************************************************
*
*  Function:    msg_RemakeMap
*  Description: Recreate message map to mirror the world
*
***************************************************************************/
PRIVATE t_RetCode msg_RemakeMap( void )
{
    CALL( SQUISH_RemakeMap() );
    return OK;
}
