/***************************************************************************
*
*  PRIVATE #includes
*
***************************************************************************/
#define INCL_DOSFILEMGR
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "msgapi32/msgapi.h"

#include "stdcode.h"
#include "comm.h"
#include "system.h"
#include "mystring.h"
#include "files.h"
#include "os.h"

#define MSG_SUBSYSTEM
#include "m_squish.h"
#include "message.h"

/***************************************************************************
*
*  PRIVATE typedefs
*
***************************************************************************/

typedef struct {
    XMSG    Xmsg_S;
    char*   Body_PC;
    char*   Control_PC;
} t_Message;

/***************************************************************************
*
*  GLOBAL variables
*
***************************************************************************/

extern t_Area*     msg_FirstArea_PS    = NULL;
extern t_Area*     msg_ThisArea_PS     = NULL;
extern char*       msg_Username_PC     = NULL;

/***************************************************************************
*
*  PRIVATE variables
*
***************************************************************************/
static HAREA       squish_AreaHandle_PS   = NULL;
static t_Message   squish_OutMsg_S;
static byte*       squish_TextBuffer_PC   = NULL;

/***************************************************************************
*
*  PRIVATE functions
*
***************************************************************************/

PRIVATE t_RetCode squish_UnlinkMsg( int Msg_I );
PRIVATE t_RetCode squish_MarkUid  ( int Uid_I, int Command_I,t_Bool* Read_PB );
PRIVATE t_RetCode squish_GetHeader( int Uid_I, XMSG* XHeader_PS );
PRIVATE t_RetCode squish_UidToMsg ( int Uid_I, int* MsgNo_PI );
PRIVATE t_RetCode squish_UidExists( int Uid_I, t_Bool* Exists_PB );
PRIVATE t_RetCode squish_SetReply ( int Msg_I, int Reply_I );
PRIVATE t_RetCode squish_CloseArea( void );
PRIVATE t_RetCode squish_FindNextMessage( int ThisMsg_I, int* NextMsg_PI );

/***************************************************************************
*
*  Code
*
***************************************************************************/

t_RetCode SQUISH_Init( void )
{
    struct _minf minf={1,0,0,NULL,NULL,NULL,NULL,NULL,NULL};

    if ( -1 == MsgOpenApi( &minf ) )
        RET_ERR( "MSGAPI ERROR: MsgOpenApi() failed.", 0 );

    squish_TextBuffer_PC = (byte*) malloc( MSG_BUFSIZE );
    if ( NOT squish_TextBuffer_PC )
        RET_ERR( "ERROR: Out of memory!", 0 );

    return OK;
}

t_RetCode SQUISH_Exit( void )
{
    if ( squish_TextBuffer_PC )
	free( squish_TextBuffer_PC );

    if ( squish_OutMsg_S.Body_PC )
	free( squish_OutMsg_S.Body_PC );

    if ( squish_AreaHandle_PS )
	CALL( squish_CloseArea() );

    if ( -1 == MsgCloseApi() )
        RET_ERR( "MSGAPI ERROR: MsgCloseApi() failed.", 0 );

    return OK;
}

t_RetCode SQUISH_SetAreaSize( int MsgCount_I )
{
    if (!squish_AreaHandle_PS)
	return OK;

    SquishSetMaxMsg( squish_AreaHandle_PS, MsgCount_I,0,0);

    return OK;
}

t_RetCode SQUISH_OpenArea( char* Filename_PC )
{
    squish_AreaHandle_PS = MsgOpenArea( Filename_PC,
				        MSGAREA_CRIFNEC,
				        MSGTYPE_SQUISH );
    if ( NOT squish_AreaHandle_PS )
	RET_ERR("MSGAPI ERROR: MsgOpenArea() failed.", 0 );

    return OK;
}


t_RetCode SQUISH_DelMsg( int MsgNo_I )
{
    squish_UnlinkMsg( MsgNo_I );

    if ( -1 == MsgKillMsg( squish_AreaHandle_PS, MsgNo_I ) )
	RET_ERR( "Error killing message!", MsgNo_I );

    return OK;
}

t_RetCode SQUISH_AreaInfo( int AreaInfo_I, int* Result_PI )
{
    int Int_I;
    int High_I,y,i;

    THISMAP->FirstMsg_I = MsgMsgnToUid( squish_AreaHandle_PS, 1 );
    THISMAP->MsgCount_I = MsgGetNumMsg( squish_AreaHandle_PS );
    High_I = MsgGetHighMsg( squish_AreaHandle_PS );

    switch ( AreaInfo_I ) {

      case -1:
	return OK;

      case 0:
	*Result_PI = THISMAP->MsgCount_I;
	break;

      case 1:
	y=0;
	if ( THISMAP->Read_PB )
	    for ( i = 0; i < THISMAP->MsgCount_I; i++ )
		if ( NOT THISMAP->Read_PB[i] )
		    y++;
	*Result_PI = y;
	break;
	
      case 2:
	*Result_PI = High_I;
	break;
	
      default:
	RET_ERR( "Bad area info nr", AreaInfo_I );
    }
    
    return OK;
}


/***************************************************************************
*
*  Function:    SQUISH_NextMessage
*  Description: UMSGID shell for FindNextMessage
*
***************************************************************************/
t_RetCode SQUISH_NextMessage( int MsgNo_I, int* NextMsgNo_PI )
{
    int     ThisMsg_I;
    int     ThatMsg_I;

    if ( MsgNo_I > 0 )
	ThisMsg_I = MsgMsgnToUid( squish_AreaHandle_PS, MsgNo_I );
    else
	ThisMsg_I = MsgNo_I;
    
    CALL( squish_FindNextMessage( ThisMsg_I, &ThatMsg_I ) );
    
    if ( ThatMsg_I > 0 )
	*NextMsgNo_PI =
	    MsgUidToMsgn( squish_AreaHandle_PS, ThatMsg_I, UID_EXACT );
    else
	*NextMsgNo_PI = ThatMsg_I;
    
    return OK;
}


/***************************************************************************
*
*  Function:    squish_FindNextMessage
*  Description: Find the logical "next message" in message reply tree
*
***************************************************************************/
PRIVATE t_RetCode squish_FindNextMessage( int ThisMsg_I, int* NextMsg_PI )
{
    XMSG    XHeader_S;
    int     LastMsg_I;
    int     NextMsg_I;
    t_Bool  FirstRead_B;
    t_Bool  Read_B = FALSE;
    int     LoopCount_I = 0;

    DBG( "FindNextMessage", ThisMsg_I );
    
    if ( 0 == ThisMsg_I ) {
	*NextMsg_PI = 0;
	return OK;
    }
    
    if ( -1 == ThisMsg_I ) {
	int i;
	
	NextMsg_I = 0;
	/* desperately, find ANY unread message in area! */
	for ( i=0; i < THISMAP->MsgCount_I; i++ )
	    if ( NOT THISMAP->Read_PB[ i ] ) {
		NextMsg_I = i + 1;
		break;
	    }
	
	/* we've got a live one, or maybe we don't.
	   exit nevertheless. */
	if ( NextMsg_I )
	    NextMsg_I = MsgMsgnToUid( squish_AreaHandle_PS, NextMsg_I );
	*NextMsg_PI = NextMsg_I;
	return OK;
    }

    CALL( squish_MarkUid( ThisMsg_I, MSG_NOP, &Read_B ) );
    if ( NOT Read_B ) {
	*NextMsg_PI = ThisMsg_I;
	return OK;
    }
    
    /* do a quickie to see if there's a first comment unread */
    CALL( squish_GetHeader( ThisMsg_I, &XHeader_S ) );
    if ( XHeader_S.replies[0] ) {
	
	if ( XHeader_S.replies[0] == ThisMsg_I ) {
	    *NextMsg_PI = -1;
	    return OK;
	}
	
	CALL( squish_MarkUid( XHeader_S.replies[0], MSG_NOP, &Read_B ) );
	if ( NOT Read_B ) {
	    *NextMsg_PI = XHeader_S.replies[0];
	    return OK;
	}
    }
    
    FirstRead_B = TRUE;
    LastMsg_I = 0;

    while ( 1 ) {
	t_Bool  Exists_B;
	t_Bool  Read_B;
	int Reply_I = 0;
	
	if ( LoopCount_I++ > 500 ) {
	    DBG( "FindNextMessage loop escape!", 500 );
	    *NextMsg_PI = -1;
	    return OK;
	}
	
	/* don't read first header twice */
	if ( NOT FirstRead_B )
	    CALL( squish_GetHeader( ThisMsg_I, &XHeader_S ) );
	FirstRead_B = FALSE;
	
	NextMsg_I = 0;
	
	/* if we are coming from below, find next branch */
	if ( LastMsg_I ) {
	    t_Bool  Found_B = FALSE;
	    int i;
	    
	    for ( i = 0; i < (MAX_REPLY-1); i++ ) {
		if ( XHeader_S.replies[ i ] == LastMsg_I ) {
		    NextMsg_I = XHeader_S.replies[ i+1 ];
		    break;
		}
	    }
	    
	    LastMsg_I = 0;
	    
	    /* if we've got a fresh branch, go read it */
	    if ( NextMsg_I ) {
		ThisMsg_I = NextMsg_I;
		continue;
	    }
	}
	/* if we aint rising (just cruising...) fetch first reply */
	else {
	    /* check if message is unread */
	    CALL( squish_MarkUid( ThisMsg_I, MSG_NOP, &Read_B ) );
	    if ( NOT Read_B ) {
		/* we've got a live one. exit. */
		*NextMsg_PI = ThisMsg_I;
		return OK;
	    }
	    
	    /* dive into first reply */
	    NextMsg_I = XHeader_S.replies[0];
	    if ( NextMsg_I ) {
		ThisMsg_I = NextMsg_I;
		continue;
	    }
	}
	
	/* no branches or replies. move up. */
	
	/* see if there is a mail "above" */
	/* if we've got one above, go read it */
	if ( XHeader_S.replyto ) {
	    LastMsg_I = ThisMsg_I;
	    ThisMsg_I = XHeader_S.replyto;
	    continue;
	}
	else {
	    /* no more mail in thread */
	    *NextMsg_PI = -1;
	    return OK;
	}
    }
    
    RET_ERR( "Hey! You're not supposed to be here!", __LINE__ );
    
    return OK;
}


/***************************************************************************
*
*  Function:    SQUISH_CreateMessage
*  Description: Clear message buffer
*
***************************************************************************/
t_RetCode SQUISH_CreateMessage( void )
{
    if ( squish_OutMsg_S.Body_PC )
	free( squish_OutMsg_S.Body_PC );
    squish_OutMsg_S.Body_PC = NULL;
    
    if ( squish_OutMsg_S.Control_PC )
	free( squish_OutMsg_S.Control_PC );
    squish_OutMsg_S.Control_PC = NULL;

    memset( squish_OutMsg_S.Xmsg_S.replies, 0, sizeof( UMSGID ) * 9 );

    squish_OutMsg_S.Xmsg_S.from[0]   = 0;
    squish_OutMsg_S.Xmsg_S.to[0]     = 0;
    squish_OutMsg_S.Xmsg_S.subj[0]   = 0;
    squish_OutMsg_S.Xmsg_S.attr      = 0;
    squish_OutMsg_S.Xmsg_S.replyto   = 0;
    squish_OutMsg_S.Xmsg_S.orig.zone = 0;
    squish_OutMsg_S.Xmsg_S.orig.net  = 0;
    squish_OutMsg_S.Xmsg_S.orig.node = 0;
    squish_OutMsg_S.Xmsg_S.orig.point= 0;
    squish_OutMsg_S.Xmsg_S.dest.zone = 0;
    squish_OutMsg_S.Xmsg_S.dest.net  = 0;
    squish_OutMsg_S.Xmsg_S.dest.node = 0;
    squish_OutMsg_S.Xmsg_S.dest.point= 0;

    return OK;
}

/***************************************************************************
*
*  Function:    SQUISH_SendMessage
*  Description: Post message in message base
*
***************************************************************************/
t_RetCode SQUISH_SendMessage( void )
{
    HMSG    Msg_PS;
    int     Len_I, CLen_I, MsgNo_I, ReplyTo_I;
    time_t  Time_I;
    struct tm*  Time_PS;
    
    if ( !squish_OutMsg_S.Body_PC )
	RET_ERR( "msgsend(): Message lacks body text!", 0 );
    
    /* Set time */
    Time_I = time( NULL );
    Time_PS= localtime( &Time_I );
    
    squish_OutMsg_S.Xmsg_S.date_written.date.yr = Time_PS->tm_year - 80;
    squish_OutMsg_S.Xmsg_S.date_written.date.mo = Time_PS->tm_mon + 1;
    squish_OutMsg_S.Xmsg_S.date_written.date.da = Time_PS->tm_mday;
    squish_OutMsg_S.Xmsg_S.date_written.time.hh = Time_PS->tm_hour;
    squish_OutMsg_S.Xmsg_S.date_written.time.mm = Time_PS->tm_min;

    Msg_PS = MsgOpenMsg( squish_AreaHandle_PS, MOPEN_CREATE, 0 );
    if ( !Msg_PS )
	RET_ERR( "Couldn't create new message!", 0 );
    
    Len_I = strlen( squish_OutMsg_S.Body_PC ) + 1;
    if ( squish_OutMsg_S.Control_PC )
	CLen_I = strlen( squish_OutMsg_S.Control_PC ) + 1;
    else
	CLen_I = 0;
    
    /* Write message */
    if ( MsgWriteMsg(Msg_PS, FALSE, &squish_OutMsg_S.Xmsg_S, squish_OutMsg_S.Body_PC,
		     Len_I, Len_I, CLen_I, squish_OutMsg_S.Control_PC ) )
	RET_ERR( "Error writing message!", 0 );
    
    if ( MsgCloseMsg( Msg_PS ) )
	RET_ERR( "Error closing message!", -1 );
    
    if (( THISMAP->FirstMsg_I != MsgMsgnToUid( squish_AreaHandle_PS, 1 )) ||
	( THISMAP->MsgCount_I != MsgGetNumMsg( squish_AreaHandle_PS )) )
	CALL( SQUISH_RemakeMap() );
    
    /* if this is a reply */
    if ( squish_OutMsg_S.Xmsg_S.replyto ) {
	MsgNo_I = MsgGetHighMsg( squish_AreaHandle_PS );
	ReplyTo_I = MsgUidToMsgn( squish_AreaHandle_PS,
				 squish_OutMsg_S.Xmsg_S.replyto,
				 UID_EXACT );
	CALL( squish_SetReply( ReplyTo_I, MsgNo_I ) );
    }
    
    CALL( SQUISH_CreateMessage() );
    
    return OK;
}


/**************************************************************************
*
*  Function:    SQUISH_PutMessageFlag
*
***************************************************************************/
t_RetCode SQUISH_PutMessageFlag( int Flag_I, int Action_I )
{
    unsigned int Mask_I;

    switch ( Flag_I ) {
      case 0: Mask_I = MSGPRIVATE; break;
      case 1: Mask_I = MSGCRASH; break;
      case 2: Mask_I = MSGREAD; break;
      case 3: Mask_I = MSGSENT; break;
      case 4: Mask_I = MSGFILE; break;
      case 5: Mask_I = MSGFWD; break;
      case 6: Mask_I = MSGORPHAN; break;
      case 7: Mask_I = MSGKILL; break;
      case 8: Mask_I = MSGLOCAL; break;
      case 9: Mask_I = MSGHOLD; break;
      case 10:Mask_I = MSGFRQ; break;
      case 11:Mask_I = MSGRRQ; break;
      case 12:Mask_I = MSGCPT; break;
      case 13:Mask_I = MSGURQ; break;
      case 14:Mask_I = MSGSCANNED; break;
      case 15:Mask_I = MSGUID; break;
      default:
	RET_ERR( "Unknown message flag", Flag_I );
    }

    switch ( Action_I ) {
      case -1:
	if (squish_OutMsg_S.Xmsg_S.attr & Mask_I)
	    squish_OutMsg_S.Xmsg_S.attr ^= Mask_I;
	break;
	
      case 1:
	squish_OutMsg_S.Xmsg_S.attr |= Mask_I;
	break;
	
      default:
	RET_ERR( "Unknown message flag action", Action_I );
    }

    return OK;
}

/***************************************************************************
*
*  Function:    SQUISH_PutMessage
*  Description: Put part of a message into message buffer
*
*               MsgPart_I is one of the following:
*
*  #define    Value Description           Example
*  ------------------------------------------------------------------------
*  M_BODY       0 - Message body filename "c:\\tmp\\node01.msg"
*  M_FROM       1 - From Name             "Joe User"
*  M_TO         2 - To Name               "Sysop"
*  M_SUBJECT    3 - Subject               "How to create a message"
*  M_ORIG       4 - From Address          "39:162/102.0"
*  M_DEST       5 - To Address            "2:201/328.0"
*  M_REPLYTO    7 - Reply to              "14171"
*  M_CONTROL    9 - Control info          "PID: Squish v1.01"
*
*  Flag_I is used for resetting reply counter
*
***************************************************************************/
t_RetCode SQUISH_PutMessage(int MsgPart_I, char* Data_PC )
{
    int   Addr_AI[4], i=0, Len_I;
    char* Tmp_PC;
    char  TmpStr_AC[1024];

    switch ( MsgPart_I ) {
      case M_BODY:
	Len_I = strlen( Data_PC );
	Tmp_PC = (char*) malloc( Len_I );
	if ( !Tmp_PC )
	    RET_ERR( "Out of memory!", Len_I );
	
	strcpy( Tmp_PC, Data_PC );
	
	/* Ugly, but WTH. Replace all LF with CR. */
	for ( i=0; i < Len_I; i++ )
	    if ( Tmp_PC[i] == 0x0a )
		Tmp_PC[i] = 0x0d;
	
	squish_OutMsg_S.Body_PC = Tmp_PC;
	break;
	
      case M_FROM:
	strcpy( squish_OutMsg_S.Xmsg_S.from, Data_PC );
	break;
	
      case M_TO:
	strcpy( squish_OutMsg_S.Xmsg_S.to, Data_PC );
	break;
	
      case M_SUBJECT:
	strcpy( squish_OutMsg_S.Xmsg_S.subj, Data_PC );
	break;
	
      case M_ORIG:
      case M_DEST:
	strcpy( TmpStr_AC, Data_PC );
	Data_PC = TmpStr_AC;
	memset( Addr_AI, 0, 4 * sizeof( int ) );
	for ( Data_PC = strtok( Data_PC, " :,/." );
	     Data_PC && i<4;
	     Data_PC = strtok( NULL, " :,/." ) )
	    Addr_AI[i++] = atoi( Data_PC );
	
	if ( M_ORIG == MsgPart_I ) {
	    squish_OutMsg_S.Xmsg_S.orig.zone = Addr_AI[0];
	    squish_OutMsg_S.Xmsg_S.orig.net  = Addr_AI[1];
	    squish_OutMsg_S.Xmsg_S.orig.node = Addr_AI[2];
	    squish_OutMsg_S.Xmsg_S.orig.point= Addr_AI[3];
	}
	else {
	    squish_OutMsg_S.Xmsg_S.dest.zone = Addr_AI[0];
	    squish_OutMsg_S.Xmsg_S.dest.net  = Addr_AI[1];
	    squish_OutMsg_S.Xmsg_S.dest.node = Addr_AI[2];
	    squish_OutMsg_S.Xmsg_S.dest.point= Addr_AI[3];
	}
	break;
	
      case M_REPLYTO:
	i = MsgMsgnToUid( squish_AreaHandle_PS, atoi( Data_PC ) );
	if ( i )
	    squish_OutMsg_S.Xmsg_S.replyto = i;
	break;
	
      case M_CONTROL:
	i = strlen( Data_PC ) + strlen( "\1PID: " MSG_PID );
	squish_OutMsg_S.Control_PC = malloc( i );
	if ( !squish_OutMsg_S.Control_PC )
	    RET_ERR( "Out of memory!", i );
	strcpy( squish_OutMsg_S.Control_PC, "\1PID: " MSG_PID );
	strcat( squish_OutMsg_S.Control_PC, Data_PC );
	break;
	
      default:
	RET_ERR( "MSG_PutMessage: MsgPart not supported", MsgPart_I );
	break;
    }
    
    return OK;
}

/**************************************************************************
*
*  Function:    SQUISH_GetMessageFlag
*
***************************************************************************/
t_RetCode SQUISH_GetMessageFlag ( int MsgNo_I, int Flag_I, int* Result_PI )
{
    HMSG    Message_PS;
    XMSG    XHeader_S;
    int         Err_I;
    unsigned    Mask_I;

    Message_PS = MsgOpenMsg( squish_AreaHandle_PS, MOPEN_READ, MsgNo_I );
    if ( NOT Message_PS )
	RET_ERR( "MSGAPI ERROR: MsgOpenMsg() failed.", MsgNo_I );
    
    Err_I = MsgReadMsg( Message_PS, &XHeader_S, 0, 0, NULL, 0, NULL );
    if ( -1 == Err_I )
	RET_ERR( "MSGAPI ERROR: MsgReadMsg() failed.", MsgNo_I );
    
    if ( -1 == MsgCloseMsg( Message_PS ) )
	RET_ERR( "MSGAPI ERROR: MsgCloseMsg() failed.", MsgNo_I );
    
    switch ( Flag_I ) {
      case 0: Mask_I = MSGPRIVATE; break;
      case 1: Mask_I = MSGCRASH; break;
      case 2: Mask_I = MSGREAD; break;
      case 3: Mask_I = MSGSENT; break;
      case 4: Mask_I = MSGFILE; break;
      case 5: Mask_I = MSGFWD; break;
      case 6: Mask_I = MSGORPHAN; break;
      case 7: Mask_I = MSGKILL; break;
      case 8: Mask_I = MSGLOCAL; break;
      case 9: Mask_I = MSGHOLD; break;
      case 10:Mask_I = MSGFRQ; break;
      case 11:Mask_I = MSGRRQ; break;
      case 12:Mask_I = MSGCPT; break;
      case 13:Mask_I = MSGURQ; break;
      case 14:Mask_I = MSGSCANNED; break;
      case 15:Mask_I = MSGUID; break;
      default:
	RET_ERR( "Unknown message flag", Flag_I );
    }

    *Result_PI = XHeader_S.attr & Mask_I;
    return OK;
}

/***************************************************************************
*
*  Function:    SQUISH_GetMessage
*  Description: Get part of a message
*
*               MsgPart_I is one of the following:
*
*  #define    Value Description           Example
*  ------------------------------------------------------------------------
*  M_BODY       0 - Message body          "Hi!\n  I just got your..."
*  M_FROM       1 - From Name             "Joe User"
*  M_TO         2 - To Name               "Sysop"
*  M_SUBJECT    3 - Subject               "How to create a message"
*  M_ORIG       4 - From Address          "39:162/102.0"
*  M_DEST       5 - To Address            "2:201/328.0"
*  M_DATE       6 - Date/time of writing  "94-03-10 12:22"
*  M_REPLYTO    7 - Reply to              "171"
*  M_REPLIES    8 - Replies               "172"
*  M_CONTROL    9 - Control info          "PID: Squish v1.01"
*  M_REPYCNT   10 - Number of replies     "4"
*          11 - Unique message id     "37682"
*
*  Flag_I is used for resetting reply counter
*
***************************************************************************/
t_RetCode SQUISH_GetMessage(int MsgNo_I,
			    int MsgPart_I,
			    int Flag_I,
			    char** Text_PPC )
{
    static  int     OldMsgNum_I      = -1;
    static  HAREA   OldAreaHandle_PS = NULL;
    static  t_Bool  HasReadXHeader_B = FALSE;
    static  int LastReply_I      = 0;
    static      int LastReplyMsg_I   = 0;


    int     i;
    HMSG    Message_PS;

    XMSG    XHeader_S;
    PXMSG   XHeader_PS = NULL;
    long    Offset_I = 0;
    dword   Bytes_I  = 0;
    byte*   Text_PC  = NULL;
    long    CBytes_I = 0;
    byte*   CText_PC = NULL;

#if 0
    t_Bool  NeedToOpen_B = ( OldAreaHandle_PS != squish_AreaHandle_PS ) OR
                           ( OldMsgNum_I      != MsgNo_I );
#else
    t_Bool  NeedToOpen_B = FALSE;
#endif
    int     Error_I, ReadBytes_I;
    int     TextSize_I;

    *Text_PPC = NULL;

    if ( (!MsgNo_I) || ( MsgNo_I > THISMAP->MsgCount_I ) ) {
	sprintf( squish_TextBuffer_PC, "<n/%d/a>", MsgNo_I );
	*Text_PPC = squish_TextBuffer_PC;
	return OK;
    }
    
    /* prepare MsgReadMsg() parameters */

    switch ( MsgPart_I ) {
      case M_BODY:
	Bytes_I = MSG_BUFSIZE - 1;
	squish_TextBuffer_PC[0] = 0;
	Text_PC = squish_TextBuffer_PC;
	XHeader_PS = &XHeader_S;
	NeedToOpen_B = TRUE;
	break;

      case M_FROM:
      case M_TO:
      case M_SUBJECT:
      case M_ORIG:
      case M_DEST:
      case M_DATE:
      case M_REPLYTO:
      case M_REPLIES:
      case M_REPLYCNT:
	XHeader_PS = &XHeader_S;
	NeedToOpen_B = TRUE;
	break;

      case M_CONTROL:
	CBytes_I = MSG_BUFSIZE;
	CText_PC = squish_TextBuffer_PC;
	NeedToOpen_B = TRUE;
	break;
	
      case 11:
	NeedToOpen_B = FALSE;
	break;
	
      case -1:
	break;
	
      default:
	RET_ERR( "ERROR: Bad message part no.", MsgPart_I );
    }
    
    /* do we really need to read the message again? */
#if 0
    if ( NeedToOpen_B OR Bytes_I OR
	( ( NOT HasReadXHeader_B ) AND XHeader_PS ) ) {
#else
    if ( 1 ) {
#endif
	Message_PS = MsgOpenMsg( squish_AreaHandle_PS, MOPEN_READ, MsgNo_I );
	if ( NOT Message_PS )
	    RET_ERR( "MSGAPI ERROR: MsgOpenMsg() failed.", MsgNo_I );
	    
	HasReadXHeader_B = FALSE;
	OldAreaHandle_PS = squish_AreaHandle_PS;
	OldMsgNum_I      = MsgNo_I;
	    
#if 0
	printf( "Header:%s %Offs:%d, Bytes:%d, Text:%s, CBytes:%d, CText:%s\n",
	       XHeader_PS?"Yes":"No", Offset_I, Bytes_I,
	       Text_PC?"Yes":"No", CBytes_I, CText_PC?"Yes":"No" );
#endif
	    
	ReadBytes_I = MsgReadMsg( Message_PS, XHeader_PS, Offset_I, Bytes_I,
				     Text_PC, CBytes_I, CText_PC );
	if ( -1 == ReadBytes_I )
	    RET_ERR( "MSGAPI ERROR: MsgReadMsg() failed.", MsgNo_I );
	    
#if 0
	if ( ReadBytes_I < Bytes_I )
	    printf( "Requested %d bytes body, got %d.\n", Bytes_I, ReadBytes_I );
#endif
	    
	if ( XHeader_PS )
	    HasReadXHeader_B = TRUE;
	    
	if ( -1 == MsgCloseMsg( Message_PS ) )
	    RET_ERR( "MSGAPI ERROR: MsgCloseMsg() failed.", MsgNo_I );
	    
    }
	
    switch ( MsgPart_I ) {
      case M_BODY: {
	  char* Pos_PC = strchr( squish_TextBuffer_PC, 0x01 );
	  if ( Pos_PC )
	      Pos_PC[0] = 0;
	      
	  /* Null terminate message at size */
	  squish_TextBuffer_PC[ ReadBytes_I ] = 0;
	      
#if 0
	  /* Strip all LF chars */
	  for ( i = 0; i < ReadBytes_I; i++ )
	      if ( 0x0a == squish_TextBuffer_PC[i] )
		  squish_TextBuffer_PC[i] = 0x20;
#endif
#if 1
	  /* is there an origin line? */
	  Pos_PC = squish_TextBuffer_PC;
	  while ( 1 ) {
	      Pos_PC = strstr( Pos_PC, " * Origin: " );
	      if ( !Pos_PC )
		  break;
		  
	      if ( ( 0x0d == Pos_PC[-1] ) || ( 0x0a == Pos_PC[-1] ) ) {
		  Pos_PC = strchr( Pos_PC, 0x0d );
		  if ( Pos_PC )
		      Pos_PC[1] = 0;
	      }
	      else
		  Pos_PC++;
	  }
#endif
#if 0
	  int len = strlen( squish_TextBuffer_PC );
	  printf( "Message is %d bytes long:\n", len );
	  COM_DumpMem( squish_TextBuffer_PC, len );
#endif
	      
	  break;
      }
	    
      case M_FROM:
	strcpy( squish_TextBuffer_PC, XHeader_S.from );
	break;
	
      case M_TO:
	strcpy( squish_TextBuffer_PC, XHeader_S.to );
	break;
	
      case M_SUBJECT:
	strcpy( squish_TextBuffer_PC, XHeader_S.subj );
	break;
	
      case M_ORIG:
	sprintf( squish_TextBuffer_PC, "%d:%d/%d.%d",
		XHeader_S.orig.zone,
		XHeader_S.orig.net,
		XHeader_S.orig.node,
		XHeader_S.orig.point );
	break;
	
      case M_DEST:
	sprintf( squish_TextBuffer_PC, "%d:%d/%d.%d",
		XHeader_S.dest.zone,
		XHeader_S.dest.net,
		XHeader_S.dest.node,
		XHeader_S.dest.point );
	break;
	
      case M_DATE:
	sprintf( squish_TextBuffer_PC, "%02d-%02d-%02d %02d:%02d",
		XHeader_S.date_written.date.yr + 80,
		XHeader_S.date_written.date.mo,
		XHeader_S.date_written.date.da,
		XHeader_S.date_written.time.hh,
		XHeader_S.date_written.time.mm );
	break;
	    
      case M_REPLYTO: {
	  int Num_I = MsgUidToMsgn( squish_AreaHandle_PS, XHeader_S.replyto,
				   UID_EXACT );
	  if ( Num_I )
	      sprintf( squish_TextBuffer_PC, "%d", Num_I );
	  else
	      squish_TextBuffer_PC[0] = 0;
	  break;
      }
	    
      case M_REPLIES: {
	  int i, Num_I;
	      
#if 0
	  if ( LastReplyMsg_I != MsgNo_I )
	      LastReply_I = 0;
	  else
	      LastReply_I++;
	  
	  if ( Flag_I )
	      LastReply_I = 0;
	  
	  LastReplyMsg_I = MsgNo_I;
	  
	  i = XHeader_S.replies[LastReply_I];
#else
	  i = XHeader_S.replies[Flag_I];
#endif
	  Num_I = MsgUidToMsgn( squish_AreaHandle_PS, i, UID_EXACT );
	  sprintf( squish_TextBuffer_PC, "%d", Num_I );
	  break;
      }
	    
      case M_CONTROL: {
	  int i, y;
	  
	  Message_PS = MsgOpenMsg( squish_AreaHandle_PS, MOPEN_READ, MsgNo_I );
	  y = MsgGetCtrlLen( Message_PS );
	  MsgCloseMsg( Message_PS );
	  
	  if ( -1 == y )
	      RET_ERR( "Error check Control Data length!", 0 );
	  
	  y--;
	  for ( i=0; i<y; i++ )
	      if ( ( 0x01 == squish_TextBuffer_PC[i] ) ||
		  ( 0x00 == squish_TextBuffer_PC[i] ) )
		  squish_TextBuffer_PC[i] = '\n';
	  break;
      }
	
      case M_REPLYCNT:
	for ( i=0; i < MAX_REPLY; i++ )
	    if ( !XHeader_S.replies[i] )
		break;
	
	sprintf( squish_TextBuffer_PC, "%d", i );
	break;
	
      case 11:
	i = MsgMsgnToUid( squish_AreaHandle_PS, MsgNo_I );
	sprintf( squish_TextBuffer_PC, "%d;%d", i,
		MsgUidToMsgn( squish_AreaHandle_PS, i, UID_EXACT ) );
	break;
	
      case -1:
	sprintf( squish_TextBuffer_PC, "%d", System_S.Ypos_I );
	break;
	
      default:
	RET_ERR( "ERROR: Bad message part no.", MsgPart_I );
	
    }
    
    CALL( COM_Import( squish_TextBuffer_PC, System_S.DataCharset_I ) );
    
    *Text_PPC = squish_TextBuffer_PC;
    
    return OK;
}


/***************************************************************************
*
*  Function:    squish_UnlinkMsg
*  Description: Remove any reply links to the message
*
***************************************************************************/
PRIVATE t_RetCode squish_UnlinkMsg( int Msg_I )
{
    HMSG   Msg_PS;
    XMSG   XHeader_S;
    UMSGID Replies_AI[20];
    int    i,j, Reply_I, MsgUid_I;

    Msg_PS = MsgOpenMsg( squish_AreaHandle_PS, MOPEN_RW, Msg_I );
    if ( !Msg_PS )
	RET_ERR( "Error opening message!", Msg_I );

    MsgReadMsg( Msg_PS, &XHeader_S, 0,0,0,0,0 );
    MsgCloseMsg( Msg_PS );

    Reply_I = XHeader_S.replyto;
    if ( !Reply_I )
	return OK;

    MsgUid_I = MsgMsgnToUid( squish_AreaHandle_PS, Msg_I );
    Reply_I  = MsgUidToMsgn( squish_AreaHandle_PS, Reply_I, UID_EXACT );

    Msg_PS = MsgOpenMsg( squish_AreaHandle_PS, MOPEN_RW, Reply_I );
    if ( !Msg_PS )
	RET_ERR( "Error opening message!", Reply_I );

    MsgReadMsg( Msg_PS, &XHeader_S, 0,0,0,0,0 );

    for ( i=0; i<9; i++ ) {
	if ( XHeader_S.replies[i] == MsgUid_I ) {
	    /* Must we *move* the replies? */
	    if ( ( i < 8 ) && XHeader_S.replies[i+1] ) {
		j = i+1;
		memcpy( Replies_AI, &(XHeader_S.replies[j]),
		       (9-j)*sizeof(UMSGID));
		memcpy( &(XHeader_S.replies[i]), Replies_AI,
		       (9-j)*sizeof(UMSGID));
		i=8;
		while( !XHeader_S.replies[i] ) i--;
	    }
	    XHeader_S.replies[i] = 0;
	    break;
	}
    }

    if ( MsgCloseMsg( Msg_PS ) )
	RET_ERR( "Error closing message!", Msg_I );

    return OK;
}

t_Bool SQUISH_NewMail( void )
{
    return ( THISMAP->FirstMsg_I != MsgMsgnToUid( squish_AreaHandle_PS, 1 )) ||
	   ( THISMAP->MsgCount_I != MsgGetNumMsg( squish_AreaHandle_PS ));
}

/***************************************************************************
*
*  Function:    squish_MarkUid
*  Description: Mark a UMSGID as (un)read
*
***************************************************************************/
PRIVATE t_RetCode squish_MarkUid( int Uid_I, int Command_I, t_Bool* Read_PB )
{
    int    MsgNo_I = MsgUidToMsgn( squish_AreaHandle_PS, Uid_I, UID_EXACT );

    CALL( MSG_MarkMessage( MsgNo_I, Command_I, Read_PB ) );

    return OK;
}

/***************************************************************************
*
*  Function:    SQUISH_CloseArea
*  Description: Close current area
*
***************************************************************************/
t_RetCode SQUISH_CloseArea( void )
{
    DBG( "SQUISH_CloseArea", 0 );

    if ( MsgCloseArea( squish_AreaHandle_PS ) )
	RET_ERR( "MSGAPI ERROR: MsgCloseArea failed.", 0 );

    squish_AreaHandle_PS = NULL;

    return OK;
}

/***************************************************************************
*
*  Function:    SQUISH_WriteMapLine
*  Description: Create a one-line map of read messages
*
***************************************************************************/
t_RetCode SQUISH_WriteMapLine( FILE* OutFile_PS )
{
    char*   Line_PC;
    int     Offset_I = 0;
    int     Size_I = 2048;
    int     MsgCount_I  = THISMAP->MsgCount_I;
    int     Msg_I;
    int     Uid_I;
    t_Bool  First_B = TRUE;

    if ( (!THISMAP->Touched_B) && (!THISMAP->Member_B ) )
	return OK;

    fprintf( OutFile_PS, "%s%s: ",
	    THISMAP->Member_B ? "*" : "",
	    msg_ThisArea_PS->Tag_PC );

    for ( Msg_I = 0; Msg_I < MsgCount_I; Msg_I++ ) {

	if ( ! THISMAP->Read_PB[ Msg_I ] )
	    continue;

	/* is it a range? */
	if ( THISMAP->Read_PB[ Msg_I + 1 ] ) {
#if 0
	    Uid_I = MsgMsgnToUid( squish_AreaHandle_PS, Msg_I + 1 );
#else
	    Uid_I = Msg_I + THISMAP->FirstMsg_I;
#endif
	    fprintf( OutFile_PS, "%d-", Uid_I );

	    /* keep going until we find an unread */
	    while ( THISMAP->Read_PB[ Msg_I ] AND ( Msg_I < MsgCount_I ) )
		Msg_I++;

	    /* step back to the last one read */
#if 0
	    if ( Msg_I < MsgCount_I )
#endif
		Msg_I--;
	}
#if 0
	Uid_I = MsgMsgnToUid( squish_AreaHandle_PS, Msg_I + 1 );
#else
	Uid_I = Msg_I + THISMAP->FirstMsg_I;
#endif
	fprintf( OutFile_PS, "%d,", Uid_I );
    }
    
    fprintf( OutFile_PS, "\n" );

    return OK;
}

/***************************************************************************
*
*  Function:    SQUISH_RemakeMap
*  Description: Recreate message map to mirror the world
*
***************************************************************************/
t_RetCode SQUISH_RemakeMap( void )
{
    static t_Bool Map_B = FALSE;
    int     NewFirst_I = MsgMsgnToUid( squish_AreaHandle_PS, 1 );
    int     NewCount_I = MsgGetNumMsg( squish_AreaHandle_PS );
    t_Area* NewArea_PS;
    t_Bool* NewMap_PB;
    t_Bool  Member_B   = THISMAP->Member_B;
    int     Count_I, Diff_I, Pos_I;

    if(!NewCount_I) {
	free( THISMAP->Read_PB );
	THISMAP->Read_PB    = (t_Bool*) malloc( 1 );
	THISMAP->FirstMsg_I = NewFirst_I;
	THISMAP->MsgCount_I = NewCount_I;
	THISMAP->Member_B   = Member_B;
	return OK;
    }

    NewMap_PB = (t_Bool*) malloc( NewCount_I );
    if (!NewMap_PB)
	RET_ERR( "Out of memory!", NewCount_I );
    memset( NewMap_PB, 0, NewCount_I );

    Diff_I = NewFirst_I - THISMAP->FirstMsg_I;

    for ( Count_I = 0; Count_I < THISMAP->MsgCount_I; Count_I++ ) {
	Pos_I = Count_I + Diff_I;
	if ( Pos_I >= NewCount_I )
	    break;
	NewMap_PB[ Count_I ] = THISMAP->Read_PB[ Pos_I ];
    }

    THISMAP->Read_PB    = NewMap_PB;
    THISMAP->FirstMsg_I = NewFirst_I;
    THISMAP->MsgCount_I = NewCount_I;
    THISMAP->Member_B   = Member_B;

    return OK;
}

/***************************************************************************
*
*  Function:    squish_GetHeader
*  Description: Read message header (only) for a message
*               (internal use ONLY!)
*
***************************************************************************/
PRIVATE t_RetCode squish_GetHeader( int Uid_I, XMSG* XHeader_PS )
{
    HMSG    Message_PS;
    int     Error_I;
    int     MsgNo_I;

    CALL( squish_UidToMsg( Uid_I, &MsgNo_I ) );

    if ( 0 == MsgNo_I ) {
    XHeader_PS->replies[0] = 0;
    XHeader_PS->replyto    = 0;
    return OK;
    }

    Message_PS = MsgOpenMsg( squish_AreaHandle_PS, MOPEN_READ, MsgNo_I );
    if ( NOT Message_PS )
    RET_ERR( "MSGAPI ERROR: MsgOpenMsg() failure.", MsgNo_I );

    Error_I = MsgReadMsg( Message_PS, XHeader_PS, 0, 0, 0, 0, 0 );
    if ( -1 == Error_I )
    RET_ERR( "MSGAPI ERROR: MsgReadMsg() failure.", MsgNo_I );

    if ( -1 == MsgCloseMsg( Message_PS ) )
    RET_ERR( "MSGAPI ERROR: MsgCloseMsg() failure.", MsgNo_I );

    return OK;
}


/***************************************************************************
*
*  Function:    squish_UidToMsg
*  Description: Convert UID to msg number, while checking for invalid no
*
***************************************************************************/
PRIVATE t_RetCode squish_UidToMsg( int Uid_I, int* MsgNo_PI )
{
    t_Bool  Exists_B;
    dword   MsgNo_I;

#if 0
    CALL( squish_UidExists( Uid_I, &Exists_B ) );
    if ( NOT Exists_B ) {
    *MsgNo_PI = 0;
    return OK;
    }
#endif

    MsgNo_I = MsgUidToMsgn( squish_AreaHandle_PS, (UMSGID) Uid_I, UID_EXACT );
    *MsgNo_PI = (int) MsgNo_I;

    return OK;
}

/***************************************************************************
*
*  Function:    squish_UidExists
*  Description: Check if UID is an existing message
*
***************************************************************************/
PRIVATE t_RetCode squish_UidExists( int Uid_I, t_Bool* Exists_PB )
{
    int Error_I, MsgNo_I;

    MsgNo_I = MsgUidToMsgn( squish_AreaHandle_PS, (UMSGID) Uid_I, UID_EXACT );
    if ( !MsgNo_I ) {
    *Exists_PB = FALSE;
    return OK;
    }

#if 0
    /* if a too low msg number is specified */
    if ( Uid_I < THISMAP->FirstMsg_I ) {
    *Exists_PB = FALSE;
    return OK;
    }

    /* if a too high msg number is specified */
    if ( Uid_I > ( THISMAP->FirstMsg_I + THISMAP->MsgCount_I ) ) {

    /* check for newly imported mail */
    CALL( MSG_AreaInfo( -1, NULL ) );
    if ( Uid_I > THISMAP->FirstMsg_I + THISMAP->MsgCount_I ) {
        *Exists_PB = FALSE;
        return OK;
    }
    }
#endif

    *Exists_PB = TRUE;

    return OK;
}

/***************************************************************************
*
*  Function:    squish_SetReply
*  Description: Set
*
***************************************************************************/
PRIVATE t_RetCode squish_SetReply( int Msg_I, int Reply_I )
{
    HMSG   Msg_PS;
    XMSG   XHeader_S;
    int    i;

    if ( Msg_I == Reply_I )
	RET_ERR( "A message cannot be a reply to itself!", Msg_I );

    Reply_I = MsgMsgnToUid( squish_AreaHandle_PS, Reply_I );

    Msg_PS = MsgOpenMsg( squish_AreaHandle_PS, MOPEN_RW, Msg_I );
    if ( !Msg_PS )
	RET_ERR( "Error opening message!", Msg_I );

    MsgReadMsg( Msg_PS, &XHeader_S, 0,0,0,0,0 );

    for ( i=0; i<9; i++ ) {
	if ( !XHeader_S.replies[i] ) {
	    XHeader_S.replies[i] = Reply_I;
	    break;
	}
    }

    /* if reply array isn't full */
    if ( i < 9 ) {
	if ( MsgWriteMsg( Msg_PS, FALSE, &XHeader_S, 0,0,0,0,0 ) )
	    RET_ERR( "Error writing message header!", 0 );
    }

    if ( MsgCloseMsg( Msg_PS ) )
	RET_ERR( "Error closing message!", Msg_I );

    return OK;
}

/***************************************************************************
*
*  Function:    squish_CloseArea
*  Description: Close current area
*
***************************************************************************/
PRIVATE t_RetCode squish_CloseArea( void )
{
    CALL( SQUISH_CloseArea() );
    return OK;
}


