/***************************************************************************
*
*  PRIVATE #includes
*
***************************************************************************/
#define INCL_DOSFILEMGR
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "stdcode.h"
#include "msgapi.h"
#include "comm.h"
#include "system.h"
#include "mystring.h"
#include "files.h"

#include "message.h"

/***************************************************************************
*
*  PRIVATE #defines
*
***************************************************************************/

#define MSG_BUFSIZE	65536
#define M_BODY		0
#define M_FROM		1
#define M_TO		2
#define M_SUBJECT	3
#define M_ORIG		4
#define M_DEST		5
#define M_DATE		6
#define M_REPLYTO	7
#define M_REPLIES	8
#define M_CONTROL	9
#define M_REPLYCNT 	10

#define A_FIRST		0
#define A_TOTAL		1
#define A_UNREAD	2

#define MSG_MAPPATH	"msgmap\\"

/***************************************************************************
*
*  PRIVATE typedefs
*
***************************************************************************/

typedef struct {
    UMSGID		FirstMsg_I;
    ulong		MsgCount_I;
    t_Bool*		Read_PB;
} t_AreaMap;

typedef struct {
    XMSG	Xmsg_S;
    char*	Body_PC;
} t_Message;

/***************************************************************************
*
*  PRIVATE variables
*
***************************************************************************/

	HAREA		msg_AreaHandle_PS	= NULL;
PRIVATE byte* 		msg_TextBuffer_PC	= NULL;
	t_AreaMap	msg_AreaMap_S		= { 0, 0, NULL };
PRIVATE	char		msg_Tagname_AC[80];
	t_Message	msg_OutMsg_S;

/***************************************************************************
*
*  PRIVATE functions
*
***************************************************************************/

PRIVATE t_RetCode msg_OpenArea	( char* Name_PC );
PRIVATE t_RetCode msg_CloseArea	( void );
PRIVATE t_RetCode msg_ReadMap	( void );
PRIVATE t_RetCode msg_MakeMap	( FILE* OutFile_PS );
PRIVATE t_RetCode msg_GetHeader	( int MsgNo_I, XMSG* XHeader_PS );
#if 0
PRIVATE t_RetCode msg_UidToMsg	( int Uid_I, int* Msg_PI );
PRIVATE t_RetCode msg_UidExists	( int Uid_I, t_Bool* Exists_PB );
#endif
PRIVATE t_RetCode msg_LoadFile  ( char*, char** );


/*==========================================================================
=
=  GLOBAL functions
=
===========================================================================*/

/***************************************************************************
*
*  Function:	MSG_Init
*  Description: Initialize MSGAPI.DLL
*
***************************************************************************/
t_RetCode MSG_Init( void )
{
    struct _minf minf={1,0,0,NULL,NULL,NULL,NULL,NULL,NULL};

    if ( -1 == MsgOpenApi( &minf ) )
        RET_ERR( "MSGAPI ERROR: MsgOpenApi() failed.", 0 );

	msg_TextBuffer_PC = (byte*) malloc( MSG_BUFSIZE );
	if ( NOT msg_TextBuffer_PC )
		RET_ERR( "ERROR: Out of memory!", 0 );

    return OK;
}

/***************************************************************************
*
*  Function:	MSG_Exit
*  Description: Deinitialize MSGAPI.DLL
*
***************************************************************************/
t_RetCode MSG_Exit( void )
{
    if ( msg_TextBuffer_PC )
	free( msg_TextBuffer_PC );

    if ( msg_OutMsg_S.Body_PC )
	free( msg_OutMsg_S.Body_PC );


    if ( msg_AreaMap_S.Read_PB ) {
	CALL( MSG_WriteMap() );
	free( msg_AreaMap_S.Read_PB );
	CALL( msg_CloseArea() );
    }

    if ( -1 == MsgCloseApi() )
        RET_ERR( "MSGAPI ERROR: MsgCloseApi() failed.", 0 );

    return OK;
}

/***************************************************************************
*
*  Function:	MSG_GotoArea
*  Description: Close current area and open new area
*
***************************************************************************/
t_RetCode MSG_GotoArea( char* Filename_PC, char* Tagname_PC )
{
    int Status_I;

    /* close old area */
    if ( msg_AreaHandle_PS )
	CALL( msg_CloseArea() );

    /* copy area name */
    strncpy( msg_Tagname_AC, Tagname_PC, 80 );

    CALL( msg_OpenArea( Filename_PC ) );

    return OK;
}

/***************************************************************************
*
*  Function:	MSG_AreaInfo
*  Description: Get info about current message area
*
*				AreaInfo_I is one of the following
*
*  #define    Value Description           Example
*  ------------------------------------------------------------------------
*  A_FIRST      0 - Number of first msg   3254
*  A_TOTAL      1 - Total number of msgs  400
*  A_UNREAD	2 - Number of unread msgs 113
*  A_????	3 - Highest message	  6500
*
****************************************************************************/
t_RetCode MSG_AreaInfo( int AreaInfo_I, int* Result_PI )
{
    int Int_I;
    int i,y;

    msg_AreaMap_S.FirstMsg_I = 1;
    msg_AreaMap_S.MsgCount_I = MsgGetNumMsg( msg_AreaHandle_PS );
    i = MsgGetHighMsg( msg_AreaHandle_PS );

    switch ( AreaInfo_I ) {
	    
      case A_FIRST:
	*Result_PI = 1;
	break;

      case A_TOTAL:
	*Result_PI = msg_AreaMap_S.MsgCount_I;
	break;
	
      case A_UNREAD:
	y=0;
	for ( i = 0; i < msg_AreaMap_S.MsgCount_I; i++ )
	    if ( NOT msg_AreaMap_S.Read_PB[i] )
		y++;
	*Result_PI = y;
	break;
	
      case 3:
	*Result_PI = msg_AreaMap_S.MsgCount_I;
	break;

      default:
	RET_ERR( "Bad area info nr", AreaInfo_I );
    }

    return OK;
}


/***************************************************************************
*
*  Function:	MSG_WriteMessage
*  Description: Write message to file
*
***************************************************************************/
t_RetCode MSG_WriteMessage( int MsgNo_I, char* Filename_PC )
{
    char*   Brace_PC;
    char*   NextParagraph_PC;
    char    LineFeed_AC[15] = "\n";
    char    LFLength_I = strlen( LineFeed_AC );
    t_Bool  Quote_B = FALSE;
    FILE*   File_PS;
    char*   String_PC;
    int	    Len_I, i;

    File_PS = fopen( Filename_PC, "w" );
    if ( !File_PS )
	RET_ERR( "Can't create file!", errno );

    CALL( MSG_GetMessage( MsgNo_I, 0, 0, &String_PC ) );

    for ( NextParagraph_PC = strchr( String_PC, '\r' );
	  NextParagraph_PC;
	  NextParagraph_PC = strchr( String_PC, '\r' ) ) {
	NextParagraph_PC[0] = 0;

	/* strip control chars */
	Len_I = strlen( String_PC ) - 1;
	for ( i=0; i < Len_I; i++ )
		if ( ( 32 > String_PC[i] ) OR ( String_PC[i] == 0x8d ) )
			String_PC[i] = 32;

	Brace_PC = strchr( String_PC, '>' );
	if ( Brace_PC )
	    Quote_B = ( ( Brace_PC - String_PC ) < 10 );
	else
	    Quote_B = FALSE;
	
	fprintf( File_PS, "%s\n", String_PC );

	if ( 0 == strncmp( " * Origin:", String_PC, 10 ) )
	    break;

	String_PC = NextParagraph_PC + 1;
    }

    fclose( File_PS );

    return OK;
}


/***************************************************************************
*
*  Function:	MSG_MarkMessage
*  Description: Mark a message as (un)read
*
***************************************************************************/
t_RetCode MSG_MarkMessage( int MsgNo_I, int Command_I, t_Bool* Read_PB )
{
    int Offset_I = MsgNo_I - msg_AreaMap_S.FirstMsg_I;

    if ( ( Offset_I > msg_AreaMap_S.FirstMsg_I + msg_AreaMap_S.MsgCount_I ) OR
	( Offset_I < 0 ) )
	return OK;

    if ( MSG_NOP != Command_I )
	msg_AreaMap_S.Read_PB[ Offset_I ] =
	    MSG_MARK == Command_I ? TRUE : FALSE;

    *Read_PB = msg_AreaMap_S.Read_PB[ Offset_I ];

    return OK;
}


/***************************************************************************
*
*  Function:	MSG_FetchNextMessage
*  Description: MSGNO (!uid) shell for NextMessage
*
***************************************************************************/
t_RetCode MSG_FetchNextMessage( int ThisMsg_I, int* NextMsg_PI )
{
    int Num_I, Num2_I;

    Num_I = MsgUidToMsgn( msg_AreaHandle_PS, ThisMsg_I, UID_EXACT );
    CALL( MSG_NextMessage( Num_I, &Num2_I ) );

    *NextMsg_PI = Num2_I;

    return OK;
}

/***************************************************************************
*
*  Function:	MSG_NextMessage
*  Description: Find the logical "next message" in message reply tree
*
***************************************************************************/
t_RetCode MSG_NextMessage( int ThisMsg_I, int* NextMsg_PI )
{
    XMSG	XHeader_S;
    int		LastMsg_I;
    int		NextMsg_I;
    t_Bool	FirstRead_B;

    if ( 0 == ThisMsg_I ) {
	*NextMsg_PI = 0;
	return OK;
    }

    if ( -1 == ThisMsg_I ) {
	int i;
	
	NextMsg_I = 0;
	/* desperately, find ANY unread message in area! */
	for ( i=0; i < msg_AreaMap_S.MsgCount_I; i++ )
	    if ( NOT msg_AreaMap_S.Read_PB[ i ] ) {
		NextMsg_I = i + msg_AreaMap_S.FirstMsg_I;
		break;
	    }
	
	/* we've got a live one, or maybe we don't.
	   exit nevertheless. */
	*NextMsg_PI = NextMsg_I;
	return OK;
    }

    /* do a quickie to see if there's a first comment unread */
    CALL( msg_GetHeader( ThisMsg_I, &XHeader_S ) );
    if ( XHeader_S.replies[0] ) {
	t_Bool	Read_B = FALSE;
	CALL( MSG_MarkMessage( XHeader_S.replies[0], MSG_NOP, &Read_B ) );
	if ( NOT Read_B ) {
	    *NextMsg_PI = XHeader_S.replies[0];
	    return OK;
	}
    }
    
    FirstRead_B = TRUE;
    LastMsg_I = 0;
    
    while ( 1 ) {
	t_Bool	Exists_B;
	t_Bool	Read_B;
	int		Reply_I = 0;
	
	/* don't read first header twice */
	if ( NOT FirstRead_B )
	    CALL( msg_GetHeader( ThisMsg_I, &XHeader_S ) );
	FirstRead_B = FALSE;
	
	NextMsg_I = 0;
	
	/* if we are coming from below, find next branch */
	if ( LastMsg_I ) {
	    t_Bool	Found_B = FALSE;
	    int i;
	    
	    for ( i = 0; i < (MAX_REPLY-1); i++ ) {
		if ( XHeader_S.replies[ i ] == LastMsg_I ) {
		    NextMsg_I = XHeader_S.replies[ i+1 ];
		    break;
		}
	    }
	    
	    LastMsg_I = 0;
	    
	    /* if we've got a fresh branch, go read it */
	    if ( NextMsg_I ) {
		ThisMsg_I = NextMsg_I;
		continue;
	    }
	}
	/* if we aint rising (just cruising...) fetch first reply */
	else {
	    /* check if message is unread */
	    CALL( MSG_MarkMessage( ThisMsg_I, MSG_NOP, &Read_B ) );
	    if ( NOT Read_B ) {
		/* we've got a live one. exit. */
		*NextMsg_PI = ThisMsg_I;
		return OK;
	    }
	    
	    /* dive into first reply */
	    NextMsg_I = XHeader_S.replies[0];
	    if ( NextMsg_I ) {
		ThisMsg_I = NextMsg_I;
		continue;
	    }
	}
	
	/* no branches or replies. move up. */
	
#if 0
	/* see if there is a mail "above" */
	if ( XHeader_S.replyto )
	    CALL( msg_UidExists( XHeader_S.replyto, &Exists_B ) );
	    Exists_B = TRUE;
	else
	    Exists_B = FALSE;
#else
	Exists_B = XHeader_S.replyto == 0 ? FALSE : TRUE;
#endif
	
	/* if we've got one above, go read it */
	if ( Exists_B ) {
	    LastMsg_I = ThisMsg_I;
	    ThisMsg_I = XHeader_S.replyto;
	    continue;
	}
	else {
	    /* no more mail in thread */
	    *NextMsg_PI = -1;
	    return OK;
	}
    }
    
    RET_ERR( "Hey! You're not supposed to be here!", __LINE__ );
    
    return OK;
}

/***************************************************************************
*
*  Function:	MSG_CreateMessage
*  Description: Clear message buffer
*
***************************************************************************/
t_RetCode MSG_CreateMessage( void )
{
    if ( msg_OutMsg_S.Body_PC )
	free( msg_OutMsg_S.Body_PC );
    msg_OutMsg_S.Body_PC = NULL;

    msg_OutMsg_S.Xmsg_S.from[0]   = 0;
    msg_OutMsg_S.Xmsg_S.to[0]     = 0;
    msg_OutMsg_S.Xmsg_S.subj[0]   = 0;
    msg_OutMsg_S.Xmsg_S.orig.zone = 0;
    msg_OutMsg_S.Xmsg_S.orig.net  = 0;
    msg_OutMsg_S.Xmsg_S.orig.node = 0;
    msg_OutMsg_S.Xmsg_S.orig.point= 0;
    msg_OutMsg_S.Xmsg_S.dest.zone = 0;
    msg_OutMsg_S.Xmsg_S.dest.net  = 0;
    msg_OutMsg_S.Xmsg_S.dest.node = 0;
    msg_OutMsg_S.Xmsg_S.dest.point= 0;

    return OK;
}

/***************************************************************************
*
*  Function:	MSG_SendMessage
*  Description: Post message in message base
*
***************************************************************************/
t_RetCode MSG_SendMessage( void )
{
    HMSG  Msg_PS;
    int   Len_I;

    if ( !msg_OutMsg_S.Body_PC )
	RET_ERR( "msgsend(): Message lacks body text!", 0 );

#if 0
    printf( "---Message begin---\n" );
    if ( msg_OutMsg_S.Body_PC )
	printf( "Body: 0x%08x '%-30s'\n",
	        msg_OutMsg_S.Body_PC, msg_OutMsg_S.Body_PC );

    if ( msg_OutMsg_S.Xmsg_S.from[0] )
	printf( "From: %s\n", msg_OutMsg_S.Xmsg_S.from );

    if ( msg_OutMsg_S.Xmsg_S.from[0] )
	printf( "To  : %s\n", msg_OutMsg_S.Xmsg_S.to );

    if ( msg_OutMsg_S.Xmsg_S.from[0] )
	printf( "Subj: %s\n", msg_OutMsg_S.Xmsg_S.subj );

    if ( msg_OutMsg_S.Xmsg_S.orig.zone )
	printf( "Orig: %d:%d/%d.%d\n",
	       msg_OutMsg_S.Xmsg_S.orig.zone,
	       msg_OutMsg_S.Xmsg_S.orig.net,
	       msg_OutMsg_S.Xmsg_S.orig.node,
	       msg_OutMsg_S.Xmsg_S.orig.point );

    if ( msg_OutMsg_S.Xmsg_S.dest.zone )
	printf( "Dest: %d:%d/%d.%d\n",
	       msg_OutMsg_S.Xmsg_S.dest.zone,
	       msg_OutMsg_S.Xmsg_S.dest.net,
	       msg_OutMsg_S.Xmsg_S.dest.node,
	       msg_OutMsg_S.Xmsg_S.dest.point );

    printf( "---Message end---\n" );

#else
    Msg_PS = MsgOpenMsg( msg_AreaHandle_PS, MOPEN_CREATE, 0 );
    if ( !Msg_PS )
	RET_ERR( "Couldn't create new message!", 0 );

    Len_I = strlen( msg_OutMsg_S.Body_PC );

    if ( MsgWriteMsg(Msg_PS, 0, &msg_OutMsg_S.Xmsg_S, msg_OutMsg_S.Body_PC,
		     Len_I, Len_I, 0, 0 ) )
	RET_ERR( "Error writing message!", 0 );
#endif

    CALL( MSG_CreateMessage() );

    return OK;
}

/***************************************************************************
*
*  Function:	MSG_PutMessage
*  Description: Put part of a message into message buffer
*
*               MsgPart_I is one of the following:
*
*  #define    Value Description           Example
*  ------------------------------------------------------------------------
*  M_BODY       0 - Message body filename "c:\\tmp\\node01.msg"
*  M_FROM       1 - From Name             "Joe User"
*  M_TO         2 - To Name               "Sysop"
*  M_SUBJECT    3 - Subject               "How to create a message"
*  M_ORIG       4 - From Address          "39:162/102.0"
*  M_DEST       5 - To Address            "2:201/328.0"
*  M_REPLYTO    7 - Reply to              "14171"
*  M_CONTROL    9 - Control info          "PID: Squish v1.01"
*
*  Flag_I is used for resetting reply counter
*
***************************************************************************/
t_RetCode MSG_PutMessage(int MsgPart_I, char* Data_PC )
{
    int	Addr_AI[4], i=0;

    memset( Addr_AI, 0, 4 * sizeof( int ) );

    switch ( MsgPart_I ) {
      case M_BODY:
	CALL( msg_LoadFile( Data_PC, &msg_OutMsg_S.Body_PC ) );
	break;

      case M_FROM:
	strcpy( msg_OutMsg_S.Xmsg_S.from, Data_PC );
	break;

      case M_TO:
	strcpy( msg_OutMsg_S.Xmsg_S.to, Data_PC );
	break;

      case M_SUBJECT:
	strcpy( msg_OutMsg_S.Xmsg_S.subj, Data_PC );
	break;

      case M_ORIG:
      case M_DEST:
	for ( Data_PC = strtok( Data_PC, " :,/." );
	      Data_PC && i<4;
	      Data_PC = strtok( NULL, " :,/." ) )
	    Addr_AI[i++] = atoi( Data_PC );

	if ( M_ORIG == MsgPart_I ) {
	    msg_OutMsg_S.Xmsg_S.orig.zone = Addr_AI[0];
	    msg_OutMsg_S.Xmsg_S.orig.net  = Addr_AI[1];
	    msg_OutMsg_S.Xmsg_S.orig.node = Addr_AI[2];
	    msg_OutMsg_S.Xmsg_S.orig.point= Addr_AI[3];
	}
	else {
	    msg_OutMsg_S.Xmsg_S.dest.zone = Addr_AI[0];
	    msg_OutMsg_S.Xmsg_S.dest.net  = Addr_AI[1];
	    msg_OutMsg_S.Xmsg_S.dest.node = Addr_AI[2];
	    msg_OutMsg_S.Xmsg_S.dest.point= Addr_AI[3];
	}
	break;

      case M_REPLYTO:
	break;

      case M_CONTROL:
	break;

      default:
	RET_ERR( "MSG_PutMessage: MsgPart not supported", MsgPart_I );
	break;
    }

    return OK;
}

/***************************************************************************
*
*  Function:	MSG_GetMessage
*  Description: Get part of a message
*
*               MsgPart_I is one of the following:
*
*  #define    Value Description           Example
*  ------------------------------------------------------------------------
*  M_BODY       0 - Message body          "Hi!\n  I just got your..."
*  M_FROM       1 - From Name             "Joe User"
*  M_TO         2 - To Name               "Sysop"
*  M_SUBJECT    3 - Subject               "How to create a message"
*  M_ORIG       4 - From Address          "39:162/102.0"
*  M_DEST       5 - To Address            "2:201/328.0"
*  M_DATE       6 - Date/time of writing  "94-03-10 12:22"
*  M_REPLYTO    7 - Reply to              "171"
*  M_REPLIES    8 - Replies               "172"
*  M_CONTROL    9 - Control info          "PID: Squish v1.01"
*  M_REPYCNT   10 - Number of replies     "4"
*
*  Flag_I is used for resetting reply counter
*
***************************************************************************/
t_RetCode MSG_GetMessage(int Uid_I,
			 int MsgPart_I, 
			 int Flag_I, 
			 char** Text_PPC )
{
    static  int		OldMsgNum_I  	 = -1;
    static  HAREA	OldAreaHandle_PS = NULL;
    static  t_Bool	HasReadXHeader_B = FALSE;
    static	int	LastReply_I      = 0;
    static     	int	LastReplyMsg_I   = 0;

    HMSG 	Message_PS;

    XMSG	XHeader_S;
    PXMSG 	XHeader_PS = NULL;
    long	Offset_I = 0;
    dword	Bytes_I  = 0;
    byte*	Text_PC  = NULL;
    long	CBytes_I = 0;
    byte*	CText_PC = NULL;

    t_Bool	NeedToOpen_B = ( OldAreaHandle_PS != msg_AreaHandle_PS ) OR
	                       ( OldMsgNum_I      != Uid_I );
    int		Error_I, ReadBytes_I;
    int		TextSize_I;
    int		MsgNum_I;

#if 0
    sprintf( msg_TextBuffer_PC, "%d", MsgPart_I );
    *Text_PPC = msg_TextBuffer_PC;
    return OK;
#endif

    *Text_PPC = NULL;

#ifdef USE_UID
    CALL( msg_UidToMsg( Uid_I, &MsgNum_I ) );

    if ( 0 == MsgNum_I ) {
	strcpy( msg_TextBuffer_PC, "<n/a>" );
	*Text_PPC = msg_TextBuffer_PC;
	return OK;
    }
#else
    MsgNum_I = Uid_I;

    if ( !MsgNum_I || ( MsgNum_I > msg_AreaMap_S.MsgCount_I ) ) {
	sprintf( msg_TextBuffer_PC, "<n/%d/a>", MsgNum_I );
	*Text_PPC = msg_TextBuffer_PC;
	return OK;
    }
#endif

    /* prepare MsgReadMsg() parameters */

    switch ( MsgPart_I ) {
      case M_BODY:
	Bytes_I = MSG_BUFSIZE - 1;
	msg_TextBuffer_PC[0] = 0;
	Text_PC = msg_TextBuffer_PC;
	XHeader_PS = &XHeader_S;
	break;

      case M_FROM:
      case M_TO:
      case M_SUBJECT:
      case M_ORIG:
      case M_DEST:
      case M_DATE:
      case M_REPLYTO:
      case M_REPLIES:
      case M_REPLYCNT:
	XHeader_PS = &XHeader_S;
	break;
	
      case M_CONTROL:
	CBytes_I = MSG_BUFSIZE;
	CText_PC = msg_TextBuffer_PC;
	break;
	
      case -1:
	break;
	
      default:
	RET_ERR( "ERROR: Bad message part no.", MsgPart_I );
    }

    /* do we really need to read the message again? */
#if 0
    if ( NeedToOpen_B OR Bytes_I OR
	( ( NOT HasReadXHeader_B ) AND XHeader_PS ) ) {
#else
    if ( 1 ) {
#endif
	Message_PS = MsgOpenMsg( msg_AreaHandle_PS, MOPEN_READ, MsgNum_I );
	if ( NOT Message_PS )
	    RET_ERR( "MSGAPI ERROR: MsgOpenMsg() failed.", MsgNum_I );

	HasReadXHeader_B = FALSE;
	OldAreaHandle_PS = msg_AreaHandle_PS;
	OldMsgNum_I      = MsgNum_I;

#if 0
	printf( "Header:%s %Offs:%d, Bytes:%d, Text:%s, CBytes:%d, CText:%s\n",
	       XHeader_PS?"Yes":"No", Offset_I, Bytes_I,
	       Text_PC?"Yes":"No", CBytes_I, CText_PC?"Yes":"No" );
#endif

	ReadBytes_I = MsgReadMsg( Message_PS, XHeader_PS, Offset_I, Bytes_I, 
				  Text_PC, CBytes_I, CText_PC );
	if ( -1 == ReadBytes_I )
	    RET_ERR( "MSGAPI ERROR: MsgReadMsg() failed.", MsgNum_I );

#if 0
	if ( ReadBytes_I < Bytes_I )
	    printf( "Requested %d bytes body, got %d.\n", Bytes_I, ReadBytes_I );
#endif

	if ( XHeader_PS )
	    HasReadXHeader_B = TRUE;

	if ( -1 == MsgCloseMsg( Message_PS ) )
	    RET_ERR( "MSGAPI ERROR: MsgCloseMsg() failed.", MsgNum_I );
	
    }

    switch ( MsgPart_I ) {
      case M_BODY: {
	  char* Pos_PC = strchr( msg_TextBuffer_PC, 0x01 );
	  if ( Pos_PC )
	      Pos_PC[0] = 0;
#if 0
	  int len = strlen( msg_TextBuffer_PC );
	  printf( "Message is %d bytes long:\n", len );
	  COM_DumpMem( msg_TextBuffer_PC, len );
#endif	
	  
	  break;
      }
	
      case M_FROM:
	strcpy( msg_TextBuffer_PC, XHeader_S.from );
	break;
	
      case M_TO:
	strcpy( msg_TextBuffer_PC, XHeader_S.to );
	break;

      case M_SUBJECT:
	strcpy( msg_TextBuffer_PC, XHeader_S.subj );
	break;
	
      case M_ORIG:
	sprintf( msg_TextBuffer_PC, "%d:%d/%d.%d",
		XHeader_S.orig.zone,
		XHeader_S.orig.net,
		XHeader_S.orig.node,
		XHeader_S.orig.point );
	break;
	
      case M_DEST:
	sprintf( msg_TextBuffer_PC, "%d:%d/%d.%d",
		XHeader_S.dest.zone,
		XHeader_S.dest.net,
		XHeader_S.dest.node,
		XHeader_S.dest.point );
	break;
	
      case M_DATE:
	sprintf( msg_TextBuffer_PC, "%02d-%02d-%02d %02d:%02d",
		XHeader_S.date_written.date.yr + 80,
		XHeader_S.date_written.date.mo,
		XHeader_S.date_written.date.da,
		XHeader_S.date_written.time.hh,
		XHeader_S.date_written.time.mm );
	break;
	
      case M_REPLYTO: {
	  int Num_I = MsgUidToMsgn( msg_AreaHandle_PS, XHeader_S.replyto, 
				    UID_EXACT );
	  if ( Num_I )
	      sprintf( msg_TextBuffer_PC, "%d", Num_I );
	  else
	      msg_TextBuffer_PC[0] = 0;
	  break;
      }
	
      case M_REPLIES: {
	  int		i, Num_I;
#if 1
	  if ( LastReplyMsg_I != MsgNum_I )
	      LastReply_I = 0;
	  else
	      LastReply_I++;
	  
	  if ( Flag_I )
	      LastReply_I = 0;
	  
	  LastReplyMsg_I = MsgNum_I;
	  
	  i = XHeader_S.replies[LastReply_I];
	  Num_I = MsgUidToMsgn( msg_AreaHandle_PS, i, UID_EXACT );

	  sprintf( msg_TextBuffer_PC, "%d", Num_I );
#else
	  char	TmpBuf_AC[10];
	  
	  msg_TextBuffer_PC[0] = 0;
	  
	  /* do it twice to make the commas look nice */
	  Num_I = XHeader_S.replies[0];
	  if ( 0 == Num_I )
	      break;
	  
	  sprintf( TmpBuf_AC, "%d", Num_I );
	  strcpy( msg_TextBuffer_PC, TmpBuf_AC );
	  
	  for ( i = 1; i < MAX_REPLY; i++ ) {
	      if ( 0 == XHeader_S.replies[i] )
		  break;
	      
	      sprintf( TmpBuf_AC, ", %d", XHeader_S.replies[i] );
	      strcat( msg_TextBuffer_PC, TmpBuf_AC );
	  }
#endif
	  break;
      }

      case M_CONTROL: {
	  int i=0;
	  
	  for ( i=0; msg_TextBuffer_PC[i]; i++ )
	      if ( 0x01 == msg_TextBuffer_PC[i] )
		  msg_TextBuffer_PC[i] = '\n';
	  break;
      }
	
      case M_REPLYCNT: {
	  int i;
	  for ( i=0; i < MAX_REPLY; i++ )
	      if ( NOT XHeader_S.replies[i] )
		  break;
	  
	  sprintf( msg_TextBuffer_PC, "%d", i );
	  break;
      }
	
      case -1:
	sprintf( msg_TextBuffer_PC, "%d", System_S.Ypos_I );
	break;
	
      default:
	RET_ERR( "ERROR: Bad message part no.", MsgPart_I );
	
    }
    
    CALL( COM_Import( msg_TextBuffer_PC, System_S.DataCharset_I ) );
    
    *Text_PPC = msg_TextBuffer_PC;
    
    return OK;
}


/***************************************************************************
*
*  Function:	MSG_WriteMap
*  Description: Write map of read messages in current area
*
***************************************************************************/
t_RetCode MSG_WriteMap( void )
{
    FILE*	InFile_PS;
    FILE*	OutFile_PS;
    char	Filename1_AC[128];
    char	Filename2_AC[128];
    char*	Line_PC;
    int		Size_I = 2048;
    int		TagLength_I = strlen( msg_Tagname_AC );
    t_Bool	TagFound_B = FALSE;
    t_Bool	InputFile_B = FALSE;
    
    /* create input map filename */
    sprintf( Filename1_AC, MSG_MAPPATH "%d", System_S.UserNum_I );
    InFile_PS = fopen( Filename1_AC, "r" );
    if ( InFile_PS )
	InputFile_B = TRUE;
    
    /* create output map filename */
    sprintf( Filename2_AC, MSG_MAPPATH "%d.new", System_S.UserNum_I );
    OutFile_PS = fopen( Filename2_AC, "w" );
    if ( NOT OutFile_PS )
	RET_ERR( "Can't create new user message map file!", errno );
    
    /* get memory for line */
    Line_PC = (char*) malloc( Size_I );
    if ( NOT Line_PC )
	RET_ERR( "Out of memory!", Size_I );
    
    if ( InputFile_B ) {
	t_Bool	More_B;
	
	/* copy map file until tagname */
	CALL( STR_ReadLine( InFile_PS, &Line_PC, Size_I, &More_B ) );
	while ( More_B ) {
	    if ( 0 == strncmp( Line_PC, msg_Tagname_AC, TagLength_I ) ) {
		TagFound_B = TRUE;
		break;
	    }
	    
	    if ( EOF == fputs( Line_PC, OutFile_PS ) )
		RET_ERR( "Error writing map file!", errno );
	    
	    CALL( STR_ReadLine( InFile_PS, &Line_PC, Size_I, &More_B ) );
	};
	
	CALL( msg_MakeMap( OutFile_PS ) );
	
	if ( TagFound_B ) {
	    
	    /* copy rest of map file */
	    CALL( STR_ReadLine( InFile_PS, &Line_PC, Size_I, &More_B ) );
	    while ( More_B ) {
			    
		if ( EOF == fputs( Line_PC, OutFile_PS ) )
		    RET_ERR( "Error writing map file!", errno );
		
		CALL( STR_ReadLine( InFile_PS, &Line_PC, Size_I, &More_B ) );
	    }
	}
    }
    else
	CALL( msg_MakeMap( OutFile_PS ) );
    
    if ( InputFile_B )
	fclose( InFile_PS );
    fclose( OutFile_PS );
    
    if ( InputFile_B ) {
	
	/* make a backup */
	sprintf( Filename1_AC, MSG_MAPPATH "%d", System_S.UserNum_I );
	sprintf( Filename2_AC, MSG_MAPPATH "%d.bak", System_S.UserNum_I );
	rename( Filename1_AC, Filename2_AC );
    }
    
    /* rename new master file */
    sprintf( Filename1_AC, MSG_MAPPATH "%d.new", System_S.UserNum_I );
    sprintf( Filename2_AC, MSG_MAPPATH "%d", System_S.UserNum_I );
    if ( NOT rename( Filename1_AC, Filename2_AC ) AND InputFile_B ) {
	/* remove backup */
	sprintf( Filename1_AC, MSG_MAPPATH "%d.bak", System_S.UserNum_I );
	remove( Filename1_AC );
    }
    
    return OK;
}


/*==========================================================================
=
=  LOCAL functions
=
===========================================================================*/
    
/***************************************************************************
*
*  Function:	msg_OpenArea
*  Description: Open a new area
*
***************************************************************************/
PRIVATE t_RetCode msg_OpenArea( char* Name_PC )
{
    msg_AreaHandle_PS = MsgOpenArea( Name_PC,
				    MSGAREA_NORMAL,
				    MSGTYPE_SQUISH );

    if ( NOT msg_AreaHandle_PS )
	RET_ERR("MSGAPI ERROR: MsgOpenArea() failed.", 0 );

    msg_AreaMap_S.FirstMsg_I = MsgMsgnToUid( msg_AreaHandle_PS, 1 );
    msg_AreaMap_S.MsgCount_I = MsgGetNumMsg( msg_AreaHandle_PS );

    CALL( msg_ReadMap() );

    return OK;
}

/***************************************************************************
*
*  Function:	msg_CloseArea
*  Description: Close current area
*
***************************************************************************/
PRIVATE t_RetCode msg_CloseArea( void )
{
    CALL( MSG_WriteMap() );

    if ( MsgCloseArea( msg_AreaHandle_PS ) )
	RET_ERR( "MSGAPI ERROR: MsgCloseArea failed.", 0 );
    
    msg_AreaHandle_PS = NULL;

    return OK;
}


/***************************************************************************
*
*  Function:	msg_ReadMap
*  Description: Read map file of unread messages
*
***************************************************************************/
PRIVATE t_RetCode msg_ReadMap( void )
{
    FILE*	File_PS;
    char	Filename_AC[128];
    char*	Line_PC;
    int		Size_I = 2048;
    int		TagLength_I = strlen( msg_Tagname_AC );
    t_Bool	TagFound_B = FALSE, More_B;
    int		FirstMsg_I, MsgCount_I, Dummy_I;


    /* get area sizes */
    CALL( MSG_AreaInfo( A_FIRST, &Dummy_I ) );
    
    FirstMsg_I = msg_AreaMap_S.FirstMsg_I;
    MsgCount_I = msg_AreaMap_S.MsgCount_I;
    
    /* remove old map */
    if ( msg_AreaMap_S.Read_PB )
	free( msg_AreaMap_S.Read_PB );
    
    /* make space for map */
    msg_AreaMap_S.Read_PB = (t_Bool*) malloc( MsgCount_I );
    if ( NOT msg_AreaMap_S.Read_PB )
	RET_ERR( "Out of memory!", MsgCount_I );
    
    /* clear the new map */
    memset( msg_AreaMap_S.Read_PB, FALSE, MsgCount_I );
    msg_AreaMap_S.Read_PB[0] = TRUE;
    
    
    /* open map file */
    sprintf( Filename_AC, MSG_MAPPATH "/%d", System_S.UserNum_I );
    File_PS = fopen( Filename_AC, "r" );
    
    /* if no map file exists */
    if ( NOT File_PS )
	return OK;
    
    
    /* make space for line */
    Line_PC = (char*) malloc( Size_I );
    if ( NOT Line_PC )
	RET_ERR( "Out of memory!", Size_I );
    
    /* scan map file for area tagname */
    CALL( STR_ReadLine( File_PS, &Line_PC, Size_I, &More_B ) );
    do {
	if ( 0 == strncmp( Line_PC, msg_Tagname_AC, TagLength_I ) ) {
	    TagFound_B = TRUE;
	    break;
	}
	CALL( STR_ReadLine( File_PS, &Line_PC, Size_I, &More_B ) );
    } while ( More_B );
    
    fclose( File_PS );
    
    /* if area line exists */
    if ( TagFound_B ) {
	
	char* Ptr_PC = Line_PC;
	
	/* if map line exists */
	
	/* move across the "FREXXNET: " tag name part of the line */
	Ptr_PC = strchr( Ptr_PC, ':' ) + 1;
	
	while ( 1 ) {
	    int	  	FirstNum_I, NextNum_I;
	    t_Bool	Tmp_B;
	    
	    /* find out the first message number */
	    FirstNum_I = strtol( Ptr_PC, &Ptr_PC, 10 );
	    if ( 0 == FirstNum_I )
		break;
	    
	    /* mark message as read */
	    CALL( MSG_MarkMessage( FirstNum_I, MSG_MARK, &Tmp_B ) );
	    
	    /* is it a range? */
	    if ( '-' == Ptr_PC[0] ) {
		int Msg_I;
		
		Ptr_PC++;
		
		NextNum_I = strtol( Ptr_PC, &Ptr_PC, 10 );
		if ( 0 == NextNum_I )
		    break;
		
		/* mark all messages in range */
		for ( Msg_I = FirstNum_I + 1; Msg_I <= NextNum_I; Msg_I++ )
		    CALL( MSG_MarkMessage( Msg_I, MSG_MARK, &Tmp_B ) );
	    }
	    
	    /* skip the trailing comma */
	    Ptr_PC++;
	}
    }
    
    free( Line_PC );
    
    return OK;
}


/***************************************************************************
*
*  Function:	msg_MakeMap
*  Description: Create a one-line map of read messages
*
***************************************************************************/
PRIVATE t_RetCode msg_MakeMap( FILE* OutFile_PS )
{
    char*	Line_PC;
    int		Offset_I = 0;
    int		Size_I = 2048;
    int		MsgCount_I	= msg_AreaMap_S.MsgCount_I - 1;
    int		FirstMsg_I	= msg_AreaMap_S.FirstMsg_I;
    int		Msg_I;
    
    fprintf( OutFile_PS, "%s: ", msg_Tagname_AC );
    
    for ( Msg_I = 0; Msg_I < MsgCount_I; Msg_I++ ) {
	
	if ( NOT msg_AreaMap_S.Read_PB[ Msg_I ] )
	    continue;
	
	/* is it a range? */
	if ( msg_AreaMap_S.Read_PB[ Msg_I + 1 ] ) {
	    fprintf( OutFile_PS, "%d-", Msg_I + FirstMsg_I );
	    
	    /* keep going until we find an unread */
	    while ( msg_AreaMap_S.Read_PB[ Msg_I ] AND ( Msg_I < MsgCount_I ) )
		Msg_I++;
	    
	    /* step back to the last one read */
	    if ( Msg_I < MsgCount_I )
		Msg_I--;
	}
	
	fprintf( OutFile_PS, "%d,", Msg_I + FirstMsg_I );
    }
    
    fprintf( OutFile_PS, "\n" );
    
    return OK;
}


/***************************************************************************
*
*  Function:	msg_GetHeader
*  Description: Read message header (only) for a message
*               (internal use ONLY!)
*
***************************************************************************/
PRIVATE t_RetCode msg_GetHeader( int MsgNo_I, XMSG* XHeader_PS )
{
    HMSG 	Message_PS;
    int		Error_I;

    Message_PS = MsgOpenMsg( msg_AreaHandle_PS, MOPEN_READ, MsgNo_I );
    if ( NOT Message_PS )
	RET_ERR( "MSGAPI ERROR: MsgOpenMsg() failure.", MsgNo_I );
    
    Error_I = MsgReadMsg( Message_PS, XHeader_PS, 0, 0, 0, 0, 0 );
    if ( -1 == Error_I )
	RET_ERR( "MSGAPI ERROR: MsgReadMsg() failure.", MsgNo_I );
    
    if ( -1 == MsgCloseMsg( Message_PS ) )
	RET_ERR( "MSGAPI ERROR: MsgCloseMsg() failure.", MsgNo_I );
    
    return OK;
}


#if 0
/***************************************************************************
*
*  Function:	msg_UidToMsg
*  Description: Convert UID to msg number, while checking for invalid no
*
***************************************************************************/
PRIVATE t_RetCode msg_UidToMsg( int Uid_I, int* MsgNo_PI )
{
    t_Bool	Exists_B;
    dword	MsgNo_I;
    
    CALL( msg_UidExists( Uid_I, &Exists_B ) );
    if ( NOT Exists_B ) {
	*MsgNo_PI = 0;
	return OK;
    }

    MsgNo_I = MsgUidToMsgn( msg_AreaHandle_PS, (UMSGID) Uid_I, UID_EXACT );
    *MsgNo_PI = (int) MsgNo_I;
    
    return OK;
}	


/***************************************************************************
*
*  Function:	msg_UidExists
*  Description: Check if UID is an existing message
*
***************************************************************************/
PRIVATE t_RetCode msg_UidExists( int Uid_I, t_Bool* Exists_PB )
{
    int Error_I;

    /* if a too low msg number is specified */
    if ( Uid_I < msg_AreaMap_S.FirstMsg_I ) {
	*Exists_PB = FALSE;
	return OK;
    }

    /* if a too high msg number is specified */
    if ( Uid_I > ( msg_AreaMap_S.FirstMsg_I + msg_AreaMap_S.MsgCount_I ) ) {

	/* check for newly imported mail */
	CALL( MSG_AreaInfo( A_FIRST, &Error_I ) );
	if ( Uid_I > msg_AreaMap_S.FirstMsg_I + msg_AreaMap_S.MsgCount_I ) {
	    *Exists_PB = FALSE;
	    return OK;
	}
    }
    
    *Exists_PB = TRUE;
    
    return OK;
}
#endif

/***************************************************************************
*
*  Function:	msg_LoadFile
*  Description: Load a file into a buffer
*
***************************************************************************/
PRIVATE t_RetCode msg_LoadFile( char* Filename_PC, char** Buffer_PPC )
{
    FILE* File_PS;
    char  Line_AC[512];
    int	  LineLen_I = 0, BufLen_I = 0;
    char* Buf_PC;

#if 1
    int		Size_I;
    t_FileInfo  Info_S;

    File_PS = fopen( Filename_PC, "r" );
    if ( !File_PS )
	RET_ERR( "Can't open file!", errno );

    CALL( FIL_FileInfo( Filename_PC, &Info_S ) );
    Size_I = Info_S.Size_I;

    Buf_PC = (char*) malloc( Size_I + 1 );
    if (!Buf_PC)
	RET_ERR( "Out of memory!", Size_I );

    fread( Buf_PC, Size_I, 1, File_PS );
    fclose( File_PS );

    *Buffer_PPC = Buf_PC;
#else

    while ( fgets( Line_AC, 511, File_PS ) ) {
	LineLen_I = strlen( Line_AC );

	if ( !BufLen_I ) {
	    BufLen_I += LineLen_I;
	    Buf_PC = malloc( LineLen_I );
	}
	else {
	    BufLen_I += LineLen_I;
	    Buf_PC = realloc( Buf_PC, BufLen_I );
	}

	if ( !Buf_PC ) {
	    fclose( File_PS );
	    RET_ERR( "Out of memory!", BufLen_I );
	}

	memcpy( Buf_PC + (BufLen_I - LineLen_I ), Line_AC, LineLen_I );
    }

    fclose( File_PS );
#endif

    *Buffer_PPC = Buf_PC;

    return OK;
}
