/***************************************************************************
 *                                                                         *
 *   GMELOC.C                                                              *
 *                                                                         *
 *   Copyright (c) 1994-1995 GALACTICOMM, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   This is the Galacticomm Messaging Engine local message base manager.  *
 *                                                                         *
 *                                            - J. Alvrus   6/5/94         *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "majorbbs.h"
#include "gcspsrv.h"
#include "galme.h"
#include "gme.h"
#include "gmeutl.h"
#include "gmeloc.h"
#include "threadu.h"

#define FILREV "$Revision: 1.9 $"

#define EMLADIR  "EMAIL"           /* E-mail attachment directory          */
#define GMECIF   "GALCUPI.SAV"     /* file name for GME cleanup info file  */
#define FORDEFN  "GALFOR.DAT"      /* forum definition file name           */
#define VIRFDFNM "GALFDF.VIR"      /* forum data file template name        */
#define QSCFNM   "GALQSCAN.DAT"    /* user quickscan file name             */
#define QIKFNM   "GALQUICK.DAT"    /* !QUICK list file name                */

/* GME State Codes */
#define AVLFID   SRCHING+1         /* gme1crf() state(s)                   */
#define CRFDF    SRCHING+2
#define CRFAD    SRCHING+3
#define INSFD    SRCHING+4
#define ATTACH   SRCHING+5         /* gme1wnm(), gme1wap() state(s)        */
#define HDLHOOK  SRCHING+6

#define MAXFID  65535U             /* max allowable forum ID               */
#define MXTHRC  64                 /* maximum number of threads to cache   */

struct fndblk gmefb;               /* for finding files (one-shot only)    */

void *tmpbuf=NULL,                 /* temporary buffer, one-cycle use only */
     *usrqs=NULL;                  /* head of alcblok'd quickscan array    */

struct qscfg *utlqsc;              /* GME utility quickscan buffer         */
char *utlapi;                      /* GME utility app-defined info buffer  */

char *emldp=NULL,                  /* E-mail data file path and file name  */
     killstr[]={"\1"};             /* for finding deleted forums           */
#define KILLCHR '\1'               /* for marking forums as deleted        */
#define NOADDRC '\xFF'             /* placeholder for zero-length address  */

int emllif,                        /* lifetime of an E-Mail message (days) */
    maxfdf,                        /* max forum data files allowed by Sysop*/
    cfmsgs,                        /* frequency (cups) for old msg cleanup */
    cffors,                        /* frequency (cups) for del'd Forum "   */
    bgmdly;                        /* delay factor for background cleanup  */
BOOL bgmcup,                       /* do old message cleanup in background?*/
     bgcipg=FALSE;                 /* is background cleanup in progress?   */

struct gmestor {                   /* stored cleanup info (in GMECIF)      */
     long highmsg;                 /*   current high message number        */
     unsigned ncsmsg;              /*   # of mcus since cleaning old msgs  */
     unsigned ncsfor;              /*   # of mcus since "  deleted Forums  */
     unsigned bgmfid;              /*   current Forum-ID in bground msg cup*/
     long bgmmid;                  /*   current msgno in bground msg cup   */
} cupinf;                          /* current info (stored at shutdown)    */

long _highmsg=0L;                  /* highest message ID on system         */

struct fmidky fmidky;              /* composite keys for one-cycle use     */
struct ftidky ftidky;
struct etidky etidky;
struct umidky umidky;

struct fordef *curfdp=NULL;        /* pointer to current forum definition  */

unsigned nforums=0;                /* number of forums                     */
struct fordef **defarr=NULL;       /* array of pointers to forum defs      */
struct fidxrf {                    /* Forum-ID cross reference structure   */
     unsigned fid;                 /*   forum ID                           */
     unsigned idx;                 /*   index in defarr[]                  */
} *fidxrf=NULL;                    /* array of Forum-ID cross references   */

int nfdf=0;                        /* number of forum data files           */
BTVFILE **fdfarr=NULL;             /* forum data file array                */

BTVFILE *emlbb=NULL,               /* E-mail message data file             */
        *fordefbb=NULL,            /* forum definition data file           */
        *curfbb=NULL,              /* current forum data file              */
        *qscbb=NULL,               /* user quickscan data file             */
        *qikbb=NULL;               /* !QUICK list file                     */

#define emdptr ((struct msgdsk *)(emlbb->data))
                                   /* E-mail btvfile message data          */
#define emukptr ((struct umidky *)(emlbb->key))
                                   /* E-mail btvfile User/msg ID key       */
#define emlkptr ((long *)(emlbb->key))
                                   /* E-mail btvfile thread/msg ID key     */
#define fddptr ((struct fordsk *)(fordefbb->data))
                                   /* forum def btvfile data               */
#define cfdptr ((struct msgdsk *)(curfbb->data))
                                   /* cur forum btvfile message data       */
#define cfmkptr ((struct fmidky *)(curfbb->key))
                                   /* cur forum btvfile forumID/msgID key  */
#define cftkptr ((struct ftidky *)(curfbb->key))
                                   /* cur forum btvfile forum/thread key   */
long opneml(void);
void clseml(void);
long opnfor(void);
void clsfor(BOOL updfors);
void loadfor(void);
struct fordef *addafor(struct fordsk *newdef,char *echoes);
void fixseq(void);
BOOL newer(unsigned first,unsigned second);
void dsk2def(struct fordsk *dsk,struct fordef *def);
char *setfad(char *path);
void setcurf(unsigned fid);
void setcfdf(void);
void insxrf(unsigned fid,int idx);
void resortf(int oldidx);
int nearnmi(int *cond,char *name);
int xrfidx(unsigned fid);
void wtouid(char *userid);
struct fordef *addfarr(struct fordsk *newdef,char *echoes);
void addseq(struct fordef *newdef);
void def2dsk(struct fordef *def,struct fordsk *dsk);
void msg2dsk(struct message *msg,struct msgdsk *dsk);
void dsk2msg(struct msgdsk *dsk,struct message *msg);
void gettxt(const struct msgdsk *dskbuf,char *text,char *appinf,unsigned reclen);
void foradd(struct message *msg);
void lofadd(int flags);
void fordel(struct msgdsk *msg);
void insnmsg(struct msgdsk *wrtbuf,const struct message *msg,const char *text,const char *appinf);
void insmptr(struct msgdsk *wrtbuf,const struct message *msg,const struct fmidky *mptr);
void updamsg(struct msgdsk *wrtbuf,const struct message *msg,const char *text,const char *appinf);
void updmptr(struct msgdsk *wrtbuf,const struct message *msg,const struct fmidky *mptr);
unsigned fillbuf(struct msgdsk *datbuf,const struct message *msg,const char *text,const char *appinf);
char *textptr(struct msgdsk *dsk);
int nvmsiz(const struct msgdsk *dsk);
unsigned hifid(void);
unsigned nxtafid(unsigned fid);
void recount(void);
BOOL fndbgmf(void);
void qikdel(void);
unsigned getexpd(unsigned daysago);
int nxtqki(int pos,struct qikdat *qikbuf);
int qiklen(struct qikdat *qikbuf);
int lastqk(struct qikdat *qikbuf);
void updpri(struct gmework *work,struct message *msg);
void whackit(struct msgdsk *msg);
void lowhack(struct msgdsk *msg);
BOOL touchk(struct gmework *work,struct msgdsk *msg);
BOOL fruchk(struct gmework *work,struct msgdsk *msg);
BOOL ethchk(struct gmework *work,struct msgdsk *msg);
BOOL forchk(struct gmework *work,struct msgdsk *msg);
BOOL fthchk(struct gmework *work,struct msgdsk *msg);
BOOL acqmsg(struct msgdsk *msgbuf,void *keybuf,int keynum,int direct,
            struct gmework *work,
            BOOL (*keychk)(struct gmework *work,struct msgdsk *msg));
BOOL getbdy(struct msgdsk *dsk,struct message *msg,char *text);

void
iniloc(void)                       /* initialize GME local message base    */
{
     long emlhi,forhi;
     FILE *fp;

     inithu(MXTHRC);
     tmpbuf=alcmem(TMPBSZ);
     qscbb=opnbtv(QSCFNM,MAXQSR);
     usrqs=alcblok(nterms,qssiz);
     utlqsc=(struct qscfg *)alcmem(max(MAXQSR,
                                   sizeof(struct fordsk)-sizeof(char)+MAXFDV));
     qikbb=opnbtv(QIKFNM,MAXQKR);
     utlapi=alcmem(TXTLEN);
     cfmsgs=numopt(CFMSGS,1,365);
     cffors=numopt(CFFORS,1,365);
     bgmcup=ynopt(BGMCUP);
     bgmdly=numopt(BGMDLY,0,32767);
     setmem(&cupinf,sizeof(struct gmestor),0);
     fp=fopen(GMECIF,FOPRB);
     if (fp != NULL) {
          fread(&cupinf,sizeof(struct gmestor),1,fp);
          _highmsg=cupinf.highmsg;
          fclose(fp);
     }
     emlhi=opneml();
     forhi=opnfor();
     if (emlhi > forhi) {
          forhi=emlhi;
     }
     if (forhi == 0L) {
          forhi=1L;
     }
     if (forhi > _highmsg) {
          _highmsg=forhi;
     }
}

void
clsloc(                            /* close GME local message base         */
BOOL updfors)                      /*   update all the Forums on disk?     */
{
     FILE *fp;

     fp=fopen(GMECIF,FOPWB);
     if (fp != NULL) {
          cupinf.highmsg=_highmsg;
          fwrite(&cupinf,sizeof(struct gmestor),1,fp);
          fclose(fp);
     }
     else if (!gmeoffl()) {
          shocst("GME CLEANUP INFO ERROR",
                 "unable to write cleanup info file: \"%s\"",GMECIF);
     }
     clseml();
     clsfor(updfors);
     if (qscbb != NULL) {
          clsbtv(qscbb);
          qscbb=NULL;
     }
     if (qikbb != NULL) {
          clsbtv(qikbb);
          qikbb=NULL;
     }
     clsthu();
}

long                               /*   returns highest message number     */
opneml(void)                       /* initialize E-mail data file          */
{
     emllif=numopt(EMLLIF,-1,32767);
     emldp=stgopt(EMLDPTH);
     emlbb=opnbtv(emldp,_reclen);
     if (!fmdir(EMLADIR)) {
          catastro("Unable to find or create E-mail attachment directory:\n"
                   "\"%s\"",EMLADIR);
     }
     #ifdef DEBUG
          if (fnd1st(&gmefb,spr("%s\\*.TMP",EMLADIR),0)) {
               do {
                    unlink(spr("%s\\%s",EMLADIR,gmefb.name));
               } while (fndnxt(&gmefb));
          }
     #endif
     ASSERT(bb == emlbb);
     if (qhibtv(EMIDKY)) {
          return(*((long *)emlbb->key));
     }
     else {
          return(0L);
     }
}

void
clseml(void)                       /* close E-mail data file               */
{
     if (emlbb != NULL) {
          clsbtv(emlbb);
          emlbb=NULL;
     }
}

long                               /*   returns highest message number     */
opnfor(void)                       /* initialize forum data files & arrays */
{
     int i;
     long high=0L;

     maxfdf=gnumdb(GMEMDF);
     fdfarr=(BTVFILE **)alczer(maxfdf*sizeof(BTVFILE *));
     fordefbb=opnbtv(FORDEFN,sizeof(struct fordsk)-1+MAXFDV);
     loadfor();
     for (i=0 ; i < nforums ; ++i) {
          curfdp=defarr[i];
          setcfdf();
          fmidky.forum=curfdp->forum;
          fmidky.msgid=LASTM;
          ASSERT(bb == curfbb);
          if (qlebtv(&fmidky,FFMKY)) {
               if (cfmkptr->forum == curfdp->forum) {
                    high=max(high,cfmkptr->msgid);
               }
          }
     }
     return(high);
}

void
clsfor(                            /* close forum data files               */
BOOL updfors)                      /*   update all the Forums on disk?     */
{
     int i;

     setbtv(fordefbb);
     if (updfors) {
          for (i=0 ; i < nforums ; ++i) {
               if (acqbtv(NULL,&defarr[i]->forum,DIDKY)) {
                    movmem(fddptr,utlqsc,sizeof(struct fordsk)-1);
                    def2dsk(defarr[i],(struct fordsk *)utlqsc);
                    if (memcmp(fddptr,utlqsc,sizeof(struct fordsk)-1) != 0) {
                         movmem(utlqsc,fddptr,sizeof(struct fordsk)-1);
                         upvbtv(NULL,llnbtv());
                    }
               }
          }
     }
     if (fordefbb != NULL) {
          clsbtv(fordefbb);
          fordefbb=NULL;
     }
     for (i=0 ; i < nfdf ; ++i) {
          if (fdfarr[i] != NULL) {
               clsbtv(fdfarr[i]);
               fdfarr[i]=NULL;
          }
     }
}

void
loadfor(void)                      /* load forum definitions               */
{
     ASSERT(bb == fordefbb);
     if (agtbtv(NULL," ",DNMKY)) {
          do {
               if (!fexist(fddptr->datfil)) {
                    if (!creatfdf(fddptr->datfil)) {
                         catastro("Unable to create forum data file: %s",
                                  fddptr->datfil);
                    }
               }
               curfdp=addafor(fddptr,fddptr->info);
               setbtv(fordefbb);
          } while (qnxbtv());
     }
     else if (!gmeoffl()) {
          catastro("No Forums defined!");
     }
     fixseq();
}

struct fordef *                    /*   returns pointer to new in-mem def  */
addafor(                           /* add a forum to in-memory arrays      */
struct fordsk *newdef,             /*   pointer to forum on-disk structure */
char *echoes)                      /*   pointer to array of echo addresses */
{
     struct fordef *memdef;
     #ifdef DEBUG
          int tmpctr;
          adr_t *tmpecho;
     #endif

     ASSERT(newdef != NULL);
     ASSERT(newdef->forum != EMLID);
     if (!alcpar((void ***)&defarr,nforums,BIGBLK)
      || !alcarr((void **)&fidxrf,sizeof(struct fidxrf),nforums,BIGBLK)) {
          catastro("Not enough memory to load Forums");
     }
     memdef=(struct fordef *)alcmem(sizeof(struct fordef));
     defarr[nforums]=memdef;
     insxrf(newdef->forum,nforums);
     ++nforums;
     if (nfdf == maxfdf && !fdfisopn(newdef->datfil)) {
          catastro("Too many forum data files required. Increase \"Dynamic Btrieve files:\" in GALME.MDF");
     }
     memdef->dfnum=opnfdf(newdef->datfil);
     dsk2def(newdef,memdef);
     if (!fmdir(memdef->attpath)) {
          catastro("Unable to find or create attachment directory \"%s\" for the %s forum",
                   memdef->attpath,memdef->name);
     }
     #ifdef DEBUG
          if (fnd1st(&gmefb,spr("%s\\*.TMP",memdef->attpath),0)) {
               do {
                    unlink(spr("%s\\%s",memdef->attpath,gmefb.name));
               } while (fndnxt(&gmefb));
          }
     #endif
     if (memdef->necho > 0) {
          ASSERT(memdef->necho <= MAXECHO);
          memdef->echoes=alcmem(MAXADR*memdef->necho);
          movmem(echoes,memdef->echoes,MAXADR*memdef->necho);
          #ifdef DEBUG
               tmpecho=(adr_t *)memdef->echoes;
               for (tmpctr=0 ; tmpctr < memdef->necho ; ++tmpctr) {
                    ASSERT(strlen(tmpecho[tmpctr]) < MAXADR);
               }
          #endif
     }
     else {
          memdef->echoes=NULL;
     }
     return(memdef);
}

void
fixseq(void)                       /* resolve any sequence ID conflicts    */
{
     unsigned seq,i,j,newest;
     BOOL fix;

     for (seq=0 ; seq < nforums ; ++seq) {
          do {
               for (i=0 ; i < nforums ; ++i) {
                    if (defarr[i]->seqid == seq) {
                         break;
                    }
               }
               if (i == nforums) {
                    for (j=0 ; j < nforums ; ++j) {
                         if (defarr[j]->seqid > seq) {
                              --defarr[j]->seqid;
                         }
                    }
               }
          } while (i == nforums);
          newest=i;
          fix=FALSE;
          for (j=0 ; j < nforums ; ++j) {
               if (j != i && defarr[j]->seqid == seq) {
                    fix=TRUE;
                    if (newer(defarr[j]->forum,defarr[newest]->forum)) {
                         newest=j;
                    }
               }
          }
          if (fix) {
               for (j=0 ; j < nforums ; ++j) {
                    if (j != newest && defarr[j]->seqid >= seq) {
                         ++defarr[j]->seqid;
                    }
               }
          }
     }
}

BOOL
newer(                             /* is first forum newer than second?    */
unsigned first,                    /*   first forum                        */
unsigned second)                   /*   second forum                       */
{
     unsigned date1,time1,date2,time2;

     setbtv(fordefbb);
     geqbtv(NULL,&first,DIDKY);
     date1=fddptr->crdate;
     time1=fddptr->crtime;
     geqbtv(NULL,&second,DIDKY);
     date2=fddptr->crdate;
     time2=fddptr->crtime;
     rstbtv();
     if (date1 > date2) {
          return(TRUE);
     }
     if (date1 < date2) {
          return(FALSE);
     }
     return(time1 > time2);
}

long
himsgid(void)                      /* get current highest message ID       */
{
     return(_highmsg);
}

unsigned
numforums(void)                    /* get number of forums                 */
{
     return(nforums);
}

int                                /*   returns index of data file in array*/
opnfdf(                            /* open forum data file & add to array  */
char *name)                        /*   path and file name of data file    */
{
     int i;

     ASSERT(name != NULL);
     ASSERT(strlen(name) < MAXPATH);
     for (i=0 ; i < nfdf ; ++i) {
          ASSERT(fdfarr != NULL);
          ASSERT(fdfarr[i] != NULL);
          if (sameas(fdfarr[i]->filnam,name)) {
               return(i);
          }
     }
     ASSERT(nfdf < maxfdf);
     fdfarr[nfdf]=opnbtv(name,_reclen);
     return(nfdf++);
}

int                                /*   returns standard GME status codes  */
recfdf(                            /* recommend a forum data file          */
struct gmework *work,              /*   GME work space (provided by caller)*/
char *recname)                     /*   buffer for recommended name        */
{
     int i;
     long nmsgs;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(recname != NULL);
     switch (work->state1) {
     case START:
          work->counter=0;
          work->fpos=-1;
          work->fdfnum=0;
          work->state1=SRCHING;
     case SRCHING:
          for (i=0 ; work->counter < nfdf && i < 2*MAXBTV ;
               ++i,++work->counter) {
               setbtv(fdfarr[work->counter]);
               nmsgs=cntrbtv();
               if (work->fpos == -1 || nmsgs < work->fpos) {
                    work->fpos=nmsgs;
                    work->fdfnum=work->counter;
               }
          }
          break;
     default:
          state_error(1,"recfdf()",work->state1);
     }
     if (work->counter == nfdf) {
          stlcpy(recname,fdfarr[work->fdfnum]->filnam,MAXPATH);
          work->state1=START;
          return(GMEOK);
     }
     return(GMEAGAIN);
}

BOOL
fdfisopn(                          /* is forum data file already open?     */
char *name)                        /*   path+name of forum data file       */
{
     int i;

     ASSERT(name != NULL);
     ASSERT(strlen(name) < MAXPATH);
     for (i=0 ; i < nfdf ; ++i) {
          ASSERT(fdfarr != NULL);
          ASSERT(fdfarr[i] != NULL);
          if (sameas(normspec((char *)tmpbuf,fdfarr[i]->filnam),
                     normspec((char *)tmpbuf+MAXPATH,name))) {
               return(TRUE);
          }
     }
     return(FALSE);
}

BOOL                               /*   return TRUE if file created        */
creatfdf(                          /* create a new forum data file         */
char *name)                        /*   name of new forum data file        */
{
     int frsiz;
     char *pname;
     FILE *fvir,*fnew;

     ASSERT(name != NULL);
     ASSERT(tmpbuf != NULL);
     pname=strrchr(name,'.');
     /* .DAT ext is enforced in gme1crf() */
     if (pname == NULL || !sameas(pname,".DAT")) {
          return(FALSE);
     }
     if ((fvir=fopen(VIRFDFNM,FOPRB)) == NULL) {
          catastro("Unable to open forum data file template: %s",VIRFDFNM);
     }
     if ((fnew=fopen(name,FOPWB)) == NULL) {
          fclose(fvir);
          return(FALSE);
     }
     while (!feof(fvir)) {
          if ((frsiz=(int)fread(tmpbuf,1,TMPBSZ,fvir)) > 0) {
               fwrite(tmpbuf,1,frsiz,fnew);
          }
          if (ferror(fnew) || ferror(fvir)) {
               fclose(fvir);
               fclose(fnew);
               unlink(name);
               return(FALSE);
          }
     }
     fclose(fvir);
     fclose(fnew);
     return(TRUE);
}

unsigned                           /*   returns packed date                */
getexpd(                           /* get message expiration date          */
unsigned daysago)                  /*   to this many days ago              */
{
     unsigned coftod;

     if (daysago > (coftod=cofdat(today()))) {
          daysago=coftod;
     }
     return(datofc(coftod-daysago));
}

void
dsk2def(                           /* copies forum definition from         */
struct fordsk *dsk,                /*   on-disk format to                  */
struct fordef *def)                /*   in-memory format                   */
{
     ASSERT(dsk != NULL && def != NULL);
     def->forum=dsk->forum;
     ASSERT(strlen(dsk->name) < FORNSZ);
     stlcpy(def->name,dsk->name,FORNSZ);
     ASSERT(strlen(dsk->topic) < TPCSIZ);
     stlcpy(def->topic,dsk->topic,TPCSIZ);
     ASSERT(strlen(dsk->forop) < UIDSIZ);
     stlcpy(def->forop,dsk->forop,UIDSIZ);
     ASSERT(strlen(dsk->attpath) < MAXDIR);
     def->attpath=NULL;
     def->attpath=setfad(dsk->attpath);
     def->nthrs=dsk->nthrs;
     def->nmsgs=dsk->nmsgs;
     def->nfiles=dsk->nfiles;
     def->nw4app=dsk->nw4app;
     ASSERT(strlen(dsk->forlok) < KEYSIZ);
     stlcpy(def->forlok,dsk->forlok,KEYSIZ);
     def->dfnpv=(char)dsk->dfnpv;
     def->dfprv=(char)dsk->dfprv;
     def->mxnpv=(char)dsk->mxnpv;
     def->msglif=dsk->msglif;
     def->chgmsg=dsk->chgmsg;
     def->chgrdm=dsk->chgrdm;
     def->chgatt=dsk->chgatt;
     def->chgadl=dsk->chgadl;
     def->chgupk=dsk->chgupk;
     def->chgdpk=dsk->chgdpk;
     def->ccr=dsk->ccr;
     def->pfnlvl=(unsigned char)dsk->pfnlvl;
     def->crdate=dsk->crdate;
     def->crtime=dsk->crtime;
     def->seqid=dsk->seqid;
     def->necho=dsk->necho;
}

char *                             /*   pointer to att path string         */
setfad(                            /* set forum attachment dir pointer     */
char *path)                        /*   path buffer to set to              */
{
     int i;
     char *old,*new;

     ASSERT(path != NULL);
     new=(char *)tmpbuf;
     old=new+MAXPATH;
     normspec(new,path);
     for (i=0 ; i < nforums ; ++i) {
          if (defarr[i]->attpath != NULL
           && sameas(new,normspec(old,defarr[i]->attpath))) {
               return(defarr[i]->attpath);
          }
     }
     return(strdup(path));
}

void
setcurf(                           /* set current forum context            */
unsigned fid)                      /*   given forum ID                     */
{
     curfdp=getdef(fid);
     setcfdf();
}

void
setcfdf(void)                      /* set current forum data file pointer  */
{
     ASSERT(fdfarr != NULL);
     ASSERT(curfdp->dfnum >= 0 && curfdp->dfnum < nfdf);
     curfbb=fdfarr[curfdp->dfnum];
     ASSERT(curfbb != NULL);
     setbtv(curfbb);
}

void
insxrf(                            /* insert forum ID cross-ref in array   */
unsigned fid,                      /*   forum ID                           */
int idx)                           /*   index in defarr                    */
{                                  /*   (assumes nforums not yet inc'd)    */
     int i,j;

     ASSERT(fid != EMLID);
     ASSERT(fidxrf != NULL);
     for (i=0 ; i < nforums ; ++i) {
          if (fidxrf[i].fid > fid) {
               break;
          }
     }
     movmem(&fidxrf[i],&fidxrf[i+1],sizeof(struct fidxrf)*(nforums-i));
     fidxrf[i].fid=fid;
     fidxrf[i].idx=idx;
     for (j=0 ; j <= nforums ; ++j) {
          if (j != i && fidxrf[j].idx >= idx) {
               ++fidxrf[j].idx;
          }
     }
}

void
resortf(                           /* resort forum arrays after name change*/
int oldidx)                        /*   current index of forum that changed*/
{
     int i,newidx;
     BOOL gtprv,ltnxt;
     struct fordef *tmpdef;

     gtprv=(oldidx == 0
         || stricmp(defarr[oldidx]->name,defarr[oldidx-1]->name) > 0);
     ltnxt=(oldidx == nforums-1
         || stricmp(defarr[oldidx]->name,defarr[oldidx+1]->name) < 0);
     if (gtprv && ltnxt) {
          return;
     }
     newidx=oldidx;
     if (gtprv) {                  /* i.e. greater-than next (move up)     */
          do {
               ++newidx;
          } while (newidx < nforums-1
              && stricmp(defarr[oldidx]->name,defarr[newidx+1]->name) > 0);
     }
     else {                        /* i.e. less-than previous (move down)  */
          do {
               --newidx;
          } while (newidx > 0
              && stricmp(defarr[oldidx]->name,defarr[newidx-1]->name) < 0);
     }
     tmpdef=defarr[oldidx];
     movmem(&defarr[oldidx+1],&defarr[oldidx],
            sizeof(struct fordef *)*(nforums-oldidx-1));
     movmem(&defarr[newidx],&defarr[newidx+1],
            sizeof(struct fordef *)*(nforums-newidx-1));
     defarr[newidx]=tmpdef;
     for (i=0 ; i < nforums ; ++i) {
          ASSERT(xrfidx(defarr[i]->forum) != NOIDX);
          fidxrf[xrfidx(defarr[i]->forum)].idx=i;
     }
}

int                                /*   (returns NOIDX if not found)       */
fnmidx(                            /* get index of named forum in def array*/
char *name)                        /*   forum name                         */
{
     int i,cond;

     ASSERT(name != NULL);
     ASSERT(defarr != NULL);
     i=nearnmi(&cond,name);
     if (cond == 0) {
          return(i);
     }
     return(NOIDX);
}

int                                /*   (returns NOIDX if not found)       */
prvfnmi(                           /* get index of prev forum in def array */
char *name)                        /*   before this forum name             */
{
     int i,cond;

     ASSERT(name != NULL);
     ASSERT(defarr != NULL);
     i=nearnmi(&cond,name);
     if (cond < 0) {
          return(i);
     }
     if (cond >= 0 && i > 0) {
          return(i-1);
     }
     return(NOIDX);
}

int                                /*   (returns NOIDX if not found)       */
nxtfnmi(                           /* get index of next forum in def array */
char *name)                        /*   after this forum name              */
{
     int i,cond;

     ASSERT(name != NULL);
     ASSERT(defarr != NULL);
     i=nearnmi(&cond,name);
     if (cond > 0) {
          return(i);
     }
     if (cond <= 0 && i < nforums-1) {
          return(i+1);
     }
     return(NOIDX);
}

int
nearnmi(                           /* get index of "nearest" forum         */
int *cond,                         /*   buffer for result of last stricmp()*/
char *name)                        /*   name to search for                 */
{
     int lo,mid,hi;

     ASSERT(name != NULL);
     lo=0;
     hi=nforums-1;
     while (lo <= hi) {
          mid=lo+(hi-lo)/2;
          if ((*cond=stricmp(defarr[mid]->name,name)) < 0) {
               lo=mid+1;
          }
          else if (*cond > 0) {
               hi=mid-1;
          }
          else {
               break;
          }
     }
     return(mid);
}

int                                /*   (returns NOIDX if not found)       */
fididx(                            /* get index of forum ID in def array   */
unsigned fid)                      /*   forum ID                           */
{
     int i;

     ASSERT(fid != EMLID);
     ASSERT(defarr != NULL);
     i=xrfidx(fid);
     if (i != NOIDX) {
          return(fidxrf[i].idx);
     }
     return(NOIDX);
}

int                                /*   (returns NOIDX if not found)       */
xrfidx(                            /* get index of forum ID in xref array  */
unsigned fid)                      /*   forum ID                           */
{
     int i;
     unsigned start;

     ASSERT(fid != EMLID);
     ASSERT(defarr != NULL);
     start=fid-1;
     if (start >= nforums) {
          start=nforums-1;
     }
     for (i=start ; i >= 0 ; --i) {
          if (fidxrf[i].fid == fid) {
               return(i);
          }
     }
     return(NOIDX);
}

unsigned                           /*   returns EMLID if doesn't exist     */
getfid(                            /* get forum ID                         */
char *name)                        /*   given forum name                   */
{
     int i;

     ASSERT(defarr != NULL);
     i=fnmidx(name);
     if (i == NOIDX) {
          return(EMLID);
     }
     return(defarr[i]->forum);
}

unsigned
hifid(void)                        /* get highest forum ID in use          */
{
     unsigned hi;
     int i;

     ASSERT(defarr != NULL);
     for (i=0,hi=0 ; i < nforums ; ++i) {
          ASSERT(defarr[i] != NULL);
          if (defarr[i]->forum > hi) {
               hi=defarr[i]->forum;
          }
     }
     return(hi);
}

unsigned
nxtafid(                           /* get next available forum ID          */
unsigned fid)                      /*   starting at forum ID               */
{
     ASSERT(fid != 0);
     do {
          ++fid;
     } while (fid != 0 && fidxst(fid));
     return(fid);
}

struct fordef *                    /*   returns NULL if doesn't exist      */
getdef(                            /* get pointer to forum definition      */
unsigned fid)                      /*   given forum ID                     */
{
     int i;

     ASSERT(fid != EMLID);
     if ((i=fididx(fid)) != NOIDX) {
          return(defarr[i]);
     }
     return(NULL);
}

struct fordef *                    /*   returns NULL if doesn't exist      */
idxdef(                            /* get pointer to forum definition      */
int idx)                           /*   given def array index              */
{
     if (idx >= 0 && idx < nforums) {
          return(defarr[idx]);
     }
     return(NULL);
}

struct fordef *                    /*   returns NULL if doesn't exist      */
fiddef(                            /* get pointer to forum definition      */
int idx)                           /*   given xrf array index              */
{
     if (idx >= 0 && idx < nforums) {
          return(defarr[fidxrf[idx].idx]);
     }
     return(NULL);
}

struct fordef *                    /*   returns NULL if doesn't exist      */
seqdef(                            /* get pointer to forum definition      */
unsigned seqid)                    /*   given sequence ID                  */
{
     unsigned i;

     for (i=0 ; i < nforums ; ++i) {
          if (defarr[i]->seqid == seqid) {
               return(defarr[i]);
          }
     }
     return(NULL);
}

void
gmeclean(                          /* do full GME cleanup                  */
BOOL crashed)                      /*   have we crashed some time today?   */
{
     bgcipg=FALSE;
     if (++cupinf.ncsfor >= cffors) {
          cleandlf();
          cupinf.ncsfor=0;
     }
     if (crashed || ++cupinf.ncsmsg >= cfmsgs) {
          if (crashed) {
               recount();
          }
          else {
               bgmdly=0;
               while (pumpcup()) {
               }
          }
          cupinf.bgmfid=EMLID;
          cupinf.bgmmid=FIRSTM;
          cupinf.ncsmsg=0;
          qikdel();
     }
}

void
recount(void)                      /* do complete recount message cleanup  */
{
     int i,forlif;
     unsigned expdate,thrcnt;
     BOOL moinfo,notify;

     if (!gmeoffl()) {
          shocst("E-MAIL/FORUMS: RE-COUNTING",
                 "Re-counting all messages due to a crash today.");
     }
     notify=!gmeoffl() && nexthook(GMEHOOK_NOT_DELMSG,NOIDX) != NOIDX;
     sv.emlopn=0;
     if (emllif >= 0) {
          expdate=getexpd(emllif);
     }
     setbtv(emlbb);
     if (alobtv(NULL,EMIDKY)) {
          do {
               if (emllif >= 0 && emdptr->crdate < expdate) {
                    if (notify) {
                         dsk2msg(emdptr,utlmsg);
                         gettxt(emdptr,utltxt,utlapi,llnbtv());
                         notdlm(DELMSG_CLNUP,utlwork,utlmsg,utltxt);
                    }
                    lowhack(emdptr);
               }
               else {
                    ++sv.emlopn;
               }
          } while (agtbtv(NULL,&emdptr->msgid,EMIDKY));
     }
     sv.sigopn=0;
     for (i=0 ; i < nforums ; ++i) {
          setcurf(ftidky.forum=defarr[i]->forum);
          curfdp->nthrs=0;
          curfdp->nmsgs=0;
          curfdp->nfiles=0;
          curfdp->nw4app=0;
          if ((forlif=curfdp->msglif) >= 0) {
               expdate=getexpd(forlif);
          }
          ftidky.thrid=0L;
          ftidky.msgid=0L;
          moinfo=agebtv(NULL,&ftidky,FFTKY);
          while (moinfo && cfdptr->forum == ftidky.forum) {
               ftidky.thrid=cfdptr->thrid;
               thrcnt=0;
               do {
                    ftidky.msgid=cfdptr->msgid;
                    if (forlif >= 0 && cfdptr->crdate < expdate
                     && !(cfdptr->flags&EXEMPT)) {
                         if (notify) {
                              dsk2msg(cfdptr,utlmsg);
                              gettxt(cfdptr,utltxt,utlapi,llnbtv());
                              notdlm(DELMSG_CLNUP,utlwork,utlmsg,utltxt);
                         }
                         lowhack(cfdptr);
                    }
                    else {
                         lofadd((int)cfdptr->flags);
                         ++sv.sigopn;
                         ++thrcnt;
                    }
                    moinfo=agtbtv(NULL,&ftidky,FFTKY);
               } while (moinfo && cfdptr->forum == ftidky.forum
                     && cfdptr->thrid == ftidky.thrid);
               if (thrcnt != 0) {
                    ++curfdp->nthrs;
               }
               setthrc(ftidky.forum,ftidky.thrid,thrcnt);
          }
     }
}

BOOL
needbgc(void)                      /* do we need to do background cleanup? */
{
     return(bgmcup && cupinf.ncsmsg+1 >= cfmsgs && fndbgmf());
}

BOOL                               /*   keep pumping?                      */
pumpcup(void)                      /* pump the normal cleanup process      */
{
     struct msgdsk *tmpdsk;
     void *key;
     int keynum;
     int msglif;
     unsigned tmplen;
     static int dlyctr=0;

     if (dlyctr++ < bgmdly) {
          return(TRUE);
     }
     dlyctr=0;
     if (!fndbgmf()) {
          return(FALSE);
     }
     if (cupinf.bgmfid == EMLID) {
          setbtv(emlbb);
          key=&cupinf.bgmmid;
          keynum=EMIDKY;
          tmpdsk=emdptr;
          msglif=emllif;
     }
     else {
          setcurf(cupinf.bgmfid);
          fmidky.forum=cupinf.bgmfid;
          fmidky.msgid=cupinf.bgmmid;
          key=&fmidky;
          keynum=FFMKY;
          tmpdsk=cfdptr;
          msglif=curfdp->msglif;
     }
     if (msglif >= 0 && agtbtv(NULL,key,keynum)
      && tmpdsk->forum == cupinf.bgmfid && tmpdsk->crdate < getexpd(msglif)) {
          if (!(tmpdsk->flags&EXEMPT)) {
               tmplen=llnbtv();
               if (gmeoffl() || !bgcipg
                || !gencfl(utlwork,tmpdsk->forum,tmpdsk->msgid)) {
                    dsk2msg(tmpdsk,utlmsg);
                    gettxt(tmpdsk,utltxt,utlapi,tmplen);
                    notdlm(DELMSG_CLNUP,utlwork,utlmsg,utltxt);
                    whackit(tmpdsk);
               }
          }
          cupinf.bgmmid=tmpdsk->msgid;
     }
     else {
          cupinf.bgmfid++;
          cupinf.bgmmid=FIRSTM;
     }
     return(TRUE);
}

BOOL                               /*   found a forid >= current one?      */
fndbgmf(void)                      /* find (and set) forid for bground cup */
{
     int i;
     unsigned start;

     if (cupinf.bgmfid == EMLID || fidxst(cupinf.bgmfid)) {
          return(TRUE);
     }
     if (cupinf.bgmfid > fidxrf[nforums-1].fid) {
          return(FALSE);
     }
     start=cupinf.bgmfid-1;
     if (start >= nforums) {
          start=nforums-1;
     }
     for (i=start ; i > 0 ; --i) {
          if (fidxrf[i-1].fid < cupinf.bgmfid) {
               break;
          }
     }
     ASSERT(cupinf.bgmfid < fidxrf[i].fid);
     cupinf.bgmfid=fidxrf[i].fid;
     cupinf.bgmmid=FIRSTM;
     return(TRUE);
}

void
qikdel(void)                       /* clean up old-style dist list att's   */
{
     int tdy,age;
     struct fndblk fb;

     tdy=cofdat(today());
     if (fnd1st(&fb,spr("%s\\*.ATQ",attpth(EMLID)),0)) {
          do {
               age=tdy-cofdat(fb.date);
               if (emllif >= 0 && age > emllif) {
                    unlink(spr("%s\\%s",attpth(EMLID),fb.name));
               }
          } while (fndnxt(&fb));
     }
}

void
cleandlf(void)                     /* removes all traces of deleted forums */
{                                  /*   (executed at clean-up)             */
     int i;
     BOOL update;
     struct fmidky *qsfm,killky;

     setbtv(fordefbb);
     if (!qgebtv(killstr,DNMKY) || fordefbb->key[0] != KILLCHR) {
          return;
     }
     setbtv(qscbb);
     if (slobtv(NULL)) {
          do {
               update=FALSE;
               qsfm=(struct fmidky *)qsdptr->accmsg;
               for (i=0 ; i < qsdptr->nforums ; ++i) {
                    while (i < qsdptr->nforums && !fidxst(qsfm[i].forum)) {
                         update=TRUE;
                         idelqs(qsdptr,i);
                    }
               }
               if (update) {
                    upvbtv(NULL,qsrlen(qsdptr->nforums));
               }
          } while (snxbtv(NULL));
     }
     setbtv(fordefbb);
     while (agebtv(NULL,killstr,DNMKY)) {
          if (fddptr->name[0] == KILLCHR) {
               if (fexist(fddptr->datfil)) {
                    setbtv(curfbb=fdfarr[opnfdf(fddptr->datfil)]);
                    killky.forum=fddptr->forum;
                    killky.msgid=FIRSTM;
                    while (agebtv(NULL,&killky,FFMKY)) {
                         if (cfdptr->forum == fddptr->forum) {
                              if ((cfdptr->flags&FILATT)
                               && !(cfdptr->flags&ISTCPY)) {
                                   normspec((char *)tmpbuf,fddptr->attpath);
                                   unlink(spr("%s\\%s",(char *)tmpbuf,
                                                       attnam(cfdptr->msgid)));
                              }
                              delbtv();
                         }
                         else {
                              break;
                         }
                    }
                    setbtv(fordefbb);
               }
               delbtv();
               cfthrs(fddptr->forum);
          }
          else {
               break;
          }
     }
}

void
cleandla(                          /* clean up after a deleted account     */
char *userid)                      /*   account to clean up after          */
{
     char tmpuid[UIDSIZ];

     stlcpy(tmpuid,userid,UIDSIZ);
     wtouid(tmpuid);
     tmpuid[0]=lwclch(tmpuid[0]);
     wtouid(tmpuid);
}

void
wtouid(                            /* whack messages to a specific user    */
char *userid)                      /*   User-ID to whack messages to       */
{
     struct umidky umkey;

     setbtv(emlbb);
     stlcpy(umkey.usrid,userid,UIDSIZ);
     umkey.msgid=FIRSTM;
     while (agtbtv(NULL,&umkey,ETOMKY) && sameas(emdptr->to,userid)) {
          umkey.msgid=emdptr->msgid;
          whackit(emdptr);
          setbtv(emlbb);
     }
}

void
def2dsk(                           /* copy fordef to fordsk header         */
struct fordef *def,                /*   source definition                  */
struct fordsk *dsk)                /*   destination definition             */
{                                  /* (only copies fields that change)     */
     stzcpy(dsk->name,def->name,FORNSZ);
     stzcpy(dsk->topic,def->topic,TPCSIZ);
     stzcpy(dsk->forop,def->forop,UIDSIZ);
     stzcpy(dsk->datfil,fdfarr[def->dfnum]->filnam,MAXPATH);
     stzcpy(dsk->attpath,def->attpath,MAXDIR);
     stzcpy(dsk->forlok,def->forlok,KEYSIZ);
     dsk->nthrs=def->nthrs;
     dsk->nmsgs=def->nmsgs;
     dsk->nfiles=def->nfiles;
     dsk->nw4app=def->nw4app;
     dsk->dfnpv=(int)def->dfnpv;
     dsk->dfprv=(int)def->dfprv;
     dsk->mxnpv=(int)def->mxnpv;
     dsk->msglif=def->msglif;
     dsk->chgmsg=def->chgmsg;
     dsk->chgrdm=def->chgrdm;
     dsk->chgatt=def->chgatt;
     dsk->chgadl=def->chgadl;
     dsk->chgupk=def->chgupk;
     dsk->chgdpk=def->chgdpk;
     dsk->ccr=def->ccr;
     dsk->pfnlvl=(int)def->pfnlvl;
     dsk->necho=def->necho;
     dsk->seqid=def->seqid;
}

void
cpymdef(                           /* copy modifyable part of in-memory def*/
struct fordef *dest,               /*   destination definition             */
struct fordef *src)                /*   source definition                  */
{
     stlcpy(dest->name,src->name,FORNSZ);
     stlcpy(dest->topic,src->topic,TPCSIZ);
     stlcpy(dest->forop,src->forop,UIDSIZ);
     stlcpy(dest->forlok,src->forlok,KEYSIZ);
     dest->dfnpv=src->dfnpv;
     dest->dfprv=src->dfprv;
     dest->mxnpv=src->mxnpv;
     dest->msglif=src->msglif;
     dest->chgmsg=src->chgmsg;
     dest->chgrdm=src->chgrdm;
     dest->chgatt=src->chgatt;
     dest->chgadl=src->chgadl;
     dest->chgupk=src->chgupk;
     dest->chgdpk=src->chgdpk;
     dest->ccr=src->ccr;
     dest->pfnlvl=src->pfnlvl;
     dest->necho=src->necho;
}

struct fordef *                    /*   returns pointer to new in-mem def  */
addfarr(                           /* add new forum to in-memory arrays    */
struct fordsk *newdef,             /*   pointer to forum on-disk structure */
char *echoes)                      /*   pointer to array of echo addresses */
{
     unsigned i;
     struct fordef *memdef;
     #ifdef DEBUG
          int tmpctr;
          adr_t *tmpecho;
     #endif

     ASSERT(newdef != NULL);
     ASSERT(newdef->forum != EMLID);
     if (!alcpar((void ***)&defarr,nforums,BIGBLK)
      || !alcarr((void **)&fidxrf,sizeof(struct fidxrf),nforums,BIGBLK)) {
          return(NULL);
     }
     memdef=(struct fordef *)malloc(sizeof(struct fordef));
     if (memdef == NULL) {
          return(NULL);
     }
     memdef->dfnum=opnfdf(newdef->datfil);
     dsk2def(newdef,memdef);
     if (memdef->necho > 0) {
          ASSERT(memdef->necho <= MAXECHO);
          memdef->echoes=(char *)malloc(MAXADR*memdef->necho);
          if (memdef->echoes == NULL) {
               free(memdef);
               return(NULL);
          }
          movmem(echoes,memdef->echoes,MAXADR*memdef->necho);
          #ifdef DEBUG
               tmpecho=(adr_t *)memdef->echoes;
               for (tmpctr=0 ; tmpctr < memdef->necho ; ++tmpctr) {
                    ASSERT(strlen(tmpecho[tmpctr]) < MAXADR);
               }
          #endif
     }
     else {
          memdef->echoes=NULL;
     }
     for (i=0 ; i < nforums ; ++i) {
          if (stricmp(defarr[i]->name,memdef->name) > 0) {
               break;
          }
     }
     movmem(&defarr[i],&defarr[i+1],sizeof(struct fordef *)*(nforums-i));
     defarr[i]=memdef;
     insxrf(memdef->forum,i);
     ++nforums;
     return(memdef);
}

void
addseq(                            /* add new forum to sequence            */
struct fordef *newdef)             /*   new forum definition (in list)     */
{
     unsigned i,seq;

     ASSERT(newdef != NULL);
     #ifdef DEBUG
          for (i=0 ; i < nforums && newdef != defarr[i] ; ++i) {
          }
          ASSERT(i < nforums);
     #endif
     for (seq=0 ; seq < nforums-1 ; ++seq) {
          for (i=0 ; i < nforums ; ++i) {
               if (defarr[i]->seqid == seq && defarr[i] != newdef) {
                    break;
               }
          }
          ASSERT(i < nforums);
          ASSERT(defarr[i]->seqid == seq);
          if (stricmp(defarr[i]->name,newdef->name) >= 0) {
               break;
          }
     }
     ASSERT(seq < nforums);
     newdef->seqid=seq;
     for (i=0 ; i < nforums ; ++i) {
          if (defarr[i] != newdef && defarr[i]->seqid >= seq) {
               ++defarr[i]->seqid;
          }
     }
}

char *                             /*   returns pointer to path string     */
attpfn(                            /* attachment path+file name            */
unsigned fid,                      /*   for specified forum                */
long mid)                          /*   and message number                 */
{
     return(strupr(spr("%s\\%s",attpth(fid),attnam(mid))));
}

char *                             /*   pointer to static name buffer      */
attnam(                            /* attachment name                      */
long mid)                          /*   given message number               */
{
     static char name[FLNSIZ];

     ASSERT(mid > 0);
     sprintf(name,"%010ldA",mid);
     name[12]='\0';
     name[11]=name[10];
     name[10]=name[9];
     name[9]=name[8];
     name[8]='.';
     return(name);
}

char *                             /*   returns pointer to path string     */
attpth(                            /* attachment path                      */
unsigned fid)                      /*   for specified forum                */
{
     static char retfil[MAXPATH];

     ASSERT(fid == EMLID || fidxst(fid));
     normspec(retfil,fid == EMLID ? EMLADIR : getdef(fid)->attpath);
     return(retfil);
}

void
msg2dsk(                           /* copy in-memory msg struct to on-disk */
struct message *msg,               /*   in-memory structure                */
struct msgdsk *dsk)                /*   on-disk structure (must be large   */
{                                  /*   enough to hold 256-byte addresses) */
     ASSERT(msg != NULL);
     ASSERT(dsk != NULL);
     dsk->forum=msg->forum;
     dsk->msgid=msg->msgid;
     dsk->gmid=msg->gmid;
     dsk->thrid=msg->thrid;
     stzcpy(dsk->topic,msg->topic,TPCSIZ);
     stzcpy(dsk->history,msg->history,HSTSIZ);
     if (msg->flags&FILATT) {
          stzcpy(dsk->attname,strupr(msg->attname),FLNSIZ);
     }
     else {
          setmem(dsk->attname,FLNSIZ,0);
     }
     dsk->crdate=msg->crdate;
     dsk->crtime=msg->crtime;
     dsk->rplto=msg->rplto;
     dsk->nrpl=msg->nrpl;
     dsk->flags=msg->flags;
     if (strlen(msg->from) >= UIDSIZ) {
          stzcpy(dsk->info,msg->from,MAXADR);
          setmem(dsk->from,UIDSIZ,0);
     }
     else if ((msg->from[0] == '\0') || (msg->from[0] == NOADDRC)) {
          dsk->from[0]=NOADDRC;
          setmem(&dsk->from[1],UIDSIZ-1,0);
     }
     else {
          stzcpy(dsk->from,msg->from,UIDSIZ);
     }
     if (strlen(msg->to) >= UIDSIZ) {
          stzcpy(dsk->info+(*dsk->from == '\0' ? MAXADR : 0),msg->to,MAXADR);
          setmem(dsk->to,UIDSIZ,0);
     }
     else if ((msg->to[0] == '\0') || (msg->to[0] == NOADDRC)) {
          dsk->to[0]=NOADDRC;
          setmem(&dsk->to[1],UIDSIZ-1,0);
     }
     else {
          stzcpy(dsk->to,msg->to,UIDSIZ);
     }
}

void
dsk2msg(                           /* copy on-disk msg struct to in-memory */
struct msgdsk *dsk,                /*   on-disk structure                  */
struct message *msg)               /*   in-memory structure                */
{
     ASSERT(msg != NULL);
     ASSERT(dsk != NULL);
     msg->forum=dsk->forum;
     msg->msgid=dsk->msgid;
     msg->gmid=dsk->gmid;
     msg->thrid=dsk->thrid;
     stlcpy(msg->topic,dsk->topic,TPCSIZ);
     stlcpy(msg->history,dsk->history,HSTSIZ);
     stlcpy(msg->attname,dsk->attname,FLNSIZ);
     msg->crdate=dsk->crdate;
     msg->crtime=dsk->crtime;
     msg->rplto=dsk->rplto;
     msg->nrpl=dsk->nrpl;
     msg->flags=dsk->flags;
     if (*dsk->from == '\0') {
          stlcpy(msg->from,dsk->info,MAXADR);
     }
     else if (dsk->from[0] == NOADDRC) {
          *msg->from='\0';
     }
     else {
          stlcpy(msg->from,dsk->from,UIDSIZ);
     }
     uclfrom(msg);
     if (*dsk->to == '\0') {
          stlcpy(msg->to,dsk->info+(*dsk->from == '\0' ? MAXADR : 0),MAXADR);
     }
     else if (dsk->to[0] == NOADDRC) {
          *msg->to='\0';
     }
     else {
          stlcpy(msg->to,dsk->to,UIDSIZ);
     }
     uclto(msg);
}

void
gettxt(                            /* get text from data file buffer       */
const struct msgdsk *dskbuf,       /*   data file buffer                   */
char *text,                        /*   buffer to read text into           */
char *appinf,                      /*   buf to read app-defined info into  */
unsigned reclen)                   /*   record length                      */
{
     unsigned nvsiz,txtsiz,apisiz;
     char *cp;

     cp=textptr((struct msgdsk *)dskbuf);
     nvsiz=nvmsiz(dskbuf);
     txtsiz=reclen-nvsiz;
     ASSERT(txtsiz < reclen);
     stlcpy(text,cp,min(txtsiz,TXTLEN));
     txtsiz=strlen(cp)+1;
     apisiz=reclen-nvsiz-txtsiz;
     if (apisiz == 0) {
          *appinf='\0';
     }
     else {
          ASSERT(apisiz < reclen);
          stlcpy(appinf,cp+txtsiz,min(apisiz,TXTLEN));
     }
}

BOOL                               /*   returns TRUE if started OK         */
iniqksnd(                          /* start up !QUICK distribution         */
struct gmework *work,              /*   work area to initialize            */
char *userid)                      /*   user ID to use                     */
{
     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     setbtv(qikbb);
     work->d.q.qikbuf=NULL;
     if (acqbtv(NULL,userid,0)) {
          if ((work->d.q.qikbuf=malloc(llnbtv())) != NULL) {
               movmem(qkdptr,work->d.q.qikbuf,llnbtv());
               work->d.q.qikctr=0;
          }
     }
     rstbtv();
     return(work->d.q.qikbuf != NULL);
}

BOOL
nxtqik(                            /* get next !QUICK entry                */
struct gmework *work,              /*   work area to use                   */
char *addr)                        /*   buffer for address                 */
{
     int i;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(work->d.q.qikbuf != NULL);
     ASSERT(work->d.q.qikctr >= 0 && work->d.q.qikctr <= MAXQIK);
     if (work->d.q.qikctr == MAXQIK) {
          return(FALSE);
     }
     i=nxtqki(work->d.q.qikctr,work->d.q.qikbuf);
     if (i == NOIDX) {
          return(FALSE);
     }
     stlcpy(addr,&work->d.q.qikbuf->list[work->d.q.qikbuf->idx[i]],MAXADR);
     work->d.q.qikctr=i+1;
     return(TRUE);
}

void
clsqik(                            /* shut down !QUICK dist                */
struct gmework *work)              /*   work area used                     */
{
     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     if (work->d.q.qikbuf != NULL) {
          free(work->d.q.qikbuf);
          work->d.q.qikbuf=NULL;
     }
}

struct qikdat *                    /*   copy of pointer to buffer          */
getqik(                            /* get a !QUICK list from disk          */
struct qikdat *qikbuf,             /*   pointer to !QUICK buffer           */
char *userid)                      /*   user ID to use                     */
{
     ASSERT(qikbuf != NULL);
     setbtv(qikbb);
     if (acqbtv(NULL,userid,0)) {
          movmem(qkdptr,qikbuf,llnbtv());
     }
     else {
          iniqik(qikbuf,userid);
     }
     rstbtv();
     return(qikbuf);
}

void
savqik(                            /* save a !QUICK list to disk           */
struct qikdat *qikbuf)             /*   pointer to !QUICK buffer           */
{
     #ifdef DEBUG
          int i;

          ASSERT(qikbuf != NULL);
          for (i=0 ; i < MAXQIK ; ++i) {
               if (qikbuf->idx[i] != NOIDX) {
                    ASSERT(strlen(&qikbuf->list[qikbuf->idx[i]]) < MAXADR);
                    if (qikbuf->idx[i] > 0) {
                         ASSERT(qikbuf->list[qikbuf->idx[i]-1] == '\0');
                    }
               }
          }
     #endif
     setbtv(qikbb);
     if (acqbtv(NULL,qikbuf->userid,0)) {
          upvbtv(qikbuf,qiklen(qikbuf));
     }
     else {
          invbtv(qikbuf,qiklen(qikbuf));
     }
     rstbtv();
}

struct qikdat *                    /*   copy of pointer to buffer          */
iniqik(                            /* initialize !QUICK record for cur user*/
struct qikdat *qikbuf,             /*   pointer to !QUICK buffer           */
char *userid)                      /*   user ID to use                     */
{
     int i;

     ASSERT(qikbuf != NULL);
     stzcpy(qikbuf->userid,userid,UIDSIZ);
     for (i=0 ; i < MAXQIK ; ++i) {
          qikbuf->idx[i]=NOIDX;
     }
     return(qikbuf);
}

void
insqik(                            /* insert entry into !QUICK buffer      */
int pos,                           /*   position to insert at              */
char *entry,                       /*   new entry for position             */
struct qikdat *qikbuf)             /*   pointer to !QUICK buffer           */
{
     int i;

     ASSERT(qikbuf != NULL);
     ASSERT(pos >= 0 && pos < MAXQIK);
     ASSERT(entry != NULL);
     ASSERT(strlen(entry) < MAXADR);
     if (qikbuf->idx[pos] != NOIDX) {
          delqik(pos,qikbuf);
     }
     if ((i=lastqk(qikbuf)) == NOIDX) {
          i=0;
     }
     else {
          i+=strlen(&qikbuf->list[i])+1;
     }
     stlcpy(&qikbuf->list[i],entry,MAXADR);
     qikbuf->idx[pos]=i;
}

void
delqik(                            /* delete an entry from !QUICK buffer   */
int pos,                           /*   position to delete                 */
struct qikdat *qikbuf)             /*   pointer to !QUICK buffer           */
{
     int i,curi,nxti;

     ASSERT(qikbuf != NULL);
     ASSERT(pos >= 0 && pos < MAXQIK);
     if ((curi=qikbuf->idx[pos]) == NOIDX) {
          return;
     }
     if (curi != lastqk(qikbuf)) {
          nxti=curi+strlen(&qikbuf->list[curi])+1;
          movmem(&qikbuf->list[nxti],&qikbuf->list[curi],
                 qiklen(qikbuf)-(sizeof(struct qikdat)-1+nxti));
          for (i=0 ; i < MAXQIK ; ++i) {
               if (qikbuf->idx[i] > curi) {
                    qikbuf->idx[i]-=nxti-curi;
               }
          }
     }
     qikbuf->idx[pos]=NOIDX;
}

int                                /*   returns index into qikdat.idx      */
nxtqki(                            /* get next non-empty !QUICK index      */
int pos,                           /*   position to check first            */
struct qikdat *qikbuf)             /*   pointer to !QUICK buffer           */
{
     int i;

     ASSERT(qikbuf != NULL);
     ASSERT(pos >= 0 && pos < MAXQIK);
     for (i=pos ; i < MAXQIK ; ++i) {
          if (qikbuf->idx[i] != NOIDX) {
               return(i);
          }
     }
     return(NOIDX);
}

int
qiklen(                            /* get !QUICK buffer space used         */
struct qikdat *qikbuf)             /*   pointer to !QUICK buffer           */
{
     int i;

     ASSERT(qikbuf != NULL);
     if ((i=lastqk(qikbuf)) == NOIDX) {
          return(sizeof(struct qikdat)-1);
     }
     else {
          return(sizeof(struct qikdat)+i+strlen(&qikbuf->list[i]));
     }
}

int                                /*   returns index in qikdat.list       */
lastqk(                            /* get largest index in !QUICK list     */
struct qikdat *qikbuf)             /*   pointer to !QUICK buffer           */
{
     int i,max;

     ASSERT(qikbuf != NULL);
     max=NOIDX;
     for (i=0 ; i < MAXQIK ; ++i) {
          if (qikbuf->idx[i] > max) {
               max=qikbuf->idx[i];
          }
     }
     return(max);
}

void
foradd(                            /* add message to forum statistics      */
struct message *msg)               /*   message to add                     */
{
     ASSERT(msg != NULL);
     ASSERT(msg->forum != EMLID);
     if (incthr(msg->forum,msg->thrid)) {
          ++curfdp->nthrs;
     }
     lofadd((int)msg->flags);
}

void
lofadd(                            /* low-level add message to forum stats */
int flags)                         /*   user flags of message to add       */
{
     ++curfdp->nmsgs;
     if (flags&FILATT) {
          if (flags&FILAPV) {
               ++curfdp->nfiles;
          }
          else {
               ++curfdp->nw4app;
          }
     }
}

void
fordel(                            /* remove deleted msg from forum stats  */
struct msgdsk *msg)                /*   message that was deleted           */
{
     ASSERT(msg != NULL);
     ASSERT(msg->forum != EMLID);
     --curfdp->nmsgs;
     if (decthr(msg->forum,msg->thrid)) {
          --curfdp->nthrs;
     }
     if (msg->flags&FILATT) {
          if (msg->flags&FILAPV) {
               --curfdp->nfiles;
          }
          else {
               --curfdp->nw4app;
          }
     }
}

void
insnmsg(                           /* insert new message in current file   */
struct msgdsk *wrtbuf,             /*   buffer to use when writing         */
const struct message *msg,         /*   new message header                 */
const char *text,                  /*   pointer to message body text       */
const char *appinf)                /*   pointer to app-defined info        */
{
     invbtv(wrtbuf,fillbuf(wrtbuf,msg,text,appinf));
}

void
insmptr(                           /* insert message pointer in cur file   */
struct msgdsk *wrtbuf,             /*   buffer to use when writing         */
const struct message *msg,         /*   message pointer header             */
const struct fmidky *mptr)         /*   pointer to body message            */
{
     ASSERT(msg != NULL);
     ASSERT(wrtbuf != NULL);
     ASSERT(mptr != NULL);
     msg2dsk((struct message *)msg,wrtbuf);
     movmem(mptr,textptr(wrtbuf),sizeof(struct fmidky));
     invbtv(wrtbuf,nvmsiz(wrtbuf)+sizeof(struct fmidky));
}

void
updamsg(                           /* update message in current file       */
struct msgdsk *wrtbuf,             /*   buffer to use when writing         */
const struct message *msg,         /*   new message header                 */
const char *text,                  /*   pointer to message body text       */
const char *appinf)                /*   pointer to app-defined info        */
{
     upvbtv(wrtbuf,fillbuf(wrtbuf,msg,text,appinf));
}

void
updmptr(                           /* update message pointer in cur file   */
struct msgdsk *wrtbuf,             /*   buffer to use when writing         */
const struct message *msg,         /*   message pointer header             */
const struct fmidky *mptr)         /*   pointer to body message            */
{
     ASSERT(msg != NULL);
     ASSERT(wrtbuf != NULL);
     ASSERT(mptr != NULL);
     msg2dsk((struct message *)msg,wrtbuf);
     movmem(mptr,textptr(wrtbuf),sizeof(struct fmidky));
     upvbtv(wrtbuf,nvmsiz(wrtbuf)+sizeof(struct fmidky));
}

unsigned                           /*   returns new record size            */
fillbuf(                           /* fill data file record buffer         */
struct msgdsk *datbuf,             /*   data file buffer                   */
const struct message *msg,         /*   message header                     */
const char *text,                  /*   message body text                  */
const char *appinf)                /*   app-defined info                   */
{
     unsigned totsiz,txtsiz;
     char *cp;

     ASSERT(msg != NULL);
     ASSERT(datbuf != NULL);
     ASSERT(text != NULL);
     msg2dsk((struct message *)msg,datbuf);
     totsiz=nvmsiz(datbuf);
     stlcpy(cp=textptr(datbuf),(char *)text,TXTLEN);
     totsiz+=(txtsiz=strlen(cp)+1);
     if (appinf != NULL && *appinf != '\0') {
          cp+=txtsiz;
          stlcpy(cp,(char *)appinf,TXTLEN);
          totsiz+=strlen(cp)+1;
     }
     if (totsiz < sizeof(struct msgdsk)) {
          totsiz=sizeof(struct msgdsk);
     }
     ASSERT(totsiz <= _reclen);
     return(totsiz);
}

char *
textptr(                           /* get pointer to text portion of msg   */
struct msgdsk *dsk)                /*   given completed message structure  */
{
     return(dsk->info+(*dsk->from == '\0' ? MAXADR : 0)
                     +(*dsk->to == '\0' ? MAXADR : 0));
}

int
nvmsiz(                            /* get size of "non-variable" msg part  */
const struct msgdsk *dsk)          /*   given completed message structure  */
{
     return(sizeof(struct msgdsk)-sizeof(struct fmidky)
           +(*dsk->from == '\0' ? MAXADR : 0)
           +(*dsk->to == '\0' ? MAXADR : 0));
}

void
setfop(                            /* set the forum-op                     */
unsigned fid,                      /*   for this forum                     */
char *uid)                         /*   to this user                       */
{
     struct fordef *tmpdef;

     ASSERT(fidxst(fid));
     ASSERT(uid != NULL);
     tmpdef=getdef(fid);
     setbtv(fordefbb);
     if (acqbtv(NULL,&tmpdef->forum,DIDKY)) {
          stlcpy(tmpdef->forop,uid,UIDSIZ);
          def2dsk(tmpdef,fddptr);
          upvbtv(NULL,llnbtv());
     }
     rstbtv();
}

char *                             /*   returns pointer to description     */
gme1fdsc(                          /* get forum description                */
unsigned fid)                      /*   given forum ID                     */
{
     setbtv(fordefbb);
     if (!acqbtv(NULL,&fid,DIDKY)) {
          rstbtv();
          return(NULL);
     }
     rstbtv();
     return(fddptr->info+MAXADR*fddptr->necho);
}

void
gme1afi(                           /* get all forum info                   */
unsigned forum,                    /*   forum ID to get                    */
struct fordsk *workdef,            /*   on-disk forum definition format    */
char *desc,                        /*   buffer for description             */
char *echoes)                      /*   buffer for echoes                  */
{                                  /*   (desc & echoes can be NULL)        */
     ASSERT(fidxst(forum));
     ASSERT(workdef != NULL);
     setbtv(fordefbb);
     if (!acqbtv(NULL,&forum,DIDKY)) {
          rstbtv();
          return;
     }
     rstbtv();
     movmem(fddptr,workdef,sizeof(struct fordsk));
     curfdp=getdef(forum);
     def2dsk(curfdp,workdef);
     if (desc != NULL) {
          stlcpy(desc,fddptr->info+MAXADR*fddptr->necho,MAXFDESC);
     }
     if (echoes != NULL && curfdp->echoes != NULL) {
          movmem(curfdp->echoes,echoes,MAXADR*curfdp->necho);
     }
}

int                                /*   returns standard GME status codes  */
gme1crf(                           /* create a new forum                   */
struct gmework *work,              /*   GME work space (provided by caller)*/
struct fordsk *newdef,             /*   new forum definition structure     */
char *desc,                        /*   descriptive text                   */
char *echoes)                      /*   pointer to array of echo addresses */
{                                  /*   (may be NULL if no echoes)         */
     int i,nbtv;
     char *pname;
     adr_t *tmpecho;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(newdef != NULL);
     ASSERT(desc != NULL);
     ASSERT(strlen(newdef->name) > 0 && strlen(newdef->name) < FORNSZ);
     ASSERT(strlen(newdef->forop) > 0 && strlen(newdef->forop) < UIDSIZ);
     ASSERT(strlen(newdef->datfil) > 0 && strlen(newdef->datfil) < MAXPATH);
     ASSERT(strlen(newdef->attpath) < MAXDIR);
     #ifdef DEBUG
          if (newdef->necho > 0) {
               ASSERT(newdef->necho <= MAXECHO);
               ASSERT(echoes != NULL);
               tmpecho=(adr_t *)echoes;
               for (i=0 ; i < newdef->necho ; ++i) {
                    ASSERT(strlen(tmpecho[i]) < MAXADR);
               }
          }
     #endif
     ASSERT(newdef->msglif >= -1);
     ASSERT(!(newdef->dfprv&1) && newdef->dfprv <= OPAXES);
     ASSERT(!(newdef->mxnpv&1) && newdef->mxnpv <= OPAXES);
     ASSERT(!(newdef->dfnpv&1) && newdef->dfnpv <= OPAXES);
     ASSERT(newdef->pfnlvl >= 0 && newdef->pfnlvl <= DFTPFN);
     ASSERT(strlen(desc)+newdef->necho*MAXADR < MAXFDV);
     nbtv=0;
     while (TRUE) {
          switch (work->state1) {
          case START:
               if (nforums == MAXPARSZ) {
                    return(GME2MFR);
               }
               normspec((char *)tmpbuf,newdef->datfil);
               if (sameas((char *)tmpbuf,normspec((char *)tmpbuf+MAXPATH,
                          emldp))
                || sameas((char *)tmpbuf,normspec((char *)tmpbuf+MAXPATH,
                          FORDEFN))) {
                    return(GMEERR);
               }
               pname=strrchr(newdef->datfil,'.');
               if (pname == NULL || !sameas(pname,".DAT")) {
                    return(GMEERR);
               }
               newdef->forum=1U;
               work->state1=AVLFID;
          case AVLFID:
               setbtv(fordefbb);
               while (newdef->forum > 0
                   && qeqbtv((char *)&newdef->forum,DIDKY)) {
                    newdef->forum=nxtafid(newdef->forum);
                    if (++nbtv == MAXBTV) {
                         return(GMEAGAIN);
                    }
               }
               if (newdef->forum == 0) {
                    work->state1=START;
                    return(GMENFID);
               }
               work->state1=CRFAD;
          case CRFAD:
               work->state1=CRFDF;
               if (!dexist(newdef->attpath)) {
                    if (!makedir(newdef->attpath)) {
                         work->state1=START;
                         return(GMEERR);
                    }
                    return(GMEAGAIN);
               }
          case CRFDF:
               work->state1=INSFD;
               if (!fdfisopn(newdef->datfil)) {
                    if (nfdf == maxfdf) {
                         work->state1=START;
                         return(GME2MFL);
                    }
                    if (!fexist(newdef->datfil)) {
                         if (!creatfdf(newdef->datfil)) {
                              work->state1=START;
                              return(GMEERR);
                         }
                         return(GMEAGAIN);
                    }
               }
          case INSFD:
               setbtv(fordefbb);
               if (qeqbtv(newdef->name,DNMKY)) {
                    work->state1=START;
                    return(GMEDUP);
               }
               if (qeqbtv((char *)&newdef->forum,DIDKY)) {
                    work->state1=START;
                    return(GMEAGAIN);
               }
               newdef->nmsgs=0;
               newdef->nthrs=0;
               newdef->nfiles=0;
               newdef->nw4app=0;
               newdef->crdate=today();
               newdef->crtime=now();
               curfdp=addfarr(newdef,echoes);
               if (curfdp == NULL) {
                    work->state1=START;
                    return(GMEMEM);
               }
               addseq(curfdp);
               newdef->seqid=curfdp->seqid;
               movmem(newdef,fddptr,sizeof(struct fordsk)-1);
               zpad(fddptr->name,FORNSZ);
               zpad(fddptr->topic,TPCSIZ);
               zpad(fddptr->forop,UIDSIZ);
               zpad(fddptr->datfil,MAXPATH);
               zpad(fddptr->attpath,MAXDIR);
               zpad(fddptr->forlok,KEYSIZ);
               if (newdef->necho > 0) {
                    movmem(echoes,fddptr->info,MAXADR*newdef->necho);
                    tmpecho=(adr_t *)fddptr->info;
                    for (i=0 ; i < newdef->necho ; ++i) {
                         zpad(tmpecho[i],MAXADR);
                    }
               }
               stlcpy(fddptr->info+MAXADR*newdef->necho,desc,
                      MAXFDV-MAXADR*newdef->necho);
               setbtv(fordefbb);
               invbtv(NULL,sizeof(struct fordsk)
                          +MAXADR*newdef->necho+strlen(desc));
               if (newdef->forum > sv.hisign) {
                    sv.hisign=newdef->forum;
               }
               work->curhook=NOIDX;
               work->state1=HDLHOOK;
          case HDLHOOK:
               work->curhook=notnwf(work->curhook,newdef,desc,echoes);
               if (work->curhook != NOIDX) {
                    return(GMEAGAIN);
               }
               work->state1=START;
               return(GMEOK);
          default:
               state_error(1,"gme1crf()",work->state1);
          }
     }
}

int                                /*   returns standard GME status codes  */
gme1dlf(                           /* delete a forum                       */
struct gmework *work,              /*   GME work space (provided by caller)*/
unsigned forum)                    /*   forum ID to delete                 */
{
     int i,j;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(forum != EMLID);
     switch (work->state1) {
     case START:
          if ((i=fididx(forum)) == NOIDX) {
               return(GMEERR);
          }
          setbtv(fordefbb);
          if (!acqbtv(NULL,&forum,DIDKY)) {
               rstbtv();
               return(GMEERR);
          }
          rstbtv();
          curfdp=defarr[i];
          notdlf(fddptr,curfdp->echoes,fddptr->info+MAXADR*fddptr->necho);
          j=xrfidx(curfdp->forum);
          --nforums;
          movmem(&defarr[i+1],&defarr[i],sizeof(struct fordef *)*(nforums-i));
          movmem(&fidxrf[j+1],&fidxrf[j],sizeof(struct fidxrf)*(nforums-j));
          for (j=0 ; j < nforums ; ++j) {
               if (fidxrf[j].idx > i) {
                    --fidxrf[j].idx;
               }
               if (defarr[j]->seqid > curfdp->seqid) {
                    --defarr[j]->seqid;
               }
          }
          free(curfdp);
          sprintf(fddptr->name,"%c%u",KILLCHR,fddptr->forum);
          setbtv(fordefbb);
          upvbtv(NULL,llnbtv());
          rstbtv();
          work->curhook=NOIDX;
          work->state1=HDLHOOK;
     case HDLHOOK:
          work->curhook=notdlfl(work->curhook,forum);
          if (work->curhook != NOIDX) {
               return(GMEAGAIN);
          }
          work->state1=START;
          return(GMEOK);
     default:
          state_error(1,"gme1dlf()",work->state1);
     }
     ASSERT(FALSE);
     work->state1=START;
     return(GMEERR);     /* should never happen */
}

int                                /*   returns standard GME status codes  */
gme1modf(                          /* modify a forum definition            */
struct gmework *work,              /*   GME work space (provided by caller)*/
struct fordef *newdef,             /*   modified forum description         */
char *desc,                        /*   descriptive text                   */
char *echoes)                      /*   pointer to array of echo addresses */
{                                  /*(desc & echoes may be NULL if no chng)*/
     int curidx,i,tmplen;
     BOOL namechg;
     adr_t *tmpecho;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(newdef != NULL);
     ASSERT(strlen(newdef->name) > 0 && strlen(newdef->name) < FORNSZ);
     ASSERT(strlen(newdef->forop) > 0 && strlen(newdef->forop) < UIDSIZ);
     #ifdef DEBUG
          if (newdef->necho > 0) {
               ASSERT(newdef->necho <= MAXECHO);
               if (echoes != NULL) {
                    tmpecho=(adr_t *)echoes;
                    for (i=0 ; i < newdef->necho ; ++i) {
                         ASSERT(strlen(tmpecho[i]) < MAXADR);
                    }
               }
          }
          if (desc != NULL) {
               ASSERT(strlen(desc) < MAXFDESC);
               ASSERT(strlen(desc)+newdef->necho*MAXADR < MAXFDV);
          }
     #endif
     ASSERT(newdef->msglif >= -1);
     ASSERT(!(newdef->dfprv&1) && newdef->dfprv <= OPAXES);
     ASSERT(!(newdef->mxnpv&1) && newdef->mxnpv <= OPAXES);
     ASSERT(!(newdef->dfnpv&1) && newdef->dfnpv <= OPAXES);
     ASSERT(newdef->pfnlvl <= DFTPFN);
     ASSERT(newdef->necho >= 0 && newdef->necho < MAXECHO);
     switch (work->state1) {
     case START:
          curidx=fididx(newdef->forum);
          if (curidx == NOIDX) {
               work->state1=START;
               return(GMENFND);
          }
          i=fnmidx(newdef->name);
          if (i != NOIDX && i != curidx) {
               work->state1=START;
               return(GMEDUP);
          }
          setbtv(fordefbb);
          if (!acqbtv(NULL,&newdef->forum,DIDKY)) {
               rstbtv();
               work->state1=START;
               return(GMEERR);
          }
          rstbtv();
          curfdp=defarr[curidx];
          if (curfdp->necho != newdef->necho) {
               if (newdef->necho != 0 && echoes == NULL) {
                    work->state1=START;
                    return(GMEERR);
               }
               if (newdef->necho == 0) {
                    free(curfdp->echoes);
                    curfdp->echoes=NULL;
               }
               else {
                    curfdp->echoes=alcrsz(curfdp->echoes,MAXADR*curfdp->necho,
                                          MAXADR*newdef->necho);
               }
               if (desc == NULL) {
                    tmplen=strlen(fddptr->info+MAXADR*fddptr->necho);
                    if (MAXADR*newdef->necho+tmplen+1 >= MAXFDV) {
                         work->state1=START;
                         return(GMEFDV);
                    }
                    movmem(fddptr->info+MAXADR*fddptr->necho,
                           fddptr->info+MAXADR*newdef->necho,tmplen+1);
               }
          }
          if (newdef->necho != 0 && echoes != NULL) {
               movmem(echoes,curfdp->echoes,MAXADR*newdef->necho);
               tmpecho=(adr_t *)curfdp->echoes;
               for (i=0 ; i < newdef->necho ; ++i) {
                    zpad(tmpecho[i],MAXADR);
               }
               movmem(curfdp->echoes,fddptr->info,MAXADR*newdef->necho);
          }
          if (desc != NULL) {
               strcpy(fddptr->info+MAXADR*newdef->necho,desc);
          }
          namechg=!sameas(newdef->name,curfdp->name);
          cpymdef(curfdp,newdef);
          def2dsk(curfdp,fddptr);
          setbtv(fordefbb);
          upvbtv(NULL,sizeof(struct fordsk)+MAXADR*fddptr->necho
                     +strlen(fddptr->info+MAXADR*fddptr->necho));
          rstbtv();
          if (namechg) {
               resortf(curidx);
          }
          work->curhook=NOIDX;
          work->state1=HDLHOOK;
     case HDLHOOK:
          work->curhook=notmdf(work->curhook,getdef(newdef->forum),desc,echoes);
          if (work->curhook != NOIDX) {
               return(GMEAGAIN);
          }
          work->state1=START;
          return(GMEOK);
     default:
          state_error(1,"gme1modf()",work->state1);
     }
     ASSERT(FALSE);
     work->state1=START;
     return(GMEERR);     /* should never happen */
}

int                                /*   returns standard GME status codes  */
gme1wnm(                           /* send a new message                   */
struct gmework *work,              /*   GME work space (provided by caller)*/
struct message *msg,               /*   message header structure           */
char *text,                        /*   message body text                  */
char *filatt)                      /*   path+file name of attachment       */
{
     FILE *fp;
     char *attfnm;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(msg->forum == EMLID || fidxst(msg->forum));
     ASSERT(strlen(text) < TXTLEN);
     while (TRUE) {
          switch (work->state1) {
          case START:
               if (msg->flags&FILATT) {
                    work->state1=ATTACH;
               }
               else {
                    work->state1=WRITING;
               }
               break;
          case ATTACH:
               ASSERT(filatt != NULL);
               work->state1=WRITING;
               if (msg->flags&FILIND) {
                    stlcpy(work->cpyatt,attpfn(msg->forum,msg->msgid),MAXPATH);
                    fp=fopen(work->cpyatt,FOPWA);
                    if (fp == NULL) {
                         work->state1=START;
                         return(GMENOAT);
                    }
                    fputs(normspec((char *)tmpbuf,filatt),fp);
                    fclose(fp);
                    return(GMEAGAIN);
               }
               else {
                    attfnm=attpfn(msg->forum,msg->msgid);
                    unlink(attfnm);
                    if (rename(filatt,attfnm) != 0) {
                         work->state1=START;
                         return(GMENOAT);
                    }
                    break;
               }
          case WRITING:
               if (msg->forum == EMLID) {
                    setbtv(emlbb);
                    insnmsg(emdptr,msg,text,work->appinf);
                    rstbtv();
                    ++sv.emlopn;
               }
               else {
                    setcurf(msg->forum);
                    insnmsg(cfdptr,msg,text,work->appinf);
                    rstbtv();
                    foradd(msg);
                    ++sv.sigopn;
               }
               *work->cpyatt='\0';
               work->curhook=NOIDX;
               work->state1=HDLHOOK;
          case HDLHOOK:
               work->curhook=notnwml(work->curhook,msg,text,work->appinf);
               if (work->curhook != NOIDX) {
                    return(GMEAGAIN);
               }
               work->state1=START;
               return(GMEOK);
          default:
               state_error(1,"gme1wnm()",work->state1);
          }
     }
}

int                                /*   returns standard GME status codes  */
gme1wap(                           /* write message as pointer to another  */
struct gmework *work,              /*   GME work space (provided by caller)*/
unsigned orgfor,                   /*   forum ID of original message       */
long orgmid,                       /*   message ID of original message     */
BOOL truecopy,                     /*   is pointer a complete copy?        */
struct message *msg,               /*   message header structure           */
char *filatt)                      /*   path+file name of attachment       */
{
     BOOL ok;
     struct fmidky orgptr;
     FILE *fp;
     char *attfnm;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(orgfor == EMLID || fidxst(orgfor));
     ASSERT(msg->forum == EMLID || fidxst(msg->forum));
     while (TRUE) {
          switch (work->state1) {
          case START:
               orgptr.forum=orgfor;
               orgptr.msgid=orgmid;
               if (orgfor == EMLID) {
                    setbtv(emlbb);
                    ok=qeqbtv(&orgmid,EMIDKY);
               }
               else {
                    setcurf(orgfor);
                    ok=qeqbtv(&orgptr,FFMKY);
               }
               rstbtv();
               if (!ok) {
                    work->state1=START;
                    return(GMENFND);
               }
               msg->flags|=ISMPTR;
               if (truecopy) {
                    msg->flags|=ISTCPY;
               }
               if (msg->flags&FILATT && !truecopy) {
                    work->state1=ATTACH;
               }
               else {
                    work->state1=WRITING;
               }
               break;
          case ATTACH:
               ASSERT(filatt != NULL);
               work->state1=WRITING;
               if (msg->flags&FILIND) {
                    stlcpy(work->cpyatt,attpfn(msg->forum,msg->msgid),MAXPATH);
                    fp=fopen(work->cpyatt,FOPWA);
                    if (fp == NULL) {
                         work->state1=START;
                         return(GMENOAT);
                    }
                    fputs(normspec((char *)tmpbuf,filatt),fp);
                    fclose(fp);
                    return(GMEAGAIN);
               }
               else {
                    attfnm=attpfn(msg->forum,msg->msgid);
                    unlink(attfnm);
                    if (rename(filatt,attfnm) != 0) {
                         work->state1=START;
                         return(GMENOAT);
                    }
                    break;
               }
          case WRITING:
               orgptr.forum=orgfor;
               orgptr.msgid=orgmid;
               if (msg->forum == EMLID) {
                    setbtv(emlbb);
                    insmptr(emdptr,msg,&orgptr);
                    rstbtv();
                    ++sv.emlopn;
               }
               else {
                    setcurf(msg->forum);
                    insmptr(cfdptr,msg,&orgptr);
                    rstbtv();
                    foradd(msg);
                    ++sv.sigopn;
               }
               *work->cpyatt='\0';
               work->curhook=NOIDX;
               work->state1=HDLHOOK;
          case HDLHOOK:
               work->curhook=notnwml(work->curhook,msg,NULL,NULL);
               if (work->curhook != NOIDX) {
                    return(GMEAGAIN);
               }
               work->state1=START;
               return(GMEOK);
          default:
               state_error(1,"gme1wap()",work->state1);
          }
     }
}

int                                /*   returns standard GME status codes  */
gme1rti(                           /* read info on a thread (orig message) */
struct gmework *work,              /*   GME work space (provided by caller)*/
int direct,                        /*   direction (0=same, 1=next, -1=prev)*/
unsigned *nmsgs,                   /*   number of messages in thread       */
struct message *msg,               /*   message header structure buffer    */
char *text)                        /*   message body text buffer           */
{
     BOOL found;
     unsigned tmplen;
     struct msgdsk *tmpdsk;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(ctxok(&work->rdctx));
     ASSERT(work->rdctx.fid == EMLID || nmsgs != NULL);
     ASSERT(msg != NULL);
     ASSERT(text != NULL);
     ASSERT(work->rdctx.fid == EMLID || fidxst(work->rdctx.fid));
     switch (work->state1) {
     case START:
          if (work->rdctx.fid == EMLID) {
               setbtv(emlbb);
               tmpdsk=emdptr;
               etidky.thrid=work->rdctx.tid;
               switch (direct) {
               case 0:
                    etidky.msgid=FIRSTM;
                    found=agebtv(NULL,&etidky,ETIDKY)
                       && emdptr->thrid == etidky.thrid;
                    break;
               case 1:
                    etidky.msgid=LASTM;
                    found=agtbtv(NULL,&etidky,ETIDKY);
                    break;
               case -1:
                    etidky.msgid=FIRSTM;
                    found=altbtv(NULL,&etidky,ETIDKY);
                    break;
               }
               if (!found) {
                    work->state1=START;
                    rstbtv();
                    return(GMENFND);
               }
          }
          else {
               setcurf(work->rdctx.fid);
               tmpdsk=cfdptr;
               ftidky.forum=work->rdctx.fid;
               ftidky.thrid=work->rdctx.tid;
               switch (direct) {
               case 0:
                    ftidky.msgid=FIRSTM;
                    found=agebtv(NULL,&ftidky,FFTKY)
                       && cfdptr->thrid == ftidky.thrid;
                    break;
               case 1:
                    ftidky.msgid=LASTM;
                    found=agtbtv(NULL,&ftidky,FFTKY);
                    break;
               case -1:
                    ftidky.msgid=FIRSTM;
                    found=altbtv(NULL,&ftidky,FFTKY);
                    break;
               }
               if (!found) {
                    work->state1=START;
                    rstbtv();
                    return(GMENFND);
               }
          }
          tmplen=llnbtv();
          if (tmpdsk->flags&ISMPTR) {
               work->state1=START;
               if (getbdy(tmpdsk,msg,text)) {
                    if (msg->forum != EMLID) {
                         *nmsgs=ninthr(msg->forum,msg->thrid);
                    }
                    rstbtv();
                    return(GMEOK);
               }
               else {
                    whackit(tmpdsk);
                    rstbtv();
                    switch (direct) {
                    case 0:
                         return(GMENFND);
                    case -1:
                    case 1:
                         return(GMEAGAIN);
                    default:
                         ASSERT(FALSE);
                    }
               }
          }
          else {
               dsk2msg(tmpdsk,msg);
               gettxt(tmpdsk,text,utlapi,tmplen);
               if (msg->forum != EMLID) {
                    *nmsgs=ninthr(msg->forum,msg->thrid);
               }
               work->state1=START;
               rstbtv();
               return(GMEOK);
          }
     default:
          state_error(1,"gme1rti()",work->state1);
     }
     ASSERT(FALSE);
     return(GMEERR);     /* should never happen */
}

int                                /*   returns standard GME status codes  */
gme1rdm(                           /* read a message utility               */
struct gmework *work,              /*   GME work space (provided by caller)*/
int direct,                        /*   read direction code                */
struct message *msg,               /*   message header structure buffer    */
char *text)                        /*   message body text buffer           */
{
     unsigned bdyfor,tmplen;
     long bdymsg;
     long tmpfpos;
     struct msgdsk *tmpdsk;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(ctxok(&work->rdctx));
     ASSERT(msg != NULL);
     ASSERT(text != NULL);
     ASSERT(work->rdctx.fid == EMLID || fidxst(work->rdctx.fid));
     switch (work->state1) {
     case START:
          if (work->rdctx.fid == EMLID) {
               setbtv(emlbb);
               tmpdsk=emdptr;
               switch (work->rdctx.seq) {
               case ESQTOU:
                    stlcpy(umidky.usrid,work->rdctx.uid,UIDSIZ);
                    umidky.msgid=work->rdctx.mid;
                    if (!acqmsg(emdptr,&umidky,ETOMKY,direct,work,touchk)) {
                         work->state1=START;
                         rstbtv();
                         return(GMENFND);
                    }
                    break;
               case ESQFRU:
                    stlcpy(umidky.usrid,work->rdctx.uid,UIDSIZ);
                    umidky.msgid=work->rdctx.mid;
                    if (!acqmsg(emdptr,&umidky,EFRMKY,direct,work,fruchk)) {
                         work->state1=START;
                         rstbtv();
                         return(GMENFND);
                    }
                    break;
               case ESQTHR:
                    etidky.thrid=work->rdctx.tid;
                    etidky.msgid=work->rdctx.mid;
                    if (!acqmsg(emdptr,&etidky,ETIDKY,direct,work,ethchk)) {
                         work->state1=START;
                         rstbtv();
                         return(GMENFND);
                    }
                    break;
               default:
                    ASSERT(FALSE);
               }
          }
          else {
               setcurf(work->rdctx.fid);
               tmpdsk=cfdptr;
               switch (work->rdctx.seq) {
               case FSQFOR:
                    fmidky.forum=work->rdctx.fid;
                    fmidky.msgid=work->rdctx.mid;
                    if (!acqmsg(cfdptr,&fmidky,FFMKY,direct,work,forchk)) {
                         work->state1=START;
                         rstbtv();
                         return(GMENFND);
                    }
                    break;
               case FSQTHR:
                    ftidky.forum=work->rdctx.fid;
                    ftidky.thrid=work->rdctx.tid;
                    ftidky.msgid=work->rdctx.mid;
                    if (!acqmsg(cfdptr,&ftidky,FFTKY,direct,work,fthchk)) {
                         work->state1=START;
                         rstbtv();
                         return(GMENFND);
                    }
                    break;
               default:
                    ASSERT(FALSE);
               }
          }
          tmplen=llnbtv();
          tmpfpos=absbtv();
          if (tmpdsk->flags&ISMPTR) {
               work->state1=START;
               bdyfor=((struct fmidky *)tmpdsk->info)->forum;
               bdymsg=((struct fmidky *)tmpdsk->info)->msgid;
               gmelok(work,bdyfor,bdymsg);
               if (getbdy(tmpdsk,msg,text)) {
                    work->rdfpos=tmpfpos;
                    estctx(&work->rdctx,msg);
                    rstbtv();
                    return(GMEOK);
               }
               else {
                    gmeulk(work,bdyfor,bdymsg);
                    whackit(tmpdsk);
                    rstbtv();
                    switch (direct) {
                    case RDEXCT:
                         return(GMENFND);
                    case RDLEGT:
                    case RDLTGE:
                    case RDGELT:
                    case RDGTLE:
                    case RDNEXT:
                    case RDPREV:
                         return(GMEAGAIN);
                    default:
                         ASSERT(FALSE);
                    }
               }
          }
          else {
               dsk2msg(tmpdsk,msg);
               gettxt(tmpdsk,text,utlapi,tmplen);
               work->rdfpos=tmpfpos;
               estctx(&work->rdctx,msg);
               work->state1=START;
               rstbtv();
               return(GMEOK);
          }
     default:
          state_error(1,"gme1rdm()",work->state1);
     }
     ASSERT(FALSE);
     return(GMEERR);     /* should never happen */
}

int                                /*   returns standard GME status codes  */
gme1rdg(                           /* read a message given global ID       */
struct gmework *work,              /*   GME work space (provided by caller)*/
struct globid *gmid,               /*   global message ID of parent        */
struct message *msg,               /*   message header structure buffer    */
char *text)                        /*   message body text buffer           */
{
     unsigned bdyfor,tmpfid,tmplen;
     long bdymsg,tmpfpos;
     struct msgdsk *tmpdsk;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(ctxok(&work->rdctx));
     ASSERT(msg != NULL);
     ASSERT(text != NULL);
     ASSERT(work->rdctx.fid == EMLID || fidxst(work->rdctx.fid));
     switch (work->state1) {
     case START:
          if (work->rdctx.fid == EMLID) {
               tmpfid=(unsigned)gmid->sysid;
               if (tmpfid != EMLID && !fidxst(tmpfid)) {
                    return(GMENFND);
               }
          }
          else {
               tmpfid=work->rdctx.fid;
          }
          if (tmpfid == EMLID) {
               setbtv(emlbb);
               tmpdsk=emdptr;
               if (!acqbtv(NULL,&gmid->msgid,EMIDKY)
                || !sameas(tmpdsk->from,work->rdctx.uid)) {
                    rstbtv();
                    return(GMENFND);
               }
          }
          else {
               setcurf(tmpfid);
               tmpdsk=cfdptr;
               if (work->rdctx.fid == EMLID) {
                    fmidky.forum=tmpfid;
                    fmidky.msgid=gmid->msgid;
                    if (!acqbtv(NULL,&fmidky,FFMKY)) {
                         rstbtv();
                         return(GMENFND);
                    }
               }
               else {
                    if (!acqbtv(NULL,gmid,FGIDKY)
                     || tmpdsk->forum != tmpfid) {
                         rstbtv();
                         return(GMENFND);
                    }
               }
          }
          tmplen=llnbtv();
          tmpfpos=absbtv();
          if (tmpdsk->flags&ISMPTR) {
               work->state1=START;
               bdyfor=((struct fmidky *)tmpdsk->info)->forum;
               bdymsg=((struct fmidky *)tmpdsk->info)->msgid;
               gmelok(work,bdyfor,bdymsg);
               if (getbdy(tmpdsk,msg,text)) {
                    work->rdfpos=tmpfpos;
                    estctx(&work->rdctx,msg);
                    rstbtv();
                    return(GMEOK);
               }
               else {
                    gmeulk(work,bdyfor,bdymsg);
                    whackit(tmpdsk);
                    rstbtv();
                    return(GMENFND);
               }
          }
          else {
               dsk2msg(tmpdsk,msg);
               gettxt(tmpdsk,text,utlapi,tmplen);
               work->rdfpos=tmpfpos;
               estctx(&work->rdctx,msg);
               work->state1=START;
               rstbtv();
               return(GMEOK);
          }
     default:
          state_error(1,"gme1rdg()",work->state1);
     }
     ASSERT(FALSE);
     return(GMEERR);     /* should never happen */
}

BOOL
gidxst(                            /* does message with global ID exist    */
unsigned forum,                    /*   in this forum                      */
struct globid *globid)             /*   with this global ID                */
{
     BOOL found;

     if (forum == EMLID) {
          return(FALSE);
     }
     setcurf(forum);
     found=qeqbtv(globid,FGIDKY);
     rstbtv();
     return(found);
}

BOOL                               /*   returns TRUE if could get message  */
grabmsg(                           /* grab message using file position     */
struct gmework *work,              /*   GME work space (provided by caller)*/
struct message *msg,               /*   message header structure buffer    */
char *text)                        /*   message body text buffer           */
{
     unsigned bdyfor,tmplen;
     long bdymsg,tmpfpos;
     struct msgdsk *tmpdsk;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(ctxok(&work->rdctx));
     ASSERT(msg != NULL);
     ASSERT(text != NULL);
     if (work->rdctx.fid == EMLID) {
          setbtv(emlbb);
          tmpdsk=emdptr;
     }
     else {
          setcurf(work->rdctx.fid);
          tmpdsk=cfdptr;
     }
     if (work->rdfpos == 0L) {
          if (work->rdctx.fid == EMLID) {
               if (!acqbtv(NULL,&work->rdctx.mid,EMIDKY)) {
                    rstbtv();
                    return(FALSE);
               }
          }
          else {
               fmidky.forum=work->rdctx.fid;
               fmidky.msgid=work->rdctx.mid;
               if (!acqbtv(NULL,&fmidky,FFMKY)) {
                    rstbtv();
                    return(FALSE);
               }
          }
     }
     else {
          if (!aabbtv(NULL,work->rdfpos,0)) {
               work->rdfpos=0L;
               rstbtv();
               return(FALSE);
          }
     }
     tmplen=llnbtv();
     tmpfpos=absbtv();
     if (tmpdsk->flags&ISMPTR) {
          bdyfor=((struct fmidky *)tmpdsk->info)->forum;
          bdymsg=((struct fmidky *)tmpdsk->info)->msgid;
          gmelok(work,bdyfor,bdymsg);
          if (getbdy(tmpdsk,msg,text)) {
               work->rdfpos=tmpfpos;
          }
          else {
               gmeulk(work,bdyfor,bdymsg);
               whackit(tmpdsk);
               work->rdfpos=0L;
               rstbtv();
               return(FALSE);
          }
     }
     else {
          dsk2msg(tmpdsk,msg);
          gettxt(tmpdsk,text,utlapi,tmplen);
          work->rdfpos=tmpfpos;
     }
     rstbtv();
     return(TRUE);
}

BOOL                               /*   returns TRUE if update successful  */
updmsg(                            /* update message on disk               */
struct gmework *work,              /*   GME work space (provided by caller)*/
const struct message *msg,         /*   message header structure buffer    */
const char *text,                  /*   message body text buffer           */
const char *appinf)                /*   app-defined info buffer            */
{
     struct msgdsk *tmpdsk;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(ctxok(&work->rdctx));
     ASSERT(msg != NULL);
     ASSERT(text != NULL);
     ASSERT(work->rdfpos != 0L);
     if (work->rdctx.fid == EMLID) {
          setbtv(emlbb);
          tmpdsk=emdptr;
     }
     else {
          setcurf(work->rdctx.fid);
          tmpdsk=cfdptr;
     }
     if (aabbtv(NULL,work->rdfpos,0)) {
          if (tmpdsk->flags&ISMPTR) {
               if (tmpdsk->flags&ISTCPY) {
                    rstbtv();
                    if (fmidky.forum == EMLID) {
                         setbtv(emlbb);
                         if (acqbtv(NULL,&fmidky.msgid,EMIDKY)) {
                              updamsg(emdptr,msg,text,appinf);
                         }
                    }
                    else {
                         setcurf(fmidky.forum);
                         if (acqbtv(NULL,&fmidky,FFMKY)) {
                              updamsg(cfdptr,msg,text,appinf);
                         }
                    }
               }
               else {
                    fmidky=*((struct fmidky *)tmpdsk->info);
                    updmptr(tmpdsk,msg,&fmidky);
               }
          }
          else {
               updamsg(tmpdsk,msg,text,appinf);
          }
          rstbtv();
          return(TRUE);
     }
     work->rdfpos=0L;
     rstbtv();
     return(FALSE);
}

BOOL                               /*   returns TRUE if delete successful  */
gme1dlm(                           /* delete message (and attachment)      */
struct gmework *work)              /*   GME work space (provided by caller)*/
{
     struct msgdsk *tmpdsk;

     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(ctxok(&work->rdctx));
     ASSERT(work->rdfpos != 0L);
     if (work->rdctx.fid == EMLID) {
          setbtv(emlbb);
          tmpdsk=emdptr;
     }
     else {
          setcurf(work->rdctx.fid);
          tmpdsk=cfdptr;
     }
     if (aabbtv(NULL,work->rdfpos,0)) {
          whackit(tmpdsk);
          work->rdfpos=0L;
          rstbtv();
          return(TRUE);
     }
     work->rdfpos=0L;
     rstbtv();
     return(FALSE);
}

void
whackit(                           /* delete current msg & att & upd counts*/
struct msgdsk *msg)                /*   current message header             */
{
     lowhack(msg);
     if (msg->forum == EMLID) {
          --sv.emlopn;
     }
     else {
          --sv.sigopn;
          fordel(msg);
     }
}

void
lowhack(                           /* low-level whack message              */
struct msgdsk *msg)                /*   current message header             */
{
     delbtv();
     if ((msg->flags&FILATT) && !(msg->flags&ISTCPY)) {
          unlink(attpfn(msg->forum,msg->msgid));
     }
     notdlml(msg->forum,msg->msgid);
}

BOOL
touchk(                            /* check if message is to user          */
struct gmework *work,              /*   given work area                    */
struct msgdsk *msg)                /*   and message from disk              */
{
     return(sameas(work->rdctx.uid,msg->to));
}

BOOL
fruchk(                            /* check if message is from user        */
struct gmework *work,              /*   given work area                    */
struct msgdsk *msg)                /*   and message from disk              */
{
     return(sameas(work->rdctx.uid,msg->from));
}

BOOL
ethchk(                            /* check if message is in E-mail thread */
struct gmework *work,              /*   given work area                    */
struct msgdsk *msg)                /*   and message from disk              */
{
     return(work->rdctx.tid == msg->thrid
         && (sameas(work->rdctx.uid,msg->from)
          || sameas(work->rdctx.uid,msg->to)));
}

BOOL
forchk(                            /* check if message is in forum         */
struct gmework *work,              /*   given work area                    */
struct msgdsk *msg)                /*   and message from disk              */
{
     return(work->rdctx.fid == msg->forum);
}

BOOL
fthchk(                            /* check if message is in forum thread  */
struct gmework *work,              /*   given work area                    */
struct msgdsk *msg)                /*   and message from disk              */
{
     return(work->rdctx.fid == msg->forum && work->rdctx.tid == msg->thrid);
}

BOOL                               /*   returns TRUE if record found       */
acqmsg(                            /* acquire message from current file    */
struct msgdsk *msgbuf,             /*   message buffer                     */
void *keybuf,                      /*   key to acquire using               */
int keynum,                        /*   key number to acquire on           */
int direct,                        /*   direction code                     */
struct gmework *work,              /*   current work area                  */
BOOL (*keychk)(struct gmework *work,struct msgdsk *msg))
                                   /*   check function                     */
{
     ASSERT(msgbuf != NULL);
     ASSERT(work != NULL);
     ASSERT(gmerqopn(work));
     ASSERT(ctxok(&work->rdctx));
     switch (direct) {
     case RDEXCT:
          return(acqbtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf));
     case RDLEGT:
          return((alebtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf))
              || (agtbtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf)));
     case RDLTGE:
          return((altbtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf))
              || (agebtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf)));
     case RDGELT:
          return((agebtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf))
              || (altbtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf)));
     case RDGTLE:
          return((agtbtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf))
              || (alebtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf)));
     case RDNEXT:
          return(agtbtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf));
     case RDPREV:
          return(altbtv(msgbuf,keybuf,keynum) && keychk(work,msgbuf));
     default:
          ASSERT(FALSE);
     }
     return(FALSE);
}

BOOL                               /*   returns FALSE if body not found    */
getbdy(                            /* get body of pointer message          */
struct msgdsk *dsk,                /*   pointer message in on-disk format  */
struct message *msg,               /*   message buffer to use              */
char *text)                        /*   text buffer to use                 */
{
     int i;
     BOOL rc,found,truecopy;
     BTVFILE *tmpbb;

     ASSERT(dsk != NULL);
     ASSERT(msg != NULL);
     ASSERT(text != NULL);
     movmem(textptr(dsk),&fmidky,sizeof(struct fmidky));
     if (fmidky.forum == EMLID) {
          tmpbb=emlbb;
          setbtv(tmpbb);
          found=qeqbtv(&fmidky.msgid,EMIDKY);
     }
     else if ((i=fididx(fmidky.forum)) != NOIDX) {
          tmpbb=fdfarr[defarr[i]->dfnum];
          setbtv(tmpbb);
          found=qeqbtv(&fmidky,FFMKY);
     }
     else {
          return(FALSE);
     }
     if (found) {
          truecopy=((dsk->flags&ISTCPY) != 0L);
          if (!truecopy) {
               dsk2msg(dsk,msg);
          }
          gcrbtv(NULL,0);     /* don't care which key */
          dsk=(struct msgdsk *)tmpbb->data;
          if (truecopy) {
               dsk2msg(dsk,msg);
          }
          gettxt(dsk,text,utlapi,llnbtv());
          rc=TRUE;
     }
     else {
          rc=FALSE;
     }
     rstbtv();
     return(rc);
}

void
estctx(                            /* establish current read context       */
struct rdctx *rdctx,               /*   context structure to establish     */
struct message *msg)               /*   message to use as context          */
{
     rdctx->mid=msg->msgid;
     rdctx->tid=msg->thrid;
     /* User-ID and forum ID don't change */
}
