/***************************************************************************
 *                                                                         *
 *   GALTEXT.C                                                             *
 *                                                                         *
 *   Copyright (c) 1988-1996 Galacticomm, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   This module contains functions for handling formatted text.           *
 *                                                                         *
 *                                                - J. Alvrus  7/17/95     *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "majorbbs.h"
#include "galrtf.h"
#include "galtext.h"

#define FILREV "$Revision: 1.5 $"

int nfmthl=0;                      /* number of registered format handlers */
struct fmthlr **fhlarr=NULL;       /* array of format handlers             */

STATIC BOOL rtftoasc(const char *rtfsrc,char *ascdst,unsigned bufsiz);
STATIC BOOL rtfstrip(const char *rtfsrc,char *ascdst,unsigned bufsiz);

STATIC
struct fmthlr rtfhlr={             /* RTF format handler                   */
     "RTF",                        /*   RTF format name                    */
     0,                            /*   handler flags                      */
     isrtf,                        /*   detect RTF handler                 */
     rtftoasc,                     /*   convert to ASCII handler           */
     rtfinasc,                     /*   insert ASCII handler               */
     rtfstrip                      /*   format stripping handler           */
};

void
init__galtext(void)                /* initialize text handling DLL         */
{
     initrtf(RTFSTKSZ);
     register_fmthlr(&rtfhlr);
}

void
register_fmthlr(                   /* register formatted text handler      */
struct fmthlr *fmthlr)             /*   formatted text handling struct     */
{
     if (nfmthl == 0) {
          fhlarr=(struct fmthlr **)alcmem(sizeof(struct fmthlr *));
     }
     else {
          fhlarr=(struct fmthlr **)alcrsz(fhlarr,
                                          nfmthl*sizeof(struct fmthlr *),
                                          (nfmthl+1)*sizeof(struct fmthlr *));
     }
     fhlarr[nfmthl++]=fmthlr;
}

int
nfmthlr(void)                      /* number of registered format handlers */
{
     return(nfmthl);
}

struct fmthlr **
fharr(void)                        /* ptr to array of ptrs to fmt handlers */
{
     return(fhlarr);
}

BOOL                               /*   is this text formatted?            */
isfmtted(                          /* check for formatted text             */
const char *srcstr)                /*   string to examine                  */
{
     int i;

     for (i=0 ; i < nfmthl ; ++i) {
          if ((*(fhlarr[i]->detect))(srcstr)) {
               return(TRUE);
          }
     }
     return(FALSE);
}

const char *                       /*   returns name of format             */
fmtname(                           /* get name of format used in this text */
const char *fmtstr)                /*   string containing formatted text   */
{
     int i;

     for (i=0 ; i < nfmthl ; ++i) {
          if ((*(fhlarr[i]->detect))(fmtstr)) {
               return(fhlarr[i]->fmtdesc);
          }
     }
     return("");
}

BOOL                               /*   was able to convert?               */
cvt2asc(                           /* convert formatted text to ASCII      */
const char *fmtsrc,                /*   formatted source string            */
char *ascdst,                      /*   ASCII destination buffer           */
unsigned dstsiz)                   /*   destination buffer size            */
{
     int i;

     for (i=0 ; i < nfmthl ; ++i) {
          if ((*(fhlarr[i]->toasc))(fmtsrc,ascdst,dstsiz)) {
               return(TRUE);
          }
     }
     return(FALSE);
}

BOOL                               /*   was able to insert?                */
insasc(                            /* insert ASCII into formatted text     */
BOOL endflg,                       /*   insert at end (vs. at beginning)   */
const char *srcbuf,                /*   buffer containing ASCII text       */
char *dstbuf,                      /*   buffer containing formatted text   */
unsigned dstsiz)                   /*   destination buffer size            */
{
     int i;

     for (i=0 ; i < nfmthl ; ++i) {
          if ((*(fhlarr[i]->detect))(dstbuf)) {
               return((*(fhlarr[i]->insasc))(endflg,srcbuf,dstbuf,dstsiz));
          }
     }
     return(FALSE);
}

BOOL                               /*   was able to convert?               */
stripfmt(                          /* strip formatting                     */
const char *fmtsrc,                /*   formatted source string            */
char *ascdst,                      /*   ASCII destination buffer           */
unsigned dstsiz)                   /*   destination buffer size            */
{
     int i;

     for (i=0 ; i < nfmthl ; ++i) {
          if ((*(fhlarr[i]->stpfmt))(fmtsrc,ascdst,dstsiz)) {
               return(TRUE);
          }
     }
     return(FALSE);
}

STATIC BOOL                        /*   was able to convert?               */
rtftoasc(                          /* convert RTF to ASCII (fmthlr style)  */
const char *rtfsrc,                /*   RTF source buffer                  */
char *ascdst,                      /*   ASCII destination buffer           */
unsigned bufsiz)                   /*   destination buffer size            */
{
     int rc;

     if (!isrtf(rtfsrc)) {
          return(FALSE);
     }
     rc=rtf2asc(rtfsrc,ascdst,bufsiz);
     if (rc != RTFOK) {
          shocst("RTF CONVERSION ERROR","RTF reader error %d in rtftoasc",rc);
     }
     return(rc == RTFOK);
}

STATIC BOOL                        /*   was able to convert?               */
rtfstrip(                          /* super-simple RTF to ASCII converter  */
const char *rtfsrc,                /*   RTF source buffer                  */
char *ascdst,                      /*   ASCII destination buffer           */
unsigned bufsiz)                   /*   destination buffer size            */
{
     int rc;

     if (!isrtf(rtfsrc)) {
          return(FALSE);
     }
     rc=striprtf(rtfsrc,ascdst,bufsiz);
     if (rc != RTFOK) {
          shocst("RTF CONVERSION ERROR","RTF reader error %d in rtfstrip",rc);
     }
     return(rc == RTFOK);
}
