/***************************************************************************
 *                                                                         *
 *   PLSTUFF.C                                                             *
 *                                                                         *
 *   Copyright (c) 1992-1996 Galacticomm, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   Galacticomm Fault-Tolerant GP handler and GP dump routines,           *
 *   plus pltile() memory allocation routine for Phar Lap programs.        *
 *                                                                         *
 *                                    - Robert A. Rose 02/18/92            *
 *     (generalized to non-BBS app's) - Robert N. Stein 01/06/92           *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "plstuff.h"

#define FILREV "$Revision: 1.5 $"

#pragma inline

void interrupt far dzhandler(EXCEP_FRAME e);
void interrupt far sshandler(EXCEP_FRAME e);
void interrupt far sfhandler(EXCEP_FRAME e);
void interrupt far gphandler(EXCEP_FRAME e);
void interrupt far pfhandler(EXCEP_FRAME e);
void exchandler(EXCEP_FRAME e);
void dftdblflt(void);

int gphdlr=0;                      /* 0=report & abort, 1=silent & continue*/
char *dstname="GP.OUT";            /* destination for GP dumps             */
FILE *gpdst;                       /* File handle for GP dumps             */
int gpboot=0;                      /* Cold Boot after GP'ing 0=No 1=Yes    */

#define MAXSTP 20                  /* max number of steps in stack walk    */

int codes[MAXSTP];                 /* code segments of stack walk entries  */
int offs[MAXSTP];                  /* code offsets of stack walk entries   */
int nsteps=0;                      /* number of stack walk steps in above  */
void *lastexcro=NULL;              /* address of last exception            */
unsigned gpcntr=0;                 /* counter of GP's (reset if recovery)  */
unsigned gpslmt=2000;              /* how many times to continue thru GP   */
int prtdnam=0;                     /* ok to print DLL name on screen 1=yes */
int imr;                           /* save 8259's IMR, handler restores    */

#ifdef PHARLAP
static int ingph=0;                /* are we in the middle of a GP event?  */
static int fullgp=0;               /* show ALL selectors, etc?             */
static char *gptitle=NULL;         /* remember the "title" fm setgphand()  */
#ifdef SSCRASH
int excnums[]={0,1,12,13,14};      /* exception numbers monitored          */
#else
int excnums[]={0,-1,12,13,14};     /* exception numbers monitored          */
#endif
#define EXCDZ 0                    /* exc  0 index, divide-by-zero         */
#define EXCSS 1                    /* exc  1 index, single-step (et al)    */
#define EXCSF 2                    /* exc 12 index, stack fault            */
#define EXCGP 3                    /* exc 13 index, general protection     */
#define EXCPF 4                    /* exc 14 index, page fault             */
#define EXCNUM 5                   /* number of exceptions monitored       */
int excidx;                        /* index of exception currently handling*/
PEHANDLER oldexc[EXCNUM];          /* old exception handler vector storage */
PEHANDLER newexc[EXCNUM]={         /* new exception handler vectors        */
     dzhandler,
     sshandler,
     sfhandler,
     gphandler,
     pfhandler,
};
char *excnams[]={
     "DIVIDE-BY-ZERO",
     "DEBUG (SINGLE-STEP)",
     "STACK FAULT",
     "GP",
     "PAGE FAULT"
};
                                   /* long exception names (for GP.OUT)    */
char *excabbr[]={"DZ","SS","SF","GP","PF"};
                                   /* short exception names (for CRT)      */
unsigned int oldimsk;         /* old 8259 interrupt mask                   */

struct dllmod {
     char filnam[9];          /* DLL filename (less extension)             */
     int handle;              /* DLL handle (when loaded)                  */
} *dlls=NULL;

int maxdlls=20;               /* maximum # of dlls without reallocing      */
int numdlls=0;                /* # of dlls in "dlls" array                 */
int curdlls=0;                /* # of dlls currently loaded                */

void interrupt far gphandler(EXCEP_FRAME e);
void gpreport(EXCEP_FRAME e);
STATIC void gprebt(void);
STATIC void dftgpr(char *rfname);
char * seldetail(unsigned sel);
void plcatr(char *filnam);
void (*hdlgpr)(char *)=dftgpr;/* GP report interceptible vector            */
void (*oldcat)(char *filnam); /* saved (*catrpt)() vector                  */
void (*gpdblflt)(void)=       /* GP double fault exit vector               */
      dftdblflt;              /* (default handler, exit(99))               */
#endif

void fndnear(FILE *dst,int code,int off);
void fndproc(int code,int off,long *rou1,char *rnam1,long *rou2,char *rnam2);
void shobits(FILE *dst,char *ptr,int len);
void shostack(FILE *dst,char *ptr,int rows,int ibpval);
void sdllinfo(FILE *dst);

void
protinit(                                  /* initialize for protected mode */
char *pgmname)                  /* program name (ending with a space) or "" */
{
     #ifdef PHARLAP
          extern int colseg,monseg;
          extern unsigned _maxfarrealloc;

          setgphand(pgmname);
          _maxfarrealloc=0;             /* max # of reallocs for huge allocs    */
          DosMapRealSeg(0xb800,(ULONG)160*25,(SEL *)&colseg);
          DosMapRealSeg(0xb000,(ULONG)160*25,(SEL *)&monseg);
          oldcat=catrpt;
          catrpt=plcatr;
     #else
          pgmname=pgmname;
     #endif
     #ifdef ECLIPSE
          extern int colseg,monseg;

          colseg=0x00B8;
          monseg=0x00B0;
     #endif
     asm  in   al,0x21
     asm  xor  ah,ah
     asm  mov  imr,ax
}

void *
alctile(qty,size)                  /* allocate over 64K tiled memory region*/
unsigned qty;                           /* number of tiles                 */
unsigned size;                          /* size of each tile               */
{
     void *bigptr;

     ASSERT(qty != 0);
     ASSERT(size != 0);
     #ifdef PHARLAP
          bigptr=MK_FP(pltile(qty*(long)size,0,size,size),0);
     #else
     #ifdef ECLIPSE
          int basseg;

          if ((bigptr=getml(qty*(long)size)) == NULL) {
               catastro("NOT ENOUGH MEMORY");
          }
          if ((basseg=ecltile(bigptr,size)) == 0) {
               catastro("NOT ENOUGH SELECTORS");
          }
          bigptr=MK_FP(basseg,0);
     #else
          if ((bigptr=getml(((unsigned long)qty*
                             (unsigned long)size)+2L)) == NULL) {
               catastro("NOT ENOUGH MEMORY");
          }
          *((unsigned *)bigptr)++=size;
     #endif
     #endif
     return(bigptr);
}

void *
ptrtile(bigptr,index)              /* get address of a tile in tiled region*/
void *bigptr;                           /* alctile() return value          */
unsigned index;                         /* index of the tile, 0..qty-1     */
{
     #ifdef PHARLAP
          ASSERT(bigptr != NULL);
          return((void *)((long)bigptr+(((long)index)<<19)));
     #else
     #ifdef ECLIPSE
          ASSERT(bigptr != NULL);
          return((void *)((long)bigptr+(((long)index)<<19)));
     #else
          unsigned size;

          ASSERT(bigptr != NULL);
          size=(((unsigned *)bigptr)[-1]);
          return((void *)((char huge *)bigptr+(index*(long)size)));
     #endif
     #endif
}

          /*** Phar Lap stuff only, for the rest of the file ***/

#ifdef PHARLAP
void
loaddll(                           /* load a .DLL file                     */
char *modnam)
{
     int rv;
     char *ptr;
     HMODULE handle;
     struct dllmod *tmp;
     char realnam[40+1];

     if ((rv=DosGetModHandle(modnam,&handle)) != 0) {
          if ((rv=DosLoadModule(realnam,40,modnam,&handle)) != 0) {
               if (rv == 2) {
                    catastro("File missing: \"%s\" (while loading %s)",
                             realnam,modnam);
               }
               else {
                    plcatast(rv,spr("LOADDLL: Unable to load module %s!",
                                     modnam));
               }
          }
     }
     if (dlls == NULL) {
          dlls=(struct dllmod *)alcmem(sizeof(struct dllmod)*maxdlls);
     }
     if (numdlls+1 == maxdlls) {
          tmp=dlls;
          dlls=(struct dllmod *)alcmem(sizeof(struct dllmod)*(maxdlls+20));
          movmem(tmp,dlls,sizeof(struct dllmod)*maxdlls);
          free(tmp);
          maxdlls+=20;
     }
     if ((ptr=strchr(modnam,'.')) != NULL) {
          *ptr='\0';
     }
     stzcpy(dlls[numdlls].filnam,modnam,9);
     dlls[numdlls++].handle=handle;
     curdlls++;
}

void
initdlls(p1,p2,p3)                 /* initialize all loaded DLLs           */
char *p1,*p2,*p3;                       /* params to pass to init routines */
{
     USHORT num;
     char fncnam[33];
     HMODULE handle;
     int i,rv,fnd;
     void (*iniadr)();

     for (i=0 ; i < numdlls ; i++) {
          num=0;
          fnd=0;
          handle=dlls[i].handle;
          while (DosEnumProc((USHORT)handle,fncnam,&num) == 0) {
               if (sameto("_INIT__",fncnam)) {
                    fnd=1;
                    break;
               }
          }
          if (!fnd || (rv=DosGetProcAddr(handle,fncnam,(PPFN)&iniadr)) != 0) {
               plcatast(rv,spr("%s.DLL has no initialization routine!\n"
                               "For example: \"INIT__%s()\"",
                               dlls[i].filnam,dlls[i].filnam));
          }
          if (prtdnam) {           /* this is only good for WGSERVER.EXE   */
               sstatr(0x1D);
               prat(42,20,"%8.8s",dlls[i].filnam);
          }
          (*iniadr)(p1,p2,p3);
     }
     if (dlls != NULL) {
          free(dlls);
          dlls=NULL;
     }
     maxdlls=20;
     numdlls=0;
}

void
setgphand(char *title)             /* set the GP handler routine           */
{
     int i;

     gptitle=title;
     for (i=0 ; i < EXCNUM ; i++) {
          if (newexc[i] >= 0) {
               DosSetExceptionHandler(excnums[i],newexc[i],&oldexc[i]);
          }
     }
}

void interrupt far
dzhandler(EXCEP_FRAME e)           /* Divide-by-zero handler               */
{
     excidx=EXCDZ;
     gpslmt=0;
     exchandler(e);
}

void interrupt far
sshandler(EXCEP_FRAME e)           /* Single Step handler                  */
{
     excidx=EXCSS;
     gpslmt=0;
     exchandler(e);
}

void interrupt far
sfhandler(EXCEP_FRAME e)           /* Stack Fault handler                  */
{
     excidx=EXCSF;
     gpslmt=0;
     exchandler(e);
}

void interrupt far
gphandler(EXCEP_FRAME e)           /* General Protection Fault handler     */
{
     excidx=EXCGP;
     exchandler(e);
}

void interrupt far
pfhandler(EXCEP_FRAME e)           /* Page Fault handler                   */
{
     excidx=EXCPF;
     gpslmt=0;
     exchandler(e);
}

void
exchandler(EXCEP_FRAME e)          /* generic exception handler            */
{                                  /* (excidx identifies exception)        */
     int action=0;
     int state=0;
     int skip=0;
     int op,ops[5];
     int td,od,bc;
     char *cp;

     lastexcro=MK_FP(e.ret_cs,e.ret_ip);
     if (gphdlr == 1 && ingph == 0 && gpcntr < gpslmt) {
          ingph=1;
          cp=MK_FP(e.ret_cs,e.ret_ip);
          bc=0;
          while (!action) {
               op=*cp;
               ops[bc]=op;
               td=op>>4;
               od=op&0x0F;
               switch (state) {
               case 0:
                    if (td == 0x04 || td == 0x05 || td == 0x07 || td == 0x0e) {
                         action=-1;          /* calls, pushes, jmps GP immed    */
                    }
                    else if (od >= 0x08 && (td == 0x0c || td == 0x0d)) {
                         action=-1;          /* ESC && rets and ints            */
                    }
                    else if (td == 0x0b || td == 0x09) {
                         action=-1;          /* XCHG regs, mov immediates       */
                    }
                    else if ((od == 0x06 || od == 0x07 || od == 0x0e) && (td <= 0x01)) {
                         action=-1;
                    }
                    else if (op == 0xc2 || op == 0xc3 || op == 0x68 || op == 0x6a ||
                             op == 0x1f || op == 0x8f || op == 0x60 || op == 0x61 ||
                             op == 0xf4 || op == 0xf5 ||
                             (op >= 0xf8 && op <= 0xfd)) {
                         action=-1;
                    }
                    else if (op == 0xa8 || op == 0xa9 || op == 0x2f || op == 0x3f ||
                             op == 0x27 || op == 0x37 || op == 0x82 || op == 0xd4 ||
                             op == 0xd5) {
                         action=-1;
                    }
                    else if ((od == 0x04 || od == 0x05 || od == 0x0c || od == 0x0d)
                             && (td >= 0 && td <= 3) || op == 0xd6 || op == 0xf1 ||
                             op == 0x63 || op == 0x62) {
                         action=-1;
                    }
                    else if (op == 0x26 || op == 0x36 || op == 0x2e || op == 0x3e ||
                             op == 0xf0 || op == 0xf2 || op == 0xf3 || op == 0x64 ||
                             op == 0x65 || op == 0x66 || op == 0x67) {
                         ;                   /* prefixes                   */
                    }
                    else if (op == 0x0f) {   /* 286/386 OS operations      */
                         action=-1;
                    }
                    else if (op == 0x8d) {   /* LEA shouldn't generate GP  */
                         action=-1;
                    }
                    else if (op == 0xd7) {   /* xlat                       */
                         action=1;
                    }
                    else if (op == 0x6e || op == 0x6f || op == 0x6c || op == 0x6d ||
                             (op >= 0xaa && op <= 0xaf) || (op >= 0xa4 && op <= 0xa7)) {
                         action=1;
                    }
                    else if ((od >= 0 && od <= 0x03) && (td >= 0 && td <= 0x03)) {
                         state=1;            /* /r instructions            */
                    }
                    else if (op == 0xc4 || op == 0xc5) {
                         state=1;
                    }
                    else if ((od >= 0x08 && od <= 0x0b) && (td >= 0 && td <= 0x03)) {
                         state=1;            /* more /r instructions       */
                    }
                    else if ((op >= 0x84 && op <= 0x8c) || op == 0x8e) {
                         state=1;
                    }
                    else if (op == 0xf6 || op == 0xf7 || op == 0xfe || op == 0xff) {
                         state=1;            /* /5 instructions            */
                    }
                    else if (op == 0xA0 || op == 0xA2) {
                         state=3;
                    }
                    else if (op == 0xa1 || op == 0xa3) {
                         state=2;
                    }
                    else if (op == 0x83 || op == 0x80) {
                         skip+=1;
                         state=1;
                    }
                    else if (op == 0x69 || op == 0xc7) {
                         skip+=2;
                         state=1;
                    }
                    else if (op == 0x6b || op == 0xc6) {
                         skip+=1;
                         state=1;
                    }
                    break;
               case 1:                            /* /r instructions         */
                    if ((op&0xc0) == 0x40) {   /* disp 8 instructions        */
                         state=3;
                    }
                    else if ((op&0xc0) == 0x80) {   /* disp 16 instructions  */
                         state=2;
                    }
                    else if ((op&0xc0) == 0xc0) {   /* register instructions */
                         action=1;
                    }
                    else {
                         if ((op&0x07) == 0x06) {   /* disp 16 instruction   */
                              state=2;
                         }
                         else {
                              action=1;
                         }
                    }
                    break;
               case 2:                            /* take two bytes and go      */
                    state=3;
                    break;
               case 3:                            /* take one byte and go       */
                    action=1;
                    break;
               }
               if (action >= 0) {
                    cp++;
               }
          }
          if (action == -1) {
               gphdlr=0;
          }
          cp+=skip;
          e.ret_ip=FP_OFF(cp);
          ingph=0;
     }
     if (gphdlr == 0 || ingph == 1 || gpcntr >= gpslmt) {
          gpreport(e);
     }
     appgprecd();
     gpcntr++;
}

void
gpreport(EXCEP_FRAME e)
{
     DESC desc;
     long segbal;
     extern void gsbldn(void);

     setwin(NULL,0,0,79,24,1);
     if (ingph) {
          setatr(0xF0);
          locate(53,23);
          printf("Double fault @%04X:%04X!",e.ret_cs,e.ret_ip);
          DosSetExceptionHandler(excnums[excidx],oldexc[excidx],NULL);
          (*gpdblflt)();
     }
     setatr(0x70);
     locate(40-52/2,22);
     printf(" %s @ %04X:%04X! Recording information in %s... ",
            excabbr[excidx],e.ret_cs,e.ret_ip,dstname);
     setatr(0xF0);
     locate(40-52/2,23);
     printf("                One moment please...                ");
     #ifdef GPSCREEN
          setatr(0x70);
          locate(40-63/2,20);
          printf("  AX   BX   CX   DX   SI   DI   "
                 "  DS   ES   SS : SP   BP  flag ");
          locate(40-63/2,21);
          printf(" %04x %04x %04x %04x %04x %04x  "
                 " %04x %04x %04x:%04x %04x %04x ",
                 e.ax,e.bx,e.cx,e.dx,e.si,e.di,
                 e.ds,e.es,_SS,e.ret_rsv4,e.bp,e.ret_flags);
          locate(40-50/2,24);
          printf(" Caller:  %04x %04x %04x %04x %04x %04x %04x %04x ",
                 ((int *)MK_FP(_SS,e.bp))[1],
                 ((int *)MK_FP(_SS,e.bp))[2],
                 ((int *)MK_FP(_SS,e.bp))[3],
                 ((int *)MK_FP(_SS,e.bp))[4],
                 ((int *)MK_FP(_SS,e.bp))[5],
                 ((int *)MK_FP(_SS,e.bp))[6],
                 ((int *)MK_FP(_SS,e.bp))[7],
                 ((int *)MK_FP(_SS,e.bp))[8]);
     #endif
     ingph=1;
     asm  mov  ax,imr
     asm  out  0x21,al
     asm  sti             /* restore interrupts, in case GP while disabled */
     if ((gpdst=fopen(dstname,FOPAA)) != NULL) {
          fprintf(gpdst,"- - - - - - - - - - - - - - - - - "
                        "- - - - - - - - - - - - - - - - - -\n\n"
                        "%s%s @ %04X:%04X EC %04X (recorded %s %s)",
                        gptitle,
                        excnams[excidx],
                        e.ret_cs,
                        e.ret_ip,
                        e.error_code,
                        ncdate(today()),
                        nctime(now()));
          fflush(gpdst);
          if (gpcntr > 0) {
               fprintf(gpdst," GPCNT=%d",gpcntr);
               fflush(gpdst);
          }
          fndnear(gpdst,e.ret_cs,e.ret_ip);
          fprintf(gpdst,"\nAX=%04X BX=%04X CX=%04X DX=%04X "
                        "SI=%04X DI=%04X BP=%04X ES=%04X\n",
                        e.ax,e.bx,e.cx,e.dx,e.si,e.di,e.bp,e.es);
          fflush(gpdst);
          fprintf(gpdst,"DS=%04X  (%s)\n",e.ds,seldetail(e.ds));
          fflush(gpdst);
          fprintf(gpdst,"ES=%04X  (%s)\n",e.es,seldetail(e.es));
          fflush(gpdst);
          fprintf(gpdst,"SS=%04X  (%s)\n",_SS,seldetail(_SS));
          fflush(gpdst);
          fprintf(gpdst,"Current CS:IP==>");
          fflush(gpdst);
          if (DosGetSegDesc(e.ret_cs,&desc) == 0) {
               segbal=(long)(desc.size-e.ret_ip);
               segbal=segbal > 16L ? 16L : (segbal < 0L ? 0L : segbal);
               if (segbal > 0L) {
                    shobits(gpdst,MK_FP(e.ret_cs,e.ret_ip),(int)segbal);
               }
               else {
                    fprintf(gpdst,"Bad IP, no bytes displayed");
                    fflush(gpdst);
               }
          }
          else {
               fprintf(gpdst,"Bad CS, no bytes displayed");
               fflush(gpdst);
          }
          fprintf(gpdst,"\n");
          fflush(gpdst);
          fclose(gpdst);
     }
     appgprept();
     if ((gpdst=fopen(dstname,FOPAA)) != NULL) {
          sdllinfo(gpdst);
          fclose(gpdst);
     }
     plstak(dstname,MK_FP(_SS,e.ret_rsv4),e.bp);
     if ((gpdst=fopen(dstname,FOPAA)) != NULL) {
          fprintf(gpdst,"\n");
          fclose(gpdst);
     }
     (*hdlgpr)(dstname);
     setwin(NULL,0,0,79,24,1);
     setatr(0x70);
     locate(40-52/2,23);
     if (!gpboot) {
          printf("              Please reboot now!              ");
          while (1) {
          }
     }
     else {
          printf("                Now Rebooting!                ");
          gprebt();
     }
}

void
dftdblflt(void)                    /* (*gpdblflt)() default vector         */
{
     exit(99);
}

char *
seldetail(                         /* format details on a selector         */
unsigned sel)
{
     DESC desc;
     int rc;
     char *cp;

     if ((rc=DosGetSegDesc(sel,&desc)) == 0) {
          cp=spr("base %08lX, size %04lX, attrib=%04X",
                 desc.base,desc.size,desc.attrib);
     }
     else if (rc == 490) {
          cp=spr("invalid selector");
     }
     else {
          cp=spr("Phar Lap error %d",rc);
     }
     return(cp);
}

void
plcatr(                            /* Phar Lap CATASTRO.TXT additions      */
char *filnam)
{
     FILE *fp;
     extern void *catbpsav;

     (*oldcat)(filnam);
     if ((fp=fopen(filnam,FOPAA)) != NULL) {
          sdllinfo(fp);
          fclose(fp);
     }
     plstak(filnam,catbpsav,(unsigned int)catbpsav);
}

void
plstak(                            /* Phar Lap stack report                */
char *filnam,                      /*   file name to append report to      */
void *sssp,                        /*   SS:SP where to start               */
unsigned int bp)                   /*   initial BP in chain                */
{
     int i;
     PFN nrou1,nrou2;
     char near1[80],near2[80];

     if ((gpdst=fopen(filnam,FOPAA)) != NULL) {
          shostack(gpdst,sssp,0,bp);
          if (nsteps > 0) {
               fprintf(gpdst,"\nRoutines:\n");
          }
          fclose(gpdst);
     }
     for (i=0 ; i < nsteps ; i++) {
          if ((gpdst=fopen(filnam,FOPAA)) != NULL) {
               fndproc(codes[i],offs[i],(long *)&nrou1,near1,
                                        (long *)&nrou2,near2);
               fprintf(gpdst,"%04X:%04X %-15s<   "
                             "%04X:%04X   <   "
                             "%04X:%04X %s\n",
                       FP_SEG(nrou2),FP_OFF(nrou2),near2,
                       codes[i],offs[i],
                       FP_SEG(nrou1),FP_OFF(nrou1),near1);
               fclose(gpdst);
          }
     }
}

void
dftgpr(                            /* default (*hdlgpr)() GP report handlr */
char *rfname)                      /* report file name (eg "GP.OUT")       */
{
     rfname=rfname;
}

void
sdllinfo(FILE *dst)                /* dump selector data to file when GP   */
{
     unsigned int i;
     char buff[50];
     DESC ssinfo;
     extern unsigned __base,__top,__hincr;
     struct fndblk finfo;

     fprintf(dst,"Modules:");
     for (i=7 ; i < 0xffff ; i+=8) {
          if (DosGetModName(i,50,buff) == 0) {
               if (fnd1st(&finfo,buff,0)) {
                    fprintf(dst,"\n%04x=%-45s %s %s %10s",
                                   i,buff,
                                   ncdate((int)finfo.date),
                                   nctime((int)finfo.time),
                                   commas(ul2as(finfo.size)));
               }
               else {
                    fprintf(dst,"\n%04x=%s",i,buff);
               }
               fflush(dst);
          }
          else if (fullgp) {
               if (DosGetSegDesc(i,&ssinfo) == 0) {
                    fprintf(dst,"\n%04X=%-32s"
                            "base %08lX,  size %04lX,  attr %04X",
                            i,buff,ssinfo.base,ssinfo.size,ssinfo.attrib);
                    fflush(dst);
               }
          }
          if (i == __base) {
               fprintf(dst,"\n%04X=  <heap base>",i);
          }
          if (i == __top) {
               fprintf(dst,"\n%04X=  <heap top>   (%uK total)",i,
                           (__top-__base+__hincr)*(64/__hincr));
          }
     }
}

void
shostack(dst,ptr,rows,ibpval)      /* dump stack to file when GP occurs    */
FILE *dst;
char *ptr;
int rows;
int ibpval;
{
     int i,j;
     int *ptr2;
     int bpval;
     DESC ssinfo;
     int done=0;
     char litbuf[16+1];

     ptr2=(int *)ptr;
     bpval=ibpval;
     nsteps=0;
     if (DosGetSegDesc((SEL)FP_SEG(ptr2),(PDESC)&ssinfo) != 0) {
          return;
     }
     fprintf(gpdst,"\nStack:");
     fflush(gpdst);
     for (i=0 ; (rows == 0) ? !done : (i < rows) ; i++) {
          fprintf(dst,"\n%04X:%04X ",FP_SEG(ptr2),FP_OFF(ptr2));
          fflush(dst);
          for (j=0 ; j < 16 ; j++) {
               if (FP_OFF(ptr) >= ssinfo.size) {
                    litbuf[j]='-';
               }
               else if (*ptr < ' ') {
                    litbuf[j]='.';
               }
               else {
                    litbuf[j]=*ptr;
               }
               ptr++;
               if ((j&1) == 1) {
               }
               else if ((FP_OFF(ptr2)+1) >= ssinfo.size) {
                    fprintf(dst," ---- ");
                    fflush(dst);
                    ptr2++;
                    done=1;
               }
               else if (FP_OFF(ptr2) == bpval) {
                    bpval=*(ptr2);
                    fprintf(dst,">%04X<",*ptr2++);
                    fflush(dst);
                    if (nsteps < MAXSTP) {
                         offs[nsteps]=*(ptr2);
                         codes[nsteps++]=*(ptr2+1);
                    }
               }
               else {
                    fprintf(dst," %04X ",*ptr2++);
                    fflush(dst);
               }
          }
          litbuf[16]='\0';
          fprintf(dst,"  %s",litbuf);
     }
}

void
shobits(dst,ptr,len)               /* utility routine for code dump        */
FILE *dst;
char *ptr;
int len;
{
     int i;

     for (i=0 ; i < len ; i++) {
          fprintf(dst,"%02X ",*(ptr++));
          fflush(dst);
     }
}

void
fndproc(int code,int off,long *rou1,char *rnam1,long *rou2,char *rnam2)
{                                  /* find procedure bounds for stack walk */
     HMODULE i,modhan;
     USHORT ord;
     char buff[80];
     void far (*fn)(),far (*gp)();
     char near1[80],near2[80];
     PFN nrou1,nrou2;
     long maxdif1,maxdif2;

     gp=MK_FP(code,off);
     modhan=0;
     maxdif1=maxdif2=0x7FFFFFFFL;
     strcpy(near1,"Unknown");
     strcpy(near2,"Unknown");
     nrou1=nrou2=NULL;
     for (i=code ; i > 0 && i < (HMODULE)0xFFF8U ; i-=8) {
          if (DosGetModName(i,80,(PCHAR)buff) == 0) {
               modhan=i;
               break;
          }
     }
     if (modhan != 0) {             /* we know what guy its in now....      */
          ord=0;
          while (DosEnumProc(modhan,buff,&ord) == 0) {
               fn=0L;
               DosGetProcAddr(modhan,buff,(PPFN)&fn);
               if ((long)fn >= (long)gp) {
                    if (((long)fn-(long)gp) <= maxdif1) {
                         maxdif1=(long)fn-(long)gp;
                         nrou1=(PFN)fn;
                         strncpy(near1,buff,79);
                    }
               }
               else {
                    if (((long)gp-(long)fn) < maxdif2) {
                         maxdif2=(long)gp-(long)fn;
                         nrou2=(PFN)fn;
                         strncpy(near2,buff,79);
                    }
               }
          }
     }
     strcpy(rnam1,near1);
     strcpy(rnam2,near2);
     *(rou1)=(long)nrou1;
     *(rou2)=(long)nrou2;
}

void
fndnear(FILE *dst,int code,int off)     /* find nearest exported routine   */
{
     char near1[80],near2[80];
     PFN nrou1,nrou2;

     fndproc(code,off,(long *)&nrou1,near1,(long *)&nrou2,near2);
     fprintf(dst,"\nGP'ed between %04X:%04X %s and %04X:%04X %s",
          FP_SEG(nrou2),FP_OFF(nrou2),near2,
          FP_SEG(nrou1),FP_OFF(nrou1),near1);
     fflush(dst);
}

STATIC void
gprebt(void)                       /* reboot machine after GP              */
{
     int rbdlay;
     long l;
     int sel,*ptr;
     FILE *fp;

     if ((fp=fopen("GP.FLG",FOPWA)) != NULL) {
          fprintf(fp,"GP Occurred!");
          fclose(fp);
     }
     for (rbdlay=50 ; rbdlay > 0 ; rbdlay--) {
          for (l=0L ; l < 200000L ; l++) {
          }
     }
     DosMapRealSeg(0x0040,(ULONG)0x0074,(SEL *)&sel);
     ptr=MAKEP(sel,0x0072);
     *ptr=0;                       /* this is to set up for a cold boot    */
     DosRealFarJump((REALPTR)0xFFFF0000LU,NULL,0,0);
}

int                                /* returns base selector for tiled reg  */
pltile(size,bsel,stride,tsize)     /* PharLap tile equiv                   */
unsigned long size;
int bsel;
unsigned stride,tsize;
{
     unsigned long linadd,curadr;
     int no,seginc,rsel,i;
     unsigned frac,rv;

     if (bsel != 0) {
          DosFreeSeg(bsel);        /* free already allocated chunk if any  */
     }
     if ((rv=DosAllocLinMem(size,&linadd)) != 0) {
          plcatast(rv,"PLTILE: Error Allocating Linear Memory!");
     }
     no=(unsigned)(size/tsize);
     frac=(unsigned)(size%tsize);
asm  xor  ax,ax
asm  mov  cx,no
asm  inc  cx        /* allow for fractional remainder too */
asm  int  31h
asm  mov  rsel,ax
asm  jc   nook
asm  jmp  ok
nook: ;
     plcatast(rsel,"PLTILE: Out of contiguous selectors!");
ok: ;
asm  mov  ax,3
asm  int  31h
asm  mov  seginc,ax
asm  jc   nook2
asm  jmp  ok2
nook2: ;
     plcatast(seginc,"PLTILE: Can't determine huge increment!");
ok2: ;
     curadr=linadd;
     for (i=0 ; i < no ; i++) {
          if ((rv=DosMapLinMemToSelector(rsel+(i*seginc),curadr,tsize)) != 0) {
               plcatast(rv,"PLTILE: Error mapping full selector!");
          }
          curadr+=stride;
     }
     if (frac != 0
      && (rv=DosMapLinMemToSelector(rsel+(i*seginc),curadr,frac)) != 0) {
          plcatast(rv,"PLTILE: Error mapping fractional selector!");
     }
     return(rsel);
}

void
plcatast(int plerr,char *msg)      /* pharlap catastro                     */
{
     static int errvals[]={1,2,3,4,5,6,8,9,13,30,31,87,111,126,
                           127,180,182,190,203,213,489,490,0};
     static char *msgs[]={"Invalid Function number",
                          "File not found",
                          "Path not found",
                          "Out of file handles",
                          "Access denied",
                          "Invalid handle",
                          "Insufficient memory",
                          "Invalid memory block",
                          "Invalid data",
                          "Read fault",
                          "General failure",
                          "Invalid parameter",
                          "Buffer too small",
                          "Invalid module handle",
                          "Procedure not found",
                          "Invalid segment number",
                          "Invalid function ordinal",
                          "Invalid module type",
                          "Environment variable not found",
                          "Bad Dynamic Link",
                          "Not enough selectors available",
                          "Not a selector"} ;
     int i;

     for (i=0 ; errvals[i] != 0 ; i++) {
          if (errvals[i] == plerr) {
               catastro("%s (%s)",msg,msgs[i]);
               return;
          }
     }
     catastro("%s (Phar Lap code %04X)",msg,plerr);
}

void
DosBlockIntr(void)            /* Block all interrupts at 8259              */
{
#ifdef __HUGE__
asm  push ds
asm  mov  ax,PLSTUFF_DATA               /*  Manually load DS!                    */
asm  mov  ds,ax                    /*  ...Borland 4 won't without a C ref   */
#endif
asm  cli
asm  in   al,0x21
asm  mov  oldimsk,ax
asm  mov  al,0xFF
asm  out  0x21,al
#ifdef __HUGE__
asm      pop  ds                      /*  restore DS to entry value                     */
#endif
}

void
DosUnblockIntr(void)          /* Unblock all interrupts at 8259            */
{
#ifdef __HUGE__
asm  push ds
asm  mov  ax,PLSTUFF_DATA          /*  Manually load DS!                    */
asm  mov  ds,ax                    /*  ...Borland 4 won't without a C ref   */
#endif
asm  mov  ax,oldimsk
asm  out  0x21,al
asm  sti
#ifdef __HUGE__
asm      pop  ds                      /*  restore DS to entry value                     */
#endif
}
#endif
