/***************************************************************************
 *                                                                         *
 *   WGSRBA.C                                                              *
 *                                                                         *
 *   Copyright (c) 1998 Galacticomm, Inc.    All Rights Reserved.          *
 *                                                                         *
 *   Rebuild GALALS2.DAT                                                   *
 *                                                                         *
 *   Read the WGSUSR2.DAT file and create any default GALALS2.DAT          *
 *   entries that do not exist. This basically rebuilds GALALS2.DAT.       *
 *   The ALIAS field is NEVER modified on an existing record. On a         *
 *   New record, it will be set to NULL. If the GALALS2.DAT file does      *
 *   not exist. Create a freash one. Output can be piped or redirected.    *
 *                                                                         *
 *   Possible enhancements:                                                *
 *   1. Scan from GALALS2.DAT and compare to WGSUSR2.DAT and remove        *
 *      entries from GALALS2.DAT that don't exist in WGSUSR2.DAT           *
 *   2. Command line option to make a report of GALALS2.DAT in ASCII       *
 *      comma delimited format.                                            *
 *   3. Command line option for verbose reporting.                         *
 *                                                                         *
 *                                           - D. Rhea  03/25/98           *
 *                                                                         *
 *   Added enhancement one. This has us scan the GALALS2.DAT file in       *
 *   record order and verifiy the Uder-ID. If there is no User-ID in       *
 *   the WGSUSR2.DAT for it, we delete the alias record.                   *
 *                                                                         *
 *                                           - D. Rhea  04/01/98           *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#undef printf                      /* Use standard printf for redirection  */
#include "majorbbs.h"
#include "integrou.h"
#include "excphand.h"
#include "alias.h"

#define FILREV "$Revision: 2 $"

/***************************************************************************
 * Data file defines and structures                                        *
 ***************************************************************************/

DFAFILE *alsbb=NULL;               /* Btrieve pointer to GALALS2.DAT       */
DFAFILE *usrbb=NULL;               /* Btrieve pointer to GALUSR2.DAT       */

                                   /* Btrieve record key definitions       */
#define USRKEY 0                   /*   key 0 (User-ID) for .DAT file      */
#define CNVKEY 1                   /*   key 1 (Converted User-ID) for file */
#define ALAKEY 2                   /*   key 2 (alias) for .DAT file        */

/* this struct must stay above prototypes!                                 */
struct alsrec {                    /* Btrieve record layout                */
     CHAR userid[UIDSIZ];          /*   User-ID (key 0)                    */
     CHAR convid[UIDSIZ];          /*   alias form of User-ID (key 1)      */
     CHAR alias[ALSSIZ];           /*   Internet alias (key 2)             */
     GBOOL sntmsg;                 /*   have we sent conflict message?     */
};

static struct usracc wrkusr;       /* WGSUSR2.DAT "user" struct            */
static struct alsrec wrkals;       /* GALALS2.DAT "alias" struct           */

/***************************************************************************
 * Global definitions                                                      *
 ***************************************************************************/

LONG alias_added=0L;               /* New aliases created                  */
LONG alias_ignored=0L;             /* Existing aliases ignored             */
LONG alias_modified=0L;            /* Existing alias record modified       */
LONG alias_collisions=0L;          /* Number of alias collisions           */
LONG alias_deleted=0L;             /* Bogus aliases deleted                */
LONG alias_verified=0L;            /* Number of aliases verified           */
LONG users_processed=0L;           /* Total aliases processed              */

/***************************************************************************
 * Function prototypes                                                     *
 ***************************************************************************/

VOID
CheckAlias(                        /* See if the supplied alias is valid   */
CHAR *userid);                     /*   UserID to check                    */

VOID
VerifyAlias(                       /* Verify alias has a valid user        */
CHAR *userid);                     /*   UserID to check                    */

CHAR *                             /*   returns pointer to converted ID    */
cnvuid(                            /* convert User-ID to generic alias     */
const CHAR *userid);               /*   User-ID to convert                 */

GBOOL                              /*   returns TRUE if conflict           */
uidcfl(                            /* does User-ID have alias conflict?    */
const CHAR *userid);               /*   User-ID to check                   */

INT
main(                              /* Rebuild GALALS2.DAT                  */
INT argc,
CHAR *argv[])
{
     (VOID)argc;
#ifdef GCWINNT
     if (!canRunUtil()) {
          MessageBox(NULL,NOPROCEED,
                    argv[0], MB_ICONSTOP|MB_OK|MB_TASKMODAL|MB_SETFOREGROUND);
          return(1);
     }
#else
     (VOID)argv;
#endif // GCWINNT
     printf("WGSRBA - Rebuild Alias file\n\n");
     printf("         Copyright 1998 by Galacticomm, Inc.\n");
     printf("         All Rights Reserved\n\n");
     printf("This utility will only make sure that valid default aliases\n");
     printf("exist. It will not create or alter a regular alias. The\n");
     printf("output from this utility can be redirected.\n\n");
     if (!isfile("GALALS2.DAT")) {
          printf("GALALS2.DAT does not exist.\n");
          dfaVirgin("GALALS2", NULL);
          printf("Created new GALALS2.DAT.\n\n");
     }
     alsbb=dfaOpen("GALALS2.dat", sizeof(struct alsrec), NULL);
     usrbb=dfaOpen("WGSUSR2.dat", sizeof(struct usracc), NULL);
     dfaSetBlk(usrbb);
     if (dfaStepLO(&wrkusr)) {
          // Loop through WGSUSR2.DAT in record order
          printf("Scanning WGSUSR2.DAT...\n\n");
          do {
               users_processed++;
               //printf("\"%s\",\"%s\",\"%s\",\"%s\"\n",
               //       wrkusr.userid, wrkusr.psword, wrkusr.curcls,
               //       wrkusr.birthd);
               CheckAlias(wrkusr.userid);
          } while (dfaStepNX(&wrkusr));
     }
     dfaRstBlk();
     dfaSetBlk(alsbb);
     if (dfaStepLO(&wrkals)) {
          // Loop through GALALS2.DAT in record order
          printf("\nVerifying GALALS2.DAT...\n\n");
          do {
               VerifyAlias(wrkals.userid);
          } while (dfaStepNX(&wrkals));
     }
     dfaRstBlk();
     dfaClose(usrbb);
     dfaClose(alsbb);
     printf("\nAfter USER scan...\n");
     printf("Records processed : %ld\n", users_processed);
     printf("  Aliases ignored : %ld\n", alias_ignored);
     printf("  Aliases changed : %ld\n", alias_modified);
     printf("  Aliases created : %ld\n", alias_added);
     printf("  Aliases collided: %ld\n", alias_collisions);
     printf("After ALIAS verify...\n");
     printf("  Aliases verified: %ld\n", alias_verified);
     printf("  Aliases deleted : %ld\n", alias_deleted);
     printf("\nProcess complete...\n");
     return(0);
}

VOID
CheckAlias(                        /* See if the supplied alias is valid   */
CHAR *userid)                      /*   UserID to check                    */
{
     //CHAR c;

     dfaSetBlk(alsbb);
     if (dfaAcqEQ(&wrkals, userid, USRKEY)) {
          if (*wrkals.convid == '\0') {
               //printf("%s: Default alias is missing.\n", userid);
               if (*wrkals.alias == '\0') {
                    wrkals.sntmsg=FALSE;
                    printf("Alias and default alias missing, ");
                    printf("flaged to notify %s.\n", userid);
                    dfaUpdate(&wrkals);
                    alias_modified++;
               }
               else {
                    //printf("Record ignored. Alias exists for %s.\n", userid);
                    alias_ignored++;
               }
          }
          else {
               //printf("Record ignored. Default alias is ok.\n");
               alias_ignored++;
          }
     }
     else {
          //printf("%s User-ID not found.\n",userid);
          if (uidcfl(userid)) {
               memset(&wrkals, '\0', sizeof(struct alsrec));
               stzcpy(wrkals.userid, userid, UIDSIZ);
               wrkals.sntmsg=FALSE;
               dfaInsert(&wrkals);
               printf("Alias collision. New record without ");
               printf("default alias created for %s.\n", userid);
               alias_collisions++;
               alias_added++;
          }
          else {
               stzcpy(wrkals.userid, userid, UIDSIZ);
               stzcpy(wrkals.convid, cnvuid(userid), UIDSIZ);
               stzcpy(wrkals.alias, "", ALSSIZ);
               wrkals.sntmsg=FALSE;
               dfaInsert(&wrkals);
               //printf(
               //   "New record and default alias created for %s.\n", userid);
               alias_added++;
          }
     }
     //printf("%s [%s] <%s> %s\n", wrkals.userid, wrkals.convid,
     //       wrkals.alias, wrkals.sntmsg ? "Y" : "N");
     //c=getc(stdin);
     //if (c == 'x') {
     //     exit(0);
     //}
     dfaRstBlk();
}

VOID
VerifyAlias(                       /* Verify alias has a valid user        */
CHAR *userid)                      /*   UserID to check                    */
{
     //CHAR c;

     if (!sameas(userid, NOALID)) { /* Skip default record                 */
          dfaSetBlk(usrbb);
          alias_verified++;
          //printf("Verify: %s.\n", userid);
          if (!dfaQueryEQ(userid,0)) {
               dfaSetBlk(alsbb);
               dfaDelete();
               printf("Deleted alias record for %s\n", userid);
               alias_deleted++;
               dfaRstBlk();
          }
          //c=getc(stdin);
          //if (c == 'x') {
          //     exit(0);
          //}
          dfaRstBlk();
     }
}

                                   /* Ripped from GALALIAS.C               */
CHAR *                             /*   returns pointer to converted ID    */
cnvuid(                            /* convert User-ID to generic alias     */
const CHAR *userid)                /*   User-ID to convert                 */
{
     INT i;
     static CHAR uid[UIDSIZ];

     stzcpy(uid, userid, UIDSIZ);
     for (i=0 ; i < strlen(uid) ; i++) {
          if (uid[i] == ' ' || uid[i] == '.' || uid[i] == '_') {
               uid[i]='*';
          }
     }
     return(uid);
}

                                   /* Ripped from GALALIAS.C               */
GBOOL                              /*   returns TRUE if conflict           */
uidcfl(                            /* does User-ID have alias conflict?    */
const CHAR *userid)                /*   User-ID to check                   */
{
     GBOOL rc;
     CHAR tmpuid[UIDSIZ];
     struct alsrec tmprec;

     dfaSetBlk(alsbb);
     stlcpy(tmpuid, cnvuid(userid), UIDSIZ);
     rc=((dfaAcqEQ(&tmprec, tmpuid, CNVKEY) && !sameas(userid, tmprec.userid))
      || (dfaAcqEQ(&tmprec, tmpuid, ALAKEY) && !sameas(userid, tmprec.userid)));
     dfaRstBlk();
     return(rc);
}