/***************************************************************************
 *                                                                         *
 *   WGSPARSE.C                                                            *
 *                                                                         *
 *   Copyright (c) 1997       Galacticomm, Inc.  All rights reserved.      *
 *                                                                         *
 *   This is parses text files and seperates based on command line         *
 *   -D paramaters.  You can use #include, #ifdef & #defines.              *
 *                                                                         *
 *                                                -W. Munarsky 04/04/97    *
 *                                                                         *
 ***************************************************************************/
#include "gcomm.h"
#include "excphand.h"
#include "psrcprp.h"
#include "cmdargs.h"

#define FILREV "$Revision: 8 $"

#define WGDEV       "WGDEV"        /*  Development directory               */

#define ARGSIZ 255                 /*  Argument size                       */

#define BUILDARG    "-B"           /*  Build all argument                  */
#define SYNCFILE    "WGSSYNC.OPT"  /*  Sync file options                   */
#define SYNCOPTS    "SYNCOPTS"     /*  Environment variable name           */
#define REBFILE     "REBUILD.FLG"  /*  Rebuild flag                        */
#define DOOPTS      "-O"           /*  Options                             */
#define QUIETOPT    "-Q"           /*  Quiet (Don't print anything)        */
#define INCOPTS     "-I"           /*  Include directories in search       */

/***************************************************************************
 * Variable Definitions                                                    *
 ***************************************************************************/
static CHAR *devdir=NULL;          //  Development directory
static GBOOL setup=FALSE;          //  Has setup taken place
static GBOOL quiet=FALSE;          //  Is quiet on?

static CHAR * Options[]={
     DOOPTS,                       //  Options tracking (compile time)
     QUIETOPT,                     //  Be silent when parsing
     INCOPTS,                      //  Include file area
     NULL
};

static CmdArgs *args=NULL;       //  Pointer to arguments

/***************************************************************************
 * Function Definitions                                                    *
 ***************************************************************************/
static GBOOL                       //  True if string is an option
isOption(                          //  Is this string an option
const CHAR *opt);                  //  String to check

static VOID
BuildArgs(                         //  Build arguments
CHAR *tmparg,                      //  Temporary arguments
GBOOL IncBuild);                   //  Include for build?

static VOID
SetupArgs(VOID);                   //  Set up all arguments

static VOID
CreateRebuild(VOID);               //  Create the rebuild flags

static VOID
ConvertFile(                       //  Parse a file
const CHAR *frmFileName,           //  File to parse
const CHAR *toFileName);           //  File to parse to

static GBOOL
CheckQuiet(VOID);                  //  Check for quiet option

static GBOOL
Inits(VOID);                       //  Initialize variables

/***************************************************************************
 * Source code                                                             *
 ***************************************************************************/
MARKSOURCE(galparse);              //  Source code marker

INT
main(VOID)                         // main program loop
{
     GBOOL usage=FALSE;
TRY
     initvid();

     args=new CmdArgs;
     if (!args->Initialized()) {
          return(0);
     }
     quiet=CheckQuiet();
     if (!quiet) {
          setatr(0x0B);
          printf("\n");
          printf("GALPARSE - File parsing utility.\n");
          printf("Copyright (c) 1994-1997 Galacticomm, Inc.  ");
          printf("All rights reserved.\n\n");
     }
     if (args->argc == 1) {
          usage=TRUE;
     }
     else if (sameas(DOOPTS,args->argv[1])) {
          setup=TRUE;
          SetupArgs();
     }
     else if (sameas(args->argv[1],args->argv[2])) {
          if (!quiet) {
               setatr(0x0C);
               printf("Parse files are identical");
               exit(0);
          }
     }
     else if (args->argc >= 3) {
          ConvertFile(args->argv[1],args->argv[2]);
     }
     else {
          usage=TRUE;
     }
     if (usage) {
          setatr(0x0C);
          printf("Usage: GALPARSE <infile> <outfile> [-Q] [-D ...]\n");
          printf("\r");
          printf("     -D<argument>     Directives to parse from <infile>\n");
          printf("     -Q               Queit mode (print nothing)\n");
          printf("\r");
          exit(0);
     }
     clsvid();
EXCEPT
     if (args != NULL) {
          delete args;
     }
     return(0);
}

static GBOOL
Inits(VOID)                        //  Initialize variables
{
     if ((devdir=getenv(WGDEV)) == NULL) {
          return(FALSE);
     }
     return(TRUE);
}

static VOID
BuildArgs(                         //  Build arguments
CHAR *tmparg,                      //  Temporary arguments
GBOOL IncBuild)                    //  Include for build?
{
     INT count;

     setmem(tmparg,ARGSIZ,0);
     for (count=(setup==TRUE ? 2 : 3); count < args->argc; count++) {
          if (isOption(args->argv[count])
           || (!IncBuild && sameas(args->argv[count],BUILDARG))) {
               continue;
          }
          if (count != 1 ) {
               stlcat(tmparg," ",ARGSIZ);
          }
          stlcat(tmparg,args->argv[count],ARGSIZ);
     }
}

static GBOOL                       //  True if string is an option
isOption(                          //  Is this string an option
const CHAR *opt)                   //  String to check
{
     INT count=0;

     while (Options[count] != NULL) {
          if (sameas(Options[count],opt)) {
               return(TRUE);
          }
          count++;
     }
     return(FALSE);
}

static VOID
SetupArgs(VOID)                    //  Set up all arguments
{
     CHAR tmparg[ARGSIZ];

     BuildArgs(tmparg,TRUE);
     if (!samein(BUILDARG,tmparg)) {
          CreateRebuild();
     }
}

static VOID
CreateRebuild(VOID)                //  Create the rebuild flags
{
     CHAR tmpstg[GCMAXPTH];
     FILE *tmpdat;

     if (!Inits()) {
          printf("WGDEV Environment variable not set!\n");
          exit(0);
     }

     sprintf(tmpstg,"%s\\BIN\\%s",devdir,REBFILE);
     if ((tmpdat=fopen(tmpstg,FOPWA)) != NULL) {
          fprintf(tmpdat,"Rebuild flag file\n");
          fclose(tmpdat);
     }
}

static VOID
ConvertFile(                       //  Parse a file
const CHAR *frmFileName,           //  File to parse
const CHAR *toFileName)            //  File to parse to
{
     CHAR tmpargs[ARGSIZ];
     CHAR *ptr;
     CHAR *ptr2;
     CHAR commargs[ARGSIZ];
     INT first=FALSE;
     CHAR inFile[GCMAXPTH];
     CHAR otFile[GCMAXPTH];
     struct ffblk fb;

     strupr(fileparts(GCPART_FNAM,frmFileName,inFile,GCMAXPTH));
     strupr(fileparts(GCPART_FNAM,toFileName,otFile,GCMAXPTH));

     if (!fndfile(&fb,frmFileName,0)) {
          printf("File not found: %s\n",frmFileName);
          return;
     }
     setmem(commargs,ARGSIZ,0);
     BuildArgs(tmpargs,FALSE);
     ptr=tmpargs;

     while ((ptr2=strstr(ptr,"-D")) != NULL) {
          ptr2+=2;
          ptr=ptr2;
          while (*ptr!=' ' && *ptr!=0) {
               ptr++;
          }
          *ptr=0;
          if (first) {
               stlcat(commargs,",",ARGSIZ);
          }
          first=TRUE;
          stlcat(commargs,ptr2,ARGSIZ);
          ptr=ptr+1;
     }
     if (!quiet) {
          printf("Parsing %s: ",inFile);
          setatr(0x0C);
     }
     if (prpFile(frmFileName,toFileName,commargs) && !quiet) {
          setatr(0x0F);
          printf("DONE\n");
     }

}

static GBOOL
CheckQuiet(VOID)                   //  Check for quiet option
{
     INT count;

     for (count=0; count < args->argc; count++) {
          if (sameas(args->argv[count],QUIETOPT)) {
               return(TRUE);
          }
     }
     return(FALSE);
}

