/***************************************************************************
 *                                                                         *
 *  SMBWRAP.CPP                                                            *
 *                                                                         *
 *  Copyright (c) 1997-1998 Galacticomm, Inc.     All Rights Reserved.     *
 *                                                                         *
 *  This is the implementation file for the wrapper class for the          *
 *  library of routines for accessing  memory data files with fixed        *
 *  length records. (SMBAPI)                                               *
 *                                                                         *
 *                                              - Phil Henning   6/21/97   *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "majorbbs.h"
#include "smbwrap.h"

#define FILREV "$Revision: 5 $"

const SHORT CSmb::smbInUse           = 1;
const SHORT CSmb::smbCreateError     = 2;

MARKSOURCE(smbwrap);

SHORT CSmb::m_err                    = 0;

CSmb::CSmb(                        // construct the SMB wrapper
size_t DataLength,                 // and open a new SMB
pSMBKEYTABLE keyptr,
UINT CachedRecords) : m_count(0)
                    , m_bookmark(NULL)
                    , m_lastkey(0)
                    , m_bookkey(0)
{
     smbInit();
     m_handle=smbOpen(DataLength,keyptr,CachedRecords);
}

CSmb::CSmb() : m_handle(NULL)      // construct the SMB wrapper, but don't
             , m_count(0)          // open any SMB
             , m_bookmark(NULL)
             , m_lastkey(0)
             , m_bookkey(0)
{
     smbInit();
}

CSmb::~CSmb()                      // Destructor
{
     if (m_handle != NULL) {
          smbClose(m_handle);
          m_handle=NULL;
     }
}

GBOOL                              //  Success?
CSmb::Create(                      // Create a new SMB
size_t DataLength,                 //  Record length
pSMBKEYTABLE keyptr,               //  Pointer to keys
UINT CachedRecords)                //  # Records to cache
{
     if (m_handle != NULL) {
          m_err=smbInUse;
          return(FALSE);
     }
     if ((m_handle=smbOpen(DataLength,keyptr,CachedRecords)) != NULL) {
          return(TRUE);
     }
     m_err=smbCreateError;
     return(FALSE);
}

GBOOL                              //  Success?
CSmb::CreateFrom(                  // Create a wrapper
SMBHANDLE smb)                     //  SMB handle to create wrapper around
{
     if (m_handle != NULL) {
          m_err=smbInUse;
          return(FALSE);
     }
     m_handle=smb;
     return(TRUE);
}

VOID
CSmb::Close()                      // Close the SMB
{
     if (m_handle != NULL) {
          smbClose(m_handle);
          m_handle=NULL;
     }
}

SHORT                              //  Error code
CSmb::GetLastError()               // Retreive last error
{
     return(m_err);
}


SMBHANDLE                          //  SMB Handle we're wrapping
CSmb::GetHandle()                  // Get SMB Handle
{
     return(m_handle);
}

INT                                //  Number of records in SMB
CSmb::GetCount()                   // Get Number of records
{
     return(m_count);
}

VOID *
CSmb::Insert(                      // Insert a record into datafile
const VOID *pData)                 //  Data to insert
{
     ASSERT(m_handle != NULL);
     ASSERT(pData != NULL);
     m_count++;
     return(smbInsert(m_handle,pData));
}

VOID
CSmb::Delete()                     // Deletes the current record pointer
{
     ASSERT(m_handle != NULL);
     smbDelete(m_handle);
     m_count--;
}

VOID *                             //  Pointer to data or NULL
CSmb::GetEqual(                    // Get a record from a database
const VOID *pKeyInfo,              //  Key we are searching for
UINT uiKey)                        //  Key number to look up by
{
     VOID* ptr;
     ASSERT(m_handle != NULL);
     ASSERT(pKeyInfo != NULL);
     ptr=smbGetEqual(m_handle,pKeyInfo,uiKey);
     m_lastkey=uiKey;
     return(ptr);
}

VOID *                             //  Pointer to data or NULL
CSmb::GetPrevious()                // Get previous from current record
{
     ASSERT(m_handle != NULL);
     return(smbGetPrevious(m_handle));
}

VOID *                             //  Pointer to data or NULL
CSmb::GetHigh(                     // Get highest record in the tree
UINT uiKey)                        //  Key to get highest on
{
     ASSERT(m_handle != NULL);
     m_lastkey=uiKey;
     return(smbGetHigh(m_handle,uiKey));
}

VOID *                             //  Pointer to data or NULL
CSmb::GetLow(                      // Get the lowest record in the tree
UINT uiKey)                        //  Key to get lowest on
{
     ASSERT(m_handle != NULL);
     m_lastkey=uiKey;
     return(smbGetLow(m_handle,uiKey));
}

VOID *                             //  Pointer to data or NULL
CSmb::GetNext()                    // Get next from current record
{
     ASSERT(m_handle != NULL);
     return(smbGetNext(m_handle));
}

VOID *                             //  Pointer to data or NULL
CSmb::CurrentData()                // Gets data of current record
{
     ASSERT(m_handle != NULL);
     return(smbCurrentData(m_handle));
}

SMBPTR                             //  Absolute number of current record
CSmb::CurrentNumber()              // Gets current position of record
{
     ASSERT(m_handle != NULL);
     return(smbCurrentNumber(m_handle));
}

VOID *                             //  Pointer to data or NULL
CSmb::GetBynum(                    // Get record by absolute number
SMBPTR Number,                     //  Record number to get
UINT uiKey)                        //  Key number to load as current
{
     ASSERT(m_handle != NULL);
     m_lastkey=uiKey;
     return(smbGetBynum(m_handle,Number,uiKey));
}

VOID *                             //  Pointer to data or NULL
CSmb::GetGreater(                  // Get greater than a key
const VOID *pKeyInfo,              //  Key we are searching for
UINT uiKey)                        //  Key number to look up by
{
     ASSERT(m_handle != NULL);
     m_lastkey=uiKey;
     return(smbGetGreater(m_handle,pKeyInfo,uiKey));
}

VOID *                             //  Pointer to data or NULL
CSmb::GetLess(                     // Get less than a key
const VOID *pKeyInfo,              //  Key we are searching for
UINT uiKey)                        //  Key number to look up by
{
     ASSERT(m_handle != NULL);
     m_lastkey=uiKey;
     return(smbGetLess(m_handle,pKeyInfo,uiKey));
}

SMBPTR                             //  Current Record Number
CSmb::Bookmark()                   // Bookmark the current record
{
     ASSERT(m_handle != NULL);
     m_bookkey=m_lastkey;
     return(m_bookmark=smbCurrentNumber(m_handle));
}

VOID*                              //  Data at bookmark
CSmb::RestoreBookMark()            // Restore the bookmark
{
     ASSERT(m_handle != NULL);
     return(smbGetBynum(m_handle,m_bookmark,m_bookkey));
}

