/***************************************************************************
 *                                                                         *
 *   GALRTF.H                                                              *
 *                                                                         *
 *   Copyright (c) 1988-1997 Galacticomm, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   This file contains declarations for handling Rich Text Format (RTF).  *
 *                                                                         *
 *   The reader is based on a sample RTF reader published in Microsoft     *
 *   Application Note GC0165.                                              *
 *                                                                         *
 *                                                - J. Alvrus  7/17/95     *
 *                                                                         *
 ***************************************************************************/

#ifndef __GALRTF_H
#define __GALRTF_H

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

#define TWIPERC (1440/12)          /* twips per character                  */
                                   /*   (1440 twips/in. / 12 char/in.)     */
#define MAXTABS 14                 /* maximum number of custom tabs        */

extern
struct tabsetup {                  /* tab setup structure                  */
     CHAR type;                    /*   type (left, center, right, decimal)*/
     INT pos;                      /*   position (relative to left margin) */
} temptab;                         /* temporary tab building struct        */

                                   /* tab type codes                       */
#define TTNONE    0                /*   not a tab (and no more tabs)       */
#define TTLEFT    1                /*   left                               */
#define TTCENTER  2                /*   centered                           */
#define TTRIGHT   3                /*   right                              */
#define TTDEC     4                /*   decimal                            */
#define TTBAR     5                /*   bar                                */

#define BARCHAR '|'                /* bar character for bar tabs           */

                                   /* flag properties                      */
#define SKIPFLG  0x0001            /*   ignore current output              */
#define CHPCAPS  0x0002            /*   char prop:  all-caps               */

                                   /* byte-sized properties                */
#define PAPJUST 0                  /*   para prop:  justification          */
#define NBYTEPROP 1                /* number of byte-sized properties      */

                                   /* word-sized properties                */
#define SCPPGW  0                  /*   sect prop:  page width             */
#define SCPMRGL 1                  /*   sect prop:  left margin            */
#define SCPMRGR 2                  /*   sect prop:  right margin           */
#define PAPFI   3                  /*   para prop:  first line indent      */
#define PAPLI   4                  /*   para prop:  left indent            */
#define PAPRI   5                  /*   para prop:  right indent           */
#define NWORDPROP 6                /* number of word-sized properties      */

                                   /* paragraph justification types        */
#define JUSTL 0                    /*   left                               */
#define JUSTR 1                    /*   right                              */
#define JUSTC 2                    /*   center                             */
#define JUSTF 3                    /*   full                               */

extern
struct rtfrstt {                   /* RTF reader state                     */
     INT flag;                     /*   property/destination flags         */
     SCHAR byte[NBYTEPROP];        /*   byte-sized properties              */
     INT word[NWORDPROP];          /*   word-sized properties              */
     struct tabsetup tab[MAXTABS]; /*   tab setup table                    */
} rtfrstt,                         /* current state                        */
 *rtfrstk;                         /* state stack                          */

extern UINT rtfrksz;               /* RTF reader stack size                */

extern
struct rtfrglob {                  /* RTF reader global info               */
     INT flags;                    /*   flags                              */
     INT dftpgw;                   /*   default page width (\paperw)       */
     INT dftmrgl;                  /*   default left margin (\margl)       */
     INT dftmrgr;                  /*   default right margin (\margl)      */
     INT dfttab;                   /*   default tab width (\deftab)        */
} rtfrglob;                        /* global info buffer                   */

                                   /* global info flags                    */
#define RGFPWS 0x0001              /*   default page width set             */
#define RGFMLS 0x0002              /*   default left margin set            */
#define RGFMRS 0x0004              /*   default right margin set           */
#define RGFTBS 0x0008              /*   default tab width set              */

extern
struct kwdtab {                    /* keyword parsing table structue       */
    CHAR *keyword;                 /*   RTF keyword                        */
    INT  dftval;                   /*   default value to use               */
    GBOOL usedft;                  /*   always use default value?          */
    INT  kwdtyp;                   /*   action to take                     */
    INT  idx;                      /*   flag prop mask if kwdtyp == KWTYPPF*/
                                   /*   byte prop idx if kwdtyp == KWTYPPB */
                                   /*   word prop idx if kwdtyp == KWTYPPW */
                                   /*   unused if kwdtyp == KWTYPPT        */
                                   /*   dest type if kwdtyp == KWTYPDS     */
                                   /*   char to print if kwdtyp == KWTYPCH */
                                   /*   action code if kwdtyp == KWTYPSP   */
} kwdtab[];                        /* keyword parsing table                */

#define KWTABSZ _kwtabsz           /* # entries in keyword parsing table   */
extern UINT _kwtabsz;              /* use KWTABSZ, not this                */

                                   /* keyword action type                  */
#define KWTYPPF 0                  /*   set flag property                  */
#define KWTYPPB 1                  /*   set byte-sized property            */
#define KWTYPPW 2                  /*   set word-sized property            */
#define KWTYPPT 3                  /*   set tab property                   */
#define KWTYPDS 4                  /*   set destination                    */
#define KWTYPCH 5                  /*   output special character           */
#define KWTYPSP 6                  /*   special handling required          */

                                   /* destination codes                    */
#define DESTBODY 0                 /*   body text                          */
#define DESTSKIP -1                /*   discard output                     */

                                   /* special action codes                 */
#define SPECSKPD 0                 /*   skip destination due to \*         */
#define SPECBIN  1                 /*   read binary data                   */
#define SPECDFTB 2                 /*   set default tabs                   */
#define SPECMRGL 3                 /*   set default left margin            */
#define SPECMRGR 4                 /*   set default right margin           */
#define SPECPGW  5                 /*   set default page width             */
#define SPECPAPD 6                 /*   return paragraph props to default  */
#define SPECCHPD 7                 /*   return character props to default  */
#define SPECSCPD 8                 /*   return section props to default    */
#define SPECTAB  9                 /*   set tab                            */
#define SPECTABB 10                /*   set bar tab                        */
#define SPECHEX  11                /*   read hexadecimal character         */

extern const CHAR *srcptr;         /* RTF reader global source pointer     */
extern CHAR *dstptr;               /* RTF reader global destination pointer*/
extern UINT dstsiz;                /* RTF reader global destination size   */
extern
INT rtfrgrp,                       /* RTF reader current group (stack ptr) */
    rtfris,                        /* RTF reader internal state            */
    curtbidx;                      /* current tab index in rtfrstt.tab     */
extern LONG rtfnbin;               /* # bytes to read in binary mode       */
extern GBOOL skipunk;              /* skip if next destination unknown     */

                                   /* RTF reader internal state codes      */
#define RISNORM 0                  /*   normal input mode                  */
#define RISBIN  1                  /*   reading binary info                */

                                   /* RTF reader error codes               */
#define RTFOK     0                /*   everything's fine!                 */
#define RTFNOTRTF 1                /*   source is not RTF                  */
#define RTFUNDFL  2                /*   unmatched '}'                      */
#define RTFOVRFL  3                /*   too many '{' -- stack overflow     */
#define RTFEOF    4                /*   unexpected end of source           */
#define RTFINVHX  5                /*   invalid hex character found in data*/
#define RTFXERR   100              /*   extended errors start here         */

#define RTFIDSTR "{\\rtf"          /* RTF identifier string                */

#define RTFDONE RTFXERR            /* finished successfully code for start */
                                   /* of text finder                       */
#define AFMTSTR "\\sectd\\pard\\plain\\fs24 " /* ASCII font size string    */
#define AFMTLEN (sizeof(AFMTSTR)-1)/* ASCII font size string length        */

extern INT (*rtfputc)(INT ch);     /* output character function pointer    */
extern VOID (*finout)(VOID);       /* finish output function pointer       */
extern VOID (*grpbeg)(VOID);       /* new group beginning function pointer */
extern VOID (*grpend)(VOID);       /* group ending function pointer        */

VOID
initrtf(                           /* initialize RTF handler               */
UINT stacksiz);                    /*   number of elements to stack        */

GBOOL
isrtf(                             /* is this RTF?                         */
const CHAR *s);                    /*   string to check                    */

INT                                /*   returns result code                */
rtf2asc(                           /* convert RTF text to ASCII            */
const CHAR *rtfsrc,                /*   RTF source buffer                  */
CHAR *ascdst,                      /*   ASCII destination buffer           */
UINT bufsiz);                      /*   destination buffer size            */

VOID
ascfin(VOID);                      /* finish ASCII output                  */

INT                                /*   returns result of operation        */
putasc(                            /* output character to ASCII destination*/
INT ch);                           /*   character to output                */

UINT                               /*   returns length in characters       */
linelen(                           /* compute length of line (expand tabs) */
CHAR *s,                           /*   line buffer (not nul terminated)   */
UINT n);                           /*   # characters in line buffer        */

VOID
addline(                           /* add completed line to output buffer  */
CHAR *s,                           /*   line buffer (not nul terminated)   */
UINT n,                            /*   # characters in line buffer        */
CHAR term,                         /*   line terminator character          */
GBOOL lastline);                   /*   last line in paragraph?            */

CHAR *                             /*   returns a copy of destination      */
formline(                          /* form completed line                  */
CHAR *dst,                         /*   destination buffer                 */
CHAR *src,                         /*   source buffer (not nul terminated) */
UINT n,                            /*   # characters in line buffer        */
GBOOL lastline);                   /*   last line in paragraph?            */

INT                                /*   returns index in tab array (or -1) */
nexttab(                           /* figure out next tab                  */
INT charpos);                      /*   after this character position      */

VOID
applyjust(                         /* apply full justification to a string */
CHAR *strbuf,                      /*   string buffer                      */
UINT justlen);                     /*   length after justification         */

GBOOL                              /*   was able to insert?                */
rtfinasc(                          /* insert ASCII into RTF                */
GBOOL endflg,                      /*   insert at end (vs. at beginning)   */
const CHAR *srcbuf,                /*   buffer containing ASCII text       */
CHAR *dstbuf,                      /*   buffer containing formatted text   */
UINT dstsiz);                      /*   destination buffer size            */

CHAR *                             /*   copy of pointer to destination     */
asc2rtf(                           /* "convert" ASCII to RTF (only chars)  */
const CHAR *ascsrc,                /*   buffer containing ASCII text       */
CHAR *rtfdst,                      /*   buffer to put RTF                  */
UINT dstsiz);                      /*   size of destination buffer         */
                                   /*   (single-character EOLs assumed)    */

CHAR *                             /*   pointer to start of text           */
rtfbotxt(                          /* find start of text                   */
const CHAR *rtfsrc);               /*   in this RTF text                   */

INT                                /*   returns result code                */
striprtf(                          /* super-simple RTFormat stripper       */
const CHAR *rtfsrc,                /*   RTF source buffer                  */
CHAR *ascdst,                      /*   ASCII destination buffer           */
UINT bufsiz);                      /*   destination buffer size            */

VOID
stpfin(VOID);                      /* finish RTF stripper output           */

INT                                /*   returns result of operation        */
putstp(                            /* output char to stripped destination  */
INT ch);                           /*   character to output                */

INT                                /*   returns result code                */
initrtfr(                          /* initialize RTF reader                */
const CHAR *rtfsrc);               /*   RTF source buffer pointer          */

INT                                /*   returns result code                */
rtfread(                           /* read RTF                             */
const CHAR *rtfsrc);               /*   RTF source buffer                  */

INT                                /*   returns result of operation        */
pushrtf(VOID);                     /* save current RTF parser state        */

INT                                /*   returns result of operation        */
poprtf(VOID);                      /* restore saved RTF parser state       */

INT                                /*   returns result of operation        */
xrtfkwd(VOID);                     /* extract RTF keyword and parameter    */

INT                                /*   returns result of operation        */
parsech(                           /* route character to appropriate dest  */
INT ch);                           /*   character to output                */

INT                                /*   returns result of operation        */
hrtfkwd(                           /* find and handle RTF keyword          */
const CHAR *keyword,               /*   keyword                            */
LONG param,                        /*   parameter                          */
GBOOL parmflg);                    /*   was there a param (param is valid)?*/

INT                                /*   index of keyword in table          */
findrk(                            /* find RTF keyword in table            */
const CHAR *keyword);              /*   keyword                            */

INT                                /*   returns result of operation        */
rtfpchg(                           /* change a property                    */
INT propidx,                       /*   index of property in arrays        */
LONG propval,                      /*   value to set property to           */
INT proptyp);                      /*   type of property                   */

INT                                /*   returns result of operation        */
rtfdchg(                           /* change destination                   */
INT dest);                         /*   new destination                    */

INT                                /*   returns result of operation        */
rtfspec(                           /* evaluate a special keyword           */
INT action,                        /*   action required                    */
LONG value);                       /*   keyword value                      */

INT                                /*   returns result of operation        */
hdlhex(VOID);                      /* handle hex data keyword              */

INT                                /*   returns numeric equivalent         */
hex2bin(                           /* convert a hex character to binary    */
CHAR c);                           /*   hex character to convert           */

VOID
settab(                            /* add new tab to array of tabs         */
INT tabpos);                       /*   with this position                 */

INT                                /*   returns index of open slot         */
lasttab(VOID);                     /* find open slot in tab array          */

VOID
sorttabs(                          /* sort tabs to keep in ascending order */
INT numtabs);                      /*   number of tabs in array            */

#ifdef __cplusplus
}; // extern "C"
#endif // __cplusplus

#endif                             /* __GALRTF_H                           */
