/***************************************************************************
 *                                                                         *
 *   SMCACHE.H                                                             *
 *                                                                         *
 *   Copyright (c) 1997      Galacticomm, Inc.      All Rights Reserved.   *
 *                                                                         *
 *   Caching API for searchable memory block API.                          *
 *                                                                         *
 *                                           - J. Alvrus    5/20/97        *
 *                                                                         *
 ***************************************************************************/

#ifndef __SMCACHE_H
#define __SMCACHE_H

#include "gcomm.h"

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

typedef INT SMCPOS;                /* cache position data type             */
typedef ULONG SMCLRU;              /* LRU value data type                  */

typedef struct smCacheBlock {      /* cached block tracking structure      */
     SMBPTR block;                 /*   reference to block being tracked   */
     SMCLRU lru;                   /*   LRU value of block                 */
     SMCPOS pos;                   /*   location of block in cache memory  */
} SMCBLOCK;

#define SMC_MAXSIZE ((size_t)(65520U/sizeof(struct smCacheBlock)))

typedef struct smCache {           /* cache tracking structure             */
     SMCBLOCK *track;              /*   tracking array                     */
     VOID *cache;                  /*   memory for storing blocks          */
     SMCLRU lru;                   /*   last LRU value generated           */
     SMCPOS numBlocks;             /*   number of blocks currently in cache*/
     SMCPOS maxBlocks;             /*   maximum blocks in cache            */
} SMCACHE;

SMCACHE *                          /*   pointer to tracking structure      */
smcOpen(                           /* open a cache for an SMB file         */
size_t blockSize,                  /*   size of block being cached         */
SMCPOS maxBlocks);                 /*   max blocks to be cached            */

VOID
smcClose(                          /* close an SMB file cache              */
SMCACHE *smc);                     /*   cache to free                      */

VOID *                             /*   block in cache, NULL if cache full */
smcAlloc(                          /* allocate a block in a cache          */
SMCACHE *smc,                      /*   cache to allocate in               */
SMBPTR ref);                       /*   reference of new block             */

VOID
smcFree(                           /* free a block in a cache              */
SMCACHE *smc,                      /*   cache to free in                   */
SMBPTR ref);                       /*   reference of block to free         */

VOID
smcFreeByPos(                      /* free block in cache by position      */
SMCACHE *smc,                      /*   cache to free in                   */
SMCPOS pos);                       /*   position of block                  */

VOID *                             /*   block in cache, NULL if not present*/
smcGet(                            /* get pointer an block in a cache      */
SMCACHE *smc,                      /*   cache to retrieve from             */
SMBPTR ref);                       /*   block reference to retrieve        */

VOID *                             /*   block in cache                     */
smcGetByPos(                       /* get ptr to block by cache position   */
SMCACHE *smc,                      /*   cache to retrieve from             */
SMCPOS pos);                       /*   position of block                  */

GBOOL
smcFull(                           /* is a cache full?                     */
SMCACHE *smc);                     /*   cache to check                     */

SMCPOS                             /*   cache position of block            */
smcFindLRUPos(                     /* find least-recently-used block       */
SMCACHE *smc);                     /*   cache to find in                   */

SMBPTR                             /*   returns reference to block         */
smcBlockFromPos(                   /* get block reference given cache pos  */
SMCACHE *smc,                      /*   cache to find in                   */
SMCPOS pos);                       /*   cache position to retrieve         */

#ifdef __cplusplus
}; // extern "C"
#endif // __cplusplus

#endif // __SMCACHE_H
