{$S-,R-,V-,I-,B-,F+,O+,A+,G+}

{Conditional defines that may affect this unit}
{$I APDEFINE.INC}

{*********************************************************}
{*                    APTIMER.PAS 1.12                   *}
{*     Copyright (c) TurboPower Software 1991.           *}
{*                 All rights reserved.                  *}
{*********************************************************}

unit ApTimer;
  {-BIOS timing functions}

interface

uses                                                                  {!!.10}
  Dpmi;                                                               {!!.10}

type
  {For calculating timeouts}
  EventTimer = record
    StartTics : LongInt;
    ExpireTics : LongInt;
  end;

const
  TicsPerDay = 1573040;      {Assumes 18.20648 tics/sec}
  SecsPerDay = 86400;     {Number of seconds in one day}

  {Clock frequency of 1193180/65536 is reduced to 1675/92. This}
  {allows longint conversions of tics values upto TicsPerDay}
  TicsFreq = 1675;
  SecsFreq = 92;

  {Fast machines, >= 486-33, require an extra loop for delays}         {!!.04}
  ExtraLoop = 25;                                                      {!!.04}

var
  CountsPerMs : Word;         {Tight loop count for one millisecond}

function Tics2Secs(Tics : LongInt) : LongInt;
  {-Returns seconds value for Tics tics}

function Secs2Tics(Secs : LongInt) : LongInt;
  {-Returns tics value for Secs seconds}

procedure NewTimer(var ET : EventTimer; Tics : LongInt);
  {-Returns a set EventTimer that will expire in Tics}

procedure NewTimerSecs(var ET : EventTimer; Secs : LongInt);
  {-Returns a set EventTimer that will expire in Secs seconds}

function TimerExpired(ET : EventTimer) : Boolean;
  {-Returns True if ET has expired}

function ElapsedTime(ET : EventTimer) : LongInt;
  {-Returns elapsed time, in tics, for this timer}

function ElapsedTimeInSecs(ET : EventTimer) : LongInt;
  {-Returns elapsed time, in seconds, for this timer}

function ElapsedTimeInMSecs(ET : EventTimer) : LongInt;
  {-Returns elapsed time, in milliseconds, for this timer}

function RemainingTime(ET : EventTimer) : LongInt;
  {-Returns remaining time, in tics, for this timer}

function RemainingTimeInSecs(ET : EventTimer) : LongInt;
  {-Returns remaining time, in seconds, for this timer}

function RemainingTimeInMSecs(ET : EventTimer) : LongInt;
  {-Returns remaining time, in milliseconds, for this timer}

procedure DelayTics(Tics : LongInt);
  {-Delay for Tics tics}

procedure Delay(Ms : Word);
  {-Delay for specified number of milliseconds (same as CRT unit Delay)}

procedure CalibrateDelay;
  {-Delay calibration routine, similar to CRT unit}

implementation

uses
  TotInput, multi;

var
  {Data area for BIOS tics value}
  BiosTics : ^LongInt; {$IFDEF STONYBROOK} VOLATILE {$ENDIF} {absolute $40:$6C;}{!!.10}{!!.12}
  BiosTicsLow : ^Word; {$IFDEF STONYBROOK} VOLATILE {$ENDIF} {absolute $40:$6C;}{!!.10}{!!.12}

function Tics2Secs(Tics : LongInt) : LongInt;
  {-Returns seconds value for Tics tics}
begin
  Tics2Secs := ((Tics + 9) * SecsFreq) div TicsFreq;
end;

function Secs2Tics(Secs : LongInt) : LongInt;
  {-Returns tics value for Secs seconds}
begin
  Secs2Tics := (Secs * TicsFreq) div SecsFreq;
end;

procedure NewTimer(var ET : EventTimer; Tics : LongInt);
  {-Returns a set EventTimer that will expire in Tics}
begin
  {Max acceptable value is 24 hours}
  if Tics > TicsPerDay then
    Tics := TicsPerDay;

  with ET do begin
    StartTics := BiosTics^;                                            {!!.10}
    ExpireTics := StartTics + Tics;
  end;
end;

procedure NewTimerSecs(var ET : EventTimer; Secs : LongInt);
  {-Returns a set EventTimer}
begin
  NewTimer(ET, Secs2Tics(Secs));
end;

function TimerExpired(ET : EventTimer) : Boolean;
  {-Returns True if ET has expired}
var
  CurTics : LongInt;
begin
  with ET do begin
    {Get current tics; assume timer has expired}
    CurTics := BiosTics^;                                             {!!.10}
    TimerExpired := True;

    {Check normal expiration}
    if CurTics > ExpireTics then
      Exit;
    {Check wrapped CurTics}
    if (CurTics < StartTics) and ((CurTics + TicsPerDay) > ExpireTics) then
      Exit;

    {If we get here, timer hasn't expired yet}
    TimerExpired := False;
  end;
end;

function ElapsedTime(ET : EventTimer) : LongInt;
  {-Returns elapsed time, in tics, for this timer}
var
  CurTics : LongInt;
begin
  with ET do begin
    CurTics := BiosTics^;                                              {!!.10}
    if CurTics >= StartTics then
      {No midnight wrap yet}
      ElapsedTime := CurTics - StartTics
    else
      {Got a midnight wrap, account for it}
      ElapsedTime := (TicsPerDay - StartTics) + CurTics;
  end;
end;

function ElapsedTimeInSecs(ET : EventTimer) : LongInt;
  {-Returns elapsed time, in seconds, for this timer}
begin
  ElapsedTimeInSecs := Tics2Secs(ElapsedTime(ET));
end;

function ElapsedTimeInMSecs(ET : EventTimer) : LongInt;
  {-Returns elapsed time, in milliseconds, for this timer}
begin
  ElapsedTimeInMSecs := ElapsedTime(ET) * 55;
end;

function RemainingTime(ET : EventTimer) : LongInt;
  {-Returns remaining time, in tics, for this timer}
var
  CurTics : LongInt;
  RemainingTics : LongInt;
begin
  with ET do begin
    CurTics := BiosTics^;                                              {!!.10}
    if CurTics >= StartTics then
      {No midnight wrap yet}
      RemainingTics := ExpireTics - CurTics
    else
      {Got a midnight wrap, account for it}
      RemainingTics := (ExpireTics - TicsPerDay) - CurTics;
  end;
  if RemainingTics < 0 then
    RemainingTime := 0
  else
    RemainingTime := RemainingTics;
end;

function RemainingTimeInSecs(ET : EventTimer) : LongInt;
  {-Returns remaining time, in seconds, for this timer}
begin
  RemainingTimeInSecs := Tics2Secs(RemainingTime(ET));
end;

function RemainingTimeInMSecs(ET : EventTimer) : LongInt;
  {-Returns remaining time (in milliseconds) for this timer}
begin
  RemainingTimeInMSecs := RemainingTime(ET) * 55;
end;

procedure DelayTics(Tics : LongInt);
  {-Delay for Tics tics}
var
  ET : EventTimer;
begin
  if Tics <= 0 then
    Exit
  else if Tics > TicsPerDay then
    Tics := TicsPerDay;

  NewTimer(ET, Tics);
  repeat
    Key^. vIdleHook;
  until TimerExpired(ET);
end;

procedure Delay(Ms : Word);
  {-Delay for specified number of milliseconds}
var
  DummyLow : ^Word;  {absolute $0 : $0;}                               {!!.10}
  InitTicks : Word;
  Counts : Word;
  Done : Boolean;
  T : Word;
  Extra : Word;                                                        {!!.04}
begin
  DummyLow := Ptr(BiosDataSele, 0);                                    {!!.10}
  for T := 1 to Ms do begin
    InitTicks := DummyLow^;                                            {!!.10}
    Counts := 0;
    repeat
      Extra := 0;                                                      {!!.04}
      repeat                                                           {!!.04}
        Inc(Extra);                                                    {!!.04}
      until (Extra = ExtraLoop) or (DummyLow^ <> InitTicks);    {!!.04}{!!.10}
      Inc(Counts);
      Done := (Counts = CountsPerMs) or (DummyLow^ <> InitTicks);      {!!.10}
    until Done;
  end;
end;

procedure CalibrateDelay;
  {-Delay calibration routine, similar to CRT unit}
var
  InitTics : Word;
  Counts : Word;
  Done : Boolean;
  Extra : Word;                                                        {!!.04}
begin
  BeginCrit;
  inline($FB);                   {sti}
  CountsPerMs := 65535;          {Upper limit for counts per tic}

  {Wait for tick to change}
  InitTics := BiosTicsLow^;                                            {!!.10}
  repeat
  until BiosTicsLow^ <> InitTics;                                      {!!.10}

  {Now count until it changes again or it reaches a limit}
  InitTics := BiosTicsLow^;                                            {!!.10}
  Counts := 0;
  repeat
    Extra := 0;                                                        {!!.04}
    repeat                                                             {!!.04}
      Inc(Extra);                                                      {!!.04}
    until (Extra = ExtraLoop) or (BiosTicsLow^ <> InitTics);    {!!.04}{!!.10}
    Inc(Counts);                                                       {!!.04}
    Done := (Counts = CountsPerMs) or (BiosTicsLow^ <> InitTics);      {!!.10}
  until Done;

  {Convert to counts per millisecond}
  CountsPerMs := Counts div 55;
  EndCrit;
end;

begin
  BiosTics := Ptr(BiosDataSele, $6C);                                  {!!.10}
  BiosTicsLow := Ptr(BiosDataSele, $6C);                               {!!.10}
  CalibrateDelay;  {initializes CountsPerMs for Delay}
end.
