{$O+}
UNIT TXTMsg;

INTERFACE

Uses
  Overlay,Dos,OPCrt;

{$I LinkVar.Pas}

{- Initialise an empty buffer -}
PROCEDURE NewBuffer(Var T :TextBuffer);

{- Return true if the buffer is empty -}
FUNCTION EmptyBuffer(Var T :TextBuffer) :Boolean;

{- Return a pointer to the first line of a buffer -}
FUNCTION FirstLine(Var T :TextBuffer) :TextNodePtr;

{- Return a pointer to the last line of a buffer -}
FUNCTION LastLine(Var T :TextBuffer) :TextNodePtr;

{- Return the next line in a buffer -}
FUNCTION NextLine(Var T :TextBuffer;  Pos :TextNodePtr) :TextNodePtr;

{- Return the previous line in a buffer -}
FUNCTION PrevLine(Var T :TextBuffer;  Pos :TextNodePtr) :TextNodePtr;

{- Add a line to the end of a buffer -}
PROCEDURE AddToEnd(Var T :TextBuffer;  Line :String);

{- Insert a line before another line -}
PROCEDURE AddInsert(Var T :TextBuffer;  Pos :TextNodePtr;  Line :String);

{- Delete a line and return the next line or Nil if it was the last line -}
FUNCTION DeleteLine(Var T :TextBuffer;  Var Pos :TextNodePtr) :TextNodePtr;

{- Delete a buffer -}
PROCEDURE DeleteBuffer(Var T :TextBuffer);

{- Retrieve the text value from a line -}
FUNCTION GetTextLine(Var T: TextBuffer;  Pos :TextNodePtr) :String;

{- Assign a new string to a line of text -}
PROCEDURE ModifyTextLine(Var T: TextBuffer;  Pos :TextNodePtr;  Line :String);

{- Word Wrap the buffer -}
PROCEDURE WrapBuffer(Var T :TextBuffer;  Margin :Byte);

{- Create a new buffer with maximum length (255) lines -}
PROCEDURE UnWrapBuffer(Var T,W :TextBuffer);

{- Count the number of lines in a buffer -}
FUNCTION BufferLength(Var T :TextBuffer) :Word;

PROCEDURE DeleteBlanks(Var T :TextBuffer);

IMPLEMENTATION

Uses Common;

PROCEDURE NewBuffer(Var T :TextBuffer);
  Begin
    T.First := Nil;
    T.Last := Nil;
  End; {NewBuffer}

PROCEDURE DeleteBuffer(Var T :TextBuffer);
  Var Step,Temp :TextNodePtr;
  Begin
    Step := T.First;
    While (Step <> Nil) do
    Begin
      FreeMem(Step^.Line,Length(Step^.Line^)+1);
      Temp := Step;
      Step := Step^.Next;
      Dispose(Temp);
    End; {While}
    NewBuffer(T);
  End; {DeleteBuffer}

FUNCTION EmptyBuffer(Var T :TextBuffer) :Boolean;
  Begin
    EmptyBuffer := (T.First = Nil);
  End; {EmptyBuffer}

FUNCTION FirstLine(Var T :TextBuffer) :TextNodePtr;
  Begin
    FirstLine := T.First;
  End; {FirstLine}

FUNCTION LastLine(Var T :TextBuffer) :TextNodePtr;
  Begin
    LastLine := T.Last;
  End; {LastLine}

FUNCTION NextLine(Var T :TextBuffer;  Pos :TextNodePtr) :TextNodePtr;
  Begin
    NextLine := Nil;
    If (Pos = Nil) then Exit;
    NextLine := Pos^.Next;
  End; {NextLine}

FUNCTION PrevLine(Var T :TextBuffer;  Pos :TextNodePtr) :TextNodePtr;
  Begin
    PrevLine := Nil;
    If (Pos = Nil) then Exit;
    PrevLine := Pos^.Prev;
  End; {PrevLine}

FUNCTION DeleteLine(Var T :TextBuffer;  Var Pos :TextNodePtr) :TextNodePtr;
  Begin
    DeleteLine := NextLine(T,Pos);

    If (Pos=Nil) or EmptyBuffer(T) then Exit;

    If (Pos^.Prev <> Nil) then
      Pos^.Prev^.Next := Pos^.Next;
    If (Pos^.Next <> Nil) then
      Pos^.Next^.Prev := Pos^.Prev;

    If (Pos = T.First) then 		{Pos was first node}
      T.First := Pos^.Next;
    If (Pos = T.Last) then		{Pos was last node}
      T.Last := Pos^.Prev;

    If (Pos^.Line <> Nil) then		{Free existing line}
      FreeMem(Pos^.Line,Length(Pos^.Line^)+1);

    Dispose(Pos);
    Pos := Nil;
  End; {DeleteLine}

FUNCTION NewNode(Line :String) :TextNodePtr;
  Var Temp :TextNodePtr;
  Begin
    NewNode := Nil;
    New(Temp);
    If (Temp=Nil) then Exit;
    Temp^.Next := Nil;
    Temp^.Prev := Nil;
    GetMem(Temp^.Line,Length(Line)+1);
    If (Temp^.Line = Nil) then Exit;
    Temp^.Line^ := Line;
    NewNode := Temp;
  End; {NewNode}

PROCEDURE ModifyTextLine(Var T: TextBuffer;  Pos :TextNodePtr;  Line :String);
  Begin
    If Pos=Nil then Exit;
    If (Pos^.Line <> Nil) then		{Free existing line}
      FreeMem(Pos^.Line,Length(Pos^.Line^)+1);
    GetMem(Pos^.Line,Length(Line)+1);	{Space for new line}
    If (Pos^.Line = Nil) then Exit;
    Pos^.Line^ := Line;
  End; {ModifyTextLine}

FUNCTION GetTextLine(Var T: TextBuffer;  Pos :TextNodePtr) :String;
  Begin
    GetTextLine := '';
    If Pos=Nil then Exit;
    GetTextLine := Pos^.Line^;
  End; {GetTextLine}

PROCEDURE AddToEnd(Var T :TextBuffer;  Line :String);
  Var Temp :TextNodePtr;
  Begin
    Temp := NewNode(Line);
    If (Temp=Nil) then Exit;

    If (T.First = Nil) then Begin
      T.First := Temp;
      T.Last := T.First;
    End else Begin
      T.Last^.Next := Temp;
      Temp^.Prev := T.Last;
      T.Last := Temp;
    End; {Else}
  End; {AddToEnd}

PROCEDURE AddInsert(Var T :TextBuffer;  Pos :TextNodePtr;  Line :String);
  Var Temp :TextNodePtr;
  Begin
    If (EmptyBuffer(T) or (Pos = Nil)) then Begin
      AddToEnd(T,Line);
    End else Begin
      Temp := NewNode(Line);
      If (Temp=Nil) then Exit;

      If (Pos^.Prev <> Nil) then
        Pos^.Prev^.Next := Temp;
      Temp^.Next := Pos;
      Temp^.Prev := Pos^.Prev;
      Pos^.Prev := Temp;

      If (Pos = T.First) then		{New front}
        T.First := Temp;
    End; {Else}
  End; {AddInsert}

FUNCTION BufferLength(Var T :TextBuffer) :Word;
  Var
    Count :Word;
    Step :TextNodePtr;
  Begin
    Count := 0;
    Step := T.First;
    While (Step <> Nil) do Begin
      Step := Step^.Next;
      Inc(Count);
    End; {While}

    BufferLength := Count;
  End; {BufferLength}

{- Create a new, wrapped, buffer.  Margin must be reasonable, i.e. not too
   close to 0 or 255 -}

PROCEDURE WrapBuffer(Var T :TextBuffer;  Margin :Byte);
  Const
    Seperators = [#32..#47,#58..#64,#91..#96,#123..#126];
  Var
    W :TextBuffer;
    Line :String;
    Source :Word;	{Can't be Byte, Length may be 255}
    Step :TextNodePtr;
    Ch :Char;

  PROCEDURE FinishLine;
    Begin
      AddToEnd(W,Line);
      Line := '';
    End; {FinishLine}

  PROCEDURE AddChar(Ch :Char);
    Var
      Overflow :String;
      P :Byte;
    Begin
      If (Length(Line) >= Margin) then Begin	{Break the line}
        Overflow := '';
					{First remove excess spaces}
        If (Line[Length(Line)]=' ') then
          While (Length(Line) > 1) and (Line[Length(Line)-1]=' ') do
            Dec(Line[0]);			{Drop last space}

        If (Length(Line) >= Margin) then Begin
          P := Length(Line);
          While (P > 0) and Not (Line[P] in Seperators) do
            Dec(P);				{Look backwards for seperator}

          If (P=0) then P := Margin;		{No seperator, one huge word}

          Overflow := Copy(Line,P+1,Length(Line)-P);
          Line[0] := Char(P);
        End; {If}

        FinishLine;
        Line := Overflow+Ch;
      End else Begin
        Line := Line+Ch;
      End; {Else}
    End; {AddChar}

  PROCEDURE Tab;
    Var
      Count :Byte;
    Begin
      For Count := 1 to 8-((Length(Line)-1) mod 8) do
        AddChar(' ');
    End; {Tab}


  Begin
    NewBuffer(W);

    Step := T.First;
    Line := '';
    While (Step <> Nil) do Begin
      Source := 1;
      While (Source <= Length(Step^.Line^)) do
      Begin
        Ch := Step^.Line^[Source];
        Case Ch of
       {^J,#10,}
        ^M,#13 : Begin
                    Line := Line+^M^J;
                    FinishLine;
                   End;

  {        ^I : Tab;}

         #141 : {Soft-CR};
         else AddChar(Ch);
        End; {Case}
        Inc(Source);
      End; {While}
      Step := Step^.Next;
    End; {While}
    If (Line <> '') then FinishLine;

    DeleteBuffer(T);
    T := W;
  End; {WrapBuffer}

{- Create a new, unwrapped, buffer.  All the lines except the last one
   in the new buffer will be of length 255 -}

PROCEDURE UnWrapBuffer(Var T,W :TextBuffer);
  Var
    Line :String;
    Source :Word;	{Can't be Byte, Length may be 255}
    Step :TextNodePtr;

  PROCEDURE FinishLine;
    Begin
      AddToEnd(W,Line);
      Line := '';
    End; {FinishLine}

  PROCEDURE AddChar(Ch :Char);
    Begin
      If (Length(Line) = 255) then FinishLine;
      Line := Line+Ch;
    End; {AddChar}

  Begin
    NewBuffer(W);

    Step := T.First;
    Line := '';
    While (Step <> Nil) do Begin
      For Source := 1 to Length(Step^.Line^) do
        AddChar(Step^.Line^[Source]);
      Step := Step^.Next;
    End; {While}
    If (Line <> '') then FinishLine;
End; {UnWrapBuffer}



PROCEDURE DeleteBlanks(Var T :TextBuffer);
  Var
    i : integer;
    Tmp,Line :String;
    Step :TextNodePtr;
  Begin
    Step := FirstLine(T);
    While (Step <> Nil) do Begin
      Line := GetTextLine(T,Step);
      Tmp := StripCh(' ',Line);
      if (Tmp[1] in [^M,^J,#141,#13]) or (Tmp = '') then
      Begin
        Step := DeleteLine(T,Step);
                       {AddInsert(T,Step,' ');}
      End else
      Begin
                       {ModifyTextLine(T,Step,Line);}
         Step := NextLine(T,Step);
      End; {Else}
    End; {Step}
  End; {DeleteBlanks}

END.
