{$A+,B+,D-,E+,F+,I+,L+,N-,O-,R-,S+,V-}
UNIT tmpcom;

{Version 3.0}

{This unit is the communications port interrupt driver for the IBM-PC.
It handles handles all low-level i/o through the serial port.  It is
installed by calling com_install.  It deinstalls itself automatically
when the program exits, or you can deinstall it by calling com_deinstall.

Donated to the public domain by Wayne E. Conrad, January, 1989.
If you have any problems or suggestions, please contact me at my BBS:

    Pascalaholics Anonymous
    (602) 484-9356
    2400 bps
    The home of WBBS
    Lots of source code
}


INTERFACE

USES
  Dos;


TYPE
  com_parity = (com_none, com_even, com_odd, com_zero, com_one);

{This variable is TRUE if the interrupt driver has been installed, or FALSE
if it hasn't.  It's used to prevent installing twice or deinstalling when not
installed.}

CONST
  com_installed: Boolean = FALSE;

var
  mpcoder:boolean;
  mpcode:array[1..6] of byte;
  fosport:byte;
  regs:registers;

procedure com_flush_rx;
procedure com_flush_tx;
procedure com_purge_tx;

function com_carrier:boolean;
function com_rx:char;
function com_tx_ready:boolean;
function com_tx_empty:boolean;
function com_rx_empty:boolean;

procedure com_tx (ch: Char);
procedure com_tx_string (st: String);
procedure com_lower_dtr;
procedure com_raise_dtr;
procedure com_set_speed(speed:word);
procedure com_set_parity(parity:com_parity; stop_bits:byte);
procedure com_install(portnum:word; var error:word; dofossil:boolean);
procedure com_deinstall;


implementation


{Summary of IBM-PC Asynchronous Adapter Registers.  From:
  Compute!'s Mapping the IBM PC and PCjr, by Russ Davis
  (Greensboro, North Carolina, 1985: COMPUTE! Publications, Inc.),
  pp. 290-292.

Addresses given are for COM1 and COM2, respectively.  The names given
in parentheses are the names used in this module.


3F8/2F8 (uart_data) Read: transmit buffer.  Write: receive buffer, or baud
rate divisor LSB if port 3FB, bit 7 = 1.

3F9/2F9 (uart_ier) Write: Interrupt enable register or baud rate divisor
MSB if port 3FB, bit 7 = 1.
PCjr baud rate divisor is different from other models;
clock input is 1.7895 megahertz rather than 1.8432 megahertz.
Interrupt enable register:
    bits 7-4  forced to 0
    bit 3     1=enable change-in-modem-status interrupt
    bit 2     1=enable line-status interrupt
    bit 1     1=enable transmit-register-empty interrupt
    bit 0     1=data-available interrupt

3FA/2FA (uart_iir) Interrupt identification register (prioritized)
     bits 7-3  forced to 0
     bits 2-1  00=change-in-modem-status (lowest)
     bits 2-1  01=transmit-register-empty (low)
     bits 2-1  10=data-available (high)
     bits 2-1  11=line status (highest)
     bit 0     1=no interrupt pending
     bit 0     0=interrupt pending

3FB/2FB (uart_lcr) Line control register
     bit 7  0=normal, 1=address baud rate divisor registers
     bit 6  0=break disabled, 1=enabled
     bit 5  0=don't force parity
            1=if bit 4-3=01 parity always 1
              if bit 4-3=11 parity always 0
              if bit 3=0 no parity
     bit 4  0=odd parity,1=even
     bit 3  0=no parity,1=parity
     bit 2  0=1 stop bit
            1=1.5 stop bits if 5 bits/character or
              2 stop bits if 6-8 bits/character
     bits 1-0  00=5 bits/character
               01=6 bits/character
               10=7 bits/character
               11=8 bits/character

     bits 5..3: 000 No parity
                001 Odd parity
                010 No parity
                011 Even parity
                100 No parity
                101 Parity always 1
                110 No parity
                111 Parity always 0


3FC/2FC (uart_mcr) Modem control register
     bits 7-5  forced to zero
     bit 4     0=normal, 1=loop back test
     bits 3-2  all PCs except PCjr
     bit 3     1=interrupts to system bus, user-designated output: OUT2
     bit 2     user-designated output, OUT1
     bit 1     1=activate rts
     bit 0     1=activate dtr

3FD/2FD (uart_lsr) Line status register
     bit 7  forced to 0
     bit 6  1=transmit shift register is empty
     bit 5  1=transmit hold register is empty
     bit 4  1=break received
     bit 3  1=framing error received
     bit 2  1=parity error received
     bit 1  1=overrun error received
     bit 0  1=data received

3FE/2FE (uart_msr) Modem status register
     bit 7  1=receive line signal detect
     bit 6  1=ring indicator (all PCs except PCjr)
     bit 5  1=dsr
     bit 4  1=cts
     bit 3  1=receive line signal detect has changed state
     bit 2  1=ring indicator has changed state (all PCs except PCjr)
     bit 1  1=dsr has changed state
     bit 0  1=cts has changed state

3FF/2FF (uart_spr) Scratch pad register.}


{Maximum port number (minimum is 1) }

const
  max_port = 4;


{Base i/o address for each COM port}

const
  uart_base: ARRAY [1..max_port] OF Integer = ($3F8, $2F8, $3E8, $2E8);


{Interrupt numbers for each COM port}

const
  intnums: ARRAY [1..max_port] OF Byte = ($0C, $0B, $0C, $0B);


{i8259 interrupt levels for each port}

const
  i8259levels: ARRAY [1..max_port] OF Byte = (4, 3, 4, 3);

{UART i/o addresses.  Values depend upon which COMM port is selected.}

var
  uart_data:word;             {Data register}
  uart_ier:word;             {Interrupt enable register}
  uart_iir:word;             {Interrupt identification register}
  uart_lcr:word;             {Line control register}
  uart_mcr:word;             {Modem control register}
  uart_lsr:word;             {Line status register}
  uart_msr:word;             {Modem status register}
  uart_spr:word;             {Scratch pad register}


{Original contents of IER and MCR registers.  Used to restore UART
to whatever state it was in before this driver was loaded.}

var
  old_ier:byte;
  old_mcr:byte;


{Original contents of interrupt vector.  Used to restore the vector when
the interrupt driver is deinstalled.}

var
  old_vector:pointer;


{Original contents of interrupt controller mask.  Used to restore the
bit pertaining to the comm controller we're using.}

var
  old_i8259_mask:byte;


{Bit mask for i8259 interrupt controller}

var
  i8259bit:byte;


{Interrupt vector number}

var
  intnum:byte;


{ Receive queue.  Received characters are held here
  until retrieved by com_rx. }

const
  rx_queue_size=5120;   {Change to suit}
var
  rx_queue:array[1..rx_queue_size] of byte;
  rx_in:word;           {Index of where to store next character}
  rx_out:word;          {Index of where to retrieve next character}
  rx_chars:word;        {Number of chars in queue}


{ Transmit queue.  Characters to be transmitted are held here
  until the UART is ready to transmit them. }

const
  tx_queue_size=16;    {Change to suit}
var
  tx_queue:array[1..tx_queue_size] of byte;
  tx_in:integer;        {Index of where to store next character}
  tx_out:integer;       {Index of where to retrieve next character}
  tx_chars:integer;     {Number of chars in queue}


{This variable is used to save the next link in the "exit procedure" chain.}

var
  exit_save:pointer;


{Macro to disable interrupts.}

Procedure disable_interrupts;
Begin
  Inline($FA);  {CLI}
End;


{Macro to enable interrupts.}

Procedure enable_interrupts;
Begin
  Inline($FB);  {STI}
End;


{Interrupt driver.  The UART is programmed to cause an interrupt whenever
a character has been received or when the UART is ready to transmit another
character.}

  { flush (empty) the receive buffer. }
procedure com_flush_rx;
var ch:char;
begin
    regs.dx:=fosport;
    regs.ah:=$0A;
    intr($14,regs);
end;

  { flush (empty) transmit buffer. }
procedure com_flush_tx;
begin
    regs.dx:=fosport;
    regs.ah:=$08;
    intr($14,regs);
end;

  { purge (empty) transmit buffer. }
procedure com_purge_tx;
begin
    regs.dx:=fosport;
    regs.ah:=$09;
    intr($14,regs);
end;

  { this function returns TRUE if a carrier is present. }
function com_carrier:boolean;
begin
    regs.dx:=fosport;
    regs.ah:=$03;
    intr($14,regs);
    if (regs.ax and $0080) = 0 then
      com_carrier:=FALSE
    else
      com_carrier:=TRUE;
end;

  { get a character from the receive buffer.
    If the buffer is empty, return NULL (#0). }
function com_rx:char;
begin
  if (com_rx_empty) then com_rx:=#0
    else
    begin
      regs.dx:=fosport;
      regs.ah:=$02;
      intr($14,regs);
      com_rx:=chr(regs.al);
    end;
end;

  { this function returns TRUE if com_tx can accept a character. }
function com_tx_ready: Boolean;
begin
    com_tx_ready:=TRUE;
end;

  { this function returns TRUE if the transmit buffer is empty. }
function com_tx_empty:boolean;
begin
    regs.dx:=fosport;
    regs.ah:=$03;
    intr($14,regs);
    com_tx_empty:=((regs.ax and $4000) <> 0);
end;

  { this function returns TRUE if the receive buffer is empty. }
function com_rx_empty:boolean;
begin
    regs.dx:=fosport;
    regs.ah:=$0C;
    intr($14,regs);
    com_rx_empty:=(regs.ax = $FFFF);
end;

  { send a character.  Waits until the transmit buffer isn't full,
    then puts the character into it.  The interrupt driver will
    send the character once the character is at the head of the
    transmit queue and a transmit interrupt occurs. }
procedure com_tx(ch:char);
var result:word;
begin
    regs.dx:=fosport;
    regs.al:=ord(ch);
    regs.ah:=$01;
    intr($14,regs);
end;

  { send a whole string }
procedure com_tx_string(st:string);
var i:byte;
    result:word;
begin
  for i:=1 to length(st) do com_tx(st[i]);
end;

  { lower (deactivate) the DTR line.  Causes most modems to hang up. }
procedure com_lower_dtr;
begin
    regs.dx:=fosport;
    regs.al:=$00;
    regs.ah:=$06;
    intr($14,regs);
end;

  { raise (activate) the DTR line. }
procedure com_raise_dtr;
begin
    regs.dx:=fosport;
    regs.al:=$01;
    regs.ah:=$06;
    intr($14,regs);
end;

  { set the baud rate.  Accepts any speed between 2 and 65535.  However,
    I am not sure that extremely high speeds (those above 19200) will
    always work, since the baud rate divisor will be six or less, where a
    difference of one can represent a difference in baud rate of
    3840 bits per second or more. }
procedure com_set_speed (speed: Word);
var divisor:word;
begin
    regs.dx:=fosport;
    case speed of
      300:regs.al:=(2 shl 5)+3;
      600:regs.al:=(3 shl 5)+3;
      1200:regs.al:=(4 shl 5)+3;
      2400:regs.al:=(5 shl 5)+3;
      4800:regs.al:=(6 shl 5)+3;
      9600:regs.al:=(7 shl 5)+3;
      19200:regs.al:=(0 shl 5)+3;
      38400:regs.al:=(1 shl 5)+3;
    end;
    regs.ah:=$00;
    intr($14,regs);
end;

  { Set the parity and stop bits as follows:

     com_none    8 data bits, no parity
     com_even    7 data bits, even parity
     com_odd     7 data bits, odd parity
     com_zero    7 data bits, parity always zero
     com_one     7 data bits, parity always one }
procedure com_set_parity(parity:com_parity; stop_bits:byte);
var lcr:byte;
begin
end;

  { Install the communications driver.  Portnum should be 1..max_port.
    Error codes returned are:

      0 - No error
      1 - Invalid port number
      2 - UART for that port is not present
      3 - Already installed, new installation ignored }

procedure com_install(portnum:word; var error:word; dofossil:boolean);
var ier:byte;
begin
  if (dofossil) then
  begin
 (*   usefossil:=FALSE; *)
    fosport:=portnum-1;
    regs.dx:=fosport;
    regs.ah:=$04;
    intr($14,regs);
    if (regs.ax = $1954) then
    begin
    (*  usefossil:=TRUE; *)
      regs.dx:=fosport;
      regs.al:=$F0;
      regs.ah:=$0F;
      intr($14,regs);
    end else
    Begin
     writeln('Fossil Driver not installed! '^G^G^G);
     halt(254);
    end;
 end;
End;


  { Deinstall the interrupt driver completely.  It doesn't change
    the baud rate or mess with DTR; it tries to leave the interrupt
    vectors and enables and everything else as it was when the driver
    was installed.
    This procedure MUST be called by the exit procedure of this
    module before the program exits to DOS, or the interrupt driver
    will still be attached to its vector -- the next communications
    interrupt that came along would jump to the interrupt driver which
    is no longer protected and may have been written over. }

procedure com_deinstall;
begin
    regs.dx:=fosport;
    regs.ah:=$05;
    intr($14,regs);
end;

  { This procedure is called when the program exits for any reason.  It
    deinstalls the interrupt driver.}
{$F+} procedure exit_procedure; {$F-}
begin
  com_deinstall;
  exitproc:=exit_save;
end;

  { This installs the exit procedure. }
begin
  exit_save:=exitproc;
  exitproc:=@exit_procedure;
end.

