/******************************************************************/
/* Real48 <-> Double conversion functions                         */
/*                                                                */
/* These will directly modify and memory map a Turbo Pascal REAL  */
/* or Delphi Real48 to that of the IEEE standard Double.          */
/*                                                                */
/* Notice: Precision loss may occur during conversion of either   */
/*         direction.  The loss is usually minimal, but if you    */
/*         requiring exacting precision.  You may need to modify  */
/*         these functions to recalculate rather than memory map  */
/*         the value.                                             */
/*                                                                */
/* You may use these freely provided you do not modify any of the */
/* copyright information or other vital information of contained  */
/* herein.                                                        */
/*                                                                */
/* Use these files at your own risk.  There is no warranty for    */
/* them and they are provided AS IS.  I will not be held          */
/* responsible for any damages these programs may cause!          */
/*                                                                */
/* Copyright (C) 2000 Michael Dillon.  All Rights Reserved.       */
/* Created on May 18, 2000                                        */
/******************************************************************/

/* Must include these headers for various functions */
#include <string.h>

/* Include the required header for these functions  */
#include "r48tdbl.h"

/* This function will take a Real48 value that was read using the */
/* structure TPReal, and convert it to that of an IEEE compliant  */
/* double value.                                                  */
double Real48ToDouble( TPReal Real48 )
{
        /* Holder of the Double's value    */
        DoubleStruct DStruct;
        /* Final return value              */
        double ReturnDbl = 0.000;

        /* Deletes any contents of DStruct */
        memset( &DStruct, 0, sizeof( DStruct ) );

        /* Takes all the values in the TPReal structure   */
        /* and converts it to the double type's structure */
        DStruct.M4ES       = ( ( unsigned short int )Real48.MS.Sign << 15 );
        DStruct.M4ES      |= ( ( unsigned short int )Real48.Exponent + 894 ) << 4;
        DStruct.M4ES      |= ( ( unsigned short int )Real48.MS.Mantissa3 >> 3 );
        DStruct.Mantissa3  = ( ( unsigned short int )Real48.MS.Mantissa3 << 13 ) | ( ( unsigned short int )Real48.Mantissa2 >> 3 );
        DStruct.Mantissa2  = ( ( unsigned short int )Real48.Mantissa2 << 13 ) | ( ( unsigned short int )Real48.Mantissa1 >> 3 );
        DStruct.Mantissa1  = ( ( unsigned short int )Real48.Mantissa1 << 13 );

        /* Saves the DoubleStructure into the standard double */
        /* value.                                             */
        memcpy( &ReturnDbl, &DStruct, sizeof( DStruct ) );

        /* Returns the newly converted double */
        return ( ReturnDbl );
}

/* Function: DoubleToReal48                                         */
/*                                                                  */
/* This will take a standard double value and convert it to a       */
/* Real48 value using the structure TPReal.                         */
/*                                                                  */
/* Warning: When this function converts it will ignore the final    */
/*          13 bits of the Mantissa for the Double variable.  This  */
/*          value is very small so the effect would be minimal.     */
/*                                                                  */
TPReal DoubleToReal48( double DblValue )
{
        /* Holds the DblValue's information */
        DoubleStruct DStruct;

        /* This value will hold the converted */
        /* form from DStruct                  */
        TPReal       Real48;

        /* Clear the values of both DStruct and Real48 */
        memset( &DStruct,         0,  sizeof( DStruct ) );
        memset( &Real48,          0,  sizeof( Real48  ) );

        /* Copy the standard double to that of DStruct */
        memcpy( &DStruct, &DblValue,  sizeof( DStruct ) );

        /* Copy all the information from the standard  */
        /* double to the Real48 format.                */
        Real48.MS.Sign      = ( unsigned char )( ( DStruct.M4ES & 0x8000 ) ? 1 : 0 ); 
        Real48.Exponent     = ( unsigned char )( ( ( DStruct.M4ES & 0x7FF0 ) >> 4 ) - 894 );
        Real48.MS.Mantissa3 = ( unsigned char )( ( ( DStruct.M4ES & 0x000F ) << 3 ) |
                                                 ( ( DStruct.Mantissa3 & 0xE000 ) >> 13 ) );
        Real48.Mantissa2    = ( unsigned short int )( ( ( DStruct.Mantissa3 & 0x1FFF ) << 3 ) |
                                                      ( ( DStruct.Mantissa2 & 0xE000 ) >> 13 ) );
        Real48.Mantissa1    = ( unsigned short int )( ( ( DStruct.Mantissa2 & 0x1FFF ) << 3 ) |
                                                      ( ( DStruct.Mantissa1 & 0xE000 ) >> 13 ) );

        /* Return the new TPReal structure */
        return ( Real48 );
}
