/***************************************************************************
 *                                                                         *
 *   RESERVE.C                                                             *
 *                                                                         *
 *   Copyright (C) 1994 GALACTICOMM, Inc.    All Rights Reserved.          *
 *                                                                         *
 *   File Reservations Kernel, for avoiding read/write and write/write     *
 *   conflicts.                                                            *
 *                                                                         *
 *                                        - RNStein  7/6/94                *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "reserve.h"

STATIC int amstub(char *grp,char *fil);

struct reserve *reserve;           /* current (or only) reservations domain*/

struct reserve *
inireserve(                        /* initialize a new reservations domain */
int nusers,                        /* number of users (e.g. nterms)        */
int *iuser,                        /* ptr to user-index var (e.g. &usrnum) */
int lkpusr,                        /* write locks per user                 */
int grpsiz,                        /* size of group strings (incl NUL)     */
int filsiz,                        /* size of file strings (incl NUL)      */
int (*amrhdl)(char *grp,char *fil))/* default "amreading()" resolver       */
                                   /* (or NULL=assume NOCONFLICT)          */
                                   /* returns ptr to reservations domain   */
{
     int i;

     reserve=(struct reserve *)alczer(sizeof(struct reserve));
     reserve->nusers=nusers;
     reserve->iuser=iuser;
     reserve->lkpusr=lkpusr;
     reserve->grpsiz=grpsiz;
     reserve->filsiz=filsiz;
     reserve->writloks=alcblok(nusers,lkpusr*(grpsiz+filsiz));
     for (i=0 ; i < nusers ; i++) {
          setmem(ptrblok(reserve->writloks,i),lkpusr*(grpsiz+filsiz),0);
     }
     reserve->amreading=amrhdl == NULL ? amstub : amrhdl;
     return(reserve);
}

void
setrsv(                            /* set reservations domain              */
struct reserve *rsv)               /* return value from inireserve()       */
{
     reserve=rsv;
}

int
w2writ(                            /* "I want to write (modify) this file" */
char *grp,                         /* group name (e.g. Library name)       */
char *fil,                         /* file name (or "*"=any file in group) */
int lkn)                           /* lock number, 0 to lkpusr-1           */
                                   /* returns NOCONFLICT=reserved for you  */
{                                  /*         else=who writes or reads     */
     int me,who;
     char *lokptr;

     me=*reserve->iuser;
     if ((who=w2read(grp,fil)) != NOCONFLICT) {
          return(who);
     }
     if ((who=reserve->amreading(grp,fil)) != NOCONFLICT) {
          return(who);
     }
     lokptr=(char *)ptrblok(reserve->writloks,me)+lkn*(reserve->grpsiz+
                                                       reserve->filsiz);
     stzcpy(lokptr,grp,reserve->grpsiz);
     stzcpy(lokptr+reserve->grpsiz,fil,reserve->filsiz);
     return(NOCONFLICT);
}

void
donewrit(                          /* "I'm done writing the file"          */
int lkn)                           /* lock number, 0 to lkpusr-1, or -1=all*/
{                                  /* (un-write-reserve)                   */
     char *lokptr;
     int loksiz;

     loksiz=reserve->grpsiz+reserve->filsiz;
     if (lkn == -1) {
          setmem(ptrblok(reserve->writloks,*reserve->iuser),
                 reserve->lkpusr*loksiz,0);
     }
     else {
          lokptr=(char *)ptrblok(reserve->writloks,*reserve->iuser)+lkn*loksiz;
          lokptr[0]='\0';
          lokptr[reserve->grpsiz]='\0';
     }
}

int
w2read(                            /* "I want to read this file"           */
char *grp,                         /* group name (e.g. Library name)       */
char *fil)                         /* file name (or "*"=any file in group) */
{                                  /* returns NOCONFLICT=nobody, else=who  */
     int who;
     int loksiz;
     int grpsiz;
     int lkpusr;
     int lok;
     char *lokptr;

     lkpusr=reserve->lkpusr;
     grpsiz=reserve->grpsiz;
     loksiz=grpsiz+reserve->filsiz;
     for (who=0 ; who < reserve->nusers ; who++) {
          lokptr=(char *)ptrblok(reserve->writloks,who);
          if (sameas(fil,"*")) {
               for (lok=0 ; lok < lkpusr ; lok++) {
                    if (sameas(grp,lokptr)) {
                         return(who);
                    }
                    lokptr+=loksiz;
               }
          }
          else {
               for (lok=0 ; lok < lkpusr ; lok++) {
                    if (sameas(grp,lokptr) && sameas(fil,lokptr+grpsiz)) {
                         return(who);
                    }
                    lokptr+=loksiz;
               }
          }
     }
     return(NOCONFLICT);
}

int
whouse(                            /* Who is using this group?             */
char *grp)                         /* group name (e.g. Library name)       */
{                                  /* return NOCONFLICT=nobody, else=who   */
     int who;

     if ((who=w2read(grp,"*")) != NOCONFLICT) {
          return(who);
     }
     if ((who=reserve->amreading(grp,"*")) != NOCONFLICT) {
          return(who);
     }
     return(NOCONFLICT);
}

STATIC int
amstub(                            /* stub for reserve->amreading()        */
char *grp,                         /* group name (e.g. Library name)       */
char *fil)                         /* file name (or "*"=any file in group) */
{                                  /* returns NOCONFLICT=nobody, else=who  */
     grp=grp;
     fil=fil;
     return(NOCONFLICT);
}

