/***************************************************************************
 *                                                                         *
 *   LOCKNKEY.C                                                            *
 *                                                                         *
 *   Copyright (C) 1991-1994 GALACTICOMM, Inc.    All Rights Reserved.     *
 *                                                                         *
 *   This is the module and suite of routines for handling locks & keys.   *
 *                                                                         *
 *                                   - S. Brinker and C. Robert  10/3/91   *
 *                                                                         *
 ***************************************************************************/

#include "gcomm.h"
#include "majorbbs.h"

#define bitint(x)   ((x)>>4)  /* index into array of ints for this bit x   */

char **locks;                 /* dynamically allocated array of lock names */
int nlocks=0;                 /* total number of locks defined             */
int nlalcd=0;                 /* total number of locks allocated for       */
int npkeys=0;                 /* current number of pseudokeys in memory    */
int npalcd=0;                 /* total number of pseudokeys alloc'd for    */
int nkalcd=1;                 /* total number of keys alc'd 4 (ints 4 bits)*/

struct pskeys *pkeys;         /* in-memory array of registered pseudokeys  */

char *kysbuf;                 /* pointer to one-shot internal keys buffer  */

BTVFILE *keysbb;              /* Btrieve data block ptr for "keys" file    */

STATIC void setkeys(char *keylist),edtonl(char *uid,char *keynam,int rmv);

void
inikys(void)                       /* initialize the locks & keys interface*/
{
     keysbb=opnbtv("bbsk.dat",RINGSZ);
     kysbuf=alcmem(RINGSZ);
}

int
register_pseudok(pskbeg,pskrou)    /* register a new pseudokey             */
char *pskbeg;                           /* beginning of pseudokey          */
int (*pskrou)(int unum,char *lock);     /* routine for validating pseudokey*/
{
     if (npalcd < npkeys+1) {
          if (npalcd == 0) {
               pkeys=(struct pskeys *)alcmem(sizeof(struct pskeys)*RSZBLK);
          }
          else {
               pkeys=(struct pskeys *)alcrsz(pkeys,
                                       sizeof(struct pskeys)*npalcd,
                                       sizeof(struct pskeys)*(npalcd+RSZBLK));
          }
          npalcd+=RSZBLK;
     }
     stzcpy(pkeys[npkeys].pskbeg,pskbeg,KEYSIZ);
     pkeys[npkeys].pskrou=pskrou;
     return(npkeys++);
}

void
nkyrec(                            /* create a new key record              */
char *uid)                              /* for this user-id or keyring     */
{
     struct keyrec temp;

     setbtv(keysbb);
     setmem(&temp,sizeof(struct keyrec),0);
     strcpy(temp.userid,uid);
     if (qeqbtv(temp.userid,0)) {
          gcrbtv(NULL,0);
          upvbtv(&temp,sizeof(struct keyrec));
     }
     else {
          invbtv(&temp,sizeof(struct keyrec));
     }
     rstbtv();
}


void
loadkeys(                          /* load a user's keys into memory       */
char *keyring)                          /* keyring name for this user      */
{
     char stg[KEYSIZ+1];

     if (usrptr->keys != NULL) {
          free(usrptr->keys);
     }
     usrptr->keys=(int *)alcmem(nkalcd*sizeof(int));
     setmem(usrptr->keys,nkalcd*sizeof(int),0);
     setbtv(keysbb);
     if (isuidc(usaptr->userid[0])) {
          if (!acqbtv(kysbuf,usaptr->userid,0)) {
               nkyrec(usaptr->userid);
               shocst("MISSING A USER'S KEYRING RECORD",
                      "(%s has been given a blank keyring record.)",
                      usaptr->userid);
          }
          else {
               setkeys(&kysbuf[KLSTOF]);
          }
     }
     if (keyring != NULL) {
          setmem(stg,KEYSIZ+1,0);
          stg[0]=RINGID;
          strncat(stg,keyring,KEYSIZ-1);
          if (!acqbtv(kysbuf,stg,0)) {
               nkyrec(stg);
               shocst("MISSING A CLASS KEYRING RECORD",
                      "(%s class has been given a blank keyring record.)",
                      stg+1);
          }
          else {
               setkeys(&kysbuf[KLSTOF]);
          }
     }
     rstbtv();
}

void
setkey(                            /* turn cur user's bit on for this key  */
char *lock)                             /* key name to turn on             */
{
     int bit;

     bit=lockbit(lock,1);
     usrptr->keys[bit>>4]|=(1<<(bit&15));
}

STATIC void
setkeys(                           /* turn user's bits on list of locks    */
char *keylist)                          /* pointer to list of locks        */
{
     char *keyptr,*ptr;

     keyptr=ptr=keylist;
     while (*ptr != '\0') {
          if (*ptr == ' ') {
               *ptr='\0';
               setkey(keyptr);
               keyptr=ptr+1;
          }
          ptr++;
     }
     setkey(keyptr);
}

int
haskno(                            /* does this user have this key number  */
int bitno)                              /* bit number for this key         */
{
     if (bitno >= 0) {
          return((usrptr->keys[bitno>>4]&(1<<(bitno&15))) != 0);
     }
     return(0);
}

int
gen_haskey(                        /* general haskey() for any user routine*/
char *lock,                             /* lock name the key is for        */
int unum,                               /* user number we're dealing with  */
struct user *uptr)                      /* user pointer we're dealing with */
{
     int bit;

     if (uptr->keys == NULL) {
          return(uptr->class == BBSPRV);
     }
     if (lock[0] == '\0') {
          return(1);
     }
     if ((bit=scnpsk(unum,lock)) != -1) {
          return(bit);
     }
     if (uptr->flags&MASTER) {
          return(1);
     }
     if ((bit=lockbit(lock,0)) == -1) {
          return(0);
     }
     return((uptr->keys[bit>>4]&(1<<(bit&15))) != 0);
}

int
scnpsk(unum,lock)                  /* scan through pseudo-keys for a lock  */
int unum;                               /* user number trying to use key   */
char *lock;                             /* lock name the key is for        */
{
     int i;

     for (i=0 ; i < npkeys ; i++) {
          if (sameto(pkeys[i].pskbeg,lock)) {
               return((*(pkeys[i].pskrou))(unum,lock));
          }
     }
     return(-1);
}

int
uhsmkey(                           /* passed user have this key (in msg)?  */
char *uid,                              /* user-id (online or offline)     */
int mnum)                               /* message number with key name    */
{
     char *lock;

     lock=rawmsg(mnum);
     return(onsysn(uid,1) ? othkey(lock) : uidkey(uid,lock));
}

int
hasmkey(                           /* does cur usr have key in message?    */
int mnum)                               /* message number with key name    */
{
     return(gen_haskey(rawmsg(mnum),usrnum,usrptr));
}

int
uhskey(                            /* does passed user have this key?      */
char *uid,                              /* user-id (online or offline)     */
char *lock)                             /* key name to check for           */
{
     return(onsysn(uid,1) ? othkey(lock) : uidkey(uid,lock));
}

int
haskey(                            /* does the current user have this key? */
char *lock)                             /* lock name to check for          */
{
     return(gen_haskey(lock,usrnum,usrptr));
}

int
othkey(                            /* does other user have this key(othusn)*/
char *lock)                             /* lock name to check for          */
{
     return(gen_haskey(lock,othusn,othusp));
}

int
uidkey(                            /* does passed user have this key?      */
char *uid,                              /* user-id to look up (not online) */
char *lock)                             /* lock name to check for          */
{
     int rv=0;
     char stg[KEYSIZ+1];

     if (lock[0] == '\0') {
          return(1);
     }
     rv=(getlst(uid,NULL) && fndkey(lock,&kysbuf[KLSTOF],0));
     if (!rv) {
          setbtv(accbb);
          if (acqbtv(&acctmp,uid,0)) {
               if (acctmp.flags&HASMST) {
                    rv=1;
               }
               else {
                    setmem(stg,KEYSIZ+1,0);
                    stg[0]=RINGID;
                    strncat(stg,acctmp.curcls,KEYSIZ-1);
                    rv=(getlst(stg,NULL) && fndkey(lock,&kysbuf[KLSTOF],0));
               }
          }
          rstbtv();
     }
     return(rv);
}

int
getlst(                            /* get a list of keys from BTRIEVE      */
char *uid,                              /* user-id or keyring name to get  */
char *buffer)                           /* buffer to put end result        */
{
     int rv=0;

     setbtv(keysbb);
     if ((rv=acqbtv(kysbuf,uid,0)) != 0 && buffer != NULL) {
          strcpy(buffer,&kysbuf[KLSTOF]);
     }
     rstbtv();
     return(rv);
}

int
fndkey(                            /* find a key (or remove) in list       */
char *lock,                             /* lock name to search for         */
char *keylist,                          /* pointer to list of keys         */
int remove)                             /* remove? (1=yes 0=no)            */
{
     int found=0;
     char *keyptr,*ptr;

     keyptr=ptr=keylist;
     while (*ptr != '\0' && !found) {
          if (*ptr == ' ') {
               *ptr='\0';
               found=sameas(lock,keyptr);
               *ptr=' ';
               if (found && remove) {
                    movmem(ptr+1,keyptr,strlen(ptr+1)+1);
               }
               keyptr=ptr+1;
          }
          ptr++;
     }
     if (!found && (found=sameas(lock,keyptr)) != 0 && remove) {
          if (keyptr == keylist) {
               *keyptr='\0';
          }
          else {
               *(keyptr-1)='\0';
          }
     }
     return(found);
}

int
givkey(                            /* give user or keyring list of keys    */
char *uid,                              /* uid or keyring name to update   */
char *keylist)                          /* list of keys to give            */
{
     char *keyptr,*ptr;

     keyptr=ptr=keylist;
     setbtv(keysbb);
     geqbtv(kysbuf,uid,0);
     while (*ptr != '\0') {
          if (*ptr == ' ') {
               *ptr='\0';
               if (!addkyu(keyptr,0)) {
                    return(0);
               }
               edtonl(uid,keyptr,0);
               keyptr=ptr+1;
          }
          ptr++;
     }
     if (!addkyu(keyptr,1)) {
          return(0);
     }
     edtonl(uid,keyptr,0);
     rstbtv();
     return(1);
}

int
addkyu(                            /* add a key to the kysbuf buffer       */
char *keyptr,                           /* pointer to the key name         */
int update)                             /* update the BTRIEVE record?      */
{
     if (!fndkey(keyptr,&kysbuf[KLSTOF],0)) {
          if (strlen(&kysbuf[KLSTOF])+32+strlen(keyptr) >= RINGSZ) {
               return(0);
          }
          if (kysbuf[KLSTOF] != '\0') {
               strcat(&kysbuf[KLSTOF]," ");
          }
          strcat(&kysbuf[KLSTOF],keyptr);
     }
     if (update) {
          setbtv(keysbb);
          upvbtv((struct keyrec *)kysbuf,KLSTOF+strlen(&kysbuf[KLSTOF])+1);
          rstbtv();
     }
     return(1);
}

void
ustkey(                            /* turn off user's bit for this key     */
char *lock)                             /* key name to turn off            */
{
     int bit;

     if ((bit=lockbit(lock,0)) != -1) {
          usrptr->keys[bit>>4]&=~(1<<(bit&15));
     }
}

void
rmvkey(                            /* remove a list of keys from user      */
char *uid,                              /* user (or keyring) to remove from*/
char *keylist)                          /* list of keys to remove          */
{
     char *keyptr,*ptr;

     keyptr=ptr=keylist;
     setbtv(keysbb);
     if (acqbtv(kysbuf,uid,0)) {
          while (*ptr != '\0') {
               if (*ptr == ' ') {
                    *ptr='\0';
                    fndkey(keyptr,&kysbuf[KLSTOF],1);
                    edtonl(uid,keyptr,1);
                    keyptr=ptr+1;
               }
               ptr++;
          }
          fndkey(keyptr,&kysbuf[KLSTOF],1);
          edtonl(uid,keyptr,1);
          upvbtv((struct keyrec *)kysbuf,KLSTOF+strlen(&kysbuf[KLSTOF])+1);
     }
     rstbtv();
}

int
lockbit(                           /* find bit number for a certain lock   */
char *lock,                             /* lock name to look for           */
int ins)                                /* insert if not found? (1 or 0)   */
{
     int cond;
     int low,mid,high;
     int i,j;
     int newalc;

     strupr(lock);
     low=0;
     mid=0;
     high=nlocks-1;
     while (low <= high) {
          mid=low+(high-low)/2;
          if ((cond=strcmp(lock,locks[mid])) < 0) {
               if (mid == low) {
                    break;
               }
               high=mid-1;
          }
          else if (cond > 0) {
               if (mid == high) {
                    mid++;
                    break;
               }
               low=mid+1;
          }
          else {
               return(mid);
          }
     }
     if (ins) {
          if (nlocks == 0) {
               locks=(char **)alcmem(sizeof(char *)*(nlalcd=RSZBLK));
          }
          else if (nlocks == nlalcd) {
               locks=(char **)alcrsz(locks,nlocks*sizeof(char *),
                                     (nlocks+RSZBLK)*sizeof(char *));
               nlalcd+=RSZBLK;
          }
          if (mid != nlocks) {
               movmem(&locks[mid],&locks[mid+1],(nlocks-mid)*sizeof(char *));
          }
          locks[mid]=alcdup(lock);
          if (nlocks != 0 && nliniu()) {
               newalc=nkalcd;
               for (othusp=user,i=0 ; i < nterms ; othusp++,i++) {
                    if (othusp->class >= ONLINE && othusp->keys != NULL) {
                         if (bitint(nlocks)+1 == nkalcd) {
                              othusp->keys=(int *)alcrsz(othusp->keys,
                                                  nkalcd*sizeof(int),
                                                  (nkalcd+RSZBLK)*sizeof(int));
                              if (newalc == nkalcd) {
                                   newalc+=RSZBLK;
                              }
                         }
                         for (j=nlocks-1 ; j >= mid ; j--) {
                              if (othusp->keys[j>>4]&(1<<(j&15))) {
                                   othusp->keys[(j+1)>>4]|=(1<<((j+1)&15));
                              }
                              else {
                                   othusp->keys[(j+1)>>4]&=~(1<<((j+1)&15));
                              }
                         }
                         othusp->keys[mid>>4]&=~(1<<(mid&15));
                    }
               }
               nkalcd=newalc;
               redolocks(mid);
          }
          nlocks++;
          return(mid);
     }
     return(-1);
}

int
keynam(                            /* validate a proposed keyname          */
char *keyname)                     /* name to validate, 0=bad 1=ok         */
{
     if (strlen(keyname) < 3 || strlen(keyname) > KEYSIZ-1) {
          return(0);
     }
     for (strupr(keyname) ; *keyname != '\0' ; keyname++) {
          if (!istxvc(*keyname) && *keyname != '#' && *keyname != '=') {
               return(0);
          }
     }
     return(1);
}

void
dlkeys(                            /* locks & keys delete account routine  */
char *uid)                              /* user's account that was deleted */
{
     setbtv(keysbb);
     geqbtv(NULL,uid,0);
     delbtv();
     rstbtv();
}

void
clslnk(void)                       /* locks & keys system shutdown routine */
{
     clsbtv(keysbb);
}

STATIC
void
edtonl(uid,keynam,rmv)             /* edit key for possibly online user    */
char *uid;                              /* user-id to edit key for         */
char *keynam;                           /* key name being edited           */
int rmv;                                /* remove this key? (1=rmv, 0=add) */
{
     int savusn;

     if (*uid != RINGID) {
          othusn=usrnum;
          if (sameas(uid,usaptr->userid) || onsysn(uid,1)) {
               savusn=usrnum;
               curusr(othusn);
               if (rmv) {
                    ustkey(keynam);
               }
               else {
                    setkey(keynam);
               }
               curusr(savusn);
          }
     }
}

