(*****************************************************************************)
(*>                                                                         <*)
(*>                     Telegard Bulletin Board System                      <*)
(*>        Copyright 1988-1994 by Tim Strike.  All rights reserved.         <*)
(*>                                                                         <*)
(*>  Module name:       TELEGARD.INC                                        <*)
(*>  Module purpose:    Data types and structures                           <*)
(*>                                                                         <*)
(*****************************************************************************)

(*
                                                          November 20, 1994

   Note to all programmers:

   These structures are the tentative release structures for Telegard 3.0.  
   While they are presented to you in this form, please remember that last 
   minute changes may still occur, and that these structures have not been 
   finalized.  As we move closer to a release date, a secondary set of 
   structures will be released which will likely be our final structures.

   The purpose of these tentative structures is to provide a means for 
   programmers to provide feedback, comments and ideas.  The structures are 
   also being released so that programmers can begin modifying and creating
   new programs so that when Telegard 3.0 is released, there will be
   utilities available at the release time, or shortly thereafter.

   I have commented these structures as best as possible.  If you need
   further explanation on how to use or accomplish something with these
   structures, please forward those questions so that we can address them
   for the next release of our structures.

   There are portions of these structures which will obviously bring up
   some questions.  One of the most commonly asked questions deals with
   the future support of multinode.  As development has progressed, I have
   been adding support for a multinode environment.  This support is 
   evident in many of the structures; such as node # storage fields, node 
   restrictions, modem setup, etc.  I'm not prepared at this time to add 
   information beyond what is already laid out in the structures.  As I 
   move toward a multinode release, either in 3.0 or a later version, I may 
   need to consider other small changes to accommodate problems or 
   unplanned situations.

   Please also note that with Telegard 3.0, programming examples will be 
   supplied on how to access certain structures and variables.  If you know
   you are going to need some of these examples, forward your requests now.

   Thanks for your interest in Telegard 3.0.  Please look over the 
   information and especially the comments for each field.  We can use all 
   the questions, comments and ideas that you can think of.

   Tim Strike,
   Telegard 3.0 Author

*)

const

  maxarcs      = 10;                          { archive commands (1..max) }
  maxevents    = 15;                          { scheduled events (0..max) }
                                              {  0 = -X[n] cmd (RESERVED) }
  maxmenucmds  = 75;                          { menu commands    (1..max) }
  numvoteas    = 15;                          { voting answers   (0..max) }
  numvoteqs    = 20;                          { voting questions (1..max) }

  mJAM         = 1;                           { JAM format }
  mFido        = 2;                           { Fido format }
  mSquish      = 3;                           { Squish format }

  mLocal       = 1;                           { Local }
  mNetmail     = 2;                           { Netmail }
  mEchomail    = 3;                           { Echomail }
  mGroupmail   = 4;                           { Groupmail }

  mPublic      = 1;                           { Public Messages }
  mPrivate     = 2;                           { Private Messages }

  atNo         = 1;                           {  No anonymous allowed }
  atYes        = 2;                           {  Anonymous allowed }
  atForced     = 3;                           {  Forced anonymous }
  atDearabby   = 4;                           {  "Dear Abby" message base }
  atAnyname    = 5;                           {  Anyname posting }

(*---------------------------------------------------------------------------*)

type

  astr      = string[160];                    { short-typed string }
  acstring  = string[40];                     { Access Control System string }
  intrange  = array[0..255] of integer;       { security tables - integers }
  byterange = array[0..255] of byte;          { security tables - bytes }
  unixtime  = longint;                        { unix time stamp - seconds
                                                  since 01/01/70 }

  acrq      = '@'..'Z';                       { AR flags }                
  acrqs     = set of acrq;                    { AR flags set }
  uflag =                                     { user AC flags }
  (
    rlogon,                                   { L - One call per day }
    rchat,                                    { C - Can't page the SysOp }
    rvalidate,                                { V - Posts marked unvalid }
    ramsg,                                    { A - Can't post AutoMessage }
    rpostan,                                  { * - Can't post anonymously }
    rpostpvt,                                 { E - Can't post private }
    rpostnet,                                 { N - Can't post NetMail }
    rpost,                                    { P - Can't post at all }
    rvoting,                                  { K - Can't vote }
    rmsg,                                     { M - Force mail deletion }
    rpostecho,                                { G - Can't post EchoMail }
    fnodlratio,                               { 1 - No UL/DL ratio }
    fnopostratio,                             { 2 - No post/call ratio }
    fnofilepts,                               { 3 - No file points checking }
    fnodeletion,                              { 4 - Protection from deletion }
    fnodailyratio,                            { 5 - No daily DL ratio }
    fnodltime                                 { 6 - No DL time check }
  );
  uflags = set of uflag;                      { set of user AC flags }

(*--------------------------------------------------------------------------*)

  mpointerrec=        { *.MSI : Message scan/lastread pointers }
  record
     userID    : longint;                     { user-ID - from user record }
     lastread  : longint;                     { lastread offset* }
     mailscan,                                { base in mail scan }
     qwkscan   : boolean;                     { base in qwk scan }
  end;

 { * The lastread field contains file offsets for *.MSG (LASTREAD.) and 
     Squish (*.SQL) last read files.  The lastread field is not used for
     JAM bases -- the userID of the JAM (*.JLR) records can be searched
     for the correct record }

(*---------------------------------------------------------------------------*)

  filstat=            { *.FB status flags }
  (
    freedl,                                   { if file is free }
    notime,                                   { if file is time check free }
    validated,                                { if file is validated }
    available,                                { if file is available }
    offline,                                  { if file is offline }
    hatch                                     { if file hatched via SDS }
  );

  ulfrec=             { *.FB : File records }
  record
     filename    : string[12];                { filename }
     description : string[45];                { description }
     extended,                                { pointer to extended desc }
     size        : longint;                   { length of file (bytes) }
     uldate,                                  { date uploaded }
     filedate,                                { date on file }
     dldate      : unixtime;                  { date last downloaded }
     filestat    : set of filstat;            { file status }
     filepoints,                              { points }
     downloads   : Word;                      { number downloads }
     uploader    : string[36];                { uploader's name }
     passwordCRC : longint;                   { CRC of password to dl }
     reserved    : array[1..20] of byte;      { RESERVED }
  end;

  extendedrec=        { *.FBE : Extended descriptions }
  record
    filename : string[12];                    { filename (for reference) }
    desc     : array[1..5] of string[45];     { description }
    nextlink : longint;                       { offset to next record }
                                              {    -1 = last record }
    reserved : array[1..10] of byte;          { RESERVED }
  end;

  filescanrec=        { *.FSI : File scan records }
  record
     userID    : longint;                     { user-ID - from user record }
     filescan  : boolean;                     { base in file scan }
  end;

(*---------------------------------------------------------------------------*)

  menuflags=          { Menu header status flags }
  (
    clrscrbefore,                             { C: clrscr before menu? }
    dontcenter,                               { D: don't center titles! }
    forcefullline,                            { F: force full line input }
    nomenuprompt,                             { N: no menu prompt }
    forcepause,                               { P: force pause before menu? }
    autotime,                                 { T: auto time display? }
    lightbar                                  { L: lightbar menu }
  );

  menurec=            { *.MNU : Menu header }
  record
     menutitle : array[1..3] of               { menu titles }
                       string[100];           
     dispfile,                                { help file displayed }
                                              {   ''       Generic }
     helpfile  : string[12];                  { tutorial help file }
                                              {   ''       Generic }
                                              {   '*OFF*'  Disabled }
     prompt    : string[120];                 { menu prompt }
     menuacs   : acstring;                    { access requirements }
     password  : string[15];                  { password required }
     fallback  : string[8];                   { fallback menu }
     helplevel : byte;                        { forced help level for menu }
     columns   : byte;                        { generic menu: # of columns }
     colours   : array[1..6] of byte;         { generic menu: colours }
     menuflags : set of menuflags;            { menu status variables }
     reserved  : array[1..50] of byte;        { RESERVED }
  end;

  commandrec=         { *.MNU : Command records }
  record
     ldesc    : string[70];                   { long description }
     sdesc    : string[35];                   { short description }
     lightbar : string[15];                   { lightbar description }
     keys     : string[14];                   { execute keys }
                                              {   'FIRSTCMD' = Exec. Menu Loaded }
                                              {   'TITLE' = Sub-title }
     acs      : acstring;                     { access requirements }
     cmdtype  : string[2];                    { command type }
     options  : string[50];                   { command options }
     viewtype : byte;                         { display of command }
                                              {    0  ACS dependant }
                                              {    1  Always visbile }
                                              {    2  Always hidden }
  end;

(*---------------------------------------------------------------------------*)

  {
    Upload/Downloads queues - temp_dir\*.QQQ & temp_dir\*.QQD

    The upload/download queues store information about flagged files that 
    are pending upload or download by the user.  Information may be 
    added/removed from these queues during a shell or in an offline 
    process, since the queues are processed on the fly and are not stored 
    in memory.

    The extended descriptions queues (*.QQD) use the same format as the
    filearea extended description files (*.FBE).  
  }

   queuerec=         { *.QQQ transfer queue record }
      record
      filename  : string[12];    { filename }
      filearea  : longint;       { file base # }
                                 { DL: -1  Mail Packet }
                                 { DL: -2  Unlisted DL }
                                 { UL: -1  SysOp Area }
      desc      : string[45];    { description }
      extended  : longint;       { extended descriptions in *.QQD }
      path      : string[60];    { DL: path of file }
      points    : integer;       { DL: # points assigned to file }
      filesize  : longint;       { DL: filesize (bytes) }
      freeratio,                 { DL: file is FREE ratio (no dl pts/k, #) }
      freetime  : boolean;       { DL: file is FREE time (no time cost) }
      readonly  : boolean;       { Dl: file is on read-only media }
      end;

(*---------------------------------------------------------------------------*)

  (* Language defintion strings - *.TGL

  The language defintion files are stored in the systatrec.langpath, and 
  are named with the language.filename.  These language files store all the 
  strings which are user definable.

  Telegard 3.0 strings will all be configurable.  This structure reflects the
  earliest format for the language files.  As the 3.0 project comes to 
  completion, this structure will take on a radically different (and larger)
  format.  Do not assume that the final version of Telegard will have the
  format below.
  *)

  fstringrec=         { *.TGL : Language definition strings }
  record
    ansiq:string[80];                         { "Do you want ANSI? " string }
    note:array[1..2] of string[80];           { Logon notes (L #1-2) }
    lprompt:string[80];                       { Logon prompt (L #3) }
    echoc:char;                               { Echo char for PWs }
    sysopin:string[80];                       { "SysOp In" (inside SysOp hours)}
    sysopout:string[80];                      { "SysOp Out" }
    engage:string[80];                        { "Engage Chat" }
    endchat:string[80];                       { "End Chat" }
    wait:string[80];                          { "SysOp Working" }
    pause:string[80];                         { "Pause" }
    entermsg1:string[80];                     { "Enter Message" line #1 }
    entermsg2:string[80];                     { "Enter Message" line #2 }
    newscan1:string[80];                      { "NewScan begin" }
    newscan2:string[80];                      { "NewScan done" }
    scanmessage:string[80];                   { Message scanning prompt }
    automsgt:string[80];                      { Auto-Message title }
    autom:char;                               { Auto-Message border characters }

    shelldos1:string[80];                     { " >> SysOp shelling to DOS ..." }
    shelldos2:string[80];                     { " ... thank you for waiting." }
    chatcall1:string[80];                     { "Paging SysOp, please wait..." }
    chatcall2:string[80];                     { ">>><*><<<" }
    guestline:string[80];                     { "To be a guest ... " }
    namenotfound:string[80];                  { "Name NOT found in user list." }
    bulletinline:string[80];                  { Bulletin line }
    thanxvote:string[80];                     { "Thanks for voting" }

    listline:string[80];                      { "List files - P to pause" }
    newline:string[80];                       { "Search for new files -" }
    searchline:string[80];                    { "Search all dirs for file mask" }
    findline1:string[80];                     { "Search for descriptions... " }
    findline2:string[80];                     { "Enter the string to search for.." }
    downloadline:string[80];                  { "Download - " }
    uploadline:string[80];                    { "Upload - " }
    viewline:string[80];                      { "View interior files - " }
    nofilepts:string[80];                     { "Insufficient file points." }
    unbalance:string[80];                     { "Your UL/DL ratio is unbalanced" }

    pninfo:string[80];                        { "P to Pause, N for next dir" }
    gfnline1:string[80];                      { "[Enter]=All files" }
    gfnline2:string[80];                      { "File mask: " }
    batchadd:string[80];                      { "File added to batch queue." }

    quotestr1:string[50];                     { message quote header #1 }
    quotestr2:string[50];                     { message quote header #2 }

    filescan:string[80];                      { "Scanning file base #nn..." }
    fnotfound:string[80];                     { "File not found" }
    note1,
    note2,
    note3:string[50];                         { Optional Logon Notes }
  end;

(*---------------------------------------------------------------------------*)

  {
    BBS listings, since they are not language specific, have been made into
    data files.  This allows the data files to be shared across languages
    (since not all the languages will point to the same text\*.MSG 
    directory).  Support was also added for multiple BBS listings - the
    filename is passed as a parameter, and references the data_path\*.BBS
    file.  This can be used for various purposes.
  }

  bbslistrec=          { *.BBS : BBS Listings }
  record
     phone          : string[20];             { BBS Phone Number }
     name,                                    { BBS Name }
     sysop          : string[30];             { BBS SysOp Name }
     description    : string[30];             { Other Description }
     speed          : string[4];              { Speed }
     software       : string[8];              { Software }
     firstdate,                               { First date added }
     lastedit       : unixtime;               { Last date edited }
  end;

(*---------------------------------------------------------------------------*)

  archiverec=         { ARCHIVE.DAT : Archiver Setup }
     array[1..maxarcs] of
     record
        active     : boolean;                 { is this format active? }
        extension  : string[3];               { archive extension }
        description,                          { archiver description }
        compress,                             { pack archive command }
        decompress,                           { unpack archive command }
        test,                                 { test archive command }
        view,                                 { view archive command }
                                              {   ''=internal (if avail) }
        comment    : string[50];              { comment archive command }
        success    : byte;                    { success errorlevel }
        reserved   : array[1..35]
                        of byte;              { RESERVED }
     end;

(*---------------------------------------------------------------------------*)

  (* Archive definition file - *.ADF

  This file is a simple text file containing one (or several) archive
  definitions.  These definition files can be imported into Telegard to be
  used as a default archive configuration.  Following are the accepted
  keywords for this file; any other lines are considered comments and
  won't be parsed.

  ARCHIVE    <string>                         { Archive description (50) }
  EXTENSION  <string>                         { Archive extension (3) }

  COMPRESS   <string>                         { Compression Command (50) }
  DECOMPRESS <string>                         { Decompression Command (50) }
  TEST       <string>                         { Testing Command (50) }
  VIEW       <string>                         { Viewing Command (50) }
  COMMENT    <string>                         { Add Comment Command (50) }
  PREFIX     <string>                         { List Prefix }

  SUCCESS    <num>                            { Errorlevel of SUCCESS }
                                              { (0..255) }

  NOTE       <string>                         { Any line marked note will be }
   :                                          { sent to the user when the }
   :                                          { archive is first setup - }
   :                                          { these could be related to }
   :                                          { specific hardware, software } 
  NOTE       <string>                         { esp: EXTERNAL SETTINGS }
                                              { /GLOBAL SWITCHES }

  The archive definition files may also be created by exporting a configured
  setup to an .ADF file.  If you do not find a archive listed in the
  TELEGARD.ADF file (default archive definitions), then please export your
  configuration and attach to the closest beta/support site.

  The first keyword for any archive definition must be ARCHIVE.  The
  definition for that archive follows that keyword.
  *)

(*---------------------------------------------------------------------------*)

  systatrec=          { CONFIG.TG : System configuration }
  record
    versionid:word;                           { version (eg $3000 = v3.0) }

                                              { system paths }
    datapath,                                 { DATA path }
    textpath,                                 { TEXT path }
    langpath,                                 { LANGUAGES path }
    menupath,                                 { MENUS path }
    logspath,                                 { LOG path }
    msgpath,                                  { MSGS path }
    semaphore,                                { SEMAPHORE path }
    attchpath,                                { FILE ATTACH path }
    swappath    : string[79];                 { SWAP path }

                                              { external utility paths }
    arcpath,                                  { ARCHIVE path }
    protpath    : string[40];                 { PROTOCOL path }

    filepath    : string[79];                 { FILE PATH }

    reserved1   : array[1..120] of byte;      { RESERVED }

                                              { site information }
    bbsname     : string[80];                 { BBS name }
    bbsphone    : string[20];                 { BBS phone number }
    bbslocation : string[50];                 { BBS location }
    sysopname   : string[30];                 { SysOp's full name or alias }
    multinode   : boolean;                    { System is Multinode? }
    bulletprefix: string[8];                  { default bulletin prefix }
    highbitinfo : boolean;                    { High-bit in user info? }
    extrasite1  : array[1..8] of byte;        {}
    sysoppw     : string[20];                 { SysOp PW }
    timeoutbell,                              { minutes before timeout bell }
    timeout,                                  { minutes before timeout logoff }
    backsysoplogs,                            { days to keep SYSOP##.LOG }
    slogtype    : byte;                       { SysOp log type
                                                   0 - File
                                                   1 - Printer
                                                   2 - Both }
    stripclog,                                { strip colors from SysOp log? }
    compressbases: boolean;                   { "compress" base numbers? }
    reserved2    : array[1..100] of byte;     { RESERVED }


                                              { hardware }
    audiolowtime,                             { allow audio hours }
    audiohitime  : word;
    monoscreen,                               { Monochrome screen }
    useems,                                   { use EMS for overlay }
    usebios,                                  { Use BIOS for local video output }
    cgasnow      : boolean;                   { Suppress snow on CGA systems }
    screensize,                               { screen size
                                                   0 - Default
                                                   1 - 25
                                                   2 - 43/50 lines
                                              }
    wfcblanktime : byte;                      { minutes to blank WFC menu screen }
    reserved3    : array[1..25] of byte;      { RESERVED }

                                              { chat }
    chatpagelen,                              { Page length in seconds }
    chatattempts,                             { # of chat attempts }
    sysopcolor,                               { SysOp chat colour }
    usercolor    : byte;                      { User chat colour }
    chatreason,                               { Ask for a chat reason? }
    chattime     : boolean;                   { Refund chat time? }
    externalchat : string[79];                { External chat command }
    chatdoortype : byte;                      { Chat door file type:        }
                                              {   0: none       3: Spitfire }
                                              {   1: GAP        4: Wildcat! }
                                              {   2: RBBS-PC    5: WWIV     }    
    reserved4    : array[1..74] of byte;      { RESERVED }

                                              { external utilities }
    fsecmd       : string[40];                { full screen editor command }
    fsedoortype,                              { FSE door file type:         }
                                              {   0: none       3: Spitfire }
                                              {   1: GAP        4: Wildcat! }
                                              {   2: RBBS-PC    5: WWIV     }
    swapshelltype: byte;                      { swap shell type: }
                                              {   0: disk       2: XMS }
                                              {   1: EMS               }
    swaponarchive,                            { swap on archive command }
    swapondoor,                               { swap on door command }
    swapondosshell,                           { swap on DOS shell }
    swaponevent,                              { swap on DOS Shell event }
    swaponfse,                                { swap on full-screen editor }
    swaponsysbat,                             { swap on system batch files }
    swaponxfer,                               { swap on file transfer }
    swapshell    : boolean;                   { swap shell function enabled? }
    envspace     : word;                      { DOS Environment Space }
    reserved5    : array[1..48] of byte;      { RESERVED }

                                              { restrictions/allowances }
    timeallow    : intrange;                  { daily time }
    callallow,                                { daily calls }
    dlratio      : byterange;                 { # ULs/# DLs ratios }
    dlkratio     : intrange;                  { DLk/ULk ratios }
    postratio,                                { post/call ratios }
    dailyratio   : byterange;                 { daily DL #files ratios }                                         
    dailykratio,                              { daily DL k ratios }
    timebankmax  : intrange;                  { timebank max limit }
    timedeposit,                              { timebank daily deposit }
    timewithdraw : byterange;                 { timebank daily withdraw }

                                              { logon information }
    ansitype,                                 { ANSI Logon -
                                                   0  Ask
                                                   1  Detect
                                                   2  Force NO
                                                   3  Force YES
                                              }
    lastcallers,                              { days of last callers }
    maxlogontries,                            { tries for PW's at logon }
    maxlogonmins : byte;                      { minutes allowed for logon }
    allowalias,                               { allow alias's? (handles) }
    datapw,                                   { data phone # password }
    voicepw,                                  { voice phone # password }
    localsec,                                 { is local security ON? }
    localscreensec,                           { is local screen-security ON? }
    globaltrap,                               { trap ALL USER'S activity? }
    autochatopen,                             { does chat buffer auto-open? }
    autominlogon,                             { Auto-Message in logon? }
    bullinlogon  : boolean;                   { bulletins in logon? }
    lcallinlogon : byte;                      { "Last Calls" list in logon? }
    yourinfoinlogon,                          { "Your Info" in logon? }
    offhooklocallogon,                        { phone off-hook for local? }
    forcevoting,                              { logon voting mandatory? }
    iemsi,                                    { allow IEMSI logons }
    checkmulti   : boolean;                   { check for mutliple logons }
    guestuser    : string[36];                { guest user name }
    reserved8    : array[1..4] of byte;       { RESERVED }
    logonmenu,                                { logon menu to start users on }
    watcharea,                                { area for pw attempt warnings }
    inquiryarea  : string[8];                 { area for forgotten pw inquiry }
    passdays     : word;                      { # of days to change password }
    updatelocal  : boolean;                   { local calls in statistics }
    timeleftbank : boolean;                   { timebank prompt w/no time? }
    minbaud,                                  { minimum logon baudrate }
    graphicsbaud : longint;                   { minimum graphics baudrate }
    reserved9    : array[1..63] of byte;      { RESERVED }

                                              { new users }
    newapp       : string[36];                { new user letter to user }
    opensystem,                               { allow new users? }
    newiemsi     : boolean;                   { Allow IEMSI new user logons }
    linelen,                                  { default video line length }
    pagelen,                                  { default video page length }
    datefmt      : byte;                      { date format - 0 ASK }
    scanphone,                                { Scan for duplicate phone #'s }
    autopassword : boolean;                   { Allow auto-created pw's }
    language     : string[8];                 { Default language, ''  Ask }
    ansi,                                     { ANSI               0  Ask }
    avatar,                                   { Avatar             1  Yes }
    clrscr,                                   { Clear Screen       2  No }
    pause,                                    { Screen Pause }
    hotkeys,                                  { Hotkeys }
    fse          : byte;                      { Full Screen Editor }
    mailaddress,                              { Ask for Mail Address }
    gender,                                   {         Gender }
    dataph,                                   {         Data Phone }
    voiceph,                                  {         Voice Phone }
    dob          : boolean;                   {         Date of Birth }
    reserved10   : array[1..75] of byte;      { RESERVED }

                                              { batch files }
    batchfileext : string[3];                 { batch filename extension }
    sysbatfiles,                              { enable system batch files? }
    batlogon,                                 { execute LOGON.BAT }
    batlogoff,                                { execute LOGOFF.BAT }
    batbbsstart,                              { execute BBSSTART.BAT }
    batbbsdone,                               { execute BBSDONE.BAT }
    batdaystart,                              { execute DAYSTART.BAT }
    batplogoff   : boolean;                   { execute PLOGOFF.BAT }
    reserved11   : array[1..25] of byte;      { RESERVED }

                                              { mail }
    minspaceforpost: word;                    { minimum drive space to post }
    extramail1 : array[1..3] of byte; 
    ccmail,                                   { carbon copy messages }
    rreq,                                     { return receipt requests }
    fileattach,                               { attach file }
    normpubpost,                              { post public messages }
    normprivpost,                             { post private messages }
    normnetpost,                              { post netmail messages }
    nopostratio,                              { no post/call ratio }
    anonpubread,                              { see who posted public anon }
    anonprivread,                             { see who posted private anon }
    anonpubpost,                              { post anon public messages }
    anonprivpost : acstring;                  { post anon private messages }
    rethonour,                                { honour rrq }
    msguploads   : boolean;                   { allow upload of messages }
    reserved12   : array[1..100] of byte;     { RESERVED }

                                              { miscellaneous }
    sop:acstring;                             { SysOp }
    csop:acstring;                            { Co-SysOp }
    spw:acstring;                             { SysOp PW at logon }

                                              { WFC }
    wfcwin1tit,                               { Window 1 Title }
    wfcwin1bor,                               { Window 1 Border }
    wfcwin1txt,                               { Window 1 Text }
    wfcwin2tit,                               { Window 2 Title }
    wfcwin2bor,                               { Window 2 Border }
    wfcwin2txt,                               { Window 2 Text }
    wfcwin3tit,                               { Window 3 Title}
    wfcwin3bor,                               { Window 3 Border }
    wfcwin3txt,                               { Window 3 Text }
    wfcwin3grp,                               { Window 3 Graph }
    wfcwin3st,                                { Window 3 Statistics }
    wfcsltime,                                { Status line time }
    wfcslaction : byte;                       { Status line action }

    { These colour records are being phased out }
    defcolours  : array[FALSE..TRUE,0..9] of  { colour records: }
                    byte;                     {   [FALSE][x] = mono }
                                              {   [TRUE][x]  = colour }
    reserved15  : array[1..8] of byte;        { RESERVED }

                                              { QWK }
    qwkpktname,                               { packet name }
    qwkwelcomefile,                           { BBS welcome file }
    qwknewsfile,                              { BBS news file }
    qwkgoodbyefile: string[8];                { BBS goodbye file }
    qwklocaldlpath,                           { local QWK download path }
    qwklocalulpath,                           { local REP upload path }
    qwkpath      : string[40];                { QWK working path }
    qwkmaxmsgs   : word;                      { max messages to DL (0=unlim) } 
    qwkarc       : string[3];                 { QWK default archive }
    qwkextctrl   : boolean;                   { extended CONTROL.DAT info }
    reserved14   : array[1..74] of byte;      { RESERVED }

    arccomment:
      array[1..3] of string[80];              { BBS comments for archives }

                                              { File System }
    searchdup,                                { search for dupe filenames }
    globaldl,                                 { global downloads }
    allowdldesc,                              { allow description download }
    importfileid,                             { import file descriptions }
    nodupeext,                                { no check of extension on dupe files }
    valallfiles  : boolean;                   { validate files automatically? }
    unlistfp,                                 { file points for unlisted DLs }
    ulrefund,                                 { percent time refund for ULs }
    dlchecktype  : byte;                      { DL checking (0=off,1=UL/DL,2=points) }
    maxintemp,                                { max K allowed in TEMP\ARC\ }
    minspaceul,                               { minimum K drive space left to upload }
    ulfpkbyte,                                { UL K-bytes per file point }
    ulfpforfile,                              { UL # points assigned to file }
    ulfpforuser  : word;                      { UL # points assigned to user }
    sysopulbase  : string[8];                 { file base for SysOp uploads }
    seeunval,                                 { see unvalidated files }
    dlunval,                                  { DL unvalidated files }
    ulvalreq     : acstring;                  { UL validated not-required }
    virusscanner : string[80];                { Virus Scanner Command }
    passederrlvl : byte;                      { Passed Scanner Errorlevel }
    reserved16   : array[1..79] of byte;      { reserved space }

                                              { Window Settings }
                                              { PLEASE DO NOT MODIFY!! }
    curwindow:byte;                           { type of SysOp window }
    istopwindow:boolean;                      { SysOp window on top? }
    callernum:longint;                        { total number of callers }
    windowon:boolean;                         { is SysOp window on? }
  end;

(*---------------------------------------------------------------------------*)

  confrec=              { CONFRENC.DAT : Conference data }
     array['@'..'Z'] of
     record
     acs      : acstring;                     { ACS requirement }
     name     : string[40];                   { name of conference }
     visible  : boolean;                      { visible w/o ACS }
     reserved : array[1..20] of byte;         { RESERVED }
     end; 

(*---------------------------------------------------------------------------*)

  eventflags=         { EVENTS.DAT status flags }
  (
    active,                                   { event is active }
    busyduring,                               { phone off-hook during event }
    monthly,                                  { event is executed monthly }
    forced                                    { event is forced }
  );

  eventrec=           { EVENTS.DAT : Scheduled event records }
  array[0..maxevents] of
     record
     flags:set of eventflags;                 { status flags }
     desc:string[30];                         { description }
     etype:char;                              { event type: 
                                                 A:ACS,
                                                 C:Chat,
                                                 E:External,
                                                 O:OS,
                                                 P:Pack }
     execdata:string[40];                     { event data: 
                                                 ACS:  ACS command
                                                 OS:   OS command
                                                 Exit: ERRORLEVEL for exit }
     exectime,                                { execution time (minutes) }
     duration : word;                         { event duration (minutes) }
     execdays : byte;                         { monthly: day (1-28) 
                                                  daily: days (bitwise) }
     lastexec : unixtime;                     { date event last executed }
     node     : word;                         { affected nodes (0=all) }
  end;

(*---------------------------------------------------------------------------*)

  fbflags=            { FAREAS.DAT status flags }
  (
    fbnoratio,                                { < No Ratio > active }
    fbvisible,                                { Visible to users w/o access }
    fbdirdlpath,                              { *.DIR file stored in PATH }
    fbcdrom,                                  { CD-ROM/read-only media }
    fbusegifspecs                             { whether to use GifSpecs }
  );

  ulrec=              { FAREAS.DAT : File base records }
  record
     desc      : string[40];                  { area description }
     filename,                                { filename }
     infofile  : string[8];                   { info filename }
     path      : string[40];                  { path }
     arctype   : string[3];                   { archive (''=inactive) }
     reserved  : byte;                        { RESERVED }
     fbstat    : set of fbflags;              { file base status }
     sysopacs,                                { sysop access }
     listacs,                                 { list/view access }
     nameacs,                                 { see uploader name access }
     ulacs,                                   { upload access }
     dlacs     : acstring;                    { download access }
     scantype  : byte;                        { scan type  0 = Default ON
                                                           1 = Default OFF
                                                           2 = Always }
     cdlabel   : string[11];                  { CD-ROM label }
     reserved1 : array[1..87] of byte;        { RESERVED }
   end;

(*---------------------------------------------------------------------------*)

  historyrec=            { HISTORY.DAT : History logs }
  record
    date      : string[8];                    { history date }
    userbaud  : array[0..25] of word;         { logon BPS rates   }
                                              {    0 - Local   10 - 16800  }
                                              {    1 - 300     11 - 19200  }
                                              {    2 - 1200    12 - 21600  }
                                              {    3 - 1275    13 - 24000  }
                                              {    4 - 2400    14 - 26400  }
                                              {    5 - 4800    15 - 28800  }
                                              {    6 - 7200    16 - 38400  }
                                              {    7 - 9600    17 - 57600  }
                                              {    8 - 12000   18 - 64000  }
                                              {    9 - 14400   19 - 115200 } 
    active,                                   { % activity }
    calls,                                    { # of calls }
    newusers,                                 { # of new users }
    pubpost,                                  { # of public posts }
    privpost,                                 { # of private posts }
    netpost,                                  { # of netmail posts }
    criterr,                                  { # of critical errors }
    uploads,                                  { # of uploads }
    downloads : word;                         { # of downloads }
    uk,                                       { amount ULed in K }
    dk        : longint;                      { amount DLed in K }
    reserved  : array[1..16] of byte;         { RESERVED }
  end;

(*---------------------------------------------------------------------------*)

  {
    IEMSI.DAT file is created in the \temp directory only if a user 
    connects with IEMSI.  At the end of this record structure will be the
    actual buffer dump of the IEMSI handshaking package which was received
    by the system - just in case for some reason it is required or needed. 
  }

  iemsirec =          { IEMSI.DAT : IEMSI connection record }
  record
     realname,                                { Users Real Name }
     handle       : string[36];               { Users Handle }
     location     : string[30];               { Users Location }
     dataph,                                  { Users Data Phone # }
     voiceph,                                 { Users Voice Phone # }
     password     : string[20];               { Users Password }
     birthdate    : unixtime;                 { Users Birthdate }
     linelen,                                 { Screen Width }
     pagelen      : byte;                     { Screen Length }
     screentype   : byte;
     protocols    : string[10];               { Protocol to use }
                                              {    DZA* DirectZAP Zmodem 
                                                   ZAP* ZedZap Zmodem -> 8k 
                                                   ZMO  Zmodem -> 1k (Z)
                                                   SLK* SEAlink
                                                   KER* Kermit }
     capabilities : string[25];               { System Capabilities }
                                              {    CHT*    IEMSI chats
                                                   MNU*    Image Downloading
                                                   TAB*    TAB support
                                                   ASCII8* 8-bit ASCII supprt }
     requests     : string[40];               { Requests } 
                                              {    NEWS    Bulletins, news, etc.
                                                   MAIL*   Logon - Mail Check
                                                   FILE*   Logon - File Check
                                                   HOT     Hot-Keys
                                                   CLR     Screen clearing
                                                   HUSH*   Do not disturb
                                                   MORE    Page pausing
                                                   FSED    Full-screen editor }
     software     : string[70];               { Terminal Software }
  end;

  {
    *These items are are currently not supported in Telegard because they
    are things which were provided for systems with these capabilities.  They
    are listed here for convenience and so that utility authors can make
    use of them if neccessary.  Some of these options may be used in
    future versions of Telegard.

    (Z)The protocol that is selected by having the ZMO flag is protocol Z.  
    If that protocol does not exist, it is not selected.  Protocol Z should
    be ZMODEM, otherwise the wrong protocol may be selected!
  }

(*---------------------------------------------------------------------------*)

  languagerec=        { LANGUAGE.DAT : Language records }
  record
     filename   : string[8];                  { language filename }
     desc       : string[30];                 { language description }
     acs        : acstring;                   { language ACS }
     textpath,                                { primary text path }
     menuspath  : string[75];                 { primary menus path }
                                              {   '' = use default }
     checktext  : boolean;                    { if primary path <> '' }
     checkmenus : boolean;                    {   and file not found, check }
                                              {   default path? }
  end;

(*---------------------------------------------------------------------------*)

  {
    The LASTON.DAT file is kept in sequential order - the last record is 
    therefore the newest record, and the first record is the oldest.  If
    writing a utility which handles this file, please observe the HIDDEN 
    flag and only display for users with Co-SysOp ACS or above.
  }

  lcallers=           { LASTON.DAT : Last callers records }
  record
     caller     : longint;                    { system caller number }
     realname,                                { real name of caller }
     handle     : string[36];                 { user name of caller }
     location   : string[30];                 { location of caller }
     logonspeed : longint;                    { logon speed }
     logontime  : unixtime;                   { logon time }
     newuser,                                 { was newuser? }
     guest,                                   { was guest? }
     hidden     : boolean;                    { hidden from display? }
     node       : byte;                       { node }  
     reserved   : array[1..24] of byte;       { RESERVED }
  end;

(*---------------------------------------------------------------------------*)

  levelsrec=          { LEVELS.DAT : Membership/validation levels }
     array['A'..'Z'] of
     record
        desc      : string[50];               { Description }
        sl        : byte;                     { SL level }
        flags1,                               { Access flags - Set 1 }
        flags2    : acrqs;                    { Access flags - Set 2 }
        ac        : uflags;                   { Restriction flags }
        addfp,                                { Bonus FilePoints (added) }
        addcredit,                            { Bonus NetCredits (added) }
        addtb,                                { Bonus Timebank (added) }
        expireday : word;                     { Expire in ## days 
                                                   0 = Never }
        tolevel   : char;                     { Expire to level A..Z,
                                                   ! = Delete }
        reserved  : array[1..25] of byte;     { RESERVED }
     end;

(*---------------------------------------------------------------------------*)

  macrorec=           { MACRO.DAT : Keyboard macro records }
  record
     userid : longint;                         { user-ID - from user record }
                                               {   checked to verify that }
                                               {   macro pointers are to }
                                               {   the correct macros }
     macro  : array[1..4] of string[240];      { stored keystrokes }
  end;

(*---------------------------------------------------------------------------*)

  mbflags=            { MAREAS.DAT status flags }
  (
    mbrealname,                               { real names are forced }
    mbvisible,                                { visible to users w/o access }
    mbansi,                                   { filter ANSI }
    mb8bit,                                   { filter 8-bit }
    mbstrip,                                  { strip center/title codes }
    mbaddtear,                                { add tear/origin lines }
    mbtopstar,                                { Top Star base (GroupMail) }
    mbimmune,                                 { immune to "twitting" }
    mbinternet,                               { Internet (UUCP) handling }
    mbmsghiwater                              { ignore 1.MSG (hi-water mark) }
  );

  boardrec=           { MAREAS.DAT : Message base records }
  record
    name,                                     { message base description }
    msgpath   : string[40];                   { messages pathname }
    filename,                                 { data filenames }
    infofile  : string[8];                    { base info filename }
    readacs,                                  { read access }
    postacs,                                  { post access }
    sysopacs,                                 { sysop access }
    netacs    : acstring;                     { network access }
    maxmsgs,                                  { max message count }
    maxdays   : word;                         { max days to keep messages }
    mbstat    : set of mbflags;               { status }
    mbformat,                                 { message base format }
    mbtype,                                   { message base type }
    anonymous,                                { anonymous type }
    mbpost    : byte;                         { message base posting type }
    origin    : string[50];                   { origin line }
    kludge_color,                             { color of kludge line }
    quote_color,                              { color of quoted line }
    tear_color,                               { color of tear line }
    origin_color,                             { color of origin line }
    addr,                                     { network addr # }
    scantype  : byte;                         { scan type  0 = Default ON
                                                           1 = Default OFF
                                                           2 = Mandatory
                                              }
    reserved  : array[1..49] of byte;         { RESERVED }
  end;

(*---------------------------------------------------------------------------*)

  {
    These index files are stored in the temporary directory while the user
    is online.  The index files contain a base compression table which
    point an "alias" base number, and the physical base number.
 
    There will be one entry for each record in the FAREAS.DAT and MAREAS.DAT
    files.  Do not modify these compression tables - it is important that
    they remain unchanged while the user is online (unless Telegard 
    recompiles the compression tables itself).
  }

  areaidxrec=         { temp_dir\FAREAS.IDX/MAREAS.IDX : Compression tables }
  record
     index : array[0..1] of integer;
  end;

(*---------------------------------------------------------------------------*)

  {
    Modem information for single node boards is stored in MODEM.DAT.  For
    mutlinote boards, the modem information for each node is stored in a
    file NODExxx.DAT - in this case, the MODEM.DAT file contains the
    default information which is used when a new NODExxx.DAT file is first
    created.
  }

  modemflags=         { MODEM.DAT/NODExxx.DAT status flags }
  (
    xonxoff,                                  { XON/XOFF (software) flowctl }
    ctsrts,                                   { CTS/RTS (hardware) flowctl }
    dtrdrop,                                  { drop DTR to hang up modem }
    fossilhot                                 { FOSSIL active during DOS }
  );

  modemrec=           { MODEM.DAT/NODExxx.DAT : Modem configuration }
  record
    desc         : string[50];                { modem reference description }
    comport      : byte;                      { COM port number }
    waitbaud,                                 { modem bps rate }
    lockedbaud   : longint;                   { locked bps rate }
                                              {    0 = Not Locked }
    init         : string[80];                { initialization command }
    answer,                                   { answer command }
    hangup,                                   { hangup command }
    offhook      : string[40];                { phone off-hook command }
    code_ok,                                  { OK result code }
    code_error,                               { ERROR result code }
    code_ring,                                { RING result code }
    code_nocarrier,                           { NO CARRIER result code }
    code_arq     : string[16];                { error correction string }
    code_connect : array[1..25] of            { connect strings }
                      string[16];             {    1 - 300     11 - 19200  }
                                              {    2 - 1200    12 - 21600  }
                                              {    3 - 1275    13 - 24000  }
                                              {    4 - 2400    14 - 26400  }
                                              {    5 - 4800    15 - 28800  }
                                              {    6 - 7200    16 - 38400  }
                                              {    7 - 9600    17 - 57600  }
                                              {    8 - 12000   18 - 64000  }
                                              {    9 - 14400   19 - 115200 }
                                              {   10 - 16800  }
    reinittime,                               { inactivity re-initialization }
    delay_cmd,                                { delay after cmd (1/10th) }
    delay_answer,                             { answer delay (1/10th) }
    delay_dtr    : byte;                      { delay for DTR drop (1/10th) }
    flags        : set of modemflags;         { status flags (see above) }
    logonacs     : acstring;
    reserved     : array[1..200] of byte;     { RESERVED }
  end;

(*---------------------------------------------------------------------------*)

  (* Modem definition file - *.MDF

  This file is a simple text file containing one (or several) modem
  definitions.  These definition files can be imported into Telegard to be
  used as the default modem configuration.  Following are the accepted
  keywords for this file; any other lines are considered comments and
  won't be parsed.

  MODEM      <string>                         { Description of definition (50) }
  SPEED      <num>                            { Maximum bps (0-115200) }
  LOCKED     <num>                            { Locked bps (0-115200) }

  INIT       <string>                         { Init string (80) }
  ANSWER     <string>                         { Answer string (40) }
  HANGUP     <string>                         { Hangup string (40) }
  OFFHOOK    <string>                         { Offhook string (40) }
  OK         <string>                         { 'OK' result (16) }
  ERROR      <string>                         { 'ERROR' result (16) }
  RING       <string>                         { 'RING' result (16) }
  NOCARRIER  <string>                         { 'NO CARRIER' result (16) }

  DELAY_CMD  <num>                            { Delay after command (0-255) }
  DELAY_ASW  <num>                            { Delay before answer (0-255) }
  DELAY_DTR  <num>                            { Delay before dtr (0-255) }

  CONNECT    <num> <string>                   { 'CONNECT ' result - first }
     :                                        { token is the speed value }
     :                                        { (0-115200), the second }
     :                                        { (separated by a space) is }
     :                                        { the string returned by the }
  CONNECT    <num> <string>                   { modem (16) }

  CONNECTARQ <string>                         { ARC result extension (16) }
  FLAGS      <string>                         { Modem Flags;
                                                   XONXOFF  XON/XOFF on
                                                   CTSRTS   CTS/RTS on
                                                   DTRDROP  DTR hangup on
                                                   FOSHOT   Fossil HOT
                                              }

  NOTE       <string>                         { Any line marked note will be }
   :                                          { sent to the user when the }
   :                                          { modem is first setup - these }
   :                                          { could be related to specific }
   :                                          { hardware, software, etc...  }
   :                                          { esp: SWITCH SETTINGS, NRAM }
  NOTE       <string>                         { SETTINGS }


  The modem definition files may also be created by exporting a configured
  setup to an .MDF file.  If you do not find your modem listed in the
  TELEGARD.MDF file (default modem definitions), then please export your
  configuration and attach to the closest beta/support site.

  The first keyword for any modem definition must be MODEM.  The definition 
  for that modem follows that keyword.

  *)

(*---------------------------------------------------------------------------*)

  netflags=           { NETWORK.DAT status flags }
  (
    nstrip,                                   { strip message codes? }
    naddtear,                                 { add tear/origin lines? }
    nholduser,                                { hold all non-SysOp NetMail? }
    nhold,                                    { def NetMail to "hold"? }
    ncrash,                                   { def NetMail to "crash"? }
    nkill                                     { def NetMail to "kill/sent"? }
  );

  networkrec=         { NETWORK.DAT : Network information }
  record
    netaddr      : array[1..20] of            { network addresses }
       record                           
       zone,net,node,point : word;            { (zone:net/node.point) }
       domain : string[30];                   { @domain name }
       END;
    origin       : string[50];                { origin line }
    kludge_color : byte;                      { color of kludge line }
    quote_color  : byte;                      { color of quoted text }
    tear_color   : byte;                      { color of tear line }
    origin_color : byte;                      { color of origin line }
    flags        : set of netflags;           { status flags }
    netpath,                                  { NetMail path }
    echopath,                                 { EchoMail path }
    grouppath,                                { GroupMail path }
    nodepath     : string[40];                { nodelist path }
    nodelisttype : byte;                      { Nodelist type 
                                                   0 : None
                                                   1 : Version 6
                                                   2 : Version 7
                                                   3 : FrontDoor }
    reserved     : array[1..75] of byte;      { RESERVED }
  end;

(*---------------------------------------------------------------------------*)

  protrec=            { PROTOCOL.DAT : File transfer protocols }
     array['A'..'Z'] of
        record
        desc     : string[25];                { description }
        active,                               { active? }
        batch    : boolean;                   { batch? (=filelist) }
        acs      : acstring;                  { access control }
        ulcmd,                                { UL command line }
        dlcmd    : string[60];                { DL command line }
        ulcode,                               { UL result code }
        dlcode   : array [1..3] of            { DL result code }
                      string[6];
        envcmd   : string[60];                { environment command }
        templog  : string[25];                { temporary log file }
        flist    : string[5];                 { DL file list }
        logpf,                                { log - word # (filename) }
        logpr    : byte;                      { log - word # (result) }
        success  : boolean;                   { results = success? }
        bidirec  : boolean;                   { bi-directional? }
        end;

(*---------------------------------------------------------------------------*)

  (* Protocol definition file - *.PDF

  This file is a simple text file containing one (or several) protocol
  definitions.  These definition files can be imported into Telegard to be
  used as a default protocol configuration.  Following are the accepted
  keywords for this file; any other lines are considered comments and
  won't be parsed.

  PROTOCOL    <string>                        { Protocol description (25) }

  BATCH       <Y/N>                           { Batch protocol? }
  UPLOAD      <string>                        { Upload command (60) }
  DOWNLOAD    <string>                        { Download command (60) }
  ENVIRONMENT <string>                        { Environment command (60) }
  ULCODE1     <string>                        { Upload result code }
  DLCODE1     <string>                        { Download result code }
                                              {   Single : Errorlevel }
                                              {   Batch  : Log Result }                                          
  SUCCESS     <Y/N>                           { Result codes = success }

                                              { Batch definitions only }
  TEMPLOG     <string>                        { Temp. log file for results }
  FILELIST    <string>                        { Batch "list" indicator }
  LOG_FILE    <num>                           { Word position of filename }
  LOG_RESULT  <num>                           { Word position of result }
  ULCODE2     <string>                        { Upload result codes }
  ULCODE3     <string>                        {   more... }
  DLCODE2     <string>                        { Download result codes }
  DLCODE3     <string>                        {   more... }
  BIDIREC     <Y/N>                           { Bi-directional protocol? }

  NOTE       <string>                         { Any line marked note will be }
   :                                          { sent to the user when the }
   :                                          { protocol is first setup - }
   :                                          { these could be related to }
   :                                          { specific hardware, software } 
  NOTE       <string>                         { esp: EXTERNAL SETTINGS }
                                              { /GLOBAL SWITCHES }

  The protocol definition files may also be created by exporting a configured
  setup to an .PDF file.  If you do not find a protocol listed in the
  TELEGARD.PDF file (default protocol definitions), then please export your
  configuration and attach to the closest beta/support site.

  The first keyword for any protocol definition must be PROTOCOL.  The
  definition for that protocol follows that keyword.
  *)

(*---------------------------------------------------------------------------*)

  qwkrec=             { QK<id##>.DAT : QWK user message base records }
  record
     areatag : string[8];                     { message area tag (filename) }
     areanum : word;                          { QWK area number (base #) }
  end;

(*---------------------------------------------------------------------------*)

  shortmsgrec=         { SHORTMSG.DAT : Logon messages/receipts }
  record
     userid : longint;                        { to user id (0=deleted) }
     msg    : string[160];                    { message }
  end;

(*---------------------------------------------------------------------------*)

  {
     The USERS.IDX file contains the names (real names and handles) of all
     users on the system.  Please note that this file **MUST** be kept in
     alphabetical order.

     Since this file is kept in sorted, alphabetical order, quick binary
     searches can be used to locate records in the file.  Please see the 
     examples for binary searches in the Telegard programming kit.
  }

  useridxrec=         { USERS.IDX : Sorted names listing }
  record
    name   : string[36];                      { user name }
    number : word;                            { user record # }
  end;

  { If user uses handle, there will be two records -- handle and real name }

(*---------------------------------------------------------------------------*)

  {
     The USERID.IDX file contains the ID of all users on the system.  Please
     note that this file **MUST** be kept in numerical order.

     Since this file is kept in sorted, numerical order, quick binary
     searches can be used to locate records in the file.  Please see the 
     examples for binary searches in the Telegard programming kit.

     If at any point you are creating USER-ID #'s you must check this file
     before assigning the ID.  Under no circumstances should any two users
     be assigned the same ID, otherwise you will find many system options
     will have cross-linked pointers.
  }

  IDidxrec=           { USERID.IDX : Sorted ID listing }
  record
    userID : longint;                         { User-ID from user record }
    number : word;                            { user record # }
  end;


(*---------------------------------------------------------------------------*)

  userstatus=          { USERS.DAT status flags }
  (
    lockedout,                                { if locked out }
    udeleted,                                 { if deleted }
    trapactivity,                             { if trapping users activity }
    trapseperate,                             { if trap to seperate TRAP file }
    chatauto,                                 { if auto chat trapping }
    chatseperate,                             { if seperate chat file to trap to }
    slogseperate,                             { if seperate SysOp log }
    alert                                     { alert SysOp when user logs on }
  );

  userflags=
  (
    newusermsg,                               { sent newuser message }
    clsmsg,                                   { clear screen before messages }
    flinput,                                  { full line input }
    hotkey,                                   { menu hotkeys active }
    pause,                                    { pause }
    novice,                                   { user is at novice help level }
    hiddenlog,                                { not displayed in call/online listings }
    hiddenlist                                { not displayed in user listings }
  );

  userrec=            { USERS.DAT : User account records }
  record
    name,                                     { user name }
    realname   : string[36];                  { real name }
    street,                                   { mailing address }
    location   : string[30];                  { city, province }
    postalcode : string[10];                  { postal code }
    voiceph,                                  { voice phone # }
    dataph     : string[20];                  { data phone # }
    pw         : string[20];                  { user password }

    lastdate   : unixtime;                    { laston date/time }
    firstdate,                                { firston date }
    pwdate,                                   { last date of pw change }
    birthdate  : string[8];                   { birth date }
    gender     : char;                        { gender - M/F }
    subdate,                                  { subscription started }
    expiredate : string[8];                   { subscription expires }
    expireto   : char;                        { expire to level A..Z,
                                                  !=Delete }

    comment    : string[40];                  { SysOp comment on User }
    notes      : array[1..3] of               { Sysop definable variables }
                 string[40];                  
    lockfile   : string[8];                   { lockout msg to print }

    extra1     : array[1..18] of byte;                        

    ugraphics  : byte;                        { user graphics protocol }
                                              {   0 : TTY - no fallback }
                                              {   1 : ANSI   }
                                              {   2 : Avatar }
                                              {   3 : RIP    }
    fallback   : byte;                        { 2nd fallback protocol }
                                              {   1 : ANSI   }
                                              {   2 : Avatar }
                                              {   3 : RIP    }
                                              { 3rd TTY }
    linelen,                                  { line length (# cols) }
    pagelen    : byte;                        { page length (# rows) }
    flags      : set of userflags;            { flags (see above) }
    status     : set of userstatus;           { status (see above) }
    extra2     : byte;

    sl      : byte;                           { SL }
    ac      : uflags;                         { user flags }
    aflags1,                                  { Access flags - Set 1 }
    aflags2 : acrqs;                          { Access flags - Set 2 }
    userID  : longint;                        { unique User-ID - THIS VALUE 
                                                SHOULD NOT BE CHANGED BY 
                                                ANY PROGRAM }
    level   : char;                           { current validation level }

    datefmt : byte;                           { Date Format }
                                              {   0 - MM/DD/YY }
                                              {   1 - DD/MM/YY }
                                              {   2 - YY/MM/DD }
    datesep : char;                           { Date Seperator (-,/,.) }
    macroptr : longint;                       { pointer for MACROS.DAT 
                                                  -1 = none }
    extra3  : array[1..21] of byte;

                                              { message system information }
    lastconf:char;                            { last conference }
    lastmsg,                                  { last message base # }
    pubpost,                                  { # public posts }
    privpost,                                 { # private posts }
    netpost : word;                           { # netmail posts }
    mailbox : string[36];                     { mailbox status
                                                  ''        Open/Normal
                                                  'CLOSED'  Closed
                                                  Username  Forwarded to... }
    credit,                                   { Netmail points (credit) }
    debit   : longint;                        { Netmail points (debit) }
    fsetype : byte;                           { FSE usage (0=never,1=always,2=ask) }

    extra4     : array[1..32] of byte;       
                                              { QWK system information }
    qwkarc     : string[3];                   { QWK default archiver }
    qwkfromyou,                               { bundle from user }
    qwkbwctrl  : boolean;                     { create BW control file }

                                              { file system information }
    extra5     : array[1..29] of byte;        

    defprotocol : char;                       { default protocol (A-Z) }
                                              {   ! Select at download time }
    lastfile,                                 { last file base # }
    uploads,                                  { amount ULed (files) }
    downloads,                                { amount DLed (files) }
    todaydl    : word;                        { amount DLed today (files) }
    uk,                                       { amount ULed (kbytes) }
    dk,                                       { amount DLed (kbytes) }
    todaydk,                                  { amount DLed today (kbytes) }
    filepoints : longint;                     { file points remaining }

                                              { miscellaneous information }
    totaltime  : longint;                     { total time on system }
    timebank,                                 { # mins in Time Bank }
    totalcalls : word;                        { total calls to system }
    tltoday,                                  { time left today }
    tbtoday    : integer;                     { bank activity +/- time }
    todaycalls,                               { calls to system today }
    illegal    : byte;                        { illegal logons }
    vote       : array[1..numvoteqs] of byte; { voting data }
    language   : string[8];                   { language }
    extra6     : byte;                            
  end;

(*---------------------------------------------------------------------------*)

  voterec=             { VOTING.DAT : Voting records }
  record
    active    : boolean;                      { active? }
    question  : string[79];                   { voting question }
    voteacs,                                  { voting acs }
    addacs    : acstring;                     { add answer acs }
    numusers  : word;                         { # user's answering question }
    numans    : byte;                         { # of active answers }
    answer    : array[0..numvoteas] of
    record
       desc   : string[60];                   { answer description }
       numres : word;                         { # votes for this answer }
    end;
  end;

(*****************************************************************************)
(* ------------ NODELIST FILE STRUCTURES ("VERSION 6" FORMAT) -------------- *)
(*****************************************************************************)

  nodeflags=                                  { NODELIST.DAT status flags }
  (
    hub,                                      { node is a net hub }
    host,                                     { node is a net host }
    region,                                   { node is region coord }
    zone,                                     { node is a zone coord }
    cm,                                       { runs continuous mail }
    ores1,                                    { reserved for Opus }
    ores2,                                    { reserved for Opus }
    ores3,                                    { reserved for Opus }
    ores4,                                    { reserved for Opus }
    ores5,                                    { reserved for Opus }
    nores1,                                   { reserved for non-Opus }
    nores2,                                   { reserved for non-Opus }
    nores3,                                   { reserved for non-Opus }
    nores4,                                   { reserved for non-Opus }
    nores5,                                   { reserved for non-Opus }
    nores6                                    { reserved for non-Opus }
  );

  modemtypes=         { NODELIST.DAT modem type flags }
  (
    hst,                                      { node uses a USRobotics HST modem }
    pep,                                      { node uses a Telebit PEP modem }
    v32,                                      { node uses a V.32 modem }
    v32b,                                     { node uses a V.32bis modem }
    h96                                       { node uses a Hayes Express96 modem }
  );

  nodedatarec=        { NODELIST.DAT : Version 6 nodelist data }
  record
    net:integer;                              { net number }
    node:integer;                             { node number }
    cost:integer;                             { cost per minute to call }
    name:array[0..33] of byte;                { node name }
    phone:array[0..39] of byte;               { phone number }
    city:array[0..29] of byte;                { city and state }
    password:array[0..7] of byte;             { password }
    realcost:integer;                         { phone company's charge }
    hubnode:integer;                          { node # of this node's hub (0=none) }
    rate:byte;                                { actual bps rate divided by 300 }
    modem:set of modemtypes;                  { modem type codes }
    flags:set of nodeflags;                   { set of flags }
    res:array[1..2] of byte;                  { RESERVED }
  end;

  nodeindexrec=       { NODELIST.IDX : Version 6 nodelist index }
  record
    node:integer;                             { node number }
    net:integer;                              { net number }
  end;

(*****************************************************************************)
(* ------------------- QWK OFFLINE MAIL FILE STRUCTURES -------------------- *)
(*****************************************************************************)

  qdatrec=            { MESSAGES.DAT : Message file header record }
  record
    status:char;                              { message status flag:             }
                                              { ' ' = public, unread             }
                                              { '-' = public, read               }
                                              { '+' = private, unread            }
                                              { '*' = private, read              }
                                              { '~' = comment to SysOp, unread   }
                                              { '`' = comment to SysOp, read     }
                                              { '%' = password protected, unread }
                                              { '^' = password protected, read   }
                                              { '!' = group password, unread     }
                                              { '#' = group password, read       }
                                              { '$' = group password to all      }
    number:array[1..7] of char;               { message number }
    date:array[1..8] of char;                 { date (MM-DD-YY) }
    time:array[1..5] of char;                 { time (HH:MM) }
    msgto:array[1..25] of char;               { To (uppercase) }
    msgfrom:array[1..25] of char;             { From (uppercase) }
    subject:array[1..25] of char;             { Subject }
    password:array[1..12] of char;            { password }
    reference:array[1..8] of char;            { reference message number }
    blocks:array[1..6] of char;               { # of blocks in message,  }
                                              {   including header block }
    active:byte;                              { $E1=active, $E2=to be deleted }
    areanum:word;                             { area number }
    res:array[1..2] of byte;                  { NOT USED }
    tagline:char;                             { contains network tagline:     }
                                              { '*' = network tagline present }
                                              { ' ' = no tagline present      }
  end;

  qndxrec=            { *.NDX : Message area index record }
  record
    recnum:longint;                           { record number in MESSAGES.DAT,    }
                                              {   stored in Microsoft MKS$ format }
    areanum:byte;                             { area number (unused) }
  end;

(*****************************************************************************)
(* ------------ FIDONET *.MSG FORMAT --------------------------------------- *)
(*****************************************************************************)

  fidoattrib=         { *.MSG: fido message attributes }
  (
    private,                                  { private }
    crash,                                    { crash mail }
    received,                                 { received }
    sent,                                     { sent }
    fattach,                                  { file attached }
    intransit,                                { in-transit }
    orphan,                                   { orphaned }
    kill,                                     { kill after sending }
    local,                                    { local message }
    hold,                                     { hold for pickup }
    unused,                                   { **UNUSED** }
    freq,                                     { file request }
    rreq,                                     { return receipt request }
    receipt,                                  { return receipt message }
    areq,                                     { audit request }
    fureq                                     { file update request }
  );

 fidomsgrec = record  { *.MSG files }
    msgfrom,                                  { from/to information }
    msgto        : Array[1..36] of char;
    subject      : Array[1..72] of char;      { subject }
    date_time    : Array[1..20] of char;      { date/time of message }
    timesread,                                { times read }
    destnode,                                 { destination node # }
    orignode,                                 { origin node #}
    cost,                                     { cost of message }
    orignet,                                  { origin network # }
    destnet,                                  { destination network # }
    destzone,
    origzone,
    destpoint,
    origpoint,
    replyto      : word;                      { reply to message # }
    attribute    : set of fidoattrib;         { attributes }
    nextreply    : word;                      { next reply }
    END;

 { To find the correct LASTREAD. record for the user, the MSI file will
   need to be loaded -- the value of MSI.lastread field is the file
   offset into the LASTREAD. file for that particular user }

 fidoreadrec = record { LASTREAD. files }
    lastread : word;                          { last message # read }
 end;

(*****************************************************************************)
(* -------------------------- JAM STRUCTURES ------------------------------- *)
(*****************************************************************************)

  jaminforec =        { *.JHR: JAM message base header - first record }
  record
     signature   : array[1..4]                { message Signature }
                      of char;                { <J><A><M> followed by <NUL> }
     datecreated,                             { creation Date }
     modifycount,                             { modification counter }
     activemsgs,                              { active Messages }
     passwordCRC,                             { CRC of password, -1 = none }
     basemsgnum  : longint;                   { lowest number in index }
     reserved    : array[1..1000]             { reserved by JAM authors }
                      of char;
  end;

  jammsgattr = (
  jlocal,                                     { local }
  jintransit,                                 { in-transit }
  jprivate,                                   { private }
  jread,                                      { read by receiver }
  jsent,                                      { sent }
  jkillsent,                                  { kill msg/sent }
  jarchivesent,                               { archive msg/sent }
  jhold,                                      { hold }
  jcrash,                                     { crash }
  jimmediate,                                 { immediate }
  jdirect,                                    { direct }
  jgate,                                      { gate }
  jfilereq,                                   { file requests }
  jfileattach,                                { files attached }
  jtruncfiles,                                { truncate (0 len) sent files }
  jkillfiles,                                 { kill sent files }
  jreceiptreq,                                { receipt requested }
  jconfirmreq,                                { confirmation of receipt }
  jorphan,                                    { orphaned message }
  jencrypt,                                   { encrypted message* }
  jcompress,                                  { compressed message* }
  jescaped,                                   { escaped message* }
  jforcepickup,                               { force pickup }
  jtypelocal,                                 { local only }
  jtypeecho,                                  { for echo dist'n }
  jtypenet,                                   { for netmail dist'n }
  jnomsgdisplay,                              { no message display }
  jlocked,                                    { locked message }
  jdeleted);                                  { deleted message }

  jamhdrrec =         { *.JHR: JAM message headers }
  record
     signature   : array[1..4]                { message signature }
                      of char;                { <J><A><M> followed by <NUL> }
     revision,                                { JAM revision level }
     reserved1   : word;                      { reserved by JAM authors }
     subfieldlen,                             { length of subfields }
     timesread,                               { # times message read }
     msgidCRC,                                { CRC-32 of MSGID line }
     replyCRC,                                { CRC-32 of REPLY line }
     replyto,                                 { reply to # }
     reply1st,                                { 1st reply number }
     replynext   : longint;                   { reply next }
     date,                                    { date written }
     datercvd,                                { date received }
     dateproc    : unixtime;                  { date processed (tosser/scanner) }
     msgnum      : longint;                   { message number }
     attribute   : set of jammsgattr;         { attributes }
     reserved2,                               { reserved by JAM authors }
     textoffset,                              { offset of text in *.JDT file }
     textlen,                                 { length of text }
     passwordCRC,                             { CRC-32 of password }
     cost        : longint;                   { cost of message }
  end;

  jamsubfieldrec =    { *.JHR: subfield records }
  record
     fieldid,                                 { subfield ID }
     reserved1   : word;                      { reserved by JAM authors }
     datalen     : longint;                   { length of buffer }
  end;

  {

  JAM field-id:

  0     Origin Address               9     Enclosed File                        
  1     Destination Address          10    Enclosed File w/Alias                
  2     Sender Name                  11    Enclosed File Request                
  3     Receiver Name                12    Enclosed File Wild Card              
  4     MSGID (^aMSGID: kludge)      13    Enclosed File w/List                 
  5     REPLYID (^aREPLY: kludge)    2000  FTS Kludge                           
  6     Subject                      2001  Seen-by 2D                           
  7     PID (^aPID: kludge)          2002  Path 2D                              
  8     TRACE (^aVIA kludge)         2003  Flags                                
                                     2004  TZUC Information (Time Zone Settings)
  
  Telegard field-id (these field-id have been specifically setup for 
  Telegard only and are not valid JAM field-id as defined by the authors
  of JAM.  Updates to the JAM format may include updated fields which
  represent some of the data that was added with the Telegard field-id -- 
  Telegard won't use these updated JAM field-id's until another release is
  made):

  5000  Sender "Alias" Name - For "Any Name", Posting Name
                              "Anonymous", "Anonymous"
                              etc...
        (if this field exists, this is used as the "sender" name when
         displaying messages)
  5001  Receiver "Alias" Name (see above)                         

  }

  { The *.JHR file contains most of the information that is required for
    messages.  The first record contains information about the base that
    is being looked at.  The records afterword are variable in length
    because of the numerous subfields that may or may not exist for each
    message:

    jaminforec
    jamhdrrec[1]
       subfield[1]
       subfield character buffer
          :
       subfield[n]
       subfield character buffer
    jamhdrrec[2]
       :
    jamhdrrec[n]
  }

  jamindexrec =       { *.JDX: JAM quick index }
  record
     userCRC,                                 { CRC-32 of receipients name (lower) }
     hdroffset   : longint;                   { offset to jamhdrrec }
  end;

  jamreadrec =        { *.JLR: JAM last read storage }
  record
     userCRC,                                 { CRC-32 of receipients name (lower) }
     userID,                                  { unique user-ID }
     lastread,                                { last read pointer }
     highread    : longint;                   { high read pointer }
  end;

(****************************************************************************)
(* ------------------------- SQUISH STRUCTURES ---------------------------- *)
(****************************************************************************)

 sqinforec =          { *.SQD: Squish message base header - first record }
 record
    length,                                   { length of this structure }
    reserved   : word;                        { RESERVED }
    nummsgs,                                  { number of messages }
    highmsg,                                  { highest message (=num_msg) }
    keepmsgs,                                 { # of messages to keep }
    hwmsgid,                                  { High water message msg-ID # }
    lastmsgid  : longint;                     { Last message msg-ID # }
    basename   : string[79];                  { basename for SquishFile }
    firstmsgptr,                              { offset to first message header }
    lastmsgptr,                               { offset to last message header }
    firstfreeptr,                             { offset to first FREE header }
    lastfreeptr,                              { offset to last FREE header }
    lastptr,                                  { offset of end of file }
    maxmsgs    : longint;                     { maximum number of messages }
    keepdays   : word;                        { days to keep messages }
    sqhdrsize  : word;                        { size of header record }
    reserved2  : array[1..124] of byte;       { RESERVED }
   end;

 sqfhdrrec =          { *.SQD: Message header }
 record
    sqid       : longint;                     { squish ID - $AFAE4453 }
    nextmsg,                                  { offset to last message }
    prevmsg    : longint;                     { offset to previous message }
    totallength,                              { length of header & message }
    msglength,                                { length of message }
    controllen  : longint;                    { length of control information }
    headertype  : word;                       { Header type
                                                  Message = 0
                                                  Free    = 1 }
    reserved    : word;                       { RESERVED }
  end;

  sqshmsgattr = (
  sprivate,                                   { private }
  scrash,                                     { crash }
  sread,                                      { read by receiver }
  ssent,                                      { sent }
  sfileattach,                                { files attached }
  sintransit,                                 { in-transit }
  sorphan,                                    { orphaned message }
  skillsent,                                  { kill msg/sent }
  slocal,                                     { local }
  shold,                                      { hold }
  sfilereq,                                   { file requests }
  sreceiptreq,                                { receipt requested }
  sconfirmreq,                                { confirmation of receipt }
  saudit,                                     { audit trail requested }
  supdate,                                    { update request }
  sscanned,                                   { echomail scanned }
  smsgid,                                     { valid msgid? }
  sres1,  sres2,  sres3,  sres4,              { reserved }
  sres5,  sres6,  sres7,  sres8,
  sres9,  sres10, sres11, sres12,
  sres13, sres14, sres15);                                    


 sqmhdrrec =          { *.SQD: Message Info Header }
 record
    attribute   : set of sqshmsgattr;         { Message attributes }
    msgfrom     : array[1..36] of char;       { Message from - NUL terminated }
    msgto       : array[1..36] of char;       { Message to - NUL terminated }
    subject     : array[1..72] of char;       { Message subject - NUL terminated }
    origaddr,                                 { Origin/Destination addresses }
    destaddr    : record                      
                     zone, net,               
                     node, point : word;      
                  end;
    origdate,                                 { Original date (utc) }
    msgdate     : longint;                    { Arrival (system) date (utc) }
    utcoffset   : word;                       { Minutes offset of UTC }
    replyto     : longint;                    { Reply-to msg-ID # }
    replies     : array[1..9] of              { Replies msg-ID # }
                     longint;
    msgid       : longint;                    { Message ID }
    rawdate     : array[1..20] of char;       { Ascii date - NUL terminated }
 end;

 sqindexrec =         { *.SQI: Squish Message Index }
 record
    msgptr,                                   { offset of sqfhdr record }
    msgid,                                    { msg-ID # }
    hash   : longint;                         { hash of 'To' name }
 end;

 { To find the correct *.SQL record for the user, the MSI file will
   need to be loaded -- the value of MSI.lastread field is the file
   offset into the *.SQL file for that particular user }

 sqreadrec =          { *.SQL: Squish Last Read Index }
 record
    msgid : longint;                          { msg-ID # }
 end;

