/* -------------------------------------------------------------------- */
/*	TALLYBUF.CPP					Citadel 								*/
/*						For talley buffer stuff.						*/
/* -------------------------------------------------------------------- */
#include "ctdl.h"
#pragma hdrstop

#include "auxtab.h"
#include "log.h"
#include "msg.h"
#include "tallybuf.h"
#include "cwindows.h"

/* -------------------------------------------------------------------- */
/*								Contents								*/
/* -------------------------------------------------------------------- */
/*								Public									*/
/*	Functions:															*/
/*		CreateTalleyBuffer()											*/
/*			Creates it for the current task.							*/
/*																		*/
/*		DestroyTalleyBuffer()											*/
/*			Destroys it for the current task.							*/
/*																		*/
/*		GetMessagesEnteredInRoom()										*/
/*			Returns the number of messages entered in this room on this */
/*			call.														*/
/*																		*/
/*		IncrementMessagesEnteredInRoom()								*/
/*			Increments the number of messages entered in this room on	*/
/*			this call.													*/
/*																		*/
/*		WasRoomBypassed()												*/
/*			Returns whether the room was bypassed this call or not. 	*/
/*																		*/
/*		GetRoomNewMessages()											*/
/*			Returns the number of new messages in a room.				*/
/*																		*/
/*		IncrementRoomNewMessages()										*/
/*			Increments the number of new messages in a room.			*/
/*																		*/
/*		DecrementRoomNewMessages()										*/
/*			Decrements the number of new messages in a room.			*/
/*																		*/
/*		SetRoomBypass() 												*/
/*			Sets if a room was bypassed.								*/
/*																		*/
/*		SetRoomVisited()												*/
/*			Sets if a room was visited this call.						*/
/*																		*/
/*		SetRoomHasMail()												*/
/*			Sets if a room has mail.									*/
/*																		*/
/*		SetRoomHasOldMail() 											*/
/*			Sets if a room has old mail.								*/
/*																		*/
/*		SetRoomMessagesEntered()										*/
/*			Sets number of messages entered in a room on this call. 	*/
/*																		*/
/*		SetRoomNewMessages()											*/
/*			Sets number of new messages in a room.						*/
/*																		*/
/*		GetRoomMessages()												*/
/*			Returns the number of messages in a room that the current	*/
/*			user can see.												*/
/*																		*/
/*		IncrementRoomMessages() 										*/
/*			Increments the number of messages in a room that the		*/
/*			current user can see.										*/
/*																		*/
/*		DecrementRoomMessages() 										*/
/*			Decrements the number of messages in a room that the		*/
/*			current user can see.										*/
/*																		*/
/*		GetRoomTotalMessages()											*/
/*			Returns the total number of messages in a room. 			*/
/*																		*/
/*		IncrementRoomTotalMessages()									*/
/*			Increments the total number of messages in a room.			*/
/*																		*/
/*		DecrementRoomTotalMessages()									*/
/*			Decremtns the total number of messages in a room.			*/
/*																		*/
/*		IsRoomMail()													*/
/*			Returns whether the room has mail.							*/
/*																		*/
/*		IsRoomOldMail() 												*/
/*			Returns whether the room has old mail.						*/
/*																		*/
/*		IsRoomVisited() 												*/
/*			Returns whether the room has been visited this call.		*/
/*																		*/
/* -------------------------------------------------------------------- */


#if defined(AUXMEM) || defined(WINCIT)
static void validateTalleyBuf(r_slot RoomIndex);
#endif

/* -------------------------------------------------------------------- */
/*	CreateTalleyBuffer()	Creates it for the current task.			*/
/* -------------------------------------------------------------------- */
#ifdef AUXMEM
#ifdef MULTI
#define NT()	NewTask->

Bool CreateTalleyBuffer(taskInfo *NewTask)
#else
#define NT()

Bool CreateTalleyBuffer(void)
#endif
	{
	assert(NT()talleyInfo == NULL);

	NT()talleyInfo = new talleyTaskInfo;

	if (NT()talleyInfo)
		{
		((talleyTaskInfo *) (NT()talleyInfo))->talleyBlocksInHeap = 0;
		((talleyTaskInfo *) (NT()talleyInfo))->tbList = NULL;

		r_slot i;

		for (i = 0; i < cfg.maxrooms; i += TALLEYBUFPERPAGE)
			{
			if (!AddAuxmemBlock(&(((talleyTaskInfo *) (NT()talleyInfo))->tbList),
					i,
					&(((talleyTaskInfo *) (NT()talleyInfo))->talleyBlocksInHeap),
					1))
				{
#ifdef MULTI
				DestroyTalleyBuffer(NewTask);
#else
				DestroyTalleyBuffer();
#endif
				return (FALSE);
				}
			}

		for (i = 0; i < cfg.maxrooms; i++)
			{
			SetRoomBypass(i, FALSE);
			SetRoomMessagesEntered(i, 0);
			}

		return (TRUE);
		}
	else
		{
		return (FALSE);
		}
	}
#else
#ifdef MULTI
#define NT()	NewTask->

Bool CreateTalleyBuffer(taskInfo *NewTask)
#else
#define NT()

Bool CreateTalleyBuffer(void)
#endif
	{
	assert(NT()talleyInfo == NULL);

	NT()talleyInfo = new talleyBuffer[cfg.maxrooms];

	if (NT()talleyInfo)
		{
		for (r_slot i = 0; i < cfg.maxrooms; i++)
			{
			SetRoomBypass(i, FALSE);
			SetRoomMessagesEntered(i, 0);
			}

		return (TRUE);
		}
	else
		{
		return (FALSE);
		}
	}
#endif


/* -------------------------------------------------------------------- */
/*	DestroyTalleyBuffer()	Destroys it for the current task.			*/
/* -------------------------------------------------------------------- */
#ifdef AUXMEM
#ifdef MULTI
#define OT()	OldTask->

void DestroyTalleyBuffer(taskInfo *OldTask)
#else
#define OT()

void DestroyTalleyBuffer(void)
#endif
	{
	assert(OT()talleyInfo != NULL);

	DeleteAuxmemList(&(((talleyTaskInfo *) (OT()talleyInfo))->tbList),
				&(((talleyTaskInfo *) (OT()talleyInfo))->talleyBlocksInHeap));

	delete OT()talleyInfo;

	OT()talleyInfo = NULL;
	}
#else
#ifdef MULTI
#define OT()	OldTask->

void DestroyTalleyBuffer(taskInfo *OldTask)
#else
#define OT()

void DestroyTalleyBuffer(void)
#endif
	{
	assert(OT()talleyInfo != NULL);

	delete [] OT()talleyInfo;
	OT()talleyInfo = NULL;
	}
#endif


/* -------------------------------------------------------------------- */
/*	GetMessagesEnteredInRoom()	Returns the number of messages entered	*/
/*								in this room on this call.				*/
/*																		*/
/*	Input:	RoomIndex		The room we are looking at. 				*/
/*																		*/
/*	Output: return			The number of messages entered in it.		*/
/* -------------------------------------------------------------------- */
int GetMessagesEnteredInRoom(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	return (getTalleyBuf(RoomIndex)->MsgRoom);
	}


/* -------------------------------------------------------------------- */
/*	IncrementMessagesEnteredInRoom()	Increments the number of		*/
/*										messages entered in this room	*/
/*										on this call.					*/
/*																		*/
/*	Input:	RoomIndex		The room we are changing at.				*/
/* -------------------------------------------------------------------- */
void IncrementMessagesEnteredInRoom(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	getTalleyBuf(RoomIndex)->MsgRoom++;
	}


/* -------------------------------------------------------------------- */
/*	WasRoomBypassed()		Returns whether the room was bypassed this	*/
/*							call or not.								*/
/*																		*/
/*	Input:	RoomIndex		The room we are looking at. 				*/
/*																		*/
/*	Output: return			TRUE if it was bypassed.					*/
/*							FALSE if not.								*/
/* -------------------------------------------------------------------- */
Bool WasRoomBypassed(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	return (getTalleyBuf(RoomIndex)->Bypassed);
	}


/* -------------------------------------------------------------------- */
/*	GetRoomNewMessages()	Returns the number of new messages in a 	*/
/*							room.										*/
/*																		*/
/*	Input:	RoomIndex		The room we are looking at. 				*/
/*																		*/
/*	Output: return			The number of new messages in it.			*/
/* -------------------------------------------------------------------- */
m_slot GetRoomNewMessages(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	return (getTalleyBuf(RoomIndex)->New);
	}


/* -------------------------------------------------------------------- */
/*	IncrementRoomNewMessages()	Increments the number of new messages	*/
/*								in a room.								*/
/*																		*/
/*	Input:	RoomIndex		The room we are changing.					*/
/* -------------------------------------------------------------------- */
void IncrementRoomNewMessages(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	getTalleyBuf(RoomIndex)->New++;
	}


/* -------------------------------------------------------------------- */
/*	DecrementRoomNewMessages()	Decrements the number of new messages	*/
/*								in a room.								*/
/*																		*/
/*	Input:	RoomIndex		The room we are changing.					*/
/* -------------------------------------------------------------------- */
void DecrementRoomNewMessages(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	getTalleyBuf(RoomIndex)->New--;
	}


/* -------------------------------------------------------------------- */
/*	SetRoomBypass() 		Sets if a room was bypassed.				*/
/*																		*/
/*	Input:	RoomIndex		The room we are setting.					*/
/*			BypassState 	TRUE or FALSE: was it or wasn't it?         */
/* -------------------------------------------------------------------- */
void SetRoomBypass(r_slot RoomIndex, Bool BypassState)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	getTalleyBuf(RoomIndex)->Bypassed = BypassState;
	}


/* -------------------------------------------------------------------- */
/*	SetRoomVisited()		Sets if a room was visited this call.		*/
/*																		*/
/*	Input:	RoomIndex		The room we are setting.					*/
/*			VisitState		TRUE or FALSE: was it or wasn't it?         */
/* -------------------------------------------------------------------- */
void SetRoomVisited(r_slot RoomIndex, Bool VisitState)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	getTalleyBuf(RoomIndex)->Visited = VisitState;
	}


/* -------------------------------------------------------------------- */
/*	SetRoomHasMail()		Sets if a room has mail.					*/
/*																		*/
/*	Input:	RoomIndex		The room we are setting.					*/
/*			MailState		TRUE or FALSE: does it or doesn't it?       */
/* -------------------------------------------------------------------- */
void SetRoomHasMail(r_slot RoomIndex, Bool MailState)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	getTalleyBuf(RoomIndex)->HasMail = MailState;
	}


/* -------------------------------------------------------------------- */
/*	SetRoomHasOldMail() 	Sets if a room has old mail.				*/
/*																		*/
/*	Input:	RoomIndex		The room we are setting.					*/
/*			OldMailState	TRUE or FALSE: does it or doesn't it?       */
/* -------------------------------------------------------------------- */
void SetRoomHasOldMail(r_slot RoomIndex, Bool OldMailState)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	getTalleyBuf(RoomIndex)->HasOldMail = OldMailState;
	}


/* -------------------------------------------------------------------- */
/*	SetRoomMessagesEntered()	Sets number of messages entered in a	*/
/*								room on this call.						*/
/*																		*/
/*	Input:	RoomIndex		The room we are setting.					*/
/*			Messages		The number to set it to.					*/
/* -------------------------------------------------------------------- */
void SetRoomMessagesEntered(r_slot RoomIndex, int Messages)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	getTalleyBuf(RoomIndex)->MsgRoom = Messages;
	}


/* -------------------------------------------------------------------- */
/*	SetRoomNewMessages()		Sets number of new messages in a room.	*/
/*																		*/
/*	Input:	RoomIndex		The room we are setting.					*/
/*			NumNew			The number to set it to.					*/
/* -------------------------------------------------------------------- */
void SetRoomNewMessages(r_slot RoomIndex, m_slot NumNew)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	getTalleyBuf(RoomIndex)->New = NumNew;
	}


/* -------------------------------------------------------------------- */
/*	GetRoomMessages()		Returns the number of messages in a room	*/
/*							that the current user can see.				*/
/*																		*/
/*	Input:	RoomIndex		The room we are looking at. 				*/
/*																		*/
/*	Output: return			The number of messages in it that the user	*/
/*							can see.									*/
/* -------------------------------------------------------------------- */
m_slot GetRoomMessages(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	return (getTalleyBuf(RoomIndex)->Messages);
	}


/* -------------------------------------------------------------------- */
/*	IncrementRoomMessages() Increments the number of messages in a room */
/*							that the current user can see.				*/
/*																		*/
/*	Input:	RoomIndex		The room we are changing.					*/
/* -------------------------------------------------------------------- */
void IncrementRoomMessages(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	getTalleyBuf(RoomIndex)->Messages++;
	}


/* -------------------------------------------------------------------- */
/*	DecrementRoomMessages() Decrements the number of messages in a room */
/*							that the current user can see.				*/
/*																		*/
/*	Input:	RoomIndex		The room we are changing.					*/
/* -------------------------------------------------------------------- */
void DecrementRoomMessages(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	getTalleyBuf(RoomIndex)->Messages--;
	}


/* -------------------------------------------------------------------- */
/*	GetRoomTotalMessages()	Returns the total number of messages in a	*/
/*							room.										*/
/*																		*/
/*	Input:	RoomIndex		The room we are looking at. 				*/
/*																		*/
/*	Output: return			The total number of messages in it. 		*/
/* -------------------------------------------------------------------- */
m_slot GetRoomTotalMessages(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	return (getTalleyBuf(RoomIndex)->Total);
	}


/* -------------------------------------------------------------------- */
/*	IncrementRoomTotalMessages()	Increments the total number of		*/
/*									messages in a room. 				*/
/*																		*/
/*	Input:	RoomIndex		The room we are changing.					*/
/* -------------------------------------------------------------------- */
void IncrementRoomTotalMessages(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	getTalleyBuf(RoomIndex)->Total++;
	}


/* -------------------------------------------------------------------- */
/*	DecrementRoomTotalMessages()	Decrements the total number of		*/
/*									messages in a room. 				*/
/*																		*/
/*	Input:	RoomIndex		The room we are changing.					*/
/* -------------------------------------------------------------------- */
void DecrementRoomTotalMessages(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	getTalleyBuf(RoomIndex)->Total--;
	}


/* -------------------------------------------------------------------- */
/*	IsRoomMail()			Returns whether the room has mail.			*/
/*																		*/
/*	Input:	RoomIndex		The room we are looking at. 				*/
/*																		*/
/*	Output: return			TRUE if it has mail.						*/
/*							FALSE if not.								*/
/* -------------------------------------------------------------------- */
Bool IsRoomMail(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	return (getTalleyBuf(RoomIndex)->HasMail);
	}


/* -------------------------------------------------------------------- */
/*	IsRoomOldMail() 		Returns whether the room has old mail.		*/
/*																		*/
/*	Input:	RoomIndex		The room we are looking at. 				*/
/*																		*/
/*	Output: return			TRUE if it has old mail.					*/
/*							FALSE if not.								*/
/* -------------------------------------------------------------------- */
Bool IsRoomOldMail(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	AUXMEMWINCODE(if (!getTalleyBuf(RoomIndex)->Valid) validateTalleyBuf(RoomIndex);)

	return (getTalleyBuf(RoomIndex)->HasOldMail);
	}


/* -------------------------------------------------------------------- */
/*	IsRoomVisited() 		Returns whether the room has been visited	*/
/*							this call.									*/
/*																		*/
/*	Input:	RoomIndex		The room we are looking at. 				*/
/*																		*/
/*	Output: return			TRUE if it has been visited.				*/
/*							FALSE if not.								*/
/* -------------------------------------------------------------------- */
Bool IsRoomVisited(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	return (getTalleyBuf(RoomIndex)->Visited);
	}


/* -------------------------------------------------------------------- */
/*	roomtalley()		talleys up total,messages & new for every room	*/
/* -------------------------------------------------------------------- */
void roomtalley(void)
	{
	register r_slot room;

	sysMail = FALSE;

	for (room = 0; room < cfg.maxrooms; room++)
		{
		getTalleyBuf(room)->Total = 0;
		getTalleyBuf(room)->Messages = 0;
		getTalleyBuf(room)->New = 0;
		getTalleyBuf(room)->HasMail = FALSE;
		getTalleyBuf(room)->HasOldMail = FALSE;
		getTalleyBuf(room)->Visited = FALSE;
		AUXMEMWINCODE(getTalleyBuf(room)->Valid = !cfg.FastLogin;)
		}

	AUXMEMWINCODE(if (cfg.FastLogin) return;)

	CITWINDOW *w = ScreenSaver.IsOn() ?
			NULL : CitWindowsMsg(NULL, getmsg(1675));

	int hashSysop, hashCfgSysop, hashUser;
	register m_slot i;

	m_slot num;

	hashSysop = hash(getmsg(1020));
	hashCfgSysop = hash(cfg.sysop);
	hashUser = hash(TI()CurrentUser->GetName());

	num = (m_slot) sizetable();

	for (i = 0; i < num; ++i)
		{
		room = getRoomNum(i);

		if (room >= cfg.maxrooms)
			{
			// if the message is beyond what we know about, throw it in
			// Dump> and keep going.
			changeheader(cfg.mtoldest + i, DUMP, UCHAR_MAX);
			room = DUMP;
			}

		if (!getFlags(i)->IsCopy())
			{
			if (cfg.checkSysMail && !sysMail && getFlags(i)->IsMail() &&
					!getFlags(i)->IsNet() && !getFlags(i)->IsReceived())
				{
				if (cfg.checkSysMail & 1 /* "sysop" */ &&
						getToHash(i) == hashSysop)
					{
					sysMail = TRUE;
					}

				if (cfg.checkSysMail & 2 /* "cfg.sysop" */ &&
						getToHash(i) == hashCfgSysop)
					{
					sysMail = TRUE;
					}
				}

			getTalleyBuf(room)->Total++;
			}
		else
			{
			if (getCopy(i) <= i)
				{
				getTalleyBuf(room)->Total++;
				}
			}

		if (MaySeeIndexMsg(i) == MSM_GOOD)
			{
			getTalleyBuf(room)->Messages++;

			if ((cfg.mtoldest + i) > TI()CurrentUser->GetRoomNewPointer(room))
				{
				getTalleyBuf(room)->New++;

				/* check to see if it is private mail and set flag if so */
				if (getFlags(i)->IsMail() || getFlags(i)->IsMassemail())
					{
					getTalleyBuf(room)->HasMail = TRUE;
					}
				}
			else
				{
				/* check to see if it is private mail and set flag if so */
				if (getFlags(i)->IsMail() && !getFlags(i)->IsNet() &&
						!getFlags(i)->IsReceived() &&
						(getToHash(i) == hashUser ||
						(TI()CurrentUser->IsSysop() &&
						getToHash(i) == hashSysop)))
					{
					getTalleyBuf(room)->HasOldMail = TRUE;
					}
				}
			}
		}

	if (w)
		{
		destroyCitWindow(w, FALSE);
		}
	}

#if defined(AUXMEM) || defined(WINCIT)
static void validateTalleyBuf(r_slot RoomIndex)
	{
	int hashSysop, hashCfgSysop, hashUser;
	messagetable *lmt;
	m_slot j;

	assert(FirstMessageInRoom);

	hashSysop = hash(getmsg(1020));
	hashCfgSysop = hash(cfg.sysop);
	hashUser = hash(TI()CurrentUser->GetName());

	for (j = FirstMessageInRoom[RoomIndex]; j != M_SLOT_ERROR;
			j = getNextRoomMsgLMT(indexslot(j)))
		{
		m_slot i = indexslot(j);
		assert(i != M_SLOT_ERROR);

		lmt = getMsgTab(i);

		assert(getRoomNumLMT(i) == RoomIndex);

		if (!getFlagsLMT(i)->IsCopy())
			{
			if (cfg.checkSysMail && !sysMail && getFlagsLMT(i)->IsMail() &&
					!getFlagsLMT(i)->IsNet() && !getFlagsLMT(i)->IsReceived())
				{
				if (cfg.checkSysMail & 1 /* "sysop" */ &&
						getToHashLMT(i) == hashSysop)
					{
					sysMail = TRUE;
					}

				if (cfg.checkSysMail & 2 /* "cfg.sysop" */ &&
						getToHashLMT(i) == hashCfgSysop)
					{
					sysMail = TRUE;
					}
				}

			getTalleyBuf(RoomIndex)->Total++;
			}
		else
			{
			if (getCopyLMT(i) <= i)
				{
				getTalleyBuf(RoomIndex)->Total++;
				}
			}

		if (MaySeeIndexMsg(i) == MSM_GOOD)
			{
			getTalleyBuf(RoomIndex)->Messages++;

			if ((cfg.mtoldest + i) >
					TI()CurrentUser->GetRoomNewPointer(RoomIndex))
				{
				getTalleyBuf(RoomIndex)->New++;

				/* check to see if it is private mail and set flag if so */
				if (getFlagsLMT(i)->IsMail() || getFlagsLMT(i)->IsMassemail())
					{
					getTalleyBuf(RoomIndex)->HasMail = TRUE;
					}
				}
			else
				{
				/* check to see if it is private mail and set flag if so */
				if (getFlagsLMT(i)->IsMail() && !getFlagsLMT(i)->IsReceived() &&
						(getToHashLMT(i) == hashUser ||
						(TI()CurrentUser->IsSysop() &&
						getToHashLMT(i) == hashSysop)))
					{
					getTalleyBuf(RoomIndex)->HasOldMail = TRUE;
					}
				}
			}
		}

	getTalleyBuf(RoomIndex)->Valid = TRUE;
	}
#endif
