// Citadel: RoomPriv.H
//
// Data and structures private to the room subsystem.

#ifndef ROOMPRIV_H
#define ROOMPRIV_H

#include "auxtab.h"
#include "boolexpr.h"


// --------------------------------------------------------------------------
// The next structure contains a room's flags. This is used in both the full
// structure of a room and in the index.

class rflags
	{
	Bool INUSE		: 1;		// Room in use?
	Bool PUBLIC 	: 1;		// Room public?
	Bool MSDOSDIR	: 1;		// Room directory?
	Bool PERMROOM	: 1;		// Room permanent?
	Bool GROUPONLY	: 1;		// Room Owned by a group?
	Bool READONLY	: 1;		// Room Read only?
	Bool DOWNONLY	: 1;		// Room download only?
	Bool SHARED 	: 1;		// Room Shared?
	Bool MODERATED	: 1;		// Room Moderated?
	Bool APLIC		: 1;		// Room have application?
	Bool BIO		: 1;		// By invintation only..
	Bool UPONLY 	: 1;		// Upload only
	Bool PRIVGRP	: 1;		// Has a privlage group
	Bool ANON		: 1;		// Anonymous messages in this room...
	Bool SUBJECT	: 1;		// ask user for subject in this room
	Bool GRP_MOD	: 1;		// Access group is moderators..
	Bool ARCHIVE	: 1;		// archive the room?
	Bool BOOLGROUP	: 1;		// boolean expresion for group access?
	uint UNUSED 	: 14;		// ...

public:
	void SetInuse(Bool New) 			{	INUSE = !!New;		}
	Bool IsInuse(void)					{	return (INUSE); 	}

	void SetPublic (Bool New)			{	PUBLIC = !!New; }
	Bool IsPublic (void)				{	return (PUBLIC );	}

	void SetMsDOSdir(Bool New)			{	MSDOSDIR = !!New;	}
	Bool IsMsDOSdir(void)				{	return (MSDOSDIR);	}

	void SetPermanent(Bool New) 		{	PERMROOM = !!New;	}
	Bool IsPermanent(void)				{	return (PERMROOM);	}

	void SetGroupOnly(Bool New) 		{	GROUPONLY = !!New;	}
	Bool IsGroupOnly(void)				{	return (GROUPONLY); }

	void SetReadOnly(Bool New)			{	READONLY = !!New;	}
	Bool IsReadOnly(void)				{	return (READONLY);	}

	void SetDownloadOnly(Bool New)		{	DOWNONLY = !!New;	}
	Bool IsDownloadOnly(void)			{	return (DOWNONLY);	}

	void SetShared (Bool New)			{	SHARED = !!New; }
	Bool IsShared (void)				{	return (SHARED );	}

	void SetModerated(Bool New) 		{	MODERATED = !!New;	}
	Bool IsModerated(void)				{	return (MODERATED); }

	void SetApplication(Bool New)		{	APLIC = !!New;		}
	Bool IsApplication(void)			{	return (APLIC); 	}

	void SetBIO(Bool New)				{	BIO = !!New;		}
	Bool IsBIO(void)					{	return (BIO);		}

	void SetUploadOnly (Bool New)		{	UPONLY = !!New; }
	Bool IsUploadOnly (void)			{	return (UPONLY );	}

	void SetPrivilegedGroup(Bool New)	{	PRIVGRP = !!New;	}
	Bool IsPrivilegedGroup(void)		{	return (PRIVGRP);	}

	void SetAnonymous(Bool New) 		{	ANON = !!New;		}
	Bool IsAnonymous(void)				{	return (ANON);		}

	void SetSubject(Bool New)			{	SUBJECT = !!New;	}
	Bool IsSubject(void)				{	return (SUBJECT);	}

	void SetGroupModerates(Bool New)	{	GRP_MOD = !!New;	}
	Bool IsGroupModerates(void) 		{	return (GRP_MOD);	}

	void SetArchive(Bool New)			{	ARCHIVE = !!New;	}
	Bool IsArchive(void)				{	return (ARCHIVE);	}

	void SetBooleanGroup(Bool New)		{	BOOLGROUP = !!New;	}
	Bool IsBooleanGroup(void)			{	return (BOOLGROUP); }
	};


// --------------------------------------------------------------------------
// This is everything we know about a room. This data is stored in the
// ROOM.DAT file, one entry per room.

typedef struct
	{
	rflags	rbflags;			// same bits as flags above
	label	rbname; 			// name of room
	char	rbdirname[64];		// user directory for room's files
	char	rbroomtell[13]; 	// filename for room description
	char	rbaplic[13];		// filename for room application
	int 	rbgrpno;			// group # which owns it

	int 	UNUSED1;			// (RETHINK) moderator's hash
	int 	UNUSED2;			// (RETHINK) host for BIO room
	int 	rbPgrpno;			// group number for privliges grp

	Bool	AUTOAPP : 1;		// Automatically run application.
	Bool	NOEXCLUDE : 1;		// users cannot exclude the room

	uint	UNUSED3 : 14;

	char	descript[80];		// for .kv[...]

	label	netID;				// Network ID for room
	char	archive[13];		// archive file for room
	char	dictionary[13]; 	// dictionary file for room.
	label	modHost;			// the moderator and host of the room
	label	exclNetPartner; 	// net with only here.
	BoolExpr GroupExpression;	// If BOOLGROUP
//	BoolExpr NetClass;			// Taking over the network.
	} aRoom;


// --------------------------------------------------------------------------
// The following is the room table structure. The room table is used as a
// quick in-memory lookup into ROOM.DAT, and contains all of the fields that
// we need often.

#ifndef WINCIT
typedef struct
	{
	rflags	rtflags;			// public/private flag etc
	label	rtname; 			// name of room
	label	netID;				// Network ID for room
	int 	grpno;				// group # which owns it
	int 	pgrpno; 			// group # which is privileged

	BoolExpr GroupExpression;	// If BOOLGROUP
//	BoolExpr NetClass;			// ...

	Bool	locked		: 1;	// for futurish mem/file locking stuff
	} rTable;
#else

// --------------------------------------------------------------------------
// We use huge pointers in windows, so we can have more than a 64K room
// table. This is done because we want to be able to have as many rooms in
// the Windows version as in the Auxmem version. This is, of course, a huge
// waste of memory. But, oh well. That's what segments do. If you use
// Windows, then you should have lots of memory anyway, right

typedef struct
	{
	rflags	rtflags;			// +   4 = 4
	label	rtname; 			// +  31 = 35
	label	netID;				// +  31 = 66
	int 	grpno;				// +   2 = 68
	int 	pgrpno; 			// +   2 = 70

	m_slot	tbtotal;			// +   4 = 74
	m_slot	tbmessages; 		// +   4 = 78
	m_slot	tbnew;				// +   4 = 82

	BoolExpr GroupExpression;	// +  52 = 134

	uint	tbMsgRoom	: 8;	// +
	Bool	tbhasmail	: 1;	//
	Bool	tbhasoldmail: 1;	//	  2
	Bool	tbvisited	: 1;	//		 =
	Bool	tbbypass	: 1;	//		   136
	Bool	locked		: 1;	// .........

	char	pad[120];			// + 120 = 256
	} rTable;
#endif


// private data - defined in ROOM.CPP

#ifdef AUXMEM
	#define ROOMTABPERPAGE	(AUXPAGESIZE / sizeof(rTable))

	extern auxTabList	*rtList;			// base of our linked list
	extern int			roomBlocksInHeap;	// how many we have
#else
	#ifdef WINCIT
		extern rTable huge	*roomTab;		// RAM index of rooms
	#else
		extern rTable		*roomTab;		// RAM index of rooms
	#endif
#endif



// shortcuts
#define roomBuf 		((aRoom *) (TI()roomInfo))

#ifdef AUXMEM
	#define getRoomTab(s)	((rTable *) (LoadAuxmemBlock((s), &rtList, \
									ROOMTABPERPAGE, sizeof(rTable))))
#else
	#define getRoomTab(s)	(&roomTab[(r_slot)(s)])
#endif




// private functions

// ROOMLOW.C
void getRoom(int rm, aRoom *buffer);
void putRoom(int rm, aRoom *buffer);
void checkdir(void);
#endif
