/* -------------------------------------------------------------------- */
/*	ROOM.CPP					Citadel 								*/
/*						The room database subsystem.					*/
/* -------------------------------------------------------------------- */
/*	One of Citadel's main concepts is that of rooms. Rooms have many    */
/*	attributes (name, group, application, etc.), and contain messages.	*/
/*	Rooms are grouped together into Halls, another of the main concepts */
/*	which Citadel is built on top of. Because rooms are so key to the	*/
/*	operation of the program, the interface to this subsystem is quite	*/
/*	extensive: there is a lot of data involved with rooms. Many of the	*/
/*	routines are in other .CPP files, because there are too many to fit */
/*	into one file and have the overlay system work well. It may seem	*/
/*	that the grouping of the routines into different source files is	*/
/*	arbitrary. This is because the first critereon for grouping these	*/
/*	routines is in terms of overlay friendlyness, which is not always	*/
/*	what makes the most since in terms of function groupings.			*/
/* -------------------------------------------------------------------- */
#include "ctdl.h"
#pragma hdrstop

#include "room.h"
#include "roompriv.h"

#include "log.h"

/* -------------------------------------------------------------------- */
/*								Contents								*/
/* -------------------------------------------------------------------- */
/*								Public									*/
/*	Functions:															*/
/*		UserCanModerateRoom()											*/
/*			Returns whether the user can moderate a room or not.		*/
/*																		*/
/*		GetRoomName()													*/
/*			Returns the name of a specified room.						*/
/*																		*/
/*		GetRoomNetID()													*/
/*			Returns the net ID of a specified room. 					*/
/*																		*/
/*		IsRoomGroupOnly()												*/
/*			Returns if a room is group-only.							*/
/*																		*/
/*		IsRoomMultiGroup()												*/
/*			Returns if a room is multi-group-only.						*/
/*																		*/
/*		IsRoomPrivGroup()												*/
/*			Returns if a room has a priviliges group.					*/
/*																		*/
/*		IsRoomReadOnly()												*/
/*			Returns if a room is read-only. 							*/
/*																		*/
/*		IsRoomModerated()												*/
/*			Returns if a room is moderated. 							*/
/*																		*/
/*		IsRoomDownloadOnly()											*/
/*			Returns if a room is download-only. 						*/
/*																		*/
/*		IsRoomUploadOnly()												*/
/*			Returns if a room is upload-only.							*/
/*																		*/
/*		GetRoomGroupNumber()											*/
/*			Returns the number of a room's group.                       */
/*																		*/
/*		GetRoomGroupExpression()										*/
/*			Returns the boolean expression for a room's group.          */
/*																		*/
/*		GetRoomPrivNumber() 											*/
/*			Returns the number of a room's proviliges group.            */
/*																		*/
/*		IsRoomShared()													*/
/*			Returns if a room is shared (networked).					*/
/*																		*/
/*		IsRoomDirectory()												*/
/*			Returns if a room is a directory room.						*/
/*																		*/
/*		IsRoomHidden()													*/
/*			Returns if a room is hidden.								*/
/*																		*/
/*		IsRoomBIO() 													*/
/*			Returns if a room is by-invitation-only.					*/
/*																		*/
/*		IsRoomSubject() 												*/
/*			Returns if a room forces a subject. 						*/
/*																		*/
/*		IsRoomInuse()													*/
/*			Returns if a room is in use.								*/
/*																		*/
/*		GetCurrentRoomDescription() 									*/
/*			Returns the description of the current room.				*/
/*																		*/
/*		IsRoomAnonymous()												*/
/*			Returns if a room is anonymous. 							*/
/*																		*/
/*		IsRoomArchive() 												*/
/*			Returns if a room is being archived.						*/
/*																		*/
/*		IsRoomPermanent()												*/
/*			Returns if a room is permanent. 							*/
/*																		*/
/*		IsRoomApplication() 											*/
/*			Returns if a room has an application.						*/
/*																		*/
/*		IsCurrentRoomExcludable()										*/
/*			Returns if the current room can be excluded.				*/
/*																		*/
/*		GetCurrentRoomDirectory()										*/
/*			Returns the directory of the current directory room.		*/
/*																		*/
/*		GetCurrentRoomDictionary()										*/
/*			Returns the dictionary for the current room.				*/
/*																		*/
/*		IsCurrentRoomAutoApp()											*/
/*			Returns if the current room has an auto application.		*/
/*																		*/
/*		IsRoomGroupModerated()											*/
/*			Returns if a room is moderated by a group.					*/
/*																		*/
/*	Data types: 														*/
/*																		*/
/*	Data:																*/
/* -------------------------------------------------------------------- */
/*								Private 								*/
/*	Functions:															*/
/*																		*/
/*	Data types: 														*/
/*		aRoom		A room's structure.                                 */
/*		rTable		The room table's structure.                         */
/*																		*/
/*	Data:																*/
/*		roomPos 														*/
/*			The room position table.									*/
/*																		*/
/*		roomTab 														*/
/*			The room table. 											*/
/*																		*/
/*		rtList															*/
/*			AUXMEM only: the room table.								*/
/*																		*/
/*		RoomChunksInHeap												*/
/*			AUXMEM only: Number of room table blocks in the heap.		*/
/*																		*/
/* -------------------------------------------------------------------- */


/* -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- */


r_slot		*roomPos;					/* room position table			*/

#ifdef AUXMEM
	auxTabList		*rtList;
	int 			roomBlocksInHeap;
#else
	#ifdef WINCIT
		rTable huge *roomTab;			/* RAM index of rooms			*/
	#else
		rTable *roomTab;				/* RAM index of rooms			*/
	#endif
#endif


/* -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- */


/* -------------------------------------------------------------------- */
/*	UserCanModerateRoom()	Returns whether the user can moderate a 	*/
/*							room or not.								*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the current user can moderate it.	*/
/*							FALSE if he cannot. 						*/
/* -------------------------------------------------------------------- */
Bool LogEntry::CanModerateRoom(r_slot RoomIndex) const
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return
		(
		IsSysop() ||

		(IsAide() && cfg.moderate) ||

			(
			getRoomTab(RoomIndex)->rtflags.IsGroupModerates() &&
			HasRoomPrivileges(RoomIndex)
			)
		);
	}


/* -------------------------------------------------------------------- */
/*	GetRoomName()			Returns the name of a specified room.		*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		Pointer to string containing room's name.   */
/*							This is an index into the room table: do	*/
/*							not write to this pointer.					*/
/* -------------------------------------------------------------------- */
const char *GetRoomName(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtname);
	}


/* -------------------------------------------------------------------- */
/*	GetRoomNetID()			Returns the name of a specified room.		*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		Pointer to string containing room's name.   */
/*							This is an index into the room table: do	*/
/*							not write to this pointer.					*/
/* -------------------------------------------------------------------- */
const char *GetRoomNetID(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->netID);
	}


/* -------------------------------------------------------------------- */
/*	IsRoomGroupOnly()		Returns if a room is group-only.			*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is group-only. 			*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomGroupOnly(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsGroupOnly());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomMultiGroup()		Returns if a room is multi-group-only.		*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is multi-group-only.		*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomMultiGroup(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsGroupOnly() &&
			getRoomTab(RoomIndex)->rtflags.IsBooleanGroup());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomPrivGroup()		Returns if a room has a priviliges group.	*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room has a priviliges group.	*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomPrivGroup(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsPrivilegedGroup());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomReadOnly()		Returns if a room is read-only. 			*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is read-only.				*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomReadOnly(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsReadOnly());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomModerated()		Returns if a room is moderated. 			*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is moderated.				*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomModerated(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsModerated());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomDownloadOnly()	Returns if a room is download-only. 		*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is download-only.			*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomDownloadOnly(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsDownloadOnly());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomUploadOnly()		Returns if a room is upload-only.			*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is upload-only.			*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomUploadOnly(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsUploadOnly());
	}


/* -------------------------------------------------------------------- */
/*	GetRoomGroupNumber()	Returns the number of a room's group.       */
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about. The room should already have */
/*							been checked with IsRoomGroupOnly to verify */
/*							that the room is group-only.				*/
/*																		*/
/*	Output: 	return		Index of room's group.                      */
/* -------------------------------------------------------------------- */
int GetRoomGroupNumber(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));
	assert(IsRoomGroupOnly(RoomIndex));
	assert(!IsRoomMultiGroup(RoomIndex));

	return (getRoomTab(RoomIndex)->grpno);
	}


/* -------------------------------------------------------------------- */
/*	GetRoomGroupExpression()Returns the exression a room's multi-group. */
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about. The room should already have */
/*							been checked with IsRoomGroupOnly to verify */
/*							that the room is group-only.				*/
/*																		*/
/*	Output: 	return		const pointer to room's expression.         */
/* -------------------------------------------------------------------- */
const uint *GetRoomGroupExpression(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));
	assert(IsRoomGroupOnly(RoomIndex));
	assert(IsRoomMultiGroup(RoomIndex));

	return (getRoomTab(RoomIndex)->GroupExpression);
	}


/* -------------------------------------------------------------------- */
/*	GetRoomPrivNumber() 	Returns the number of a room's priviliges   */
/*							group.										*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about. The room should already have */
/*							been checked with IsRoomPrivGroup to verify */
/*							that the room has a priviliges group.		*/
/*																		*/
/*	Output: 	return		Index of room's group.                      */
/* -------------------------------------------------------------------- */
int GetRoomPrivNumber(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));
	assert(IsRoomPrivGroup(RoomIndex));

	return (getRoomTab(RoomIndex)->pgrpno);
	}


/* -------------------------------------------------------------------- */
/*	IsRoomShared()			Returns if a room is shared.				*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is shared. 				*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomShared(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsShared());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomDirectory()		Returns if a room is a directory room.		*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is a directory room.		*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomDirectory(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsMsDOSdir());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomHidden()			Returns if a room is hidden.				*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is hidden. 				*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomHidden(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (!getRoomTab(RoomIndex)->rtflags.IsPublic());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomBIO() 			Returns if a room is by-invitation-only.	*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is BIO.					*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomBIO(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsBIO());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomSubject() 		Returns if a room forces a subject. 		*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room forces a subject.			*/
/*							FALSE if it does not.						*/
/* -------------------------------------------------------------------- */
Bool IsRoomSubject(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsSubject());
	}


/* -------------------------------------------------------------------- */
/*	GetCurrentRoomDescription() 	Returns the description of the		*/
/*									current room.						*/
/*																		*/
/*	Output: 	return		Pointer to string containing room's name.   */
/*							This is an index into the room table: do	*/
/*							not write to this pointer.					*/
/* -------------------------------------------------------------------- */
const char *GetCurrentRoomDescription(void)
	{
	assert(roomBuf->rbflags.IsInuse());

	return (roomBuf->descript);
	}


/* -------------------------------------------------------------------- */
/*	IsRoomInuse()			Returns if a room is in use.				*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is in use. 				*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomInuse(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);

	return (getRoomTab(RoomIndex)->rtflags.IsInuse());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomAnonymous()		Returns if a room anonymous.				*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is anonymous.				*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomAnonymous(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsAnonymous());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomArchive() 		Returns if a room is being archived.		*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is being archived. 		*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomArchive(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsArchive());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomPermanent()		Returns if a room is permanent. 			*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is permanent.				*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomPermanent(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsPermanent());
	}


/* -------------------------------------------------------------------- */
/*	IsRoomApplication() 	Returns if a room has an application.		*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room has an application.		*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomApplication(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsApplication());
	}


/* -------------------------------------------------------------------- */
/*	IsCurrentRoomExcludable()	Returns if the current room can be		*/
/*								excluded.								*/
/*																		*/
/*	Output: 	return		TRUE if the room can be excluded.			*/
/*							FALSE if it cannot. 						*/
/* -------------------------------------------------------------------- */
Bool IsCurrentRoomExcludable(void)
	{
	assert(roomBuf->rbflags.IsInuse());

	return (!roomBuf->NOEXCLUDE);
	}


/* -------------------------------------------------------------------- */
/*	GetCurrentRoomDirectory()	Returns the directory of the current	*/
/*								directory room. 						*/
/*																		*/
/*	Output: 	return		A pointer to a string containing the room's */
/*							directory.									*/
/*																		*/
/*	Note:	Make sure the room is a directory room before calling this	*/
/*			function.													*/
/* -------------------------------------------------------------------- */
const char *GetCurrentRoomDirectory(void)
	{
	assert(roomBuf->rbflags.IsInuse());
	assert(roomBuf->rbflags.IsMsDOSdir());

	return (roomBuf->rbdirname);
	}


/* -------------------------------------------------------------------- */
/*	GetCurrentRoomDictionary()	Returns the dictionary for the current	*/
/*								room.									*/
/*																		*/
/*	Output: 	return		A pointer to a string containing the room's */
/*							dictionary. 								*/
/* -------------------------------------------------------------------- */
const char *GetCurrentRoomDictionary(void)
	{
	assert(roomBuf->rbflags.IsInuse());

	return (roomBuf->dictionary);
	}


/* -------------------------------------------------------------------- */
/*	IsCurrentRoomAutoApp()		Returns if the current room has an auto */
/*								application.							*/
/*																		*/
/*	Output: 	return		TRUE if the room has an auto appliction.	*/
/*							FALSE if it does not.						*/
/*																		*/
/*	Note:	The room should already have been checked to make sure it	*/
/*			has an application. 										*/
/* -------------------------------------------------------------------- */
Bool IsCurrentRoomAutoApp(void)
	{
	assert(roomBuf->rbflags.IsInuse());
	assert(roomBuf->rbflags.IsApplication());

	return (roomBuf->AUTOAPP);
	}


/* -------------------------------------------------------------------- */
/*	GetCurrentRoomApplication() Returns the application for the current */
/*								room.									*/
/*																		*/
/*	Output: 	return		A pointer to a string containing the room's */
/*							dictionary. 								*/
/*																		*/
/*	Note:	Make sure the room is an application room before calling	*/
/*			this function.												*/
/* -------------------------------------------------------------------- */
const char *GetCurrentRoomApplication(void)
	{
	assert(roomBuf->rbflags.IsInuse());
	assert(roomBuf->rbflags.IsApplication());

	return (roomBuf->rbaplic);
	}


/* -------------------------------------------------------------------- */
/*	IsRoomGroupModerated()	Returns if a room is moderated by a group.	*/
/*																		*/
/*	Input:		RoomIndex	Room's index in the database that we want   */
/*							to know about.								*/
/*																		*/
/*	Output: 	return		TRUE if the room is group-moderated.		*/
/*							FALSE if it is not. 						*/
/* -------------------------------------------------------------------- */
Bool IsRoomGroupModerated(r_slot RoomIndex)
	{
	assert(RoomIndex < cfg.maxrooms);
	assert(RoomIndex >= 0);
	assert(IsRoomInuse(RoomIndex));
	assert(IsRoomPrivGroup(RoomIndex));

	return (getRoomTab(RoomIndex)->rtflags.IsGroupModerates());
	}
