/* -------------------------------------------------------------------- */
/*	This structure specifies the contents of LOG5.DAT. This is used to	*/
/*	keep track of each user's message pointers for each room. For each	*/
/*	room, each user has an unsigned long that is the number of the		*/
/*	newest message on the system the last time the user was in the		*/
/*	room. Any messages newer than this were posted after he was last in */
/*	the room, and are therefore "new."									*/
/* -------------------------------------------------------------------- */
class LogEntry5 : public LogStarter
	{
	ulong	*NewPointer;
	r_slot	MaxRooms;

public:
	void Clear(void)
		{
		assert(this);
		VerifyHeap();

		if (IsValid())
			{
			memset(NewPointer, 0, MaxRooms * sizeof(*NewPointer));
			VerifyHeap();
			}
		}

	LogEntry5(r_slot NumRooms)
		{
		assert(this);
		VerifyHeap();

		MaxRooms = NumRooms;
		NewPointer = new ulong[NumRooms];
		Clear();
		ResetFileName();
		VerifyHeap();
		}

	~LogEntry5()
		{
		assert(this);
		VerifyHeap();
		delete [] NewPointer;
		VerifyHeap();
		}

	LogEntry5(const LogEntry5 &Original)
		{
		assert(this);
		VerifyHeap();
		MaxRooms = Original.MaxRooms;
		NewPointer = new ulong[MaxRooms];

		if (IsValid() && Original.IsValid())
			{
			memcpy(NewPointer, Original.NewPointer, sizeof(*NewPointer) * MaxRooms);
			}

		SetFileName(Original.GetFileName());
		VerifyHeap();
		}

	LogEntry5& operator =(const LogEntry5 &Original)
		{
		assert(this);
		VerifyHeap();
		delete [] NewPointer;

		MaxRooms = Original.MaxRooms;
		NewPointer = new ulong[MaxRooms];

		if (IsValid() && Original.IsValid())
			{
			memcpy(NewPointer, Original.NewPointer, sizeof(*NewPointer) * MaxRooms);
			}

		SetFileName(Original.GetFileName());

		VerifyHeap();
		return (*this);
		}

	Bool Load(l_index Index);
	Bool Save(l_index Index) const;

	void ResetFileName(void)
		{
		assert(this);
		SetFileName(log5Dat);
		}

	void Resize(r_slot NewNumRooms)
		{
		assert(this);
	#if defined(AUXMEM) || defined(WINCIT)
		assert(NewNumRooms <= 16376);
	#else
		assert(NewNumRooms <= 65532u / sizeof(rTable));
	#endif

		VerifyHeap();
		delete [] NewPointer;

		MaxRooms = NewNumRooms;
		NewPointer = new ulong[MaxRooms];
		Clear();
		VerifyHeap();
		}

	Bool IsValid(void) const
		{
		assert(this);
		return (NewPointer != NULL);
		}

	ulong GetRoomNewPointer(r_slot RoomSlot) const
		{
		assert(this);
		assert(RoomSlot >= 0);
		assert(RoomSlot < MaxRooms);

		return (NewPointer[RoomSlot]);
		}

	void SetRoomNewPointer(r_slot RoomSlot, ulong NewNewPointer)
		{
		assert(this);
		assert(RoomSlot >= 0);
		assert(RoomSlot < MaxRooms);

		if (IsValid())
			{
			NewPointer[RoomSlot] = NewNewPointer;
			}
		}

	void CopyRoomNewPointers(const LogEntry5 &ToCopy)
		{
		assert(this);
		if (IsValid() && ToCopy.IsValid())
			{
			r_slot NumCopy;

			NumCopy = min(MaxRooms, ToCopy.MaxRooms);

			memcpy(NewPointer, ToCopy.NewPointer, NumCopy * sizeof(*NewPointer));
			}
		}

	const void *GetPointer(void) const
		{
		assert(this);
		return (NewPointer);
		}
	};
