/* -------------------------------------------------------------------- */
/*  TERM.C                   Dragon Citadel                             */
/* -------------------------------------------------------------------- */
/*                       Terminal emulation code                        */
/* -------------------------------------------------------------------- */

/* -------------------------------------------------------------------- */
/*  Includes                                                            */
/* -------------------------------------------------------------------- */
#include <string.h>
#include <time.h>
#include "ctdl.h"
#include "proto.h"
#include "global.h"

/* -------------------------------------------------------------------- */
/*                              Contents                                */
/* -------------------------------------------------------------------- */
/*  setTerm()       Setup the terminal                                  */
/*  termCap()       Does a terminal command                             */
/*  putCode()       Sends the escape code to the modem                  */
/* -------------------------------------------------------------------- */

/* -------------------------------------------------------------------- */
/*  HISTORY:                                                            */
/*                                                                      */
/*  10/18/90    (PAT)   Created to have all the terminal code           */
/*                                                                      */
/* -------------------------------------------------------------------- */

/* -------------------------------------------------------------------- */
/*  Static Data                                                         */
/* -------------------------------------------------------------------- */
static void putCode(char *str);
void setOutTrans(BOOL ibmAnsi);

/* Name, Descript, 
   Normal, Bold, Inverse, Blink, Underline, 
   IBM-Extended, IBM-Color 
 */

static  TERMINAL    trm_tty = 
    {"TTY", "Generic terminal",
     "", "", "", "", "", 
     FALSE, FALSE };
     
static  TERMINAL    trm_ansi_bbs = 
    {"ANSI-BBS", "IBM system with monocrome ANSI",
     "[0m", "[1m", "[7m", "[5m", "[4m", 
     TRUE, FALSE };
     
static  TERMINAL    trm_ansi_col = 
    {"ANSI-CLR", "IBM system with color ANSI",    
     "[0;1;37;40m", "[1;33;40m", "[1;33;41m", "[5m", "[1;35;40m", 
     TRUE, TRUE };

static  TERMINAL    trm_vt100 = 
    {"VT100", "Non-IBM supporting vt100 graphics",
     "[0m", "[1m", "[7m", "[5m", "[4m", 
     FALSE, FALSE };

static TERMINAL *terms[] = {
    &trm_tty,
    &trm_ansi_bbs,
    &trm_ansi_col,
    &trm_vt100,
    NULL
};

/* -------------------------------------------------------------------- */
/*  putCode()       Sends the escape code to the modem                  */
/* -------------------------------------------------------------------- */
static void putCode(char *str)
{
    while(*str)
    {
        outMod(*str);
        str++;
    }
}


/* -------------------------------------------------------------------- */
/*  localTermCap()  Does a local (non-networked) terminal command       */
/* -------------------------------------------------------------------- */
void localTermCap(char c)
{
    setio(whichIO, echo, outFlag);
    
    switch (toupper(c))
    {
    case TERM_BS:
        doBS();
        pause(5);
        break;
        
    case TERM_IMPERV:
        outFlag = NOSTOP;
        break;
        
    case TERM_PAUSE:
        pause(100);
        break;
    
    case TERM_HANGUP:
        Hangup();
        break;
    
    default:
        break;  
    }
}

/* -------------------------------------------------------------------- */
/*  setTerm()       Setup the terminal                                  */
/* -------------------------------------------------------------------- */
void setTerm(char *t)
{
    int i;
    
    for (i=0; terms[i] != NULL; i++)
    {
        if (strcmpi(terms[i]->name, t) == SAMESTRING)
        {
            term = *terms[i];
            setOutTrans(term.ibmAnsi);
            return;
        }
    }
    
    term = *terms[0];
    setOutTrans(term.ibmAnsi);
}

/* -------------------------------------------------------------------- */
/*  askTerm()       Setup the terminal                                  */
/* -------------------------------------------------------------------- */
void askTerm(void)
{
    int i, i2;
    
    for (i=0; terms[i] != NULL; i++)
    {
        mPrintf(" %d - %-20s %s", i+1, terms[i]->name, terms[i]->descript); 
        doCR();
    }

    i2 = (int)getNumber("Terminal", 0L, (long)i, 0L);

    if (i2)
    {
        term = *terms[i2-1];
        setOutTrans(term.ibmAnsi);
    }
}

/* -------------------------------------------------------------------- */
/*  askTerm()       Setup the terminal                                  */
/* -------------------------------------------------------------------- */
void setOutTrans(BOOL ibmAnsi)
{
    uint i;
    
    for (i=0; i<0xFF; i++)
    {
        outFilter[i] = (uchar)i;
    }
    
    if (!ibmAnsi)
    {
        for (i=127; i<0xFF; i++)
            outFilter[i] = 0x00;
    
        memcpy(outFilter + 128, "CueaaaaceeeiiiAAE  ooouuyoUc$Y f"
                                "aiounN**?++  !<>.:*|++++++|+++++"
                                "++++-++++++++-+++++++++++++*_[]\""
                                "aBr Eou         = ><|| =***/    ", 127);
    }
}

/* -------------------------------------------------------------------- */
/*  termCap()       Does a terminal command                             */
/* -------------------------------------------------------------------- */
static int iso_clr[] = {0, 0x4, 0x2, 0x6, 0x1, 0x5, 0x3, 0x7};
    
void termCap(char c)
{
    char str[10];
    static BOOL hilight = FALSE,    /* hilight is on */
                blink = FALSE,      /* blinking is on */
                dirty = TRUE;       /* the hilight (remote) is dirty */
    
    setio(whichIO, echo, outFlag);
    
    /*
     *  ISO COLOR support, should handle damn near any case.. 
     */
    
    /* 
     * Since the ^A0 may actualy be hilighted (or the ^A3 may not!) 
     * we need to reset the hilight state if the system is flagged
     * 'dirty'.
     *
     * Also if we reset to non-hilighted we need to reoutput the
     * blinking if it was set.
     */
    if (dirty && ((c >= 'A' && c <= 'H') || (c >= 'a' && c <= 'h')))
    {
        sprintf(str, "%c[%dm", 27, hilight ? 1 : 0);
        putCode(str);
        if (blink && !hilight)
            putCode("[5m");
        dirty = FALSE;
    }
    
    /*
     * Send the background
     */
    if (c >= 'A' && c <= 'H')
    {
        if (term.ibmColor)
        {
            sprintf(str, "%c[4%cm", ESC, '0' + (c - 'A'));
            putCode(str);
            ansiattr = (ansiattr & 0x0F) 
                       | (iso_clr[(c - 'A')] << 4) | (blink ? 0x80 : 0);
        }
        return;
    }
    
    /*
     * Send the forground 
     */
    if (c >= 'a' && c <= 'h')
    {
        if (term.ibmColor)
        {
            sprintf(str, "%c[3%cm", ESC, '0' + (c - 'a'));
            putCode(str);
            ansiattr = (ansiattr & 0xF0) 
                       | (iso_clr[(c - 'a')]) | (hilight ? 0x08 : 0);
        }
        return;
    }
    
    /*
     * Normal DragCit colors.. 
     */
    switch (toupper(c))
    {
    case TERM_BLINK:
        if (*term.blink)
        {
            putCode(term.blink);
            ansiattr = ansiattr | 128;
            blink = TRUE;
        }
        break;
        
    case TERM_REVERSE:
        if (*term.inverse)
        {
            putCode(term.inverse);
            ansiattr = cfg.wattr;
            dirty = TRUE;
        }
        break;
        
    case TERM_BOLD:
        if (*term.bold)
        {
            putCode(term.bold);
            ansiattr = cfg.cattr;
            hilight = TRUE;
            dirty = FALSE;
        }
        break;
        
    case TERM_UNDERLINE:
        if (*term.under)
        {
            putCode(term.under);
            ansiattr = cfg.uttr;
            dirty = TRUE;
        }
        break;
        
    case TERM_NORMAL:
    default:
        if (*term.normal)
        {
            putCode(term.normal);
            ansiattr = cfg.attr;
            hilight = FALSE;
            blink = FALSE;
            dirty = TRUE;
        }
        break;
    }
}

