/* -------------------------------------------------------------------- */
/*  CARRIER.C                Dragon Citadel                             */
/* -------------------------------------------------------------------- */
/*  Code to detect and report on Carrier.                               */
/* -------------------------------------------------------------------- */

/* -------------------------------------------------------------------- */
/*  Includes                                                            */
/* -------------------------------------------------------------------- */
#include <string.h>
#include <time.h>
#include "ctdl.h"
#include "proto.h"
#include "global.h"

/* -------------------------------------------------------------------- */
/*                              Contents                                */
/* -------------------------------------------------------------------- */
/*  haveCarrier()   do we have a connection, either console or modem?   */
/*  getModStr()     get a string from the modem, waiting for upto 3 secs*/
/*  carrier()       checks carrier                                      */
/*  carrdetect()    sets global flags for carrier detect                */
/*  carrloss()      sets global flags for carrier loss                  */
/*  checkCR()       Checks for CRs from the data port for half a second.*/
/*  findbaud()      Finds the baud from sysop and user supplied data.   */
/*  ringdetectbaud()    sets baud rate according to ring detect         */
/*  verbosebaud()   sets baud rate according to verbodse codes          */
/*  numericbaud()   sets baud rate according to numeric codes           */
/* -------------------------------------------------------------------- */

/* -------------------------------------------------------------------- */
/*  HISTORY:                                                            */
/*                                                                      */
/*  08/15/90    (PAT)   Created from modem.c, mode carrier detect code  */
/*                      out of iChar() to main prompt..                 */
/*                                                                      */
/* -------------------------------------------------------------------- */

/* -------------------------------------------------------------------- */
/*  haveCarrier()   do we have a connection, either console or modem?   */
/* -------------------------------------------------------------------- */
BOOL haveCarrier(void)
{
    return ((onConsole || (gotCarrier() && modStat)) && !ExitToMsdos);
}

/* -------------------------------------------------------------------- */
/*  carrdetect()    sets global flags for carrier detect                */
/* -------------------------------------------------------------------- */
void carrdetect(void)
{
    char temp[30];
    
    warned          = FALSE;
    modStat         = TRUE;

    time(&conntimestamp);

    connectcls();
    update25();

    sprintf(temp, "Carrier-Detect (%d)", bauds[speed]);
    trap(temp,  T_CARRIER);

    logBuf.credits = cfg.unlogbal;
}

/* -------------------------------------------------------------------- */
/*  carrloss()      sets global flags for carrier loss                  */
/* -------------------------------------------------------------------- */
void carrloss(void)
{
    outFlag         = OUTSKIP;
    modStat         = FALSE;
    pause(100);
    Initport();   /* Don't know weather that'll do anything or not...but... */

    trap("Carrier-Loss", T_CARRIER);
}

/* -------------------------------------------------------------------- */
/*  checkCR()       Checks for CRs from the data port for half a second.*/
/* -------------------------------------------------------------------- */
BOOL checkCR(void)
{
    int i;

    for (i = 0; i < 50; i++)
    {
        pause(1);
        if (MIReady()) if (getMod() == '\r') return FALSE;
    }
    return TRUE;
}

/* -------------------------------------------------------------------- */
/*  findbaud()      Finds the baud from sysop and user supplied data.   */
/* -------------------------------------------------------------------- */
int findbaud(void)
{
    char noGood = TRUE;
    int  Time = 0;
    int  baudRunner;                    /* Only try for 60 seconds      */

    while (MIReady())   getMod();               /* Clear garbage        */
    baudRunner = 0;
    while (gotCarrier() && noGood && Time < 120)
    {
        Time++;
        baud(baudRunner);
        noGood = checkCR();
        if (noGood) baudRunner = (baudRunner + 1) % (3 /* 2400 too */);
    }
    return !noGood;
}

/* -------------------------------------------------------------------- */
/*  ringdetectbaud()    sets baud rate according to ring detect         */
/* -------------------------------------------------------------------- */
void ringdetectbaud(void)
{
    baud(ringdetect());
}

/* -------------------------------------------------------------------- */
/*  verbosebaud()   sets baud rate according to verbodse codes          */
/* -------------------------------------------------------------------- */
int verbosebaud(void)
{
    char c, f=0;
    long t;

    if (debug)
        outCon('[');

    time(&t);
    
    while(gotCarrier() && time(NULL) < (t + 4) && !KBReady())
    {
        if(MIReady())
        {
            c = (char)getMod();
        }
        else 
        {
            c = 0;
        }

        if (debug && c)
        {
            outCon(c);
        }

        if (f)
        {
            switch(c)
            {
            case '\n':
            case '\r':  /* CONNECT */
                baud(0);
                if (debug)
                    outCon(']');
                return TRUE;
                
            case '1':   /* CONNECT 1200 */
                baud(1);
                if (debug)
                    outCon(']');
                return TRUE;
                
            case '2':   /* CONNECT 2400 */
                baud(2);
                if (debug)
                    outCon(']');
                return TRUE;
                
            case '4':   /* CONNECT 4800 */
                baud(3);
                if (debug)
                    outCon(']');
                return TRUE;
                
            case '9':   /* CONNECT 9600 */
                baud(4);
                if (debug)
                    outCon(']');
                return TRUE;
                
            default:
                break;
            }
        }

        if (c == 'C')
        {
            if (debug)
            {
                outCon('@');
            }
            f = 1;
        }
    }
    return FALSE;
}

/* -------------------------------------------------------------------- */
/*  numericbaud()   sets baud rate according to numeric codes           */
/* -------------------------------------------------------------------- */
int numericbaud(void)
{
    char str[42]; /* for baud detect */
    int i   ;
    
    while (getModStr(str) && gotCarrier())
    {
        i =  atoi(str);
        
        switch(i)   
        {
        case 13:  /* 9600   13 or 17   */
        case 17:
            baud(4);
            return TRUE;

        case 10:  /* 2400   10 or 16   */
        case 16:
            baud(2);
            return TRUE;

        case 5:   /* 1200   15 or  5   */
        case 15:
            baud(1);
            return TRUE;

        case 1:   /* 300    1  */
            baud(0);
            return TRUE;

        default:  /* something else */
            break;
        }
    }
    
    return FALSE;
}

/* -------------------------------------------------------------------- */
/*  getModStr()     get a string from the modem, waiting for upto 3 secs*/
/*                  for it. Returns TRUE if it gets one.                */
/* -------------------------------------------------------------------- */
int getModStr(char *str)
{
    long tm;
    int  l = 0, c;

    time(&tm);

    if (debug) cPrintf("[");

    while (
             (time(NULL) - tm) < 4 
          && !KBReady() 
          && l < 40 
          )
    {
        if (MIReady())
        {
            c = getMod();

            if (c == 13 || c == 10) /* CR || LF */
            {
                str[l] = NULL;
                if (debug) cPrintf("]\n");
                return TRUE;
            }
            else
            {
                if (debug) cPrintf("%c", c);
                str[l] = (char)c;
                l++;
            }
        }
    }

    if (debug) cPrintf(":F]\n");

    str[0] = NULL;

    return FALSE;
}

/* -------------------------------------------------------------------- */
/*  carrier()       checks carrier                                      */
/* -------------------------------------------------------------------- */
#ifndef TERM
int carrier(void)
{
    unsigned char c;

    if (disabled)
    {
        return FALSE;
    }
    
    if (gotCarrier())
    {
        switch(cfg.dumbmodem)
        {
        case 0:     /* numeric */
            c = numericbaud();
            break;

        case 1:     /* returns */
            c = findbaud();
            break;

        case 2:     /* HS on RI */
            ringdetectbaud();
            c = TRUE;
            break;

        case 3:     /* verbose */
            c = verbosebaud();
            break;

        default:
        case 4:     /* forced */
            baud(cfg.initbaud);
            c = TRUE;
            break;
        }

        if (c)
        {
            update25();
            carrdetect();
            update25();
            return(TRUE);
        }
        else
        {
            update25();
            Initport();
            return(FALSE);
        }
    } 
    else
    {
        return FALSE;
    }
}
#endif 

