/*
**  --- xymodem.c ---
**
**  Copyright (C) 1996 by MarshallSoft Computing, Inc.
**
**  NOTES: (1) Assumes 8 data bits.
**         (2) XYMODEM.C and other protocol code depends on
**             XYZ_IO and CRC.
**         (3) All down loads are from the sub-directory DNLOAD.ALL
**         (4) All up loads are to the sub-directory UPLOAD.?
**             where ? is the port number [0,1,2,...]
*/

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <malloc.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <errno.h>
#include <sys\types.h>
#include <sys\stat.h>

#include "find.h"
#include "pcl4c.h"
#include "ascii.h"
#include "xyz_io.h"
#include "crc.h"
#include "tsd.h"
#include "xymodem.h"

/* state function constants */

#define WAITFORINCOMING    0x0010
#define WAITFORINCOMING_1  WAITFORINCOMING+1

#define TXYSTARTUP         0x0020
#define TXYSTARTUP_1       TXYSTARTUP+1
#define TXYSTARTUP_2       TXYSTARTUP+2

#define RXYSTARTUP         0x0030
#define RXYSTARTUP_1       RXYSTARTUP+1
#define RXYSTARTUP_2       RXYSTARTUP+2

#define TXYEOT             0x0040
#define TXYEOT_1           TXYEOT+1
#define TXYEOT_2           TXYEOT+2

#define TXYPACKET          0x0050
#define TXYPACKET_1        TXYPACKET+1
#define TXYPACKET_2        TXYPACKET+2
#define TXYPACKET_3        TXYPACKET+3
#define TXYPACKET_4        TXYPACKET+4

#define RXYPACKET          0x0060
#define RXYPACKET_1        RXYPACKET+1
#define RXYPACKET_2        RXYPACKET+2
#define RXYPACKET_3        RXYPACKET+3
#define RXYPACKET_4        RXYPACKET+4
#define RXYPACKET_5        RXYPACKET+5
#define RXYPACKET_6        RXYPACKET+6

#define TXYMODEM           0x0070
#define TXYMODEM_1         TXYMODEM+1
#define TXYMODEM_2         TXYMODEM+2
#define TXYMODEM_3         TXYMODEM+3
#define TXYMODEM_4         TXYMODEM+4
#define TXYMODEM_5         TXYMODEM+5
#define TXYMODEM_6         TXYMODEM+6

#define RXYMODEM           0x0080
#define RXYMODEM_1         RXYMODEM+1
#define RXYMODEM_2         RXYMODEM+2
#define RXYMODEM_3         RXYMODEM+3
#define RXYMODEM_4         RXYMODEM+4

#define TYMODEM            TY_MODEM
#define TYMODEM_1          TY_MODEM+1

#define RYMODEM            RY_MODEM
#define RYMODEM_1          RY_MODEM+1

#define TXMODEM            TX_MODEM

#define RXMODEM            RX_MODEM

/* constants */

#define DEBUG  0

#ifndef NULL
#define NULL (void *)0
#endif

#define FALSE 0
#define TRUE !FALSE
#define MAXTRY    3
#define LIMIT    60
#define XYDELAY   1
#define ABORT_CHAR CAN
#define ONE_SECOND  18
#define SHORT_WAIT   4
#define LONG_WAIT   3

#define NBR_CHANS 16

typedef struct NameTag
{char Name[13];
 struct NameTag *Next;
} NameType;

typedef struct ChannelTag
{long FileSize;       /* file size */
 long ByteCount;      /* bytes remaining to send or bytes received */
 long Timeout;        /* WaitForIncoming: timeout time */
 int  BytesWanted;    /* WaitForIncoming: # bytes wanted */
 int  Port;           /* port # */
 int  Handle;         /* file handle */
 int  PacketSeq;      /* packet sequence number # */
 int  PacketNbr;      /* packet number (PacketSeq % 256) */
 int  PacketSize;     /* packet size (128 or 1024) */
 int  PacketType;     /* packet type */
 int  RxPacketNbr;    /* RX packet number */
 int  Attempt;        /* packet retry attempt # */
 int  NAKcount;       /* number of NAKs send or received */
 int  Loop;           /* loop iterator */
 int  Result;         /* state function result */
 unsigned short CheckSum;  /* check sum or CRC */
 char NCGchar;        /* NAK or 'C' */
 char OneKflag;       /* TRUE: send 1K packets */
 char BatchFlag;      /* TRUE: send filename in packet 0 */
 char EOTseen;        /* TRUE: EOT was seen */
 char EOFseen;        /* TRUE: EOF was seen */
 char EmptyFlag;      /* TRUE: Filename is empty */
 char FileName[13];   /* file name */
 NameType *FirstName; /* first YMODEM name */
 char Buffer[1024];   /* 1K packet buffer */
} ChannelType;

#define CALL_WAITFOR(Tics,Required) {ChanPtr->Timeout = SioTimer()+(Tics);ChanPtr->BytesWanted = (Required);ThreadCallState(WAITFORINCOMING);}

/* globals */

static int OneSecond = ONE_SECOND;
static int ShortWait = SHORT_WAIT * ONE_SECOND;
static int LongWait  = LONG_WAIT * ONE_SECOND;
static int NAKrate   = 10 * ONE_SECOND;
static int DebugFlag = 0;

static int NbrChannels = 0;
static ChannelType *ChanPtr = NULL;
static ChannelType *ChannelData[NBR_CHANS] =
   {NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,
    NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL};
static char PathName[40];

static int CurrentChannel;  /* same as current thread */


/*** set string ***/

static void SetString(int Thread,char *MsgPtr)
{xyzClear(Thread);
 xyzPutString(Thread,MsgPtr);
}

/*** call SioGetc ***/

static int GetChar(void)
{int Code;
 Code = SioGetc(ChanPtr->Port,0);
 if(Code<-1)
   {SetString(ChanPtr->Port,"SioGetc error:");
    xyzPutInteger(ChanPtr->Port,Code);
   }
 return Code;
}

/*** call SioPutc ***/

static int PutChar(char Chr)
{int Code;
 Code = SioPutc(ChanPtr->Port,Chr);
 if(Code<-1)
   {SetString(ChanPtr->Port,"SioPutc error:");
    xyzPutInteger(ChanPtr->Port,Code);
   }
 return Code;
}

/*** display error ***/

static void SayError(int Thread,char *String)
{SetString(Thread,"ERROR: COM");
 xyzPutInteger(Thread,1+ChanPtr->Port);
 xyzPutChar(Thread,':');
 xyzPutString(Thread,String);
}

/*** transmit CANs ***/

static void TxCAN(void)
{int i;
 for(i=0;i<6;i++) PutChar(CAN);
}

/*** display packet error ***/

static void PacketError(int Thread,char *MsgPtr)
{SetString(Thread,"ERROR: COM");
 xyzPutInteger(Thread,Thread);
 xyzPutChar(Thread,':');
 xyzPutString(Thread,"P=");
 xyzPutInteger(Thread,ChanPtr->PacketSeq);
 xyzPutString(Thread,",A=");
 xyzPutInteger(Thread,ChanPtr->Attempt);
 xyzPutString(Thread," ");
 xyzPutString(Thread,MsgPtr);
 TxCAN();
}

/*** display NCG char ***/

static void SayNCGchar(int Thread)
{ChanPtr = ChannelData[Thread];
 switch(ChanPtr->NCGchar)
   {case NAK:
      SetString(Thread,"Using Checksum");
      break;
    case 'C':
      SetString(Thread,"Using CRC");
      break;
    default:
      SetString(Thread,"What is ");
      xyzPutHexInt(Thread,ChanPtr->NCGchar);
   }
}

/*
** waits for incoming character(s)
**
** ENTRY:     Timeout : timer tics before timeout
**        BytesWanted : # bytes wanted before returning
*/

static void WaitForIncoming(int Thread)
{ChanPtr = ChannelData[Thread];
 if(SioRxQue(ChanPtr->Port) >= ChanPtr->BytesWanted)
  {/* incoming byte(s) ! */
   ThreadReturnState(); return;
  }
 ThreadNextState();
}

static void WaitForIncoming_1(int Thread)
{ChanPtr = ChannelData[Thread];
 /* have we timed out ? */
 if(SioTimer() > ChanPtr->Timeout)
   {/* timed out */
    ThreadReturnState(); return;
   }
 ThreadJumpState(WAITFORINCOMING);
}

/*** start up transmitter side ***/

static void TxyStartup(int Thread)
{ChanPtr = ChannelData[Thread];
 if(DebugFlag) SetString(Thread,"### TxyStartup");
 /* clear Rx buffer */
 SioRxClear(ChanPtr->Port);
 /* wait for receiver's start up NAK or 'C' */
 ChanPtr->Loop = 0;
 ThreadNextState();
}

static void TxyStartup_1(int Thread)
{ChanPtr = ChannelData[Thread];
 CALL_WAITFOR(OneSecond,1);
}

static void TxyStartup_2(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 Code = GetChar();
 if(Code>=0)
   {/* received a byte */
    if(((char)Code==NAK)||((char)Code=='C'))
       {ChanPtr->NCGchar = (char)Code;
        ChanPtr->Result = TRUE;
        ThreadReturnState(); return;
       }
     /* did other side cancel ? */
     if((char)Code==CAN)
       {SayError(Thread,"Cancelled by receiver");
        ChanPtr->Result = FALSE;
        ThreadReturnState(); return;
       }
    /* else dump the byte */
   }
 /* Code < 0 */
 if(++ChanPtr->Loop > LIMIT)
   {/* fails */
    SayError(Thread,"No response from receiver");
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 /* try again */
 ThreadJumpState(TXYSTARTUP_1);
} /* end -- TxyStartup */

/*** start up receiver side ***/

static void RxyStartup(int Thread)
{ChanPtr = ChannelData[Thread];
 if(DebugFlag)
   {SetString(Thread,"### RxyStartup:");
    xyzPutChar(Thread,ChanPtr->NCGchar);
   }
 /* clear RX buffer */
 SioRxClear(ChanPtr->Port);
 SayNCGchar(Thread);
 /* Send NAKs or 'C's */
 ChanPtr->Loop = 0;
 ThreadNextState();
}

static void RxyStartup_1(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 if(++ChanPtr->Loop > LIMIT)
   {/* no response */
    SayError(Thread,"No response from sender");
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 /* stop attempting CRC after 1st 4 tries */
 if((ChanPtr->NCGchar!=NAK)&&(ChanPtr->Loop==5))
   {SetString(Thread,"Switching to NAKs");
    ChanPtr->NCGchar = NAK;
   }
 /* send next NAK or 'C' */
 PutChar(ChanPtr->NCGchar);
 CALL_WAITFOR(NAKrate,1);
}

static void RxyStartup_2(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 Code = GetChar();
 if(Code>=0)
    {/* got byte -- did other side cancel ? */
     if((char)Code==CAN)
       {SayError(Thread,"Cancelled by receiver");
        ChanPtr->Result = FALSE;
        ThreadReturnState(); return;
       }
     /* push byte back onto queue ! */
     SioUnGetc(ChanPtr->Port,(char)Code);
     ChanPtr->Result = TRUE;
     ThreadReturnState(); return;
    }
 /* send another NAK / 'C' */
 ThreadJumpState(RXYSTARTUP_1);
} /* end -- RxyStartup */

/*** send EOT ***/

static void TxyEOT(int Thread)
{ChanPtr = ChannelData[Thread];
 ChanPtr->Loop = 0;
 ThreadNextState();
}

static void TxyEOT_1(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 if(++ChanPtr->Loop==10)
   {ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 PutChar(EOT);
 /* await response */
 ThreadDelay(ShortWait);
 ThreadNextState();
}

static void TxyEOT_2(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 Code = GetChar();
 if((char)Code==ACK)
   {ChanPtr->Result = TRUE;
    ThreadReturnState(); return;
   }
 else ThreadJumpState(TXYEOT_1);
}

/* send packet */

static void TxyPacket(int Thread)
{/* begin */
 ChanPtr = ChannelData[Thread];
 if(DebugFlag)
   {SetString(Thread,"TxP: COM");
    xyzPutInteger(Thread,1+ChanPtr->Port);
    xyzPutString(Thread,":Packet=");
    xyzPutInteger(Thread,ChanPtr->PacketSeq);
    xyzPutString(Thread,",PacketSize=");
    xyzPutInteger(Thread,ChanPtr->PacketSize);
    xyzPutString(Thread,",NCGchar=");
    xyzPutChar(Thread,ChanPtr->NCGchar);
   }
 else
   {SetString(Thread,"Packet ");
    xyzPutInteger(Thread,ChanPtr->PacketNbr);
   }
 /* better be 128 or 1024 packet size */
 if(ChanPtr->PacketSize==1024) ChanPtr->PacketType = STX;
 else ChanPtr->PacketType = SOH;
 ChanPtr->PacketNbr &= 0x00ff;
 /* make up to MAXTRY attempts to send this packet */
 ChanPtr->Attempt = 0;
 ThreadNextState();
}

static void TxyPacket_1(int Thread)
{ChanPtr = ChannelData[Thread];
 if(++ChanPtr->Attempt==MAXTRY)
   {ChanPtr->Result = FALSE;
    /* can't send packet ! */
    SayError(Thread,"TxP: Packet timeout (3 NAKs)");
    ThreadReturnState(); return;
   }
 /* send SOH/STX  */
 PutChar((char)ChanPtr->PacketType);
 ThreadDelay(XYDELAY);
 ThreadNextState();
}

static void TxyPacket_2(int Thread)
{int i;
 ChanPtr = ChannelData[Thread];
 /* send packet # */
 PutChar((char)ChanPtr->PacketNbr);
 /* send 1's complement of packet */
 PutChar((char)(255-ChanPtr->PacketNbr));
 /* send data */
 ChanPtr->CheckSum = 0;
 for(i=0;i<ChanPtr->PacketSize;i++)
   {PutChar(ChanPtr->Buffer[i]);
    /* update checksum / CRC */
    if(ChanPtr->NCGchar==NAK) ChanPtr->CheckSum += ChanPtr->Buffer[i];
    else ChanPtr->CheckSum = ComputeCRC(ChanPtr->Buffer[i],ChanPtr->CheckSum);
   }
 /* flush reverse channel */
 SioRxClear(ChanPtr->Port);
 /* send checksum */
 if(ChanPtr->NCGchar==NAK)
   PutChar((char)(ChanPtr->CheckSum & 0x00ff) );
 else
   {PutChar( (char)((ChanPtr->CheckSum>>8) & 0x00ff) );
    PutChar( (char)(ChanPtr->CheckSum & 0x00ff) );
   }
 /* wait for receivers ACK */
 ThreadDelay(XYDELAY);     /* ??? */
 ThreadNextState();
}

static void TxyPacket_3(int Thread)
{int i;
 int Code;
 ChanPtr = ChannelData[Thread];
 CALL_WAITFOR(LongWait,1);
}

static void TxyPacket_4(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 Code = GetChar();
 if(Code==-1)
   {SetString(Thread,"TxP: No response from receiver");
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 /* check incoming */
 if((char)Code==CAN)
   {SetString(Thread,"TxP: Cancelled by REMOTE");
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 if((char)Code==ACK)
   {ChanPtr->Result = TRUE;
    ThreadReturnState(); return;
   }
 if((char)Code!=NAK)
   {PacketError(Thread,"TxP: Expecting ACK/NAK not ");
    xyzPutHexInt(Thread,Code);
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 /* again */
 SetString(Thread,"Packet  ");
 xyzPutInteger(Thread,ChanPtr->PacketNbr);
 xyzPutString(Thread," NAKed");
 ChanPtr->NAKcount++;
 ThreadJumpState(TXYMODEM_1);
}

/*** receive packet ***/

static void RxyPacket(int Thread)
{int Code;
 /* begin */
 ChanPtr = ChannelData[Thread];
 if(!DebugFlag)
   {SetString(Thread,"Packet ");
    xyzPutInteger(Thread,ChanPtr->PacketNbr);
   }
 ChanPtr->PacketNbr &= 0x00ff;
 ChanPtr->Attempt = 0;
 ThreadNextState();
}

static void RxyPacket_1(int Thread)
{ChanPtr = ChannelData[Thread];
 if(++ChanPtr->Attempt==MAXTRY)
   {ChanPtr->Result = FALSE;
    /* can't receive packet ! */
    SayError(Thread,"RxP: packet retries exceeded");
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 /* wait for incoming SOH or STX */
 CALL_WAITFOR(ShortWait,1);
}

static void RxyPacket_2(int Thread)
{int i;
 int Code;
 ChanPtr = ChannelData[Thread];
 /* expect SOH or STX */
 Code = GetChar();
 if(Code==-1)
   {PacketError(Thread,"RxP: Timed out waiting for SOH/STX/EOT/CAN");
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 switch((char)Code)
   {case SOH:
      /* 128 byte buffer incoming */
      ChanPtr->PacketSize = 128;
      break;
    case STX:
      /* 1024 byte buffer incoming */
      ChanPtr->PacketSize = 1024;
      break;
    case EOT:
      /* all packets have been sent */
      PutChar(ACK);
      ChanPtr->EOTseen = TRUE;
      ChanPtr->Result = TRUE;
      if(DebugFlag&&ChanPtr->EOTseen)
        {SetString(Thread,"RxP: COM");
         xyzPutInteger(Thread,1+ChanPtr->Port);
         xyzPutString(Thread,":EOT");
        }
      ThreadReturnState(); return;
    case CAN:
      /* sender has cancelled ! */
      SetString(Thread,"RxP: Cancelled by REMOTE");
      ChanPtr->Result = FALSE;
      ThreadReturnState(); return;
    default:
      /* error ! */
      PacketError(Thread,"RxP: Expected SOH/STX/EOT/CAN not ");
      xyzPutHexInt(Thread,Code);
      ChanPtr->Result = FALSE;
      ThreadReturnState(); return;
   }
 if(DebugFlag&&(!ChanPtr->EOTseen))
   {SetString(Thread,"RxP: COM");
    xyzPutInteger(Thread,1+ChanPtr->Port);
    xyzPutString(Thread,":Packet=");
    xyzPutInteger(Thread,ChanPtr->PacketSeq);
    xyzPutString(Thread,",NCGchar=");
    xyzPutChar(Thread,ChanPtr->NCGchar);
    xyzPutString(Thread,",PacketSize=");
    xyzPutInteger(Thread,ChanPtr->PacketSize);
   }
 /* wait for incoming packet number & packet # compliment */
 CALL_WAITFOR(ShortWait,2);
}

static void RxyPacket_3(int Thread)
{int Code;
 int RxPacketNbr;
 int RxPacketNbrComp;
 ChanPtr = ChannelData[Thread];
 /* expect packet # & packet # compliment */
 Code = GetChar();
 if(Code==-1)
    {PacketError(Thread,"RxP: Timed out waiting for packet number");
     ChanPtr->Result = FALSE;
     ThreadReturnState(); return;
    }
 RxPacketNbr = 0x00ff & Code;
 /* expect 1's complement */
 Code = GetChar();
 if(Code==-1)
   {PacketError(Thread,"RxP: Timed out waiting for complement of packet #");
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 RxPacketNbrComp = 0x00ff & Code;
 /* verify packet number */
 if(RxPacketNbr+RxPacketNbrComp!=255)
   {PacketError(Thread,"RxP: Packet sequence error");
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 ChanPtr->RxPacketNbr = RxPacketNbr;
 ThreadNextState();
}

static void RxyPacket_4(int Thread)
{ChanPtr = ChannelData[Thread];
 /* receive data */
 ChanPtr->CheckSum = 0;
 ChanPtr->Loop = 0;
 ThreadNextState();
}

static void RxyPacket_5(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 if(++ChanPtr->Loop==ChanPtr->PacketSize)
   {/* can't receive packet */
    SayError(Thread,"RxP: Packet retries exceeded");
    ChanPtr->Result = FALSE;
    ThreadReturnState(); return;
   }
 /* wait for incoming data packet */
 CALL_WAITFOR(ShortWait,ChanPtr->PacketSize+1);
}

static void RxyPacket_6(int Thread)
{int i;
 int Code;
 unsigned short RxCheckSum;
 unsigned short RxCheckSum1;
 unsigned short RxCheckSum2;
 ChanPtr = ChannelData[Thread];
 /* read the data packet */
 for(i=0;i<ChanPtr->PacketSize;i++)
   {Code = GetChar();
    if(Code==-1)
      {PacketError(Thread,"RxP: Timed out waiting for data. Index=");
       xyzPutInteger(Thread,i);
       ChanPtr->Result = FALSE;
       ThreadReturnState(); return;
      }
    ChanPtr->Buffer[i] = (char)Code;
    /* compute CRC or checksum */
    if(ChanPtr->NCGchar!=NAK) ChanPtr->CheckSum =
       ComputeCRC((unsigned char)Code,ChanPtr->CheckSum);
    else ChanPtr->CheckSum = (ChanPtr->CheckSum + Code) & 0x00ff;
   }
 /* read CRC / checksum */
 if(ChanPtr->NCGchar!=NAK)
   {/* receive 2 byte CRC */
    Code = GetChar();
    if(Code==-1)
      {PacketError(Thread,"RxP: Timed out waiting for 1st CRC byte");
       ChanPtr->Result = FALSE;
       ThreadReturnState(); return;
      }
    RxCheckSum1 = Code & 0x00ff;
    Code = GetChar();
    if(Code==-1)
      {PacketError(Thread,"RxP: Timed out waiting for 2nd CRC byte");
       ChanPtr->Result = FALSE;
       ThreadReturnState(); return;
      }
    RxCheckSum2 = Code & 0x00ff;
    RxCheckSum = (RxCheckSum1<<8) | RxCheckSum2;
   }
 else
   {/* receive one byte checksum */
    Code = GetChar();
    if(Code==-1)
      {PacketError(Thread,"RxP: Timed out waiting for checksum");
       ChanPtr->Result = FALSE;
       ThreadReturnState(); return;
      }
    RxCheckSum = Code & 0x00ff;
   }
 /* is checksum OK ? */
 if(RxCheckSum!=ChanPtr->CheckSum)
   {/* bad checksum */
    PutChar(NAK);
    ChanPtr->NAKcount++;
    SetString(Thread,"RxP: NAKing packet ");
    xyzPutInteger(Thread,ChanPtr->PacketNbr);
    if(DebugFlag)
      {SetString(Thread,"RxP: Packet ");
       xyzPutInteger(Thread,ChanPtr->PacketNbr);
       xyzPutString(Thread,": Checksum: RX=");
       xyzPutHexInt(Thread,RxCheckSum);
       xyzPutString(Thread,", Computed=");
       xyzPutHexInt(Thread,ChanPtr->CheckSum);
      }
    /* NAK the packet */
    PutChar(NAK);
    ThreadJumpState(RXYMODEM_1); return;
   }
 else
   {/* is packet number OK ? */
    if(ChanPtr->RxPacketNbr!=ChanPtr->PacketNbr)
      {PutChar(NAK);
       ChanPtr->NAKcount++;
       SetString(Thread,"RxP: NAKing packet ");
       xyzPutInteger(Thread,ChanPtr->PacketNbr);
       /* NAK the packet */
       PutChar(NAK);
       ThreadJumpState(RXYMODEM_1); return;
      }
   }
 /* good packet. flush reverse channel */
 SioRxClear(ChanPtr->Port);
 /* ACK the packet */
 PutChar(ACK);
 ChanPtr->Result = TRUE;
 ThreadReturnState();
}

/*** start xymodem send ***/

static void TxyModem(int Thread)
{int  Code;
 /* begin */
 ChanPtr = ChannelData[Thread];
 ChanPtr->EmptyFlag = FALSE;
 ChanPtr->PacketSize = 128;
 if(ChanPtr->BatchFlag) if(ChanPtr->FileName[0]=='\0') ChanPtr->EmptyFlag = TRUE;
 if(!ChanPtr->EmptyFlag)
   {/* Filename is NOT empty */
    ChanPtr->EmptyFlag = FALSE;
    sprintf(PathName,"DNLOAD.ALL\\%s",ChanPtr->FileName);
    ChanPtr->Handle = open(PathName, O_BINARY|O_RDONLY, S_IREAD);
    if(ChanPtr->Handle<0)
      {SetString(Thread,"Cannot open ");
       xyzPutString(Thread,PathName);
       TxCAN();
       ChanPtr->Result = FALSE;
       ThreadReturnState(); return;
      }
    SetString(Thread,"Opened/R ");
    xyzPutString(Thread,PathName);
   }
 SetString(Thread,"XYMODEM send: waiting for Receiver ");
 if(!ChanPtr->EmptyFlag)
   {/* compute filesize */
    ChanPtr->FileSize = lseek(ChanPtr->Handle,0L,2);
    lseek(ChanPtr->Handle,0L,0);
    ChanPtr->ByteCount = ChanPtr->FileSize;
    SetString(Thread,"Filesize=");
    xyzPutLong(Thread,ChanPtr->FileSize);
   }
 else
   {/* empty file */
    ChanPtr->ByteCount = 0L;
    SetString(Thread,"Empty File");
   }
 /* clear comm port ( there may be several NAKs queued up ) */
 SioRxClear(ChanPtr->Port);
 /* get receivers start up NAK  or 'C' */
 ThreadCallState(TXYSTARTUP);
}


static void TxyModem_1(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 if(!ChanPtr->Result)
   {ThreadReturnState(); return;
   }
 if(DebugFlag)
   {SetString(Thread,"Sending: ");
    if(ChanPtr->EmptyFlag) SetString(Thread,"(empty)");
    else xyzPutString(Thread,ChanPtr->FileName);
   }
 /* loop over all packets */
 ThreadDelay(OneSecond/4);
 ThreadNextState();
}

static void TxyModem_2(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 if(ChanPtr->BatchFlag) ChanPtr->PacketSeq = 0 - 1;
 else ChanPtr->PacketSeq = 1 - 1;
 ChanPtr->EOFseen = FALSE;
 ThreadNextState();
}

static void TxyModem_3(int Thread)
{int i, k;
 int Code;
 ChanPtr = ChannelData[Thread];
 ChanPtr->PacketSeq++;
 /* load up buffer */
 if(ChanPtr->PacketSeq==0)
   {/* zero packet buffer */
    for(i=0;i<128;i++) ChanPtr->Buffer[i] = '\0';
    /* this is a Filename packet */
    ChanPtr->PacketSize = 128;
    k = 0;
    for(i=0;i<strlen(ChanPtr->FileName);i++)
           ChanPtr->Buffer[k++] = ChanPtr->FileName[i];
    ChanPtr->Buffer[k++] = '\0';
    sprintf(&ChanPtr->Buffer[k],"%ld",ChanPtr->FileSize);
   }
 else /* PacketSeq > 0 */
   {/* this is a DATA packet: use 1K or 128 byte block ? */
    if(ChanPtr->OneKflag&&(ChanPtr->NAKcount<2)&&(ChanPtr->ByteCount>=1024)) ChanPtr->PacketSize = 1024;
    else ChanPtr->PacketSize = 128;
    /* read next block from disk */
    Code = read(ChanPtr->Handle,(void *)&ChanPtr->Buffer[0],
                (unsigned)ChanPtr->PacketSize);
    /* error ? */
    if(Code<0)
      {SetString(Thread,"read error: ");
       xyzPutString(Thread, sys_errlist[errno]);
       ChanPtr->Result = FALSE;
       ThreadReturnState(); return;
      }
#if DEBUG
    if(DebugFlag)
      {SetString(Thread,"Read ");
       xyzPutInteger(Thread,Code);
      }
#endif
    /* have we read it all ? */
    if(Code==0)
      {/* all done */
       ChanPtr->EOFseen = TRUE;
       ThreadJumpState(TXYMODEM_5); return;
      }
    /* recalculate bytes left to send */
    ChanPtr->ByteCount -= Code;
    /* pad packet */
    for(i=Code;i<ChanPtr->PacketSize;i++) ChanPtr->Buffer[i] = 0x1a;
   }
 /* send this packet */
 ChanPtr->PacketNbr = (ChanPtr->PacketSeq) % 256;
 ThreadCallState(TXYPACKET);
}

static void TxyModem_4(int Thread)
{ChanPtr = ChannelData[Thread];
 if(!ChanPtr->Result)
   {ThreadReturnState(); return;
   }
 /* success */
 if(ChanPtr->PacketSeq==0)
   {/* done if just sent empty packet 0 */
    if(ChanPtr->EmptyFlag)
       {/* all done. */
        ThreadJumpState(TXYMODEM_5); return;
       }
    /* 'restart' after non-empty packet 0 */
    else
      {ThreadCallState(TXYSTARTUP); return;
      }
   }
 ThreadJumpState(TXYMODEM_3);
}

static void TxyModem_5(int Thread)
{ChanPtr = ChannelData[Thread];
 if(ChanPtr->EmptyFlag)
   {ChanPtr->Result = TRUE;
    ThreadNextState(); return;
   }
 /* all data sent ? */
 if(ChanPtr->EOFseen)
   {/* close the input file */
    close(ChanPtr->Handle);
    SetString(Thread,"Closed ");
    xyzPutString(Thread,ChanPtr->FileName);
    /* send EOT */
    ThreadCallState(TXYEOT);
   }
 else ThreadJumpState(TXYMODEM_3);
}

static void TxyModem_6(int Thread)
{ChanPtr = ChannelData[Thread];
 if(!ChanPtr->Result)
   {SayError(Thread,"EOT not acknowledged");
    ThreadReturnState(); return;
   }
 /*xyzWriteCPS(Tics,ChanPtr->FileSize,ChanPtr->FileName,FALSE);*/
 SetString(Thread,"Transfer Complete");
 ChanPtr->Result = TRUE;
 ThreadReturnState();
} /* end -- TxyModem */

/* start xymodem receive */

static void RxyModem(int Thread)
{int  Code;           /* return code */
 /* begin */
 ChanPtr = ChannelData[Thread];
 ChanPtr->EOTseen = FALSE;
 ChanPtr->ByteCount = 0L;
 SetString(Thread,"XYMODEM Receive: Waiting for Sender ");
 /* clear comm port */
 SioRxClear(ChanPtr->Port);
 /* Send NAKs or 'C's */
 ThreadCallState(RXYSTARTUP);
}

static void RxyModem_1(int Thread)
{ChanPtr = ChannelData[Thread];
 if(!ChanPtr->Result)
   {ThreadReturnState(); return;
   }
 /* open file unless BatchFlag is on */
 if(ChanPtr->BatchFlag) ChanPtr->PacketSeq = 0 - 1;
 else
   {/* start with packet 1 */
    ChanPtr->PacketSeq = 1 - 1;
    if(DebugFlag)
      {SetString(Thread,"Receiving: ");
       xyzPutString(Thread,ChanPtr->FileName);
      }
    /* open file passed in FileName[] for write */
    sprintf(PathName,"UPLOAD.%d\\%s",ChanPtr->Port,ChanPtr->FileName);
    ChanPtr->Handle = open(PathName,O_WRONLY|O_CREAT|O_TRUNC|O_BINARY, S_IWRITE);
    if(ChanPtr->Handle<0)
      {SetString(Thread,"Cannot open ");
       xyzPutString(Thread,PathName);
       TxCAN();
       ChanPtr->Result = FALSE;
       ThreadReturnState(); return;
      }
    SetString(Thread,"Opened/W ");
    xyzPutString(Thread,PathName);
   }
 ThreadNextState();
}

static void RxyModem_2(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 ChanPtr->PacketSeq++;
 ChanPtr->PacketNbr = (ChanPtr->PacketSeq  % 256);
 /* get next packet */
 ThreadCallState(RXYPACKET);
}

static void RxyModem_3(int Thread)
{int i, n, c;
 int Code;
 ChanPtr = ChannelData[Thread];
 if(!ChanPtr->Result)
   {ThreadReturnState(); return;
   }
 /* info packet ? */
 if(ChanPtr->PacketSeq==0)
   {/* copy Filename from packet */
    strcpy(ChanPtr->FileName,ChanPtr->Buffer);
    if(DebugFlag)
      {SetString(Thread,"Receiving: ");
       if(*ChanPtr->FileName) xyzPutString(Thread,ChanPtr->FileName);
       else xyzPutString(Thread," (empty)");
      }
    /* done if null packet 0 */
    if(*ChanPtr->FileName=='\0')
      {SetString(Thread,"Batch Transfer Complete");
       ChanPtr->Result = TRUE;
       ThreadReturnState(); return;
      }
   }
  ChanPtr->ByteCount += (long)ChanPtr->PacketSize;
  /* all done if EOT was received */
  if(ChanPtr->EOTseen)
    {if(ChanPtr->FileSize>0L) ChanPtr->ByteCount = ChanPtr->FileSize;
     /*xyzWriteCPS(Tics,ChanPtr->ByteCount,ChanPtr->FileName,FALSE);*/
     close(ChanPtr->Handle);
     SetString(Thread,"Closed ");
     xyzPutString(Thread,ChanPtr->FileName);
     SetString(Thread,"Transfer Complete");
     ChanPtr->Result = TRUE;
     ThreadJumpState(RXYMODEM); return;
    }
  /* process packet */
  if(ChanPtr->PacketSeq==0)
    {/* copy filename from packet 0 buffer */
     strncpy(ChanPtr->FileName,ChanPtr->Buffer,13);
     /* open file using filename in packet 0 */
     sprintf(PathName,"UPLOAD.%d\\%s",ChanPtr->Port,ChanPtr->FileName);
     ChanPtr->Handle = open(PathName,O_WRONLY|O_CREAT|O_TRUNC|O_BINARY, S_IWRITE);
     if(ChanPtr->Handle<0)
       {SetString(Thread,"open failed");
        ChanPtr->Result = FALSE;
        ThreadReturnState(); return;
       }
     SetString(Thread,"Opened/W ");
     xyzPutString(Thread,PathName);
     /* get file size */
     n = strlen(ChanPtr->FileName);
     ChanPtr->FileSize = 0L;
     for(i=1;;i++)
       {c = (char) ChanPtr->Buffer[n+i];
        if((c==' ')||(c=='\0')) break;
        ChanPtr->FileSize = (10L * ChanPtr->FileSize) + (c - 0x30);
       }
     ChanPtr->ByteCount = ChanPtr->FileSize;
     if(DebugFlag)
       {SetString(Thread,"FileSize : ");
        xyzPutLong(Thread,ChanPtr->FileSize);
       }
     /* must 'restart' after packet 0 */
     ThreadCallState(RXYSTARTUP); return;
    }
  else /* DATA packet */
    {/* write buffer to disk */
     if(ChanPtr->BatchFlag)
       {if(ChanPtr->ByteCount < (long)ChanPtr->PacketSize)
             i = (int) ChanPtr->ByteCount;
        else i = ChanPtr->PacketSize;
        i = write(ChanPtr->Handle,(void *)&ChanPtr->Buffer[0],(unsigned)i);
        if(i>0) ChanPtr->ByteCount -= (long)i;
       }
     else i = write(ChanPtr->Handle,(void *)&ChanPtr->Buffer[0],
                    (unsigned)ChanPtr->PacketSize);
     /* did we write ? */
     if(i==-1)
       {SetString(Thread,"write error: ");
        xyzPutString(Thread, sys_errlist[errno]);
       }
#if DEBUG
     if(DebugFlag)
       {SetString(Thread,"Wrote ");
        xyzPutInteger(Thread,i);
       }
#endif
    }
 ThreadJumpState(RXYMODEM_2);
}

static void RxyModem_4(int Thread)
{
 ThreadJumpState(RXYMODEM_2);
}

/***********/


static void TxModem(int Thread)
{
#if DEBUG
 SetString(Thread,"<S XMODEM>");
#endif
 ThreadJumpState(TXYMODEM);
}

static void RxModem(int Thread)
{
#if DEBUG
 SetString(Thread,"<R XMODEM>");
#endif
 ThreadJumpState(RXYMODEM);
}

static void TyModem(int Thread)
{int Code;
 NameType *NamePtr;
#if DEBUG
 SetString(Thread,"<S YMODEM>");
#endif
 ChanPtr = ChannelData[Thread];
 NamePtr = ChanPtr->FirstName;
 if(NamePtr==NULL)
   {/* send empty filename */
    ChanPtr->FileName[0] = '\0';
    ThreadJumpState(TXYMODEM); return;
   }
 strcpy(ChanPtr->FileName,NamePtr->Name);
 ChanPtr->OneKflag = TRUE;
 ChanPtr->BatchFlag = TRUE;
 ChanPtr->FirstName = NamePtr->Next;
 ThreadCallState(TXYMODEM);
}

static void TyModem_1(int Thread)
{int Code;
 ChanPtr = ChannelData[Thread];
 ThreadJumpState(TYMODEM);
}

static void RyModem(int Thread)
{int Code;
 NameType *NamePtr;
#if DEBUG
 SetString(Thread,"<R YMODEM>");
#endif
 ChanPtr = ChannelData[Thread];
 ThreadJumpState(RXYMODEM);
}

/***[ PUBLIC Functions ]***/

/* XMODEM transmit */

int XmodemTx(int Port,char *FileName,int OneKflag)
{if((Port<COM1)||(Port>COM1+NbrChannels)) return FALSE;
 ChanPtr = ChannelData[Port];
 ChanPtr->OneKflag = OneKflag;
 ChanPtr->BatchFlag = FALSE;
 ChanPtr->NAKcount = 0;
 strcpy(ChanPtr->FileName,FileName);
 return TRUE;
}

/* XMODEM receive */

int XmodemRx(int Port,char *FileName,char NCGchar)
{if((Port<COM1)||(Port>COM1+NbrChannels)) return FALSE;
 ChanPtr = ChannelData[Port];
 ChanPtr->NCGchar = NCGchar;
 ChanPtr->BatchFlag = FALSE;
 ChanPtr->NAKcount = 0;
 strcpy(ChanPtr->FileName,FileName);
 return TRUE;
}

/* build list of names to send */

int BuildNameList(char *FileSpec)
{char FileName[13];
 int Count = 0;
 NameType *NamePtr;
 NameType *LastName;
 if(!FindFirst(FileSpec,FileName,NULL)) return Count;
 NamePtr = (NameType *) malloc( sizeof(struct NameTag) );
 if(NamePtr==NULL)
   {printf("Cannot allocate\n");
    return Count;
   }
 /* populate NamePtr struct */
 NamePtr->Next = NULL;
 ChanPtr->FirstName = NamePtr;
 LastName = NamePtr;
 strcpy(NamePtr->Name,FileName);
 Count++;
 while(1)
   {/* get next name */
    if(!FindNext(FileName,NULL)) return Count;
    NamePtr = (NameType *) malloc( sizeof(struct NameTag) );
    if(NamePtr==NULL)
      {printf("Cannot allocate\n");
       return Count;
      }
    NamePtr->Next = NULL;
    LastName->Next = NamePtr;
    LastName = NamePtr;
    NamePtr->Next = NULL;
    strcpy(NamePtr->Name,FileName);
    Count++;
   }
}

/* YMODEM send */

int YmodemTx(int Port,char *FileSpec)
{if((Port<COM1)||(Port>COM1+NbrChannels)) return FALSE;
 ChanPtr = ChannelData[Port];
 if(BuildNameList(FileSpec)==0) return FALSE;
 ChanPtr->BatchFlag = TRUE;
 ChanPtr->OneKflag = TRUE;
 ChanPtr->NAKcount = 0;
 return TRUE;
}

/* YMODEM receive */

int YmodemRx(int Port,char NCGchar)
{if((Port<COM1)||(Port>COM1+NbrChannels)) return FALSE;
 ChanPtr = ChannelData[Port];
 ChanPtr->NCGchar = NCGchar;
 ChanPtr->BatchFlag = TRUE;
 ChanPtr->NAKcount = 0;
 return TRUE;
}

/* set xy parameters  */

void xyParms(int Channels,   /* number of channels */
             int Debug,      /* debug flag */
             int Factor,     /* wait factor (tics) */
             int Rate)       /* NAK rate (seconds) */
{int i,k;
 char *Ptr;
 DebugFlag = Debug;
 if(Factor)
    {OneSecond = Factor;
     ShortWait = OneSecond * SHORT_WAIT;
     LongWait  = OneSecond * LONG_WAIT;
    }
 if(Rate>0) NAKrate = ONE_SECOND * Rate;
 NbrChannels = Channels;
 for(i=0;i<Channels;i++)
   {Ptr = (char *) malloc( sizeof(struct ChannelTag) );
    if(Ptr==NULL)
       {puts("xyParms: cannot allocate");
        abort();
       }
    ChanPtr = (ChannelType *) Ptr;
    ChannelData[i] = ChanPtr;
    for(k=0;k<sizeof(struct ChannelTag);k++) *Ptr++ = '\0';
    ChanPtr->Port = COM1 + i;
   }

 /* -- register all states --*/

 /* WAITFORINCOMING states */
 tsdRegister(WAITFORINCOMING,WaitForIncoming);
 tsdRegister(WAITFORINCOMING_1,WaitForIncoming_1);

 /* TXYSTARTUP states */
 tsdRegister(TXYSTARTUP,TxyStartup);
 tsdRegister(TXYSTARTUP_1,TxyStartup_1);
 tsdRegister(TXYSTARTUP_2,TxyStartup_2);

 /* RXYSTARTUP states */
 tsdRegister(RXYSTARTUP,RxyStartup);
 tsdRegister(RXYSTARTUP_1,RxyStartup_1);
 tsdRegister(RXYSTARTUP_2,RxyStartup_2);

 /* TXYEOT states */
 tsdRegister(TXYEOT,TxyEOT);
 tsdRegister(TXYEOT_1,TxyEOT_1);
 tsdRegister(TXYEOT_2,TxyEOT_2);

 /* TXYPACKET states */
 tsdRegister(TXYPACKET,TxyPacket);
 tsdRegister(TXYPACKET_1,TxyPacket_1);
 tsdRegister(TXYPACKET_2,TxyPacket_2);
 tsdRegister(TXYPACKET_3,TxyPacket_3);
 tsdRegister(TXYPACKET_4,TxyPacket_4);

 /* RXYPACKET states */
 tsdRegister(RXYPACKET,RxyPacket);
 tsdRegister(RXYPACKET_1,RxyPacket_1);
 tsdRegister(RXYPACKET_2,RxyPacket_2);
 tsdRegister(RXYPACKET_3,RxyPacket_3);
 tsdRegister(RXYPACKET_4,RxyPacket_4);
 tsdRegister(RXYPACKET_5,RxyPacket_5);
 tsdRegister(RXYPACKET_6,RxyPacket_6);

 /* TXYMODEM states */
 tsdRegister(TXYMODEM,TxyModem);
 tsdRegister(TXYMODEM_1,TxyModem_1);
 tsdRegister(TXYMODEM_2,TxyModem_2);
 tsdRegister(TXYMODEM_3,TxyModem_3);
 tsdRegister(TXYMODEM_4,TxyModem_4);
 tsdRegister(TXYMODEM_5,TxyModem_5);
 tsdRegister(TXYMODEM_6,TxyModem_6);

 /* RXYMODEM states */
 tsdRegister(RXYMODEM,RxyModem);
 tsdRegister(RXYMODEM_1,RxyModem_1);
 tsdRegister(RXYMODEM_2,RxyModem_2);
 tsdRegister(RXYMODEM_3,RxyModem_3);
 tsdRegister(RXYMODEM_4,RxyModem_4);

 /* TYMODEM (TY_MODEM) states */
 tsdRegister(TYMODEM,TyModem);
 tsdRegister(TYMODEM_1,TyModem_1);

 /* RYMODEM (RY_MODEM) states */
 tsdRegister(RYMODEM,RyModem);

 /* TXMODEM (TX_MODEM) states */
 tsdRegister(TXMODEM,TxModem);

 /* RXMODEM (RX_MODEM) states */
 tsdRegister(RXMODEM,RxModem);
}

#if DEBUG
void DumpChannel(int Port)
{ChannelType *ChanPtr;
 NameType *NamePtr;
 if((Port<COM1)||(Port>COM1+NbrChannels)) return;
 ChanPtr = ChannelData[Port];
 printf("   FileSize = %ld\n",ChanPtr->FileSize);
 printf("  ByteCount =%ld\n",ChanPtr->ByteCount);
 printf("    Timeout = %ld\n",ChanPtr->Timeout);
 printf("BytesWanted = %d\n",ChanPtr->BytesWanted);
 printf("       Port = %d\n",ChanPtr->Port);
 printf("     Handle = %d\n",ChanPtr->Handle);
 printf("  PacketSeq = %d\n",ChanPtr->PacketSeq);
 printf("  PacketNbr = %d\n",ChanPtr->PacketNbr);
 printf(" PacketSize = %d\n",ChanPtr->PacketSize);
 printf(" PacketType = %d\n",ChanPtr->PacketType);
 printf("RxPacketNbr = %d\n",ChanPtr->RxPacketNbr);
 printf("    Attempt = %d\n",ChanPtr->Attempt);
 printf("   NAKcount = %d\n",ChanPtr->NAKcount);
 printf("       Loop = %d\n",ChanPtr->Loop);
 printf("     Result = %d\n",ChanPtr->Result);
 printf("   CheckSum = %x\n",ChanPtr->CheckSum);
 printf("    NCGchar = %c\n",ChanPtr->NCGchar);
 printf("   OneKflag = %d\n",ChanPtr->OneKflag);
 printf("  BatchFlag = %d\n",ChanPtr->BatchFlag);
 printf("    EOTseen = %d\n",ChanPtr->EOTseen);
 printf("  EmptyFlag = %d\n",ChanPtr->EmptyFlag);
 printf("   FileName = '%s'\n",ChanPtr->FileName);
 printf("  FirstName = %x\n",ChanPtr->FirstName);
 NamePtr = ChanPtr->FirstName;
 while(NamePtr)
   {printf("\t\t'%s'\n",NamePtr);
    NamePtr = NamePtr->Next;
   }
}
#endif
