/*
**  -- win_io.c --
**
**  Version 2.0.
**
**  Implements non-overlapped screen I/O.
*/

#define FALSE 0
#define TRUE !FALSE
#define BYTE unsigned char

#include <stdio.h>
#include <dos.h>
#include <malloc.h>
#include <conio.h>
#include "dos_io.h"
#include "win_io.h"
#include "ascii.h"

#define NBR_WINS 4

#define NORMAL  0x07
#define INVERSE 0x70

static BYTE IsReady = FALSE;

typedef struct WinType
   {BYTE Attribute;
    BYTE Enabled;
    BYTE CurRow;
    BYTE CurCol;
    BYTE TopRow;
    BYTE BotRow;
    BYTE LeftCol;
    BYTE RghtCol;
   } WinType;

static WinType *Window[NBR_WINS];

/*** PRIVATE ***/

static void Initialize(void)
{int i;
 if(!IsReady)
  {IsReady = TRUE;
   for(i=0;i<NBR_WINS;i++)
     {Window[i] = (WinType *) malloc(sizeof(WinType));
#if 0
printf("Window[%d] @ 0x%x\n",i,Window[i]);
#endif
      Window[i]->Enabled = FALSE;
     }
  }
}


static int Check(int Win)
{
 if(!IsReady) Initialize();
 if((Win<0)||(Win>=NBR_WINS)) return WIN_RANGE;
 if(Window[Win]->Enabled) return WIN_AOK;
 else return WIN_DISABLED;
}


static void NewRow(WinType *WinPtr)
{if(++WinPtr->CurRow>WinPtr->BotRow)
   {Scroll(WinPtr->TopRow,WinPtr->LeftCol,WinPtr->BotRow,WinPtr->RghtCol,1,WinPtr->Attribute);
    WinPtr->CurRow = WinPtr->BotRow;
   }
}


static void Advance(WinType *WinPtr)
{if(++WinPtr->CurCol>WinPtr->RghtCol)
  {WinPtr->CurCol = WinPtr->LeftCol;
   NewRow(WinPtr);
  }
 Position(WinPtr->CurRow,WinPtr->CurCol);
}


static int WriteChar(WinType *WinPtr,char C)
{
 switch(C)
   {case LF:
      NewRow(WinPtr);
      /* fall thru */;
    case CR:
      WinPtr->CurCol = WinPtr->LeftCol;
      Position(WinPtr->CurRow,WinPtr->CurCol);
      break;
    case BS:
      if(WinPtr->CurCol > WinPtr->LeftCol)
        {/* back up */
         Position(WinPtr->CurRow,--WinPtr->CurCol);
         AttrWrite((BYTE)' ', (BYTE)WinPtr->Attribute);
        }
      break;
    default:
      if((C<' ')||(C>'~')) C = ' ';
      AttrWrite((BYTE)C, (BYTE)WinPtr->Attribute);
      Advance(WinPtr);
      break;
   }
return WIN_AOK;
}


/*** PUBLIC ***/


/* set attribute to normal */

int WinNormal(int Win)
{int rc;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Window[Win]->Attribute = NORMAL;
 return WIN_AOK;
}


/* set attribute to inverse */

int WinInverse(int Win)
{int rc;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Window[Win]->Attribute = INVERSE;
 return WIN_AOK;
}


/* create window */


int WinCreate(int Win,int r1,int c1,int r2,int c2)
{WinType *WinPtr;
 if(!IsReady) Initialize();
 if((Win<0)||(Win>=NBR_WINS)) return WIN_RANGE;
 WinPtr = Window[Win];
 if(WinPtr->Enabled) return WIN_ENABLED;
 WinPtr->Enabled = TRUE;
 WinPtr->Attribute = NORMAL;
 if(r1>r2) return FALSE;
 if(r2>24) return FALSE;
 if(c1>c2) return FALSE;
 if(c2>79) return FALSE;
 WinPtr->TopRow = r1;
 WinPtr->BotRow = r2;
 WinPtr->LeftCol = c1;
 WinPtr->RghtCol = c2;
 WinPtr->CurRow = r1;
 WinPtr->CurCol = c1;
 Position((BYTE)r1,(BYTE)c1);
 return WIN_AOK;
}

/* write char to window */

int WinPutChar(int Win,char C)
{int rc;
 WinType *WinPtr;
 WinPtr = Window[Win];
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Position(WinPtr->CurRow,WinPtr->CurCol);
 return WriteChar(WinPtr,C);
}

/* write string to window */

int WinPutString(int Win,char *String)
{int rc;
 WinType *WinPtr;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 WinPtr = Window[Win];
 Position(WinPtr->CurRow,WinPtr->CurCol);
 while(*String!='\0') WriteChar(WinPtr,*String++);
 return WIN_AOK;
}

/* clear window */

int WinClear(int Win)
{int rc;
 WinType *WinPtr;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 WinPtr = Window[Win];
 Scroll(WinPtr->TopRow,WinPtr->LeftCol,WinPtr->BotRow,WinPtr->RghtCol,0,WinPtr->Attribute);
 WinPtr->CurRow = WinPtr->TopRow;
 WinPtr->CurCol = WinPtr->LeftCol;
 return WIN_AOK;
}

 /* input string from user */

int WinGetString(int Win, char *String, int Length)
{int  i, rc;
 char c;
 WinType *WinPtr;
 if((rc=Check(Win))!=WIN_AOK) return rc;
 WinPtr = Window[Win];
 Position(WinPtr->CurRow,WinPtr->CurCol);
 /* input text from user */
 i = 0;
 while(1)
     {c = 0x00ff & getch();
      switch(c)
        {case LF:
           /* ignore LFs */
           break;
         case CR:
           /* done */
           String[i] = '\0';
           return WIN_AOK;
         case ESC:
         case CAN:
           /* aborting */
           *String = '\0';
           return WIN_AOK;
         case BS:
           if(i>0)
             {/* back up */
              i--;
              Position(WinPtr->CurRow,--WinPtr->CurCol);
              AttrWrite((BYTE)' ', (BYTE)WinPtr->Attribute);
             }
           break;
         default:
           if((c>=' ')&&(c<='~'))
              {/* save character */
               String[i++] = c;
               /* display char & advance cursor */
               AttrWrite((BYTE)c, (BYTE)WinPtr->Attribute);
               Advance(WinPtr);
               if(i==Length)
                  {/* user string done */
                   String[i] = '\0';
                   return WIN_AOK;
                  }
              }
           break;
        } /* end case */
     } /* end while */
}

/* set cursor position in window */

int WinSetPos(int Win,int Row, int Col)
{int rc;
 WinType *WinPtr;
 WinPtr = Window[Win];
 if((rc=Check(Win))!=WIN_AOK) return rc;
 if(WinPtr->TopRow  + Row > WinPtr->BotRow)  return WIN_RANGE;
 if(WinPtr->LeftCol + Col > WinPtr->RghtCol) return WIN_RANGE;
 WinPtr->CurRow = WinPtr->TopRow+Row;
 WinPtr->CurCol = WinPtr->LeftCol+Col;
 Position(WinPtr->CurRow,WinPtr->CurCol);
 return WIN_AOK;
}

/* get window cursor position */

int WinGetPos(int Win,int *RowP,int *ColP)
{int rc;
 WinType *WinPtr;
 WinPtr = Window[Win];
 if((rc=Check(Win))!=WIN_AOK) return rc;
 Position(WinPtr->CurRow,WinPtr->CurCol);
 if(RowP) *RowP = WinPtr->CurRow - WinPtr->TopRow;
 if(ColP) *ColP = WinPtr->CurCol - WinPtr->LeftCol;
 return WIN_AOK;
}