/* session.c */

#define DEBUG 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <ctype.h>
#include <malloc.h>
#include "ascii.h"
#include "pcl4c.h"
#include "session.h"
#include "logger.h"
#include "tsd.h"
#include "sio.h"
#include "win_io.h"
#include "bbs.h"
#include "buffer.h"
#if USE_XYMODEM
#include "xymodem.h"
#include "find.h"
#endif


#define FALSE    0
#define TRUE     !FALSE
#define ONE_SEC  19

#define NAME_SIZE    8
#define STRING_SIZE 15
#define FNAME_SIZE  13

/*** state function indices ***/

#define WAIT_FOR     0x0200
#define WAIT_FOR_1   WAIT_FOR+1

#define WAIT_QUIET   0x0210
#define WAIT_QUIET_1 WAIT_QUIET+1

#define SEND_TO      0x0220
#define SEND_TO_1    SEND_TO+1

#define GET_STRING   0x0230
#define GET_STRING_1 GET_STRING+1

#define HANGUP       0x0240
#define HANGUP_1     HANGUP+1
#define HANGUP_2     HANGUP+2
#define HANGUP_3     HANGUP+3

#define PROTOCOL     0x0250
#define PROTOCOL_1   PROTOCOL+1
#define PROTOCOL_2   PROTOCOL+2

#define SEND_X       0x0260
#define SEND_X_1     SEND_X+1
#define SEND_X_2     SEND_X+2

#define RECEIVE_X    0x0270
#define RECEIVE_X_1  RECEIVE_X+1
#define RECEIVE_X_2  RECEIVE_X+2

#define SEND_Y       0x0280
#define SEND_Y_1     SEND_Y+1
#define SEND_Y_2     SEND_Y+2

#define RECEIVE_Y    0x0290
#define RECEIVE_Y_1  RECEIVE_Y+1
#define RECEIVE_Y_2  RECEIVE_Y+2

#define X2A          0x02A0
#define X2B          0x02B0
#define X2C          0x02C0

#define RUN          0x02D0
#define RUN_1        RUN+1
#define RUN_2        RUN+2
#define RUN_3        RUN+3
#define RUN_4        RUN+4
#define RUN_5        RUN+5
#define RUN_6        RUN+6
#define RUN_7        RUN+7
#define RUN_8        RUN+8
#define RUN_9        RUN+9
#define RUN_10       RUN+10
#define RUN_11       RUN+11
#define RUN_12       RUN+12

/*** channel structure ***/

typedef struct ChannelTag
{long TimeMark;                /* time mark */
 int  Index;                   /* next position in String to send or match */
 int  Result;                  /* result of WaitFor or SendTo */
 int  Pace;                    /* SendTo pace, or Quiet Tics */
 int  Port;                    /* Port # (COM1==0) */
 char String[STRING_SIZE+1];   /* string to send or match */
 char User[NAME_SIZE+1];       /* user name */
 char EchoDot;                 /* T: echo '.' */
 char LastChar;                /* last incoming char [DEBUG] */
 char Protocol;                /* protocol chosen ('X','Y', or 'Z') */
 char Filename[FNAME_SIZE];    /* filename */
} ChannelType;

/*>>> channel i == thread i <<<*/

/*** globals ***/

static int NbrChannels = 0;
static ChannelType *ChanList[NBR_PORTS];
static ChannelType *ChanPtr = NULL;
static char NoUser[NAME_SIZE+1] = "No user  ";
static char Message[40];
static int CurrentChannel = 0;
static char PathName[FNAME_SIZE+15];

/*** state function prototypes ***/

void Run(int);
void Run_1(int);
void Run_2(int);
void Run_3(int);
void Run_4(int);
void Run_5(int);
void Run_6(int);
void Run_7(int);
void Run_8(int);
void Run_9(int);
void Run_10(int);
void Run_11(int);
void Run_12(int);
void WaitFor(int);
void WaitFor_1(int);
void WaitQuiet(int);
void WaitQuiet_1(int);
void SendTo(int);
void SendTo_1(int);
void GetString(int);
void GetString_1(int);
void Hangup(int);
void Hangup_1(int);
void Hangup_2(int);
void Hangup_3(int);
void Protocol(int);
void Protocol_1(int);
void Protocol_2(int);
void SendX(int);
void SendX_1(int);
void SendX_2(int);
void ReceiveX(int);
void ReceiveX_1(int);
void ReceiveX_2(int);
void SendY(int);
void SendY_1(int);
void SendY_2(int);
void ReceiveY(int);
void ReceiveY_1(int);
void ReceiveY_2(int);
int  FileExits(int,char *);


/*** PUBLIC functions ***/

int ssnInit(int NbrChans)
{int i, j;
 ChannelType *Ptr;
 /* register all states */
 tsdRegister(SEND_TO,     SendTo);
 tsdRegister(SEND_TO_1,   SendTo_1);
 tsdRegister(WAIT_FOR,    WaitFor);
 tsdRegister(WAIT_FOR_1,  WaitFor_1);
 tsdRegister(WAIT_QUIET,  WaitQuiet);
 tsdRegister(WAIT_QUIET_1,WaitQuiet_1);
 tsdRegister(RUN,         Run);
 tsdRegister(RUN_1,       Run_1);
 tsdRegister(RUN_2,       Run_2);
 tsdRegister(RUN_3,       Run_3);
 tsdRegister(RUN_4,       Run_4);
 tsdRegister(RUN_5,       Run_5);
 tsdRegister(RUN_6,       Run_6);
 tsdRegister(RUN_7,       Run_7);
 tsdRegister(RUN_8,       Run_8);
 tsdRegister(RUN_9,       Run_9);
 tsdRegister(RUN_10,      Run_10);
 tsdRegister(RUN_11,      Run_11);
 tsdRegister(RUN_12,      Run_12);
 tsdRegister(GET_STRING,  GetString);
 tsdRegister(GET_STRING_1,GetString_1);
 tsdRegister(HANGUP,      Hangup);
 tsdRegister(HANGUP_1,    Hangup_1);
 tsdRegister(HANGUP_2,    Hangup_2);
 tsdRegister(HANGUP_3,    Hangup_3);
 tsdRegister(PROTOCOL,    Protocol);
 tsdRegister(PROTOCOL_1,  Protocol_1);
 tsdRegister(PROTOCOL_2,  Protocol_2);
 tsdRegister(SEND_X,      SendX);
 tsdRegister(SEND_X_1,    SendX_1);
 tsdRegister(SEND_X_2,    SendX_2);
 tsdRegister(RECEIVE_X,   ReceiveX);
 tsdRegister(RECEIVE_X_1, ReceiveX_1);
 tsdRegister(RECEIVE_X_2, ReceiveX_2);
 tsdRegister(SEND_Y,      SendY);
 tsdRegister(SEND_Y_1,    SendY_1);
 tsdRegister(SEND_Y_2,    SendY_2);
 tsdRegister(RECEIVE_Y,   ReceiveY);
 tsdRegister(RECEIVE_Y_1, ReceiveY_1);
 tsdRegister(RECEIVE_Y_2, ReceiveY_2);
 /* construct channel structure */
 NbrChannels = NbrChans;
 for(i=0;i<NbrChans;i++)
   {Ptr = (ChannelType *) malloc(sizeof(struct ChannelTag));
    if(Ptr==NULL)
      {WinPutString(CONTROL_WIN,"ssnInit: Cannot allocate\n");
       return -1;
      }
    ChanList[i] = Ptr;
    strcpy(&Ptr->User[0],NoUser);
    Ptr->Protocol = 'X';
   }
 return 0;
}

int ssnStart(int Channel,int Port)
{int i;
 if((Port<0)||(Port>=COM20)) return -1;
 ChanList[Channel]->Port = Port;
 tsdNewThread(RUN,Channel);
 sprintf(Message,"Channel %d [port COM%d] ready\n",Channel,1+Port);
 WinPutString(CONTROL_WIN,Message);
 return 0;
}

char *ssnGetUser(int Channel)
{return &ChanList[Channel]->User[0];
}

char ssnGetProtocol(int Channel)
{return ChanList[Channel]->Protocol;
}

void ssnSetChannel(int Channel)
{/* set current port channel */
 CurrentChannel = Channel;
}

char *ssnGetString(int Channel)
{
 return &ChanList[Channel]->String[0];
}

int ssnGetIndex(int Channel)
{/* debug info [Index] */
 return ChanList[Channel]->Index;
}

int ssnDebug(int Channel)
{/* programmer defined debug */
 return (int)ChanList[Channel]->LastChar;
}

/* display char in CALLER window */

void ssnPostChar(int Channel,char Chr)
{/* save in (re-display) buffer */
 if(Chr!=BS) BufferAppend(Channel,Chr);
 /* write to log file */
 LogPutChar(Channel,Chr);
 /* write to CALLER area if currently selected */
 if(Channel==CurrentChannel)
    {WinPutChar(CALLER_WIN,Chr);
     /* restore cursor to CONTROL window */
     WinGetPos(CONTROL_WIN,NULL,NULL);
    }
}

/* display string in caller window */

void ssnPostString(int Channel,char *Text)
{int i;
 int Len;
 /* post string in caller area */
 Len = strlen(Text);
 for(i=0;i<Len;i++) ssnPostChar(Channel,*Text++);
}

/*** private ***/

/* display string in CONTROL window */

static void DisplayMessage(char *Ptr)
{WinPutString(CALLER_WIN,Ptr);
 WinGetPos(CONTROL_WIN,NULL,NULL);
}

/* get incoming character */

static int GetChar(int Channel)
{int C;
 C = SioGetc(ChanList[Channel]->Port,0);
#if DEBUG
 if(C>=0) ChanList[Channel]->LastChar = (char)C;
#endif
 return C;
}

/* send character over serial port */

static void PutChar(int Channel,char Chr)
{ChanPtr = ChanList[Channel];
 /* send over serial port */
 SioPutc(ChanPtr->Port,Chr);
 if(Chr==CR)
   {/* append line feed to CR */
    Chr = LF;
    SioPutc(ChanPtr->Port,Chr);
   }
 /* post the char */
 ssnPostChar(Channel,Chr);
}

/* send string over serial port */

static void PutString(int Channel,char *Ptr)
{int i;
 int Len;
 Len = strlen(Ptr);
 for(i=0;i<Len;i++) PutChar(Channel,*Ptr++);
}

/*** state functions ***/

void Run(int Channel)
{int i;
 ChanPtr = ChanList[Channel];
 ssnPostString(Channel,"(Awaiting logon)\n");
 sioStatus(ChanPtr->Port,'W');
 WinGetPos(CONTROL_WIN,NULL,NULL);
 /* wait for incoming "RING" */
 SioRxClear(ChanPtr->Port);
 SioPutc(ChanPtr->Port,CR);
 strcpy(&ChanPtr->String[0],"RING");
 ChanPtr->TimeMark = 0;  /* wait forever */
 ChanPtr->Index = 0;
 ThreadCallState(WAIT_FOR);
}

void Run_1(int Channel)
{ChanPtr = ChanList[Channel];
 /* got "RING" ? */
 if(!ChanPtr->Result)
   {/* error !? */

    ThreadReturnState(); return;
   }
 /* answer with ATA */
 strcpy(&ChanPtr->String[0],"!ATA!");
 ChanPtr->Pace = ONE_SEC/5;
 ThreadCallState(SEND_TO);
}

void Run_2(int Channel)
{ChanPtr = ChanList[Channel];
 /* wait up to 60 seconds for incoming "CONNECT" */
 strcpy(&ChanPtr->String[0],"CONNECT");
 ChanPtr->TimeMark = SioTimer() + 60*ONE_SEC;
 ThreadCallState(WAIT_FOR);
}

void Run_3(int Channel)
{ChanPtr = ChanList[Channel];
 /* got "CONNECT" ? */
 if(!ChanPtr->Result)
   {/* didn't get connect */
    WinPutString(CONTROL_WIN,"CONNECT failed\n");
    ThreadJumpState(RUN); return;
   }
 /* send CR & then wait for line to clear */
 PutChar(Channel,CR);
 ChanPtr->Pace = 3*ONE_SEC;
 ThreadCallState(WAIT_QUIET);
}

void Run_4(int Channel)
{ChanPtr = ChanList[Channel];
 sioStatus(ChanPtr->Port,'C');
 /* issue log on message to caller */
 PutString(Channel,"Login:");
 /* get user name */
 ThreadCallState(GET_STRING);
}

void Run_5(int Channel)
{ChanPtr = ChanList[Channel];
 /* save string in USER area */
 strncpy(&ChanPtr->User[0],&ChanPtr->String[0],NAME_SIZE);
 ChanPtr->User[ChanPtr->Index] = '\0';
 /* display if channel selected */
 if(Channel==CurrentChannel)
    sioSelect(Channel,&ChanPtr->User[0],ChanPtr->Protocol);
 /* ask for password (we don't do anything with it) */
 PutString(Channel,"Password:");
 ChanPtr->EchoDot = TRUE;
 ThreadCallState(GET_STRING);
}

void Run_6(int Channel)
{ChanPtr = ChanList[Channel];
 ChanPtr->EchoDot = FALSE;
 /* got password in String (we won't use it) */
 ThreadNextState();
}

void Run_7(int Channel)
{ChanPtr = ChanList[Channel];
 /* display menu bar for caller */
 PutString(Channel,"Q)uit P)rotocol S)end R)eceive:");
 /* get one char answer from caller */
 ThreadNextState();
}

void Run_8(int Channel)
{ChanPtr = ChanList[Channel];
 /* await single incoming char */
 if(SioRxQue(ChanPtr->Port)==0) return;
 ThreadNextState();
}

void Run_9(int Channel)
{char Chr;
 ChanPtr = ChanList[Channel];
 Chr = (char) GetChar(Channel);
 PutChar(Channel,Chr);
 PutChar(Channel,CR);
 switch(toupper(Chr))
   {case 'Q':  /* QUIT */
      strcpy(&ChanPtr->User[0],NoUser);
      ChanPtr->Protocol = 'X';
      ThreadJumpState(RUN_11); return;
    case 'P':  /* PROTOCOL */
      ThreadCallState(PROTOCOL); return;
    case 'S':  /* SEND */
      sioStatus(ChanPtr->Port,'S');
      switch(ChanPtr->Protocol)
        {case 'X':
           ThreadCallState(SEND_X); return;
         case 'Y':
           ThreadCallState(SEND_Y); return;
         case 'Z':
           /* not implemented */
           break;
        }
      break;
    case 'R':  /* RECEIVE */
      sioStatus(ChanPtr->Port,'R');
      switch(ChanPtr->Protocol)
        {case 'X':
           ThreadCallState(RECEIVE_X); return;
         case 'Y':
           ThreadCallState(RECEIVE_Y); return;
         case 'Z':
           /* not implemented */
           break;
        }
      break;
   }
 /* loop back to display menu bar again */
 sioStatus(ChanPtr->Port,'C');
 ThreadJumpState(RUN_7);
}

void Run_10(int Channel)
{ChanPtr = ChanList[Channel];
 sioStatus(ChanPtr->Port,'C');
 ssnPostChar(Channel,'\n');
 ThreadJumpState(RUN_7);
}

void Run_11(int Channel)
{ChanPtr = ChanList[Channel];
 sioStatus(ChanPtr->Port,'W');
 /* hang up modem */
 ThreadCallState(HANGUP);
}

void Run_12(int Channel)
{/* await next RING */
 ssnPostString(Channel,"(User logged off)\n");
 ThreadJumpState(RUN);
}

/*
**  match incoming string
**
**  Entry: TimeMark and String
**   Exit: Result
*/

void WaitFor(int Channel)
{ChanPtr = ChanList[Channel];
 ChanPtr->Index = 0;
 ChanPtr->Result = FALSE;
 ThreadNextState();
}

void WaitFor_1(int Channel)
{int  Code;
 char Chr;
 long TimeMark;
 ChanPtr = ChanList[Channel];
 /* look for incoming */
 while(1)
   {TimeMark = ChanPtr->TimeMark;
    if(TimeMark)
      {if(SioTimer()>=TimeMark)
        {/* timed out ! */
         ChanPtr->Result = FALSE;
         ThreadReturnState(); return;
        }
      }
    /* any incoming ? */
    Code = GetChar(Channel);
    if(Code<0) return;
    Chr = (char)Code;
    ssnPostChar(Channel,Chr);
    /* does char match ? */
    if(Chr==ChanPtr->String[ChanPtr->Index])
      {/* char matches */
       ChanPtr->Index++;
       if(ChanPtr->String[ChanPtr->Index]=='\0')
         {/* entire string matches ! */
          ssnPostChar(Channel,'\n');
          ChanPtr->Result = TRUE;
          ThreadReturnState(); return;
         }
      }
    else
      {/* char does NOT match */
       if(ChanPtr->Index > 0) SioUnGetc(ChanPtr->Port,Chr);
       ChanPtr->Index = 0;
       return;
      }
   }
}

/*
**  send string to modem (with inter-char delay)
**
**  Entry: String and Pace
**   Exit: Result
*/

void SendTo(int Channel)
{ChanPtr = ChanList[Channel];
 ChanPtr->Index = 0;
 ThreadNextState();
}

void SendTo_1(int Channel)
{char Chr;
 ChanPtr = ChanList[Channel];
 Chr = ChanPtr->String[ChanPtr->Index++];
 if(Chr=='\0')
   {/* entire string sent */
    ChanPtr->Result = TRUE;
    ThreadReturnState(); return;
   }
 /* replace '!' with carriage return */
 if(Chr=='!') Chr = CR;
 /* send char & delay  ??? */
 SioPutc(ChanPtr->Port,Chr);
 ThreadDelay(ChanPtr->Pace);
}

/*
**  Wait for continuous quiet
**
**  Entry: Pace (quiet time)
*/

void WaitQuiet(int Channel)
{ChanPtr = ChanList[Channel];
 SioRxClear(ChanPtr->Port);
 ChanPtr->TimeMark = SioTimer() + (long)ChanPtr->Pace;
 ThreadNextState();
}

void WaitQuiet_1(int Channel)
{ChanPtr = ChanList[Channel];
 if(SioRxQue(ChanPtr->Port) > 0)
   {/* got incoming ! */
    ThreadJumpState(WAIT_QUIET); return;
   }
 if(SioTimer()>=ChanPtr->TimeMark)
   {/* got quiet period ! */
    ThreadReturnState(); return;
   }
}

/*
**  Get string from user (via serial port)
**
**  Exit: String = users's response
**        Index  = # chars in String
*/

void GetString(int Channel)
{int Code;
 char Chr;
 ChanPtr = ChanList[Channel];
 ChanPtr->Index = 0;
 ThreadNextState();
}

void GetString_1(int Channel)
{int i;
 int Code;
 char Chr;
 ChanPtr = ChanList[Channel];
 Code = GetChar(Channel);
 if(Code==-1) return;
 Chr = toupper((char)Code);
 /* back space ? */
 if(Chr==BS)
   {if(ChanPtr->Index>0)
     {ChanPtr->Index--;
      BufferBack(Channel);
      PutChar(Channel,BS);
     }
    return;
   }
 /* is a CR ? */
 if(Chr==CR)
   {/* user has typed CR */
    PutChar(Channel,Chr);
    ThreadReturnState(); return;
   }
 else
   {/* save in String */
    if(ChanPtr->Index<STRING_SIZE) ChanPtr->String[ChanPtr->Index++] = Chr;
    /* echo dot ? */
    if(ChanPtr->EchoDot) Chr = '.';
    /* echo back to sender */
    PutChar(Channel,Chr);
   }
}

/*
**  Hangup up modem
*/

void Hangup(int Channel)
{ThreadDelay(ONE_SEC+1);
 ThreadNextState();
}

void Hangup_1(int Channel)
{ChanPtr = ChanList[Channel];
 strcpy(&ChanPtr->String[0],"+++");
 ChanPtr->Pace = 5;
 ThreadCallState(SEND_TO);
}

void Hangup_2(int Channel)
{ThreadDelay(ONE_SEC+1);
 ThreadNextState();
}

void Hangup_3(int Channel)
{ChanPtr = ChanList[Channel];
 strcpy(&ChanPtr->String[0],"!ATH0!");
 ThreadDelay(ONE_SEC/4);
 ThreadReturnState();
}

/*
**  Get protocol
*/

void Protocol(int Channel)
{ChanPtr = ChanList[Channel];
 /* issue log on message to caller */
 PutString(Channel,"Protocol X)modem Y)modem Z)modem:");
 ThreadNextState();
}

void Protocol_1(int Channel)
{ChanPtr = ChanList[Channel];
 /* await single incoming char */
 if(SioRxQue(ChanPtr->Port)==0) return;
 ThreadNextState();
}

void Protocol_2(int Channel)
{char Chr;
 ChanPtr = ChanList[Channel];
 Chr = (char) GetChar(Channel);
 PutChar(Channel,Chr);
 PutChar(Channel,CR);
 switch(toupper(Chr))
   {case 'X': ChanPtr->Protocol = 'X';
      PutString(Channel,"XMODEM selected\n");
      break;
    case 'Y': ChanPtr->Protocol = 'Y';
      PutString(Channel,"YMODEM selected\n");
      break;
    case 'Z': ChanPtr->Protocol = 'Z';
      PutString(Channel,"ZMODEM selected\n");
      break;
    default:
      PutString(Channel,"No such protocol\n");
      break;
   }
 /* display if channel selected */
 if(Channel==CurrentChannel)
    sioSelect(Channel,&ChanPtr->User[0],ChanPtr->Protocol);
 ThreadReturnState();
}

int FileExists(int Channel,char *Path)
{/* does file exists ? */
 if(!FindFirst(Path,NULL,NULL))
   {PutString(Channel,"Cannot open ");
    PutString(Channel,Path);
    PutString(Channel,"\n");
    return FALSE;
   }
 return TRUE;
}

/*
**  XMODEM Send
*/

void SendX(int Channel)
{ChanPtr = ChanList[Channel];
 /* ask for file name */
 PutString(Channel,"Enter filename:");
 /* get name */
 ThreadCallState(GET_STRING);
}

void SendX_1(int Channel)
{ChanPtr = ChanList[Channel];
 /* retrieve filename */
 if(*ChanPtr->String!='\0')
    {strncpy(&ChanPtr->Filename[0],&ChanPtr->String[0],FNAME_SIZE);
     strcpy(PathName,"DNLOAD.ALL\\");
     strcat(PathName,&ChanPtr->Filename[0]);
     /* does file exists ? */
     if(!FileExists(Channel,PathName))
        {ChanPtr->Result = FALSE;
         ThreadReturnState(); return;
        }
     PutString(Channel,"Start your XMODEM receive...");
     XmodemTx(ChanPtr->Port,ChanPtr->Filename,FALSE);
     ThreadCallState(TX_MODEM);
    }
 else
    {ChanPtr->Result = FALSE;
     ThreadReturnState();
    }
}

void SendX_2(int Channel)
{ChanPtr = ChanList[Channel];
 ChanPtr->Result = TRUE;
 ThreadReturnState();
}

/*
**  XMODEM Receive
*/

void ReceiveX(int Channel)
{ChanPtr = ChanList[Channel];
 /* ask for file name */
 PutString(Channel,"Enter filename:");
 /* get filename */
 ThreadCallState(GET_STRING);
}

void ReceiveX_1(int Channel)
{ChanPtr = ChanList[Channel];
 /* retrieve filename */
 if(*ChanPtr->String!='\0')
    {strncpy(&ChanPtr->Filename[0],&ChanPtr->String[0],FNAME_SIZE);
     XmodemRx(ChanPtr->Port,ChanPtr->Filename,NAK);
     PutString(Channel,"Start your XMODEM send...");
     ThreadCallState(RX_MODEM);
    }
 else
    {ChanPtr->Result = FALSE;
     ThreadReturnState();
    }
}

void ReceiveX_2(int Channel)
{ChanPtr = ChanList[Channel];
 ChanPtr->Result = TRUE;
 ThreadReturnState();
}

/*
**  YMODEM Send
*/

void SendY(int Channel)
{ChanPtr = ChanList[Channel];
 /* ask for file spec */
 PutString(Channel,"Enter filename:");
 /* get spec */
 ThreadCallState(GET_STRING);
}

void SendY_1(int Channel)
{ChanPtr = ChanList[Channel];
 /* retrieve file spec */
 if(*ChanPtr->String!='\0')
    {strncpy(&ChanPtr->Filename[0],&ChanPtr->String[0],FNAME_SIZE);
     strcpy(PathName,"DNLOAD.ALL\\");
     strcat(PathName,&ChanPtr->Filename[0]);
     /* does file exists ? */
     if(!FileExists(Channel,PathName))
        {ChanPtr->Result = FALSE;
         ThreadReturnState(); return;
        }
     PutString(Channel,"Start your YMODEM receive...");
     YmodemTx(ChanPtr->Port,ChanPtr->Filename);
     ThreadCallState(TY_MODEM);
    }
 else
    {ChanPtr->Result = FALSE;
     ThreadReturnState();
    }
}

void SendY_2(int Channel)
{ChanPtr = ChanList[Channel];
 ChanPtr->Result = TRUE;
 ThreadReturnState();
}

/*
**  YMODEM Receive
*/

void ReceiveY(int Channel)
{ChanPtr = ChanList[Channel];
 ChanPtr->Filename[0] = '\0';
 ThreadNextState();
}

void ReceiveY_1(int Channel)
{ChanPtr = ChanList[Channel];
 YmodemRx(ChanPtr->Port,'C');
 PutString(Channel,"Start your YMODEM send...");
 ThreadCallState(RY_MODEM);
}

void ReceiveY_2(int Channel)
{ChanPtr = ChanList[Channel];
 ChanPtr->Result = TRUE;
 ThreadReturnState();
}
